/*******************************************************************************
 *
 *  Copyright (c) 1999 by Thierry Lelegard
 *
 *  This software is covered by the "GNU GENERAL PUBLIC LICENSE" (GPL),
 *  version 2, June 1991. See the file named COPYING for details.
 *
 *  Project: VMSCD - OpenVMS CD-ROM Utility for Linux
 *  File:    volume.c
 *  Author:  Thierry Lelegard
 *
 *
 *  Abstract
 *  --------
 *  This module implements the ODS-2 volume processing. This includes
 *  initializing the file system structure, locating and opening files.
 *  See the file volume.h for details.
 *
 *
 *  Credits
 *  -------
 *  This work is based on the excellent book "VMS File System Internals"
 *  by Kirby McCoy, Digital Press, 1990. Thanks very much to the author.
 *
 *
 *  Modification History
 *  --------------------
 *  18 Dec 1999 - Thierry Lelegard (lelegard@club-internet.fr)
 *                Creation of the file.
 *
 *
 *******************************************************************************
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include "ods2.h"
#include "utils.h"
#include "cache.h"
#include "volume.h"


/*******************************************************************************
 *
 *  This routine computes a 16 bits checksum as used in ODS-2 structures.
 *
 *******************************************************************************
 */

static int checksum16 (const void *data, int size)
{
    int word_count = size / 2;
    const uint16 *current = data;
    int result = 0;

    while (word_count-- > 0)
        result = (result + *current++) & 0x00FFFF;

    return result;
}


/*******************************************************************************
 *
 *  This routine converts an FID from ODS-2format to a civilized format.
 *
 *******************************************************************************
 */

static void get_fid (fid_t *fid, const ods2_fid_t *ods2_fid)
{
    fid->number   = ods2_fid->fid_w_num + (ods2_fid->fid_b_nmx << 16);
    fid->sequence = ods2_fid->fid_w_seq;
    fid->volume   = ods2_fid->fid_b_rvn;
}


/*******************************************************************************
 *
 *  This routine opens a cdrom.
 *
 *******************************************************************************
 */

volume_t *open_volume (int err_flags, cache_t cache)
{
    int ok, checksum, len;
    volume_t *vol;
    static const fid_t indexf_fid = {FID_C_INDEXF, FID_C_INDEXF, 0};

    /* Issue some debugging messages if trace is on */

    trace ("sizeof (ods2_fid_t)         = %d\n", sizeof (ods2_fid_t));
    trace ("sizeof (ods2_home_block_t)  = %d\n", sizeof (ods2_home_block_t));
    trace ("sizeof (ods2_file_attr_t)   = %d\n", sizeof (ods2_file_attr_t));
    trace ("sizeof (ods2_file_header_t) = %d\n", sizeof (ods2_file_header_t));
    trace ("sizeof (ods2_dir_t)         = %d\n", sizeof (ods2_dir_t));
    trace ("sizeof (ods2_dirent_t)      = %d\n", sizeof (ods2_dirent_t));

    /* Initialize the volue structure */

    vol = utmalloc (sizeof (volume_t));
    memset (vol, 0, sizeof (volume_t));
    vol->cache = cache;

    /* Read home block of volume. Use first home block only, ignore */
    /* alternate home blocks. The first home block is at LBN 1. */

    ok = read_lbn (err_flags, cache, 1, 1, &vol->home, LBN_META) == 0;

    /* Validate home block cheksum */

    if (ok) {
        checksum = checksum16 (&vol->home, sizeof (vol->home) - 2);
        if (checksum != vol->home.hm2_w_checksum2) {
            ok = 0;
            error (err_flags, "invalid home block, not an ODS-2 volume\n");
            error (err_flags, "stored checksum is 0x%04X, expected 0x%04X\n",
                vol->home.hm2_w_checksum2, checksum);
        }
    }

    /* Get a copy of the volume name. Stop at last blank and ensure it is */
    /* nul-terminated. */

    for (len = sizeof (vol->home.hm2_t_volname);
         len > 0 && vol->home.hm2_t_volname [len - 1] == ' ';
         len--);

    vol->volname = utmalloc (len + 1);
    memcpy (vol->volname, vol->home.hm2_t_volname, len);
    vol->volname [len] = 0;

    /* Then, open INDEXF.SYS. See special implementation note in open_file(). */

    if (ok)
        ok = (vol->indexf = open_file (err_flags, vol, &indexf_fid)) != NULL;

    if (ok)
        return vol;
    else {
        free (vol->volname);
        free (vol);
        return NULL;
    }
}


/*******************************************************************************
 *
 *  This routine closes the cdrom.
 *
 *******************************************************************************
 */

void close_volume (volume_t *vol)
{
    close_file (vol->indexf);
    free (vol->volname);
    free (vol);
}


/*******************************************************************************
 *
 *  This routine decodes a list of retrieval pointers from a file header.
 *  The return value is the number of retrieval pointers.
 *
 *******************************************************************************
 */

static int get_retrieval (
    const ods2_file_header_t *header,
    vbn_t first_vbn,
    retrieval_t *rp_buffer,
    int rp_max_count)
{
    int rp_index;
    const char *fmap, *fmap_end;
    const ods2_fmap1_t *fmap1;
    const ods2_fmap2_t *fmap2;
    const ods2_fmap3_t *fmap3;

    /* Locate the map area inside the file header */

    fmap = (const char*) header + 2 * header->fh2_b_mpoffset;
    fmap_end = (const char*) header + 2 * header->fh2_b_acoffset;

    trace ("volume.get_retrieval: map area 0x%04X to 0x%04X\n",
        2 * header->fh2_b_mpoffset, 2 * header->fh2_b_acoffset - 1);

    rp_index = 0;

    while (fmap < fmap_end && rp_index < rp_max_count) {

        fmap1 = (const ods2_fmap1_t *) fmap;
        fmap2 = (const ods2_fmap2_t *) fmap;
        fmap3 = (const ods2_fmap3_t *) fmap;

        switch (fmap1->fm2_v_format) {

            case FM2_C_PLACEMENT:
                fmap += sizeof (ods2_fmap_placement_t);
                break;

            case FM2_C_FORMAT1:
                rp_buffer[rp_index].vbn = first_vbn;
                rp_buffer[rp_index].lbn =
                    (unsigned int) fmap1->fm2_w_lowlbn +
                    ((unsigned int) fmap1->fm2_v_highlbn << 16);
                rp_buffer[rp_index].count = fmap1->fm2_b_count1 + 1;

                trace ("volume.get_retrieval: got FM2_C_FORMAT1 {%u,%u}\n",
                    rp_buffer[rp_index].lbn, rp_buffer[rp_index].count);

                first_vbn += rp_buffer[rp_index].count;
                rp_index++;
                fmap += sizeof (ods2_fmap1_t);
                break;

            case FM2_C_FORMAT2:
                rp_buffer[rp_index].vbn = first_vbn;
                rp_buffer[rp_index].lbn = fmap2->fm2_l_lbn2;
                rp_buffer[rp_index].count = fmap2->fm2_v_count2 + 1;

                trace ("volume.get_retrieval: got FM2_C_FORMAT2 {%u,%u}\n",
                    rp_buffer[rp_index].lbn, rp_buffer[rp_index].count);

                first_vbn += rp_buffer[rp_index].count;
                rp_index++;
                fmap += sizeof (ods2_fmap2_t);
                break;

            case FM2_C_FORMAT3:
                rp_buffer[rp_index].vbn = first_vbn;
                rp_buffer[rp_index].lbn = fmap3->fm2_l_lbn3;
                rp_buffer[rp_index].count =
                    (unsigned int) fmap3->fm2_w_lowcount +
                    ((unsigned int) fmap3->fm2_v_count2 << 16) + 1;

                trace ("volume.get_retrieval: got FM2_C_FORMAT3 {%u,%u}\n",
                    rp_buffer[rp_index].lbn, rp_buffer[rp_index].count);

                first_vbn += rp_buffer[rp_index].count;
                rp_index++;
                fmap += sizeof (ods2_fmap3_t);
                break;
        }
    }

    trace ("volume.get_retrieval: return %d retrieval pointers\n", rp_index);

    return rp_index;
}


/*******************************************************************************
 *
 *  This routine decodes the file name from a file header.
 *
 *******************************************************************************
 */

static void get_header_name (
    const ods2_file_header_t *header,
    char *name,
    int name_size,
    int *version)
{
    const ods2_file_header_ident_t *ident;
    const char *end, *p;
    char *semi, *np = name;
    int ident_size;

    /* Locate the identification area inside the header */

    ident = (const ods2_file_header_ident_t*)
        ((const char*)header + 2 * header->fh2_b_idoffset);
    ident_size = 2 * header->fh2_b_mpoffset - 2 * header->fh2_b_idoffset;

    /* If the identification area is valid and contains a name field ... */

    if ((const char*)header < (const char*)ident &&
        (const char*)ident + ident_size <= 
        (const char*)header + sizeof (ods2_file_header_t) &&
        offsetof (ods2_file_header_ident_t, fi2_t_filename) +
        sizeof (ident->fi2_t_filename) <= ident_size) {

        /* Copy name up to first space */

        end = ident->fi2_t_filename + sizeof (ident->fi2_t_filename);

        for (p = ident->fi2_t_filename;
             p < end && *p != ' ' && name_size > 1;
             name_size--, *np++ = *p++);

        /* If there is a name extension, find first space in it. */

        if (offsetof (ods2_file_header_ident_t, fi2_t_filenamext) +
            sizeof (ident->fi2_t_filenamext) <= ident_size) {

            end = ident->fi2_t_filenamext + sizeof (ident->fi2_t_filenamext);

            for (p = ident->fi2_t_filenamext;
                 p < end && *p != ' ' && name_size > 1;
                 name_size--, *np++ = *p++);
        }
    }

    /* Terminate the name */

    if (name_size > 0)
        *np = 0;

    /* Locate the version number */

    if ((semi = strchr (name, ';')) == NULL)
        *version = 0;
    else {
        *version = atoi (semi + 1);
        *semi = 0;
    }

    trace ("volume.get_header_name: found \"%s\", version = %d\n",
        name, *version);
}


/*******************************************************************************
 *
 *  This routine reads a file header.
 *  Return 0 if success, -1 if error.
 *
 *******************************************************************************
 */

static int read_file_header (
    int err_flags,
    const volume_t *vol,
    const fid_t *fid,
    ods2_file_header_t *header)
{
    vbn_t vbn;
    lbn_t lbn;
    int status;
    fid_t fid1;

    trace ("volume.read_file_header: reading header (%d,%d,%d)\n", 
        fid->number, fid->sequence, fid->volume);

    /* If the requested header is for INDEXF.SYS, we locate the header */
    /* by LBN, using the home block. For other files, we locate the */
    /* header by VBN into INDEXF.SYS. */

    if (fid->number == FID_C_INDEXF) {
        lbn = vol->home.hm2_l_ibmaplbn + vol->home.hm2_w_ibmapsize;
        status = read_lbn (err_flags, vol->cache, lbn, 1, header, LBN_META);
    }
    else {
        vbn = vol->home.hm2_w_ibmapvbn + vol->home.hm2_w_ibmapsize +
            fid->number - 1;
        status = read_vbn (err_flags, vol->indexf, vbn, 1, header);
    }

    /* Check that the header is valid for this FID */

    if (status == 0) {

        get_fid (&fid1, &header->fh2_w_fid);

        if (fid->number != fid1.number ||
            fid->sequence != fid1.sequence ||
            fid->volume != fid1.volume) {

            error (err_flags,
                "invalid header, expected (%d,%d,%d), got (%d,%d,%d)\n",
                fid->number, fid->sequence, fid->volume,
                fid1.number, fid1.sequence, fid1.volume);
            status = -1;
        }
    }

    return status;
}


/*******************************************************************************
 *
 *  This routine opens a file.
 *
 *  Implementation note: As a special check, if the FID is the one of
 *  INDEXF.SYS, the open operation directly uses the home block of
 *  the volume to locate the file. Normal file opening uses vol->indexf
 *  to locate the file. The previous check is used to avoid recursion
 *  when opening INDEXF.SYS during the volume opening.
 *
 *******************************************************************************
 */

file_t *open_file (int err_flags, const volume_t *vol, const fid_t *fid)
{
    file_t *file;
    int rp_count, ok, checksum;
    vbn_t vbn = 1;
    fid_t fid1;
    ods2_file_header_t header;
    ods2_file_attr_t *recattr;
    retrieval_t *map, rp [ODS2_MAX_RP_PER_HEADER];
    char name [ODS2_MAX_NAME_SIZE];

    trace ("volume.open_file: opening file (%d,%d,%d)\n", 
        fid->number, fid->sequence, fid->volume);

    /* Initialize the file structure */

    file = utmalloc (sizeof (file_t));
    memset (file, 0, sizeof (file_t));
    file->cache = vol->cache;
    file->fid = *fid;

    /* Loop on all file headers (main header + optional extension headers) */

    for (fid1 = *fid, ok = 1; fid1.number != 0 && ok; ) {

        /* Read one header */

        if (read_file_header (err_flags, vol, &fid1, &header) < 0) {
            ok = 0;
            break;
        }

        /* Save first header in file_t structure */

        if (fid1.number == fid->number)
            file->header = header;

        /* Check if header is valid */

        checksum = checksum16 (&header, sizeof (header) - 2);

        if (checksum != header.fh2_w_checksum) {
            error (err_flags, "invalid file header (%d,%d,%d)\n",
                fid1.number, fid1.sequence, fid1.volume);
            error (err_flags, "stored checksum is 0x%04X, expected 0x%04X\n",
                header.fh2_w_checksum, checksum);
            ok = 0;
            break;
        }

        ok = header.fh2_b_idoffset >=
            offsetof (ods2_file_header_t, fh2_l_highwater) / 2 &&
            header.fh2_b_idoffset <= sizeof (header) / 2 &&
            header.fh2_b_idoffset <= header.fh2_b_mpoffset &&
            header.fh2_b_mpoffset <= sizeof (header) / 2 &&
            header.fh2_b_mpoffset <= header.fh2_b_acoffset &&
            header.fh2_b_acoffset <= sizeof (header) / 2 &&
            header.fh2_b_acoffset <= header.fh2_b_rsoffset &&
            header.fh2_b_rsoffset <= sizeof (header) / 2 &&
            header.fh2_b_struclev == 2 &&
            header.fh2_b_strucver >= 1;

        if (!ok) {
            error (err_flags, "invalid file header (%d,%d,%d)\n",
                fid1.number, fid1.sequence, fid1.volume);
            break;
        }

        /* Get more retrieval pointers */

        rp_count = get_retrieval (&header, vbn, rp, ODS2_MAX_RP_PER_HEADER);
        map = utmalloc ((file->map_count + rp_count) * sizeof (retrieval_t));
        memcpy (map + file->map_count, rp, rp_count * sizeof (retrieval_t));

        if (file->map != NULL) {
            memcpy (map, file->map, file->map_count * sizeof (retrieval_t));
            free (file->map);
        }

        file->map = map;
        file->map_count += rp_count;

        /* Get FID of next extension header (zero if last) */

        get_fid (&fid1, &header.fh2_w_ext_fid);
    }

    /* Exit now is something invalid has been found */

    if (!ok) {
        if (file->map != NULL)
            free (file->map);
        free (file);
        return NULL;
    }

    /* Build the file name from the header */

    get_header_name (&file->header, name, sizeof (name), &file->version);
    file->name = new_string (name);

    /* Compute file size in blocks and in bytes */

    recattr = &file->header.fh2_w_recattr;

    file->last_vbn = recattr->fat_w_hiblkl + (recattr->fat_w_hiblkh << 16);

    file->used_size = (recattr->fat_w_efblkl +
        (recattr->fat_w_efblkh << 16) - 1) * ODS2_BLOCK_SIZE +
        recattr->fat_w_ffbyte;

    /* Compute the last VBN containing data */

    file->last_data_vbn = recattr->fat_w_efblkl + (recattr->fat_w_efblkh << 16);
    if (recattr->fat_w_ffbyte == 0)
        file->last_data_vbn--;

    /* Initializes the directory fields if the file is a directory. */

    if (file->header.fh2_l_filechar & FH2_M_DIRECTORY) {

        /* Allocate one block for the directory records. Note that in ODS-2 */
        /* directories a record never crosses a block boundary ("nospan" */
        /* option). If a record does not fit into a block, it starts at the */
        /* next block. Directory blocks can consequently be read one at a */
        /* time, there is always some complete records to read. */

        file->buffer = utmalloc (ODS2_BLOCK_SIZE);

        /* The other directory fields are left to zero, which is a valid */
        /* start state: cur_record = 0 means "end of current block" and */
        /* since vbn is 0, the next operation will start at vbn 1. */
    }

    return file;
}


/*******************************************************************************
 *
 *  This routine closes a file.
 *
 *******************************************************************************
 */

void close_file (file_t *file)
{
    if (file->name != NULL)
        free (file->name);
    if (file->map != NULL)
        free (file->map);
    if (file->buffer != NULL)
        free (file->buffer);
    free (file);
}


/*******************************************************************************
 *
 *  This routine locates a file.
 *
 *******************************************************************************
 */

int locate_file (
    int err_flags,
    const volume_t *vol,
    const char *path,
    const fid_t *root,
    fid_t *fid,
    char *name,
    int name_size,
    int *version)
{
    file_t *dir;
    const char *comp;
    int n, locate, complen;

    trace ("volume.locate_file: parsing \"%s\"\n", path);

    /* Start at the root. Open it and get its name */

    if ((dir = open_file (err_flags, vol, root)) == NULL)
        return -2; /* "other" error */

    *fid = *root;
    strncpy (name, dir->name, name_size);
    name [name_size - 1] = 0;
    *version = dir->version;

    /* Loop on all components of the path */

    for (comp = path; *comp != 0; comp += complen) {

        /* Locate the next component of the path in comp/complen */

        while (*comp == '/')
            comp++;

        for (complen = 0;
             comp [complen] != 0 && comp [complen] != '/';
             complen++);

        /* If the component is empty, the current file is the target */

        if (complen == 0)
            break;

        /* If the component is "." ignore component */

        if (complen == 1 && comp[0] == '.')
            continue;

        /* Open the current file (should be a directory) */

        trace ("volume.locate_file: lookup dir %s (%d,%d,%d) for \"%.*s\"\n",
            name, fid->number, fid->sequence, fid->volume, complen, comp);

        if (dir == NULL && (dir = open_file (err_flags, vol, fid)) == NULL)
            return -2; /* "other" error */

        /* Check that the current file is actually a directory */

        if (!is_a_directory (dir)) {
            error (err_flags, "file %s (%d,%d,%d) is not a directory\n",
                name, fid->number, fid->sequence, fid->volume);
            close_file (dir);
            return -1; /* not found */
        }

        /* If the component is ".." move to parent directory  */

        if (complen == 2 && comp[0] == '.' && comp[1] == '.') {

            /* Get FID of parent directory */

            get_fid (fid, &dir->header.fh2_w_backlink);

            /* Close current directory, open parent and get its name */

            close_file (dir);
            if ((dir = open_file (err_flags, vol, fid)) == NULL)
                return -2; /* "other" error */

            strncpy (name, dir->name, name_size);
            name [name_size - 1] = 0;
            *version = dir->version;
            continue;
        }

        /* Look for the component in the current directory */

        for (;;) {

            /* Look for next file in the directory */

            locate = next_file (err_flags, dir, name, name_size, version, fid);

            trace ("volume.locate_file: got %s;%d FID=(%d,%d,%d) status=%d\n",
                name, *version, fid->number, fid->sequence, fid->volume,
                locate);

            if (locate < 0)
                /* End of directory or other error => file is not found */
                break;

            if (locate == 0)
                /* Additional version of the previous file, ignore it */
                continue;

            /* A file is returned in name/fid. Check if the name of */
            /* the component matches (case insensitive) */

            for (n = 0; n < complen && toupper (comp[n]) == name[n]; n++);

            /* If the name matches or if the name is comp.DIR, found */

            if (n == complen && (name[n] == 0 || strcmp (name+n, ".DIR") == 0)){
                locate = 0;
                break;
            }

            /* If the name is after the component in alphabetical order, */
            /* the component does not exist (the content or a directory */
            /* is sorted in alphabetical order) */

            if (n < complen && toupper (comp[n]) < name[n]) {
                locate = -1;;
                break;
            }
        }

        close_file (dir);
        dir = NULL;

        if (locate < 0) {
            error (err_flags, "%.*s not found\n", complen, comp);
            return locate;
        }
    }

    if (dir != NULL)
        close_file (dir);

    return 0; /* found */
}


/*******************************************************************************
 *
 *  This routine builds the path of a file (defined by its FID) using
 *  the baclinks and file names as stored in the file headers.
 *
 *******************************************************************************
 */

int get_file_path (
    int err_flags,
    const volume_t *vol,
    const fid_t *fid,
    char *name,
    int name_size,
    int *version)
{
    int count, version1;
    fid_t fid1;
    ods2_file_header_t header;
    char *p, *c, *n, path [ODS2_MAX_PATH_SIZE], comp [ODS2_MAX_NAME_SIZE];

    trace ("volume.get_file_path: getting path of (%d,%d,%d)\n",
        fid->number, fid->sequence, fid->volume);

    /* We use a level counter to avoid infinite recursion */

    count = ODS2_MAX_DIR_DEPTH;

    /* We build the reverse path into "path". It will be reversed later */

    p = path;

    *version = -1;

    /* Loop on file headers */

    for (fid1 = *fid; fid1.number != FID_C_MFD && count > 0; count--) {

        /* Read one file header and get its name */

        if (read_file_header (err_flags, vol, &fid1, &header) < 0)
            return -1;

        get_header_name (&header, comp, sizeof (comp), &version1);

        /* Get the version of the first file (final path component */

        if (*version < 0)
            *version = version1;

        /* Copy the name component. Convert to lower case. Remove */
        /* trailing ".DIR" or "." if they have a non-empty name. */

        *p++ = '/';

        for (c = comp; *c != 0; c++) {

            if (p >= path + sizeof (path) - 2) {
                error (err_flags, "file path is too long\n");
                return -1;
            }

            if (*c == '.' && c > comp && (c[1] == 0 || strcmp (c, ".DIR") == 0))
                break;

            *p++ = tolower (*c);
        }

        /* Get the FID of the parent directory */

        get_fid (&fid1, &header.fh2_w_backlink);
    }

    *p = 0;

    /* If we found too many levels of directories, we suspect a recursion */

    if (count <= 0) {
        error (err_flags, "infinite directory recursion detected\n");
        return -1;
    }

    /* Reverse the path into the user variable */

    trace ("volume.get_file_path: reverse path is \"%s\"\n", path);

    if (name_size <= p - name) {
        error (err_flags, "file path is too long\n");
        return -1;
    }

    for (n = name; p > path; ) {
        *p = 0;
        while (p > path && *--p != '/');
        strcpy (n, p);
        n += strlen (n);
    }
    if (n == name)
        *n++ = '/';
    *n = 0;

    return 0;
}


/*******************************************************************************
 *
 *  This routine checks if a file is a directory.
 *
 *******************************************************************************
 */

int is_a_directory (const file_t *file)
{
    return (file->header.fh2_l_filechar & FH2_M_DIRECTORY) != 0;
}


/*******************************************************************************
 *
 *  This routine checks if a file is a "system reserved" file..
 *
 *******************************************************************************
 */

int is_reserved_file (const fid_t *fid)
{
    return fid->number <= FID_C_MAXRESFIL && fid->number == fid->sequence;
}


/*******************************************************************************
 *
 *  This routine reads one or more VBN in a file.
 *
 *******************************************************************************
 */

int read_vbn (
    int err_flags,
    const file_t *file,
    vbn_t vbn,
    int vbn_count,
    void *buffer)
{
    int rp, vbn_subcount;
    lbn_t lbn = 0;
    vbn_t last_vbn = 0;
    lbn_usage_t lbn_usage;

    trace ("volume.read_vbn: VBN %d, %d blocks\n", vbn, vbn_count);

    /* Locate the first VBN in the retrieval pointers */

    for (rp = 0; rp < file->map_count && lbn == 0; rp++) {
        /* Compute last VBN in this extent */
        last_vbn = file->map[rp].vbn + file->map[rp].count - 1;
        /* Check if the VBN is in this extent */
        if (file->map[rp].vbn <= vbn && vbn <= last_vbn)
            lbn = file->map[rp].lbn + vbn - file->map[rp].vbn;
    }

    /* Check if the VBN was found */

    if (lbn == 0) {
        error (err_flags, "VBN %d not found in file\n", vbn);
        return -1;
    }

    /* Directories and reserved files contain meta-data. */
    /* Other files contain regular data. */

    lbn_usage = is_a_directory (file) || is_reserved_file (&file->fid) ?
        LBN_META : LBN_DATA;

    /* Now read all requested VBN's */

    while (vbn_count > 0) {

        /* Compute the number of VBN we can find in this extent */

        vbn_subcount = last_vbn - vbn + 1;
        if (vbn_subcount > vbn_count)
            vbn_subcount = vbn_count;

        /* Read these VBN using their LBN */

        if (read_lbn (err_flags, file->cache, lbn, vbn_subcount, buffer,
            lbn_usage) < 0)
            return -1;

        /* Prepare to read in the next extent if more blocks are needed */

        if ((vbn_count -= vbn_subcount) > 0) {
            buffer = (char*)buffer + vbn_subcount * ODS2_BLOCK_SIZE;
            vbn += vbn_subcount;
            if (++rp >= file->map_count) {
                error (err_flags, "attempt to read after end of file "
                    "(VBN %d)\n", vbn);
                return -1;
            }
            /* Check if there is no hole in the file */
            if (file->map[rp].vbn != vbn) {
                error (err_flags, "VBN %d is in a hole\n", vbn);
                return -1;
            }
            /* Compute starting LBN and last VBN in next extent */
            lbn = file->map[rp].lbn;
            last_vbn = file->map[rp].vbn + file->map[rp].count - 1;
        }
    }

    return 0;
}


/*******************************************************************************
 *
 *  This routine reads a file byte by byte.
 *
 *******************************************************************************
 */

int read_byte (int err_flags, file_t *file)
{
    /* Check that the file is not a directory */

    if (is_a_directory (file))
        return -1;

    /* If there is currently no data buffer, allocate one */

    if (file->buffer == NULL) {
        file->buffer = utmalloc (ODS2_BLOCK_SIZE);
        file->cur_byte = file->last_byte = file->buffer;
    }

    /* If we reached the end of the current block, read the next one */

    if (file->cur_byte >= file->last_byte) {

        /* Read one more VBN (if not at end of file) */

        if (file->vbn >= file->last_data_vbn)
            return -2; /* end of file */

        if (read_vbn (err_flags, file, file->vbn + 1, 1, file->buffer) < 0)
            return -1; /* error */

        file->vbn++;
        file->cur_byte = file->buffer;

        /* Compute the last meaningful byte in the block */

        if (file->vbn != file->last_data_vbn)
            file->last_byte = file->buffer + ODS2_BLOCK_SIZE;
        else
            file->last_byte = file->buffer + file->used_size % ODS2_BLOCK_SIZE;
    }

    /* Return the next character in the current block */

    if (file->cur_byte < file->last_byte)
        return *file->cur_byte++;
    else
        return -2; /* end of file */
}


/*******************************************************************************
 *
 *  This routine skips the end of the current block in the file
 *  (applies only when the file is read byte by byte).
 *
 *******************************************************************************
 */

void skip_block (file_t *file)
{
    file->cur_byte = file->last_byte;
}


/*******************************************************************************
 *
 *  This routine returns the next file name in a directory.
 *
 *******************************************************************************
 */

int next_file (
    int err_flags,
    file_t *dir,        /* directory file descriptor */
    char *name,         /* returned file name.type (without version) */
    int name_size,      /* size of name buffer */
    int *version,       /* version number of file */
    fid_t *fid)         /* fid of file */
{
    int size;

    /* Preinitialize returned value to nothing */

    *version = fid->number = fid->sequence = fid->volume = 0;
    if (name_size > 0)
        *name = 0;

    /* Check that the file is actually a directory */

    if (!is_a_directory (dir)) {
        error (err_flags, "%s is not a directory\n", dir->name);
        return -2;
    }

    /* Loop in the directory data blocks until a name is found */

    for (;;) {

        /* When cur_record is zero, this means "end of current block". */
        /* Load next block until a non-empty one is found. */

        while (dir->cur_record == NULL) {

            /* If the next block is after last VBN, this is the end. */

            if (dir->vbn >= dir->last_data_vbn)
                return -1; /* no more file */

            /* Read the next data block in the directory */

            if (read_vbn (err_flags, dir, ++(dir->vbn), 1, dir->buffer) < 0)
                return -2; /* "other" errors */

            /* Adjust current record into the new data block. */
            /* Set current entry to zero to indicate "start of record". */

            dir->cur_record = (ods2_dir_t*) dir->buffer;
            dir->cur_entry = NULL;

            /* If first record is also the last one, the block is empty */

            if (dir->cur_record->dir_w_size == 0xFFFF)
                dir->cur_record = NULL;
        }

        /* When cur_entry is zero, this means "start of record". */

        while (dir->cur_record != NULL && dir->cur_entry == NULL) {

            /* If the record is the last one in the block, stop here */

            if ((uint8*)dir->cur_record >= dir->buffer + ODS2_BLOCK_SIZE ||
                dir->cur_record->dir_w_size == 0xFFFF) {

                dir->cur_record = NULL;
                break;
            }

            /* Adjust the pointers into the record */

            dir->first_entry = (ods2_dirent_t*) (dir->cur_record->dir_t_name +
                ((dir->cur_record->dir_b_namecount + 1) & ~1));
            dir->cur_entry = dir->first_entry;
            dir->last_entry = (ods2_dirent_t*) ((uint8*)dir->cur_record +
                sizeof (dir->cur_record->dir_w_size) +
                dir->cur_record->dir_w_size);

            /* Check the validity of the directory record */

            if ((uint8*)dir->last_entry > dir->buffer + ODS2_BLOCK_SIZE ||
                (uint8*)dir->last_entry <= (uint8*)dir->cur_record ||
                (uint8*)dir->first_entry > (uint8*)dir->last_entry ||
                (uint8*)dir->first_entry <= (uint8*)dir->cur_record) {

                error (err_flags, "invalid directory entry in %s\n", dir->name);
                return -2;
            }

            /* If the record is not the kind that contains FID, or is */
            /* empty, skip it */

            if (dir->cur_record->dir_v_type != DIR_C_FID ||
                dir->first_entry >= dir->last_entry) {

                dir->cur_record = (ods2_dir_t*) dir->last_entry;
                dir->cur_entry = NULL;
            }
        }

        /* If no record found in this block, loop back */

        if (dir->cur_record == NULL)
            continue;

        /* If the current entry is after the last one, end of record */

        if (dir->cur_entry >= dir->last_entry) {
            dir->cur_record = (ods2_dir_t*) dir->last_entry;
            dir->cur_entry = NULL;
            continue;
        }

        /* The current entry is valid, Extract the informations */

        *version = dir->cur_entry->dir_w_version;
        get_fid (fid, &dir->cur_entry->dir_w_fid);

        if (name_size > 0) {
            size = dir->cur_record->dir_b_namecount < name_size ?
                dir->cur_record->dir_b_namecount : name_size - 1;
            memcpy (name, dir->cur_record->dir_t_name, size);
            name [size] = 0;
        }

        /* Return 1 if the returned file name is the first version and */
        /* zero otherwise. Then point to the next entry in the record */

        return dir->cur_entry++ == dir->first_entry && 
            ! dir->cur_record->dir_v_prevrec ? 1 : 0;
    }
}


/*******************************************************************************
 *
 *  This routine rewinds a file.
 *
 *******************************************************************************
 */

void rewind_file (file_t *file)
{
    file->vbn = 0;
    file->cur_record = NULL;
    file->first_entry = file->cur_entry = file->last_entry = NULL;
    file->cur_byte = file->last_byte;
}
