/*******************************************************************************
 *
 *  Copyright (c) 1999 by Thierry Lelegard
 *
 *  This software is covered by the "GNU GENERAL PUBLIC LICENSE" (GPL),
 *  version 2, June 1991. See the file named COPYING for details.
 *
 *  Project: VMSCD - OpenVMS CD-ROM Utility for Linux
 *  File:    volume.h
 *  Author:  Thierry Lelegard
 *
 *
 *  Abstract
 *  --------
 *  This module implements the ODS-2 volume processing. This includes
 *  initializing the file system structure, locating and opening files.
 *  See the file ods2.h for the on-disk structure.
 *
 *
 *  Modification History
 *  --------------------
 *  18 Dec 1999 - Thierry Lelegard (lelegard@club-internet.fr)
 *                Creation of the file.
 *
 *
 *******************************************************************************
 */


#ifndef VOLUME_H_LOADED
#define VOLUME_H_LOADED 1

typedef uint32 vbn_t;

/*
 *  Description of a retrieval pointer.
 *  The last entry in an array of retrieval pointers is always {0,0,0}.
 */

typedef struct {
    vbn_t vbn;       /* Virtual block number in file, start at 1 */
    lbn_t lbn;       /* Logical block number on disk, start at 0 */
    int   count;     /* Number of blocks in this extent */
} retrieval_t;

/*
 *  Description of a file identification (FID).
 */

typedef struct {
    uint32 number;    /* File number (header number) */
    uint16 sequence;  /* Sequence number (usage count of header) */
    uint16 volume;    /* Volume number in volume set (0 for CD-ROM) */
} fid_t;

/*
 *  Description of a file.
 */

typedef struct {
    cache_t cache;               /* LBN cache to cdrom device */
    fid_t fid;                   /* File identification */
    char *name;                  /* Name as present in header (mallocated) */
    int version;                 /* Version as present in header */
    ods2_file_header_t header;   /* First (or only) header */
    vbn_t last_vbn;              /* Last allocated VBN */
    vbn_t last_data_vbn;         /* Last VBN containing data */
    int used_size;               /* File size in bytes */
    retrieval_t *map;            /* Retrieval pointers (mallocated) */
    int map_count;               /* Number of RP in map array */

    /* The following fields are used only if the file is a directory */
    /* or if it is read byte by byte. */

    uint8 *buffer;               /* Data buffer (mallocated) */
    vbn_t vbn;                   /* VBN of current data block */

    /* The following fields are used only when the file is read by bytes */

    uint8 *cur_byte;             /* Next byte to read in the buffer */
    uint8 *last_byte;            /* Last meaningful byte in buffer + 1 */

    /* The following fields are used only if the file is a directory */

    ods2_dir_t *cur_record;      /* Current record in buffer */
    ods2_dirent_t *first_entry;  /* First entry in current record */
    ods2_dirent_t *cur_entry;    /* Current entry in current record */
    ods2_dirent_t *last_entry;   /* Aftet last entry in current record */
} file_t;

/*
 *  Description of an ODS-2 volume.
 */

typedef struct {
    cache_t cache;            /* LBN cache to cdrom device */
    ods2_home_block_t home;   /* Home block */
    char *volname;            /* Volume name (mallocated) */
    file_t *indexf;           /* Description of INDEXF.SYS */
} volume_t;

/*
 *  This routine opens a cdrom.
 *  Return NULL on error.
 */

volume_t *open_volume (int err_flags, cache_t cache);

/*
 *  This routine closes the cdrom.
 */

void close_volume (volume_t *vol);

/*
 *  This routine opens a file.
 *  Return NULL on error.
 */

file_t *open_file (int err_flags, const volume_t *vol, const fid_t *fid);

/*
 *  This routine closes a file.
 */

void close_file (file_t *file);

/*
 *  This routine locates a file. The name is relative to a directory
 *  specified by its FID. Get the FID and local name of the target file.
 *  Returned value:
 *     0 : Success, file found.
 *    -1 : File not found.
 *    -2 : Other errors.
 */

int locate_file (
    int err_flags,
    const volume_t *vol,
    const char *path,
    const fid_t *root,
    fid_t *fid,
    char *name,
    int name_size,
    int *version);

/*
 *  This routine builds the path of a file (defined by its FID) using
 *  the baclinks and file names as stored in the file headers.
 *  Return -1 in case of error, 0 otherwise.
 */

int get_file_path (
    int err_flags,
    const volume_t *vol,
    const fid_t *fid,
    char *name,
    int name_size,
    int *version);

/*
 *  This routine checks if a file is a directory.
 */

int is_a_directory (const file_t *file);

/*
 *  This routine checks if a file is a "system reserved" file..
 */

int is_reserved_file (const fid_t *fid);

/*
 *  This routine reads one or more VBN in a file.
 *  Return -1 in case of error, 0 otherwise.
 */

int read_vbn (
    int err_flags,
    const file_t *file,
    vbn_t vbn,
    int vbn_count,
    void *buffer);

/*
 *  This routine reads a file byte by byte.
 *  Directory files cannot be read like this, use next_file instead.
 *  Return -1 in case of error, -2 at end of file, the next byte otherwise.
 */

int read_byte (int err_flags, file_t *file);

/*
 *  This routine skips the end of the current block in the file
 *  (applies only when the file is read byte by byte).
 */

void skip_block (file_t *file);

/*
 *  This routine returns the next file name in a directory.
 *  Returned value:
 *     1 : First version (highest version number) of the file
 *     0 : Another version of the same file
 *    -1 : End of directory, no file returned.
 *    -2 : Other errors.
 */

int next_file (
    int err_flags,
    file_t *dir,        /* directory file descriptor */
    char *name,         /* returned file name.type (without version) */
    int name_size,      /* size of name buffer */
    int *version,       /* version number of file */
    fid_t *fid);        /* fid of file */

/*
 *  This routine rewinds a file.
 */

void rewind_file (file_t *file);

#endif /* VOLUME_H_LOADED */
