// value with derivatives

// The "Value" is supposed to be a function of parameters t_1,...,t_n
// This structure stores the value of the "Value" and its n derivatives.


#define WANT_STREAM
#define WANT_MATH

#include "vwd.h"

#ifdef DO_REPORT
#define REPORT { static ExeCounter ExeCount(__LINE__,11); ++ExeCount; }
#else
#define REPORT {}
#endif



void ParameterSetClass::CleanUp()
{
   REPORT
   NameSeq* ns1;
   for (NameSeq* ns = Next; ns; ns = ns1) { ns1 = ns->Next; delete ns; }
   Frozen = false; n = 0; Next = 0; last_i = 0;
}

void ParameterSetClass::DecrRef()
{
   REPORT
   if (--RefCount < 0) { REPORT  delete this; }
}

ParameterSet::ParameterSet()
{
   REPORT
   PSC = new ParameterSetClass();
   if (!PSC) Throw(Bad_alloc());
}

void ParameterSet::operator=(const ParameterSet& ps)
{
   REPORT
   ParameterSetClass* PSC1 = PSC; PSC = ps.PSC;
   PSC->IncrRef(); PSC1->DecrRef();
}

int ParameterSetClass::AddParameter(const String& name)
{
   REPORT
   NameSeq* New = new NameSeq(name);          // element to be added to list
   if (!New) Throw(Bad_alloc());              // no space
   NameSeq* ns = Next;
   if (!ns) { REPORT Next = New; }            // no names on list
   else
   {
      REPORT
      for (NameSeq* ns1 = ns->Next; ns1; ns1 = ns1->Next)
      {
         if (ns->Name==name)
            Throw(Logic_error("Duplicate name in parameter set\n"));
         ns = ns1;
      }
      ns->Next = New;
   }
   return ++n;
}

int ParameterSetClass::LocateParameter(const String& name) const
{
   REPORT
   int i = 0;
   for (NameSeq* ns = Next; ns; ns = ns->Next)
      { i++; if (ns->Name==name) return i; }
   return 0;
}

String& ParameterSetClass::operator()(int i)
{
   // this rather complicated procedure is so if we call for names
   // in increasing orders of i we don't have to keep scanning through list
   REPORT
   if (i <=0 || i > n) Throw(Logic_error("Illegal parameter index\n"));
   NameSeq* ns;
   if (i < last_i || last_i == 0) { REPORT  ns = Next; last_i = 1; }
   else { REPORT  ns = last_seq; }
   while (last_i < i) { REPORT  ns = ns->Next; ++last_i; }
   last_seq = ns;
   return ns->Name;
}

void AssertEqual(const ParameterSet& ps1, const ParameterSet& ps2)
{
   REPORT
   if (ps1!=ps2) Throw(Logic_error("Different parameter sets\n"));
}


inline Real square(Real x) { return x * x; }

ValueWithDerivatives::ValueWithDerivatives(const ParameterSet& ps)
   : PS(ps), Value(0), Derivatives(ps.Size())
   { REPORT Derivatives = 0; }

ValueWithDerivatives::ValueWithDerivatives(const ParameterSet& ps, Real v,
   const RowVector& d)
   : PS(ps), Value(v), Derivatives(d)
{
   REPORT
   if (PS.Size() != d.Ncols()) Throw(Logic_error("Incompatible dimensions\n"));
}

ValueWithDerivatives::ValueWithDerivatives(const ParameterSet& ps, Real v,
   int k)
   : PS(ps), Value(v), Derivatives(ps.Size())
{ REPORT  Derivatives = 0; Derivatives(k) = 1; }

ValueWithDerivatives::ValueWithDerivatives(const ParameterSet& ps, Real v,
   const String& name)
   : PS(ps), Value(v), Derivatives(ps.Size())
{
   REPORT
   Derivatives = 0;
   int k = PS.LocateParameter(name);
   if (!k) Throw(Logic_error("Name not recognised\n"));
   Derivatives(k) = 1;
}

ValueWithDerivatives::ValueWithDerivatives(const ValueWithDerivatives& vwd)
   : PS(vwd.PS), Value(vwd.Value), Derivatives(vwd.Derivatives) { REPORT }

void ValueWithDerivatives::operator=(const ValueWithDerivatives& vwd)
   { REPORT  PS = vwd.PS; Value = vwd.Value; Derivatives = vwd.Derivatives; }

void ValueWithDerivatives::operator+=(const ValueWithDerivatives& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Value += vwd.Value; Derivatives += vwd.Derivatives;
}

void ValueWithDerivatives::operator-=(const ValueWithDerivatives& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Value -= vwd.Value; Derivatives -= vwd.Derivatives;
}

void ValueWithDerivatives::operator*=(const ValueWithDerivatives& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Derivatives = vwd.Value * Derivatives + Value * vwd.Derivatives;
   Value *= vwd.Value;
}

void ValueWithDerivatives::operator/=(const ValueWithDerivatives& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Value /= vwd.Value;
   Derivatives = (Derivatives - Value  * vwd.Derivatives) / vwd.Value;
}

void ValueWithDerivatives::operator+=(Real r) { REPORT  Value += r; }

void ValueWithDerivatives::operator-=(Real r) { REPORT  Value -= r; }

void ValueWithDerivatives::operator*=(Real r)
   { REPORT  Value *= r; Derivatives *= r; }

void ValueWithDerivatives::operator/=(Real r)
   { REPORT  Value /= r; Derivatives /= r; }

ValueWithDerivatives ValueWithDerivatives::operator-() const
   { REPORT  return ValueWithDerivatives(PS, -Value, -Derivatives); }

ValueWithDerivatives operator+(const ValueWithDerivatives& vwd1,
   const ValueWithDerivatives& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   return ValueWithDerivatives(vwd1.PS, vwd1.Value + vwd2.Value,
      vwd1.Derivatives + vwd2.Derivatives);
}

ValueWithDerivatives operator-(const ValueWithDerivatives& vwd1,
   const ValueWithDerivatives& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   return ValueWithDerivatives(vwd1.PS, vwd1.Value - vwd2.Value,
      vwd1.Derivatives - vwd2.Derivatives);
}

ValueWithDerivatives operator*(const ValueWithDerivatives& vwd1,
   const ValueWithDerivatives& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   return ValueWithDerivatives(vwd1.PS, vwd1.Value * vwd2.Value,
      vwd2.Value * vwd1.Derivatives + vwd1.Value * vwd2.Derivatives);
}

ValueWithDerivatives operator/(const ValueWithDerivatives& vwd1,
   const ValueWithDerivatives& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   Real v = vwd1.Value / vwd2.Value;
   return ValueWithDerivatives( vwd1.PS, v,
      (vwd1.Derivatives - v * vwd2.Derivatives) / vwd2.Value );
}

ValueWithDerivatives pow(const ValueWithDerivatives& vwd1,
   const ValueWithDerivatives& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   Real v = pow(vwd1.Value, vwd2.Value);
   return ValueWithDerivatives( vwd1.PS, v,
      v * ((vwd2.Value / vwd1.Value) * vwd1.Derivatives
         + log(vwd1.Value) * vwd2.Derivatives) );
}

ValueWithDerivatives operator+(Real r, const ValueWithDerivatives& vwd)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, r + vwd.Value, vwd.Derivatives);
}

ValueWithDerivatives operator-(Real r, const ValueWithDerivatives& vwd)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, r - vwd.Value, - vwd.Derivatives);
}

ValueWithDerivatives operator*(Real r, const ValueWithDerivatives& vwd)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, r * vwd.Value, r * vwd.Derivatives);
}

ValueWithDerivatives operator/(Real r, const ValueWithDerivatives& vwd)
{
   REPORT
   Real v = r / vwd.Value;
   return ValueWithDerivatives(vwd.PS, v, (- v / vwd.Value) * vwd.Derivatives);
}

ValueWithDerivatives pow(Real r, const ValueWithDerivatives& vwd)
{
   REPORT
   Real v = pow(r, vwd.Value);
   return ValueWithDerivatives(vwd.PS, v, (v * log(r)) * vwd.Derivatives);
}

ValueWithDerivatives operator+(const ValueWithDerivatives& vwd, Real r)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, vwd.Value + r, vwd.Derivatives);
}

ValueWithDerivatives operator-(const ValueWithDerivatives& vwd, Real r)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, vwd.Value - r, vwd.Derivatives);
}

ValueWithDerivatives operator*(const ValueWithDerivatives& vwd, Real r)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, vwd.Value * r, vwd.Derivatives * r);
}

ValueWithDerivatives operator/(const ValueWithDerivatives& vwd, Real r)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, vwd.Value / r, vwd.Derivatives / r);
}

ValueWithDerivatives pow(const ValueWithDerivatives& vwd, Real r)
{
   REPORT
   Real v = pow(vwd.Value, r-1);
   return
      ValueWithDerivatives(vwd.PS, v * vwd.Value, (r * v) * vwd.Derivatives);
}

ValueWithDerivatives pow(const ValueWithDerivatives& vwd, int r)
{
   if (r == 0) { REPORT  return ValueWithDerivatives(vwd.PS) + 1.0; }
   REPORT
   Real v = pow(vwd.Value, r-1);
   return
      ValueWithDerivatives(vwd.PS, v * vwd.Value, (r * v) * vwd.Derivatives);
}

ValueWithDerivatives exp(const ValueWithDerivatives& vwd)
{
   REPORT
   Real ev = exp(vwd.Value);
   return ValueWithDerivatives(vwd.PS, ev, ev * vwd.Derivatives);
}

ValueWithDerivatives log(const ValueWithDerivatives& vwd)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, log(vwd.Value),
      vwd.Derivatives / vwd.Value);
}

ValueWithDerivatives sin(const ValueWithDerivatives& vwd)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, sin(vwd.Value),
      cos(vwd.Value) * vwd.Derivatives);
}

ValueWithDerivatives cos(const ValueWithDerivatives& vwd)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, cos(vwd.Value),
      (-sin(vwd.Value)) * vwd.Derivatives);
}

ValueWithDerivatives tan(const ValueWithDerivatives& vwd)
{
   REPORT
   return ValueWithDerivatives(vwd.PS, tan(vwd.Value),
      vwd.Derivatives / square(cos(vwd.Value)));
}


#ifdef WANT_ERF
// code provided by Tomas Gonzalez-Llarena
// assumes erf(double) is available as a library function
ValueWithDerivatives erf(const ValueWithDerivatives& vwd)
{
  REPORT
  Real e2 = exp(-square(vwd.Value));
  return ValueWithDerivatives(vwd.PS, erf(vwd.Value),
     2.0 * vwd.Derivatives * e2 / sqrt(M_PI));
}
#endif



// Gaussian numerical integration with 32 terms
ValueWithDerivatives GaussianIntegration32(VWDOfReal& f, Real Lower, Real Upper)
{
   REPORT
   double x[] =
   {
      0.048307665687738316235,
      0.144471961582796493485,
      0.239287362252137074545,
      0.331868602282127649780,
      0.421351276130635345364,
      0.506899908932229390024,
      0.587715757240762329041,
      0.663044266930215200975,
      0.732182118740289680387,
      0.794483795967942406963,
      0.849367613732569970134,
      0.896321155766052123965,
      0.934906075937739689171,
      0.964762255587506430774,
      0.985611511545268335400,
      0.997263861849481563545
   };

   double w[] =
   {
      0.096540088514727800567,
      0.095638720079274859419,
      0.093844399080804565639,
      0.091173878695763884713,
      0.087652093004403811143,
      0.083311924226946755222,
      0.078193895787070306472,
      0.072345794108848506225,
      0.065822222776361846838,
      0.058684093478535547145,
      0.050998059262376176196,
      0.042835898022226680657,
      0.034273862913021433103,
      0.025392065309262059456,
      0.016274394730905670605,
      0.007018610009470096600
   };
   Real delta = 0.5 * (Upper - Lower); Real centre = 0.5 * (Upper + Lower);
   Real dev = delta * x[0];
   ValueWithDerivatives value = w[0] * (f(centre + dev) + f(centre - dev));

   for (int i = 1; i < 16; i++)
   {
      dev = delta * x[i];
      value += w[i] * (f(centre + dev) + f(centre - dev));
   }
   return value * delta;
}

// ****************** second derivatives ***********************************

ValueWithDerivatives2::ValueWithDerivatives2(const ParameterSet& ps)
   : PS(ps), Value(0), Derivatives(ps.Size()),
      SecondDerivatives(ps.Size(), ps.Size())
   { REPORT  Derivatives = 0; SecondDerivatives = 0; }

ValueWithDerivatives2::ValueWithDerivatives2(const ParameterSet& ps, Real v,
   const RowVector& d, const Matrix& d2)
   : PS(ps), Value(v), Derivatives(d), SecondDerivatives(d2)
{
   REPORT
   if (PS.Size() != d.Ncols()) Throw(Logic_error("Incompatible dimensions\n"));
}

ValueWithDerivatives2::ValueWithDerivatives2(const ParameterSet& ps, Real v,
   int k)
   : PS(ps), Value(v), Derivatives(ps.Size()),
      SecondDerivatives(ps.Size(), ps.Size())
   { REPORT  Derivatives = 0; Derivatives(k) = 1; SecondDerivatives = 0; }

ValueWithDerivatives2::ValueWithDerivatives2(const ParameterSet& ps, Real v,
   const String& name)
   : PS(ps), Value(v), Derivatives(ps.Size()),
      SecondDerivatives(ps.Size(), ps.Size())
{
   REPORT
   Derivatives = 0; SecondDerivatives = 0;
   int k = PS.LocateParameter(name);
   if (!k) Throw(Logic_error("Name not recognised\n"));
   Derivatives(k) = 1;
}

ValueWithDerivatives2::ValueWithDerivatives2(const ValueWithDerivatives2& vwd)
   : PS(vwd.PS), Value(vwd.Value), Derivatives(vwd.Derivatives),
      SecondDerivatives(vwd.SecondDerivatives) { REPORT }

void ValueWithDerivatives2::operator=(const ValueWithDerivatives2& vwd)
{
   REPORT
   PS = vwd.PS; Value = vwd.Value; Derivatives = vwd.Derivatives;
   SecondDerivatives = vwd.SecondDerivatives;
}

void ValueWithDerivatives2::operator+=(const ValueWithDerivatives2& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Value += vwd.Value; Derivatives += vwd.Derivatives;
   SecondDerivatives += vwd.SecondDerivatives;
}

void ValueWithDerivatives2::operator-=(const ValueWithDerivatives2& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Value -= vwd.Value; Derivatives -= vwd.Derivatives;
   SecondDerivatives -= vwd.SecondDerivatives;
}

void ValueWithDerivatives2::operator*=(const ValueWithDerivatives2& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Matrix OP = Derivatives.t() * vwd.Derivatives;
   Derivatives = vwd.Value * Derivatives + Value * vwd.Derivatives;
   SecondDerivatives = vwd.Value * SecondDerivatives
      + OP + OP.t()
      + Value * vwd.SecondDerivatives;
   Value *= vwd.Value;
}

void ValueWithDerivatives2::operator/=(const ValueWithDerivatives2& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Matrix OP = Derivatives.t() * vwd.Derivatives;
   OP += OP.t() + Value * vwd.SecondDerivatives;
   Value /= vwd.Value;
   OP = 2.0 * Value * vwd.Derivatives.t() * vwd.Derivatives - OP;
   Derivatives = (Derivatives - Value  * vwd.Derivatives) / vwd.Value;
   SecondDerivatives = (SecondDerivatives + OP / vwd.Value) / vwd.Value;
}

void ValueWithDerivatives2::operator+=(Real r) { REPORT  Value += r; }

void ValueWithDerivatives2::operator-=(Real r) { REPORT  Value -= r; }

void ValueWithDerivatives2::operator*=(Real r)
   { REPORT  Value *= r; Derivatives *= r; SecondDerivatives *= r; }

void ValueWithDerivatives2::operator/=(Real r)
   { REPORT  Value /= r; Derivatives /= r; SecondDerivatives /= r; }

ValueWithDerivatives2 ValueWithDerivatives2::operator-() const
{
   REPORT
   return ValueWithDerivatives2(PS, -Value, -Derivatives, -SecondDerivatives);
}

ValueWithDerivatives2 operator+(const ValueWithDerivatives2& vwd1,
   const ValueWithDerivatives2& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   return ValueWithDerivatives2(vwd1.PS, vwd1.Value + vwd2.Value,
      vwd1.Derivatives + vwd2.Derivatives,
      vwd1.SecondDerivatives + vwd2.SecondDerivatives);
}

ValueWithDerivatives2 operator-(const ValueWithDerivatives2& vwd1,
   const ValueWithDerivatives2& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   return ValueWithDerivatives2(vwd1.PS, vwd1.Value - vwd2.Value,
      vwd1.Derivatives - vwd2.Derivatives,
      vwd1.SecondDerivatives - vwd2.SecondDerivatives);
}

ValueWithDerivatives2 operator*(const ValueWithDerivatives2& vwd1,
   const ValueWithDerivatives2& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   Matrix OP = vwd1.Derivatives.t() * vwd2.Derivatives;
   return ValueWithDerivatives2(vwd1.PS, vwd1.Value * vwd2.Value,
      vwd2.Value * vwd1.Derivatives + vwd1.Value * vwd2.Derivatives,
      vwd2.Value * vwd1.SecondDerivatives
      + OP + OP.t()
      + vwd1.Value * vwd2.SecondDerivatives);
}

ValueWithDerivatives2 operator/(const ValueWithDerivatives2& vwd1,
   const ValueWithDerivatives2& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   Matrix OP = vwd1.Derivatives.t() * vwd2.Derivatives;
   OP += OP.t() + vwd1.Value * vwd2.SecondDerivatives;
   Real v = vwd1.Value / vwd2.Value;
   OP = 2.0 * v * vwd2.Derivatives.t() * vwd2.Derivatives - OP;
   return ValueWithDerivatives2( vwd1.PS, v,
      (vwd1.Derivatives - v * vwd2.Derivatives) / vwd2.Value,
      (vwd1.SecondDerivatives + OP / vwd2.Value) / vwd2.Value );
}

ValueWithDerivatives2 pow(const ValueWithDerivatives2& vwd1,
   const ValueWithDerivatives2& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   Real v = pow(vwd1.Value, vwd2.Value);
   Real logv1 = log(vwd1.Value); Real r = (vwd2.Value / vwd1.Value);
   RowVector D1 = (r * vwd1.Derivatives + logv1 * vwd2.Derivatives);
   Matrix D2 = vwd1.Derivatives.t() * vwd2.Derivatives;
   D2 = (D2 + D2.t() - r * vwd1.Derivatives.t() * vwd1.Derivatives)
      / vwd1.Value
      + logv1 * vwd2.SecondDerivatives + r * vwd1.SecondDerivatives
      + D1.t() * D1;
   return ValueWithDerivatives2( vwd1.PS, v, v * D1, v * D2 );
}

ValueWithDerivatives2 operator+(Real r, const ValueWithDerivatives2& vwd)
{
   REPORT
   return ValueWithDerivatives2(vwd.PS, r + vwd.Value, vwd.Derivatives,
      vwd.SecondDerivatives);
}

ValueWithDerivatives2 operator-(Real r, const ValueWithDerivatives2& vwd)
{
   REPORT
   return ValueWithDerivatives2(vwd.PS, r - vwd.Value, - vwd.Derivatives,
       - vwd.SecondDerivatives);
}

ValueWithDerivatives2 operator*(Real r, const ValueWithDerivatives2& vwd)
{
   REPORT
   return ValueWithDerivatives2(vwd.PS, r * vwd.Value, r * vwd.Derivatives,
      r * vwd.SecondDerivatives);
}


ValueWithDerivatives2 operator/(Real r, const ValueWithDerivatives2& vwd)
{
   REPORT
   Real v = r / vwd.Value;
   return ValueWithDerivatives2(vwd.PS, v, (- v / vwd.Value) * vwd.Derivatives,
      (2.0 / vwd.Value * vwd.Derivatives.t() * vwd.Derivatives
         - vwd.SecondDerivatives) * r / square(vwd.Value) );
}

ValueWithDerivatives2 pow(Real r, const ValueWithDerivatives2& vwd)
{
   REPORT
   Real v = pow(r, vwd.Value); Real logr = log(r); Real vlogr = v * logr;
   return ValueWithDerivatives2( vwd.PS, v, vlogr * vwd.Derivatives,
      vlogr * vwd.SecondDerivatives
      + vlogr * logr * vwd.Derivatives.t() * vwd.Derivatives );
}

ValueWithDerivatives2 operator+(const ValueWithDerivatives2& vwd, Real r)
{
   REPORT
   return ValueWithDerivatives2(vwd.PS, vwd.Value + r, vwd.Derivatives,
      vwd.SecondDerivatives);
}

ValueWithDerivatives2 operator-(const ValueWithDerivatives2& vwd, Real r)
{
   REPORT
   return ValueWithDerivatives2(vwd.PS, vwd.Value - r, vwd.Derivatives,
      vwd.SecondDerivatives);
}

ValueWithDerivatives2 operator*(const ValueWithDerivatives2& vwd, Real r)
{
   REPORT
   return ValueWithDerivatives2(vwd.PS, vwd.Value * r, vwd.Derivatives * r,
      vwd.SecondDerivatives * r);
}

ValueWithDerivatives2 operator/(const ValueWithDerivatives2& vwd, Real r)
{
   REPORT
   return ValueWithDerivatives2(vwd.PS, vwd.Value / r, vwd.Derivatives / r,
      vwd.SecondDerivatives / r);
}


ValueWithDerivatives2 pow(const ValueWithDerivatives2& vwd, Real r)
{
   REPORT
   Real v0 = pow(vwd.Value, r - 2);
   Real v1 = v0 * vwd.Value;  Real v2 = v1 * vwd.Value;  v1 *= r;
   return
      ValueWithDerivatives2( vwd.PS, v2, v1 * vwd.Derivatives,
      v1 * vwd.SecondDerivatives
      + r * (r - 1) * v0 * vwd.Derivatives.t() * vwd.Derivatives );
}

ValueWithDerivatives2 pow(const ValueWithDerivatives2& vwd, int r)
{
   if (r == 0) { REPORT  return ValueWithDerivatives2(vwd.PS) + 1.0; }
   if (r == 1) { REPORT  return vwd; }
   REPORT
   Real v0 = pow(vwd.Value, r - 2);
   Real v1 = v0 * vwd.Value;  Real v2 = v1 * vwd.Value;  v1 *= r;
   return
      ValueWithDerivatives2( vwd.PS, v2, v1 * vwd.Derivatives,
      v1 * vwd.SecondDerivatives
      + r * (r - 1) * v0 * vwd.Derivatives.t() * vwd.Derivatives );
}

ValueWithDerivatives2 exp(const ValueWithDerivatives2& vwd)
{
   REPORT
   Real ev = exp(vwd.Value);
   return ValueWithDerivatives2( vwd.PS, ev, ev * vwd.Derivatives,
      ev * (vwd.SecondDerivatives + vwd.Derivatives.t() * vwd.Derivatives) );
}

ValueWithDerivatives2 log(const ValueWithDerivatives2& vwd)
{
   REPORT
   return ValueWithDerivatives2(vwd.PS, log(vwd.Value),
      vwd.Derivatives / vwd.Value,
      vwd.SecondDerivatives / vwd.Value
         - vwd.Derivatives.t() * (vwd.Derivatives / square(vwd.Value)) );
}

ValueWithDerivatives2 sin(const ValueWithDerivatives2& vwd)
{
   REPORT
   Real sv = sin(vwd.Value); Real cv = cos(vwd.Value);
   return ValueWithDerivatives2( vwd.PS, sv, cv * vwd.Derivatives,
      cv * vwd.SecondDerivatives - sv * vwd.Derivatives.t() * vwd.Derivatives );
}

ValueWithDerivatives2 cos(const ValueWithDerivatives2& vwd)
{
   REPORT
   Real sv = sin(vwd.Value); Real cv = cos(vwd.Value);
   return ValueWithDerivatives2(vwd.PS, cv, -sv * vwd.Derivatives,
      -sv * vwd.SecondDerivatives
      - cv * vwd.Derivatives.t() * vwd.Derivatives);
}

ValueWithDerivatives2 tan(const ValueWithDerivatives2& vwd)
{
   REPORT
   Real tv = tan(vwd.Value); Real cv2 = square(cos(vwd.Value));
   return ValueWithDerivatives2(vwd.PS, tan(vwd.Value),
      vwd.Derivatives / cv2,
      vwd.SecondDerivatives / cv2
      + 2.0 * tv / cv2 * vwd.Derivatives.t() * vwd.Derivatives );
}

#ifdef WANT_ERF
// code provided by Tomas Gonzalez-Llarena
// assumes erf(double) is available as a library function
ValueWithDerivatives2 erf(const ValueWithDerivatives2& vwd)
{ 
  REPORT
  Real e2 = exp(-square(vwd.Value)); 
  return ValueWithDerivatives2(vwd.PS, erf(vwd.Value),
     2.0 * vwd.Derivatives * e2 / sqrt(M_PI),
     2.0 * (vwd.SecondDerivatives
        - 2.0 * (vwd.Derivatives.t() * vwd.Derivatives ) * vwd.Value)
        * e2 / sqrt(M_PI) );
} 
#endif


// Gaussian numerical integration with 32 terms
ValueWithDerivatives2
   GaussianIntegration32(VWD2OfReal& f, Real Lower, Real Upper)
{
   REPORT
   double x[] =
   {
      0.048307665687738316235,
      0.144471961582796493485,
      0.239287362252137074545,
      0.331868602282127649780,
      0.421351276130635345364,
      0.506899908932229390024,
      0.587715757240762329041,
      0.663044266930215200975,
      0.732182118740289680387,
      0.794483795967942406963,
      0.849367613732569970134,
      0.896321155766052123965,
      0.934906075937739689171,
      0.964762255587506430774,
      0.985611511545268335400,
      0.997263861849481563545
   };

   double w[] =
   {
      0.096540088514727800567,
      0.095638720079274859419,
      0.093844399080804565639,
      0.091173878695763884713,
      0.087652093004403811143,
      0.083311924226946755222,
      0.078193895787070306472,
      0.072345794108848506225,
      0.065822222776361846838,
      0.058684093478535547145,
      0.050998059262376176196,
      0.042835898022226680657,
      0.034273862913021433103,
      0.025392065309262059456,
      0.016274394730905670605,
      0.007018610009470096600
   };
   Real delta = 0.5 * (Upper - Lower); Real centre = 0.5 * (Upper + Lower);
   Real dev = delta * x[0];
   ValueWithDerivatives2 value = w[0] * (f(centre + dev) + f(centre - dev));

   for (int i = 1; i < 16; i++)
   {
      dev = delta * x[i];
      value += w[i] * (f(centre + dev) + f(centre - dev));
   }
   return value * delta;
}


