// value with derivatives

// The "Value" is supposed to be a function of paramters t_1,...,t_n
// This structure stores the value of the "Value" and its n derivatives.

//#define WANT_ERF                         // activate if you want erf
                                         // you'll need erf in your library
#include "newmat.h"                      // newmat functions
#include "str.h"                         // string class


// **************** the parameter set manipulation classes *******************

// ParameterSet PS();                   start a new parameter set
// int i = PS.AddParameter("name");     add a new parameter called "name"
//                                      check it for uniqueness
//                                      return its index
// int i = PS.LocateParameter("name");  find index number (1,...,n) of "name"
//                                      return 0 if not present
// String name = PS(i);                 return name with index i
// int n = PS.Size();                   return number of parameters
// bool f = PS.Frozen();                return true if number of parameters
//                                      is "frozen" - not working yet
// bool e = PS1 == PS2;                 return true if pointing to same
//                                      ParameterSetClass
// bool e = PS1 != PS2;                 return true if not pointing to same
//                                      ParameterSetClass
// AssertEqual(PS1, PS2);               throw an exception if not pointing
//                                      to same underlying class


// a list of strings

class NameSeq
{
public:
   NameSeq* Next;
   String Name;
   NameSeq(const String& name) : Next(0), Name(name) {}
};

// the real data for the ParameterSet

class ParameterSetClass
{
   bool Frozen;
   long RefCount;
   int n;
   NameSeq* Next;
   int last_i;                          // details of last name searched for
   NameSeq* last_seq;
   ParameterSetClass()
      : Frozen(false), RefCount(0), n(0), Next(0), last_i(0) {}
   ~ParameterSetClass() { CleanUp(); }
   void CleanUp();
   void IncrRef() { RefCount++; }
   void DecrRef();
   int AddParameter(const String& name);
   int LocateParameter(const String& name) const;
   String& operator()(int i);
   friend class ParameterSet;
};

// the parameter set as seen by the user

class ParameterSet
{
   ParameterSetClass* PSC;
public:
   ParameterSet();
   ParameterSet(const ParameterSet& ps) : PSC(ps.PSC) { PSC->IncrRef(); }
   ~ParameterSet() { PSC->DecrRef(); }
   void operator=(const ParameterSet& ps);
   int AddParameter(const String& name) { return PSC->AddParameter(name); }
   int LocateParameter(const String& name) const
      { return PSC->LocateParameter(name); }
   String& operator()(int i) const { return PSC->operator()(i); }
   int Size() const { return PSC->n; }
   bool Frozen() const { return PSC->Frozen; }
   friend bool operator==(const ParameterSet& ps1, const ParameterSet& ps2)
      { return ps1.PSC == ps2.PSC; }
   friend bool operator!=(const ParameterSet& ps1, const ParameterSet& ps2)
      { return ps1.PSC != ps2.PSC; }
   friend void AssertEqual(const ParameterSet& ps1, const ParameterSet& ps2);
};

// ********************** Value with Derviatives classes *********************

class ValueWithDerivatives
{
   Real Value;
   RowVector Derivatives;
   ParameterSet PS;
public:
   ValueWithDerivatives(const ParameterSet& ps);
   ValueWithDerivatives(const ParameterSet& ps, Real v, const RowVector& d);
   ValueWithDerivatives(const ParameterSet& ps, Real v, int k);
   ValueWithDerivatives(const ParameterSet& ps, Real v, const String& name);
   ValueWithDerivatives(const ValueWithDerivatives&);
   ~ValueWithDerivatives() {}
   Real GetValue() const { return Value; }
   ReturnMatrix GetDerivatives() const { return Derivatives; }
   Real GetDerivative(int i) const { return Derivatives(i); }
   ParameterSet GetParameterSet() { return PS; }
   void operator=(const ValueWithDerivatives&);
   void operator+=(const ValueWithDerivatives&);
   void operator-=(const ValueWithDerivatives&);
   void operator*=(const ValueWithDerivatives&);
   void operator/=(const ValueWithDerivatives&);
   void operator+=(Real);
   void operator-=(Real);
   void operator*=(Real);
   void operator/=(Real);
   ValueWithDerivatives operator-() const;
   friend ValueWithDerivatives operator+(const ValueWithDerivatives&,
      const ValueWithDerivatives&);
   friend ValueWithDerivatives operator-(const ValueWithDerivatives&,
      const ValueWithDerivatives&);
   friend ValueWithDerivatives operator*(const ValueWithDerivatives&,
      const ValueWithDerivatives&);
   friend ValueWithDerivatives operator/(const ValueWithDerivatives&,
      const ValueWithDerivatives&);
   friend ValueWithDerivatives pow(const ValueWithDerivatives&,
      const ValueWithDerivatives&);
   friend ValueWithDerivatives operator+(Real, const ValueWithDerivatives&);
   friend ValueWithDerivatives operator-(Real, const ValueWithDerivatives&);
   friend ValueWithDerivatives operator*(Real, const ValueWithDerivatives&);
   friend ValueWithDerivatives operator/(Real, const ValueWithDerivatives&);
   friend ValueWithDerivatives pow(Real, const ValueWithDerivatives&);
   friend ValueWithDerivatives operator+(const ValueWithDerivatives&, Real);
   friend ValueWithDerivatives operator-(const ValueWithDerivatives&, Real);
   friend ValueWithDerivatives operator*(const ValueWithDerivatives&, Real);
   friend ValueWithDerivatives operator/(const ValueWithDerivatives&, Real);
   friend ValueWithDerivatives pow(const ValueWithDerivatives&, Real);
   friend ValueWithDerivatives pow(const ValueWithDerivatives&, int);
   friend ValueWithDerivatives exp(const ValueWithDerivatives&);
   friend ValueWithDerivatives log(const ValueWithDerivatives&);
   friend ValueWithDerivatives sin(const ValueWithDerivatives&);
   friend ValueWithDerivatives cos(const ValueWithDerivatives&);
   friend ValueWithDerivatives tan(const ValueWithDerivatives&);
#ifdef WANT_ERF
   friend ValueWithDerivatives erf(const ValueWithDerivatives&);
#endif
};

// prototype of VWD which is a function of a Real

class VWDOfReal
{
protected:
   Real x;                             // Current x value
   bool xSet;                          // true if a value assigned to x

public:
   virtual void Set(Real X) { x = X; xSet = true; }
                                       // set x
   VWDOfReal() : xSet(false) {}
   virtual ValueWithDerivatives operator()() = 0;
                                       // function value at current x
                                       // set current x
   ValueWithDerivatives operator()(Real X) { Set(X); return operator()(); }
                                       // set x, return value
};

// Gaussian numerical integration
ValueWithDerivatives
   GaussianIntegration32(VWDOfReal& function, Real Lower, Real Upper);

// ***** Value with Derviatives classes - first and second derivatives *****

class ValueWithDerivatives2
{
   Real Value;
   RowVector Derivatives;
   Matrix SecondDerivatives;
   ParameterSet PS;
public:
   ValueWithDerivatives2(const ParameterSet& ps);
   ValueWithDerivatives2(const ParameterSet& ps, Real v, const RowVector& d,
      const Matrix& d2);
   ValueWithDerivatives2(const ParameterSet& ps, Real v, int k);
   ValueWithDerivatives2(const ParameterSet& ps, Real v, const String& name);
   ValueWithDerivatives2(const ValueWithDerivatives2&);
   ~ValueWithDerivatives2() {}
   Real GetValue() const { return Value; }
   ReturnMatrix GetDerivatives() const { return Derivatives; }
   ReturnMatrix GetSecondDerivatives() const { return SecondDerivatives; }
   Real GetDerivative(int i) const { return Derivatives(i); }
   Real GetSecondDerivative(int i, int j) const
      { return SecondDerivatives(i, j); }
   ParameterSet GetParameterSet() { return PS; }
   void operator=(const ValueWithDerivatives2&);
   void operator+=(const ValueWithDerivatives2&);
   void operator-=(const ValueWithDerivatives2&);
   void operator*=(const ValueWithDerivatives2&);
   void operator/=(const ValueWithDerivatives2&);
   void operator+=(Real);
   void operator-=(Real);
   void operator*=(Real);
   void operator/=(Real);
   ValueWithDerivatives2 operator-() const;
   friend ValueWithDerivatives2 operator+(const ValueWithDerivatives2&,
      const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 operator-(const ValueWithDerivatives2&,
      const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 operator*(const ValueWithDerivatives2&,
      const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 operator/(const ValueWithDerivatives2&,
      const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 pow(const ValueWithDerivatives2&,
      const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 operator+(Real, const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 operator-(Real, const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 operator*(Real, const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 operator/(Real, const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 pow(Real, const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 operator+(const ValueWithDerivatives2&, Real);
   friend ValueWithDerivatives2 operator-(const ValueWithDerivatives2&, Real);
   friend ValueWithDerivatives2 operator*(const ValueWithDerivatives2&, Real);
   friend ValueWithDerivatives2 operator/(const ValueWithDerivatives2&, Real);
   friend ValueWithDerivatives2 pow(const ValueWithDerivatives2&, Real);
   friend ValueWithDerivatives2 pow(const ValueWithDerivatives2&, int);
   friend ValueWithDerivatives2 exp(const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 log(const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 sin(const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 cos(const ValueWithDerivatives2&);
   friend ValueWithDerivatives2 tan(const ValueWithDerivatives2&);
#ifdef WANT_ERF
   friend ValueWithDerivatives2 erf(const ValueWithDerivatives2&);
#endif
};

// prototype of VWD which is a function of a Real

class VWD2OfReal
{
protected:
   Real x;                             // Current x value
   bool xSet;                          // true if a value assigned to x

public:
   virtual void Set(Real X) { x = X; xSet = true; }
                                       // set x
   VWD2OfReal() : xSet(false) {}
   virtual ValueWithDerivatives2 operator()() = 0;
                                       // function value at current x
                                       // set current x
   ValueWithDerivatives2 operator()(Real X) { Set(X); return operator()(); }
                                       // set x, return value
};

// Gaussian numerical integration
ValueWithDerivatives2
   GaussianIntegration32(VWD2OfReal& function, Real Lower, Real Upper);


