/* miniproc.c */
/* SLICEDICE allows miniproc to cut its own source up into separate files
#__SLICEDICE "history.txt"
*/

/* miniproc.c

David Mathog, Biology Division, Caltech

Copyright 1997, 1998, 1999, 2000 by David Mathog and California Instititute of Technology.

This software may be used freely, but may not be redistributed.  
Distributions may be obtained from:

   http://seqaxp.bio.caltech.edu/www/miniproc.html

You may modify this sofware for your own use, but you may not incorporate
any part of the original code into any other piece of software which will
then be distributed (whether free or commercial) unless prior written
consent is obtained.  For more information, or to report bugs, contact: 

   mathog@seqaxp.bio.caltech.edu

For more information, see miniproc.doc

Revision history

3.04 04-JAN-2000
     Added: {{}} as alternative substitution operator.  Cannot send 
            anything with  <<>> even temporarily out to a web browser as it
            would substitute and couldn't be read back later.
     Fixed: Memory access violation - showed up only in getenv function on 
            Solaris.
3.03 20-OCT-1999
     Fixed:  Failed to initialize string arrays with '\0' strings which
               caused strange errors if they were later accessed without
               having a value assigned to them.
             Really dumb error which made [] calculations run inside
               if blocks in locations where they should have been skipped.
     Added:  .constant.  RPN operator.  Load the value of one of a 
             predefined set of integer constants onto the stack.  Constants
             are the bitmaps used for "set" find operations and
             "safety" mask, also MAXINLINE,MAXSUBLINE.  Fatal error
             if the constant doesn't exist.  Match is not case sensitive.
             [ 'letters' 'control' .constant_. ] = [ 1 8 ]
             ISDIR detection in file_info
             F$DIR_LIST - lists files in a directory

     KNOWN BUG:
             F$DIR_LIST can change the type on ANY variable, so if somebody
             feeds it SAFETY or TRACE the program will likely crash.  Need
             a general priv/protection scheme to handle this.

3.02 28-JUN-1999
     Fixed:  f$macro_return or _break called outside a macro crashed program
     Changed:RPN operator look up method.  Previously this was done by 
             scanning down the list of all operators.  That gets pretty 
             slow when > 50 operators are present.  New method is to
             presort the list at program initialization, and then do
             a binary search to find the operator.

             Removed hardcoding for MC1,MC2, etc.  Now created at run time
               based on the value of MPC_TUPLE.  Same deal for P0->P9,
               set by MPC_PNUM_MAX.

             "" are "external quotes".  Usage was made uniform in that
             they now work the same everywhere.  This will break existing
             scripts which use them in RPN expressions, but is otherwise
             more logical given the introduction of internal quotes (below.)
     Added:  internal quotes 'foo bar'.  These MAY be used in RPN 
               expressions and elsewhere.
             trailing comments, delimited by a "!" at the end of the input 
             line (or before the "-" in lines to be continued) and ending
             at the first ! left of that.  These may not include a "!"
             within them.
                command line   !this is a comment!
                command line   !this is a comment!-
                 continued     !more comments    !-
                 continuted
 
             bitwise integer operators.  [ var1 var2 .bit_and_. ]
                This may not be totally portable.  Before each operation
                the double stack variable is converted to an int, and
                then at the end it is converted back to a double. On
                some platforms Int may have more precision than a double
                can hold.
             f$macro_map.   Map the dimension of one array onto one or more
                macros.  f$macro_map array macro macro macro ...
             f$macro_body.  Puts a mark inside a macro so that code above 
               is executed only when indices are 1,1,1.  Like the first
               clause in a C for() statement.  Not required.  Without it 
               all code executes each time.
             arrays.  3 indices set by new .(). operator, created by
                p = 1       creates an integer variable
                [ &p 5 6 7 .(dim). ]  -> p is becomes matrix of size 5,6,7
                matrix indices go from 1 to dimension. (NOT 0 to, like C).
                "biggest" index is the leftmost, "smallest" is the rightmost.
                [ &p 1 2 1 .(). ]  refers to the element
                p(1,2,1)   = array element 1,2,1

                HOWEVER, be VERY careful with MC1 MC2 MC3 as
                1 0 0 are legal for these if MC2MAX,MC3MAX are both 0, but 
                1 0 0 is an ILLEGAL array index. If indices
                are to be set from Macro Counters, be sure to set the
                repeat count on the macro to 2,1,1 and not 2,0,0.

                array_var1 = array_var2

                Copies the ACTIVE CELL value from the second array to 
                the ACTIVE CELL of the first array.  One cell is always
                active, when first created, it is the first cell.  If
                the left side of this equation is a new variable, it
                is created, but it is not an array.
              

             Operators that act on double arrays.  Note that arrays can only be manipulated by 
                variable name on the stack, as arrays may not be stored in 
                the stack.  Also the result of array opertations are NOT
                stored in RESULT, but the value will be set to true or
                false, depending on whether a math error was detected or 
                not.

                .(+). , .(add).       add two or more same sized double matrices,
                  the resultant matrix must already exist and be 
                  predimensioned correctly.  One parameter, 1 or more operands.
                  [ "offsets" "moreoffsets" "addresult" .(+)_. ]
                  [ "offsets" "moreoffsets" "addresult" .(+)_. ]
                .(*). , .(multiply).  multiply two or more compatible sized double matrices
                  Multiplication is left to right, and the product of each 
                  operation must be compatible with the next.
                  One parameter (the output file), 2 or more operands.
                  [ "left" "center" "right" "multresult" .(*)_. ]
                .(scale).         multiply every element by a constant
                  Matrices are created with all zero values. Use .(scale). to
                  zero one which held nonzero values.
                  One parameter, 1 or more operands.
                .(offset).        add a constant to every element
                  Matrices are created with all zero values. Use .(offset). to
                  create a unit matrix.
                  One parameter, 1 or more operands.

                  [ "mat1" "mat2" "mat3" 0.5 .(scale)_. ]
                  [ "mat1" "mat2" "mat3" 200.1 .(offset)_. ]

             Operators which work on INTVAR,DBLVAR, or STRINGVAR matrices
                .(sort).         sort along ONE index of an array
                                 when sorting, the whole "plane" 
                                 perpindicular to the selected index axis 
                                 is swapped.  (Think of it as rotating a 
                                 cube then sorting the horizontal slices
                                 by a value along one column.)
                                 One parameter, 1 or more operands.
                  [ "mat1" "mat2" "mat3"  3 .(sort)_. ] 
                     sort on 3rd index (biggest) into ascending order
                  [ "mat1" "mat2" "mat3" -2 .(sort)_. ]
                     sort on 2nd index (smallest) into descending order
                  
                 .(showdim).     Leaves the array dimensions on the RPN stack,
                                 NOT including the zeroed higher order elements.
                                 Since the number of indices can vary, it 
                                 should not generally be run on multiple arrays at once.
                                 That is:
                  [ &p 5 6 .(dim). &p .(showdim). ]  -> [ 5 6 ]
                  

             f$macro_map array macro1 macro2 macro3...
               Like f$macro_repeat, but takes the dimensions of the array
               and sets them as the repeat count on the macro(s).
3.01 30-APR-1999
     Fixed:  Removed extra WRITE_DEBUG in exit code.
     Changed:
             AltPrefix handling.  Default is now that altprefix is #__,
             like the regular command prefix.  Setting it to '\0' makes all
             lines into commands (no need for a prefix). BUT remember that
             the test order is FIRST check the regular prefix, THEN altprefix.
3.00 15-JAN-1999
     Too many changes for a simple revision, do a full version number change.
     Changed all function names to mpcf_ prefix.
     Added:  setdepth      set the depth of an rpn operator (note, 0 means ALL)
             debug         write string(s) to the debug device
     Fixed:  RPNOPER variable used in an [] expression, picked wrong operator.
2.04 15-DEC-1998
     Added: Memory control - variables may now have different persistences
        and may be removed.  Same for Macros.  This is needed so that 
        miniproc may be embedded into another application.
        Ability to embed miniproc in another program, including
          SLAVE_MACRO, a way for the calling program to pass commands
          into miniproc not involving a file.
        Added stacksize (put size of rpn stack onto stack).
     Code Revision:  Rearranged VARIABLE structure to break out the
        (many) extra pieces for a Macro into a separate type/variable.
        Limited output of each debugging line to 4096 bytes and put in size
          checking so that this isn't exceeded.
     Fixed:  swap (range=1, operand >> 1, would erroneously swap past
          range limit down stack.)
2.03 11-DEC-1998
     Added: array  loads a set of names like name[1] to name[n] onto the stack.
            [ &name 3 .array. ] -> [ &name[3] &name[2] &name[1] ]
     Fixed:  element, elements

2.02 26-MAY-1998
     Fixed bug (extra %s in two fprintf lines), thanks: Evan.Antworth@SIL.ORG

2.01 16-MAY-1998
     Fixed bug.  f$out name number, without a disposition was leaving disposition
        undefined.
     Added:  load, store   rpn stack <-> named storage operators
             elements      break out many operands in one call -> named storage
             scale,offset  multiply,add a constant to many RPN stack locations
2.00 15-APR-1998.
     Added DBLVAR data type (double float).
     Added #__ [ ... ] result result result reverse polish built in
       calculator (easier to use than f$evaluate for complex expressions).
       Examples:
       [ 10.1 20 ] b a             b=20.0, a=10.1 (note stack unloading order).
       [ 1000 100 10 1 .add.] a    a = 11         (+ may be used instead of add)
       [ 1000 100 10 1 .add_.] a    a = 1111
       [ 1000 100 10 1 .add_3.] a    a = 111
       [ a .SIN. b .COS. .*. 2 .*.] d      d = 2*sin(a)*cos(b)
       [ a b .AND. c d .AND. .OR. ]      RESULT = (a AND b) OR (c AND d), float!= is true

       RESTRICTION:  RPN operators may be stored in strings, but will not
       be interpreted as operators unless a subsitition is forced:
         anopt = ".+."
         [1 2 anopt] a            <-  this will generate an error
            but
         [1 2 <<anopt>>] a        <-  this will work as expected
     Added formatted output for floats
     Added "deprecate" special variable.  Used for checking scripts
       for deprecated features, which will be removed at some point.
       0:  deprecated code works as before (default)
       1:  deprecated code use generates a warning
       2:  deprecated code use generates a fatal error
     Added variable convertwidth.  Specifies the minimum size for d->s and i->s
       output strings.  This is needed because a user might want to 
       repeatedly stuff a value into a very long string, which could then 
       become even longer, and the program has no way of knowing how big
       an area to set aside unless it is told.  Output strings will always 
       be converted to at least this size before a d->s or i->s operation.
       The default is 32.
     Fixed f$date, which wasn't recognized as a date.
1.08 17-FEB-1998.
     Added P0, the number of arguments passed to a macro.
     Modified F$TYPE.  It now returns a 4 for a zero length
       string variable.
     Fixed bug - eliminate, retain, stringdel were not setting
       STATUS to 2 on changes.
     Fixed bug - when calling macro state of Pn variables after
       those intentionally passed was indeterminate.  Now they
       are set to zero length strings (STRINGVAR).
     Fixed bug - divide wasn't working
     Fixed bug - modulo wasn't working
     Fixed bug - f$macro_break always returned status 1 instead of supplied 
          status.
1.07 02-FEB-1998.
     Added getenv function.  Some operating systems may not support it, in
       which case it returns a FALSE status.
     Added option to allow MPC_MAXINLINE to be set at compilation, and
       changed default to 32768.
     Added ALTPREFIX mechanism/symbol, so that scripts can run with
       a prefix different from "#__".  Note that #__ is ALWAYS active,
       but a second prefix can now be enabled as well.
     Modified f$out syntax, so that either new,append,"new","append", or
       a symbol holding new or append could be used.
     Fixed bug in endif implementation - valid statements following and
       endif were not executed if within another logical block.
     Fixed EXIT_FAILURE versus FAILURE, now consistently uses EXIT_FAILURE.
     Fixed bug.  continuation lines read from macros were not working.
     Fixed bug.  if/elseif/else/endif labels were bleeding down into other
       modules so that labels could not be reused.
     Fixed bug.  Zero length passthrough strings ( #__"") were not working.
       These now correctly output blank lines.
1.06 22-DEC-1997.
     Fixed bug.  Line continuation - buffer was not being reset so that
       if the first line of a continuation was shorter than the preceding
       input, the tail of the preceding would be inserted.
     Fixed bug.  *var was not working - at all.
     Fixed bug.  Assignment of local variables when passed to a macro
       was happening AFTER the macro name changed, so that the name
       lookup would fail.
1.05 17-DEC-1997.
     Fixed bug.  Setting an existing string variable to a null string
       left it as a NULL, rather than a pointer to the character '\0'.
1.04 09-DEC-1997.
     Added f$break.  This is an f$exit that can be used within an
       if/elseif/else/endif structure, but ONLY inside an input file.  That
       is, you cannot exit a macro through one, only an f$in file.
     Fixed bug in error reporting for end of file encountered (wrong file 
       name was returned.)
1.03 03-DEC-1997.
     Modified if logic so that if a if f$(function) works for all functions.
       (If f$evaluate returns with status 0 though, it is a fatal error.)
     Modified f$<- to return value in RESULT instead of P9.
     Modified local variables to work within macros (that is, local to
       macro, previously it was local to input file.)
     Fixed bug, f$macro_return always returned 1
1.02 01-DEC-1997.  Fixed bug in finr[] access, array was 9 elements, but
        accessed by 11-19 index.  Now it works on Solaris. (Thanks Pete.Lee@decus.ch)
     Added :name variables, that is, "local" variables and macros.
     Added #__"whatever" and #__&whatever processing. 
     Added f$<- operator.
     Added buffering in f$evaluate, so that variables may be both
       result and operand.
     Added ifnot, elseifnot
1.01 28-NOV-1997.  Fixed bug in f$macro_repeat - incorrect handling of
       less than 3 repeat count values. 
     Fixed bug in f$evaluate for lexhigh,lexlow - incorrect clean up
       code for extending result string.
     Added f$evaluate resize.
1.0  28-OCT-1997, first release version

*/

/*
#__SLICEDICE "miniproc.h"
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <errno.h>
#include <ctype.h>                    /* for toupper, tolower */
#include <sys/stat.h>                 /* no sys needed for openvms, but others are picky */
#include <setjmp.h>                   /* for setjmp,longjmp */
#include <limits.h>                   /* for UINT_MAX, INT_MAX */
#include <dirent.h>                   /* for opendir,readdir,closedir */

/* make the variables and tests global */

/* all #defines */

/* WRITE_DEBUG() and WRITE_UNCRECOVERABLE()  must be custom routines
   when Miniproc is used in an 
   embedded environment.  A conditional include for "custom_mpc_dbg.h"
   is provided here so that it can be accessed */

#define mpc_max(a,b) (a > b) ? a : b
#define mpc_min(a,b) (a < b) ? a : b
#ifndef _EMBEDDED_MINIPROC_
#define WRITE_DEBUG() (void) fprintf(stderr,"%s\n",mpc_dbg_out)
#define WRITE_UNRECOVERABLE() WRITE_DEBUG()
#else  /* _EMBEDDED_MINIPROC_ */
#include "custom_mpc_dbg.h"
#endif /* _EMBEDDED_MINIPROC_ */
#define MPC_BAILOUT(a) (void) longjmp(mpc_env, a)
#define MPC_WRITE_AND_BAILOUT(a) WRITE_DEBUG(); (void) longjmp(mpc_env, a)


#define MPC_VERSION_INFO    "miniproc Version 3.04 - 04-JAN-2000"
#define MPC_TUPLE            3    /* size of arrays and macro counters */
#define MPC_PNUM_MAX         9    /* P0->P9, this sets "9" */
#define MPC_RPNARGS          3    /* maximum number of arguments for an RPN 
                                     function, but they can have many, many 
                                     operands. */
#define MPC_STREAMS         10    /* maximum number of input streams, that 
                                     is, maximum depth of files calling files */
#define MPC_FILE_OLOW        0    /* define range of output file id's */
#define MPC_FILE_OHIGH       9    /* define range of output file id's */
#define MPC_FILE_OSIZE       (MPC_FILE_OHIGH - MPC_FILE_OLOW + 1)
#define MPC_FILE_ILOW       10    /* define range of input file id's */
#define MPC_FILE_INORMAL    11    /* first input file is peculiar, normal ones start here */
#define MPC_FILE_IHIGH      19    /* define range of input file id's */
#define MPC_FILE_ISIZE       (MPC_FILE_IHIGH - MPC_FILE_ILOW + 1)
#define MPC_HASHSPACE     4096    /* hash size for variable access */
#define MPC_MAXSTACK      1024    /* greatest number of tokens on a function line, 
                                     anybody who hits this limit should be using a 
                                     different language or method!!!
                                     Also deepest RPN stack */
#ifndef MPC_MAXINLINE             /* can set from the compiler */
#define MPC_MAXINLINE    32768    /* biggest fully substituted line, also sum of all 
                                     strings held in the action stack */
#endif
#define MPC_MAXALTPREFIX    16    /* largest alternate prefix which can be STORED in the ifstack*/
#define MPC_MAXVARLEN      256    /* biggest variable name */
#define MPC_MAXDBGLEN     4096    /* longest string that may be passed to mpc_debug */
#define MPC_MAXLABELLEN     64    /* biggest if label name */
#define MPC_MAXIFDEPTH    1024    /* if stack depth */
#define MPC_DEFMACRO      1024    /* default allocation for a macro, to start with */
#define MPC_MAXFNAME       256    /* biggest file name */
#define MPC_MAXMACRODEPTH  100    /* maximum depth for macros calling macros */
#define MPC_ZEROD        ((double) 0.0)
#define MPC_TRACE_COMMAND    1    /* flag bits for trace variable, self explanatory */
#define MPC_TRACE_INPUT      2
#define MPC_TRACE_ADDVAR     4
#define MPC_TRACE_SETVAR     8
#define MPC_TRACE_MACRO     16
#define MPC_TRACE_FUNC      32
#define MPC_TRACE_OUTPUT    64
#define MPC_TRACE_SUBS     128
#define MPC_TRACE_FULLNAME 256
#define MPC_TRACE_DELVAR   512
#define MPC_PI 3.14159265358979323846
#define MPC_SAFE_PATH        1    /* if set, use only string to the right of /\]>: 
                                     in file names, disabling paths, including on
                                     the file passed from the command line */ 
#define MPC_SAFE_IN          2    /* if set, disables f$in */
#define MPC_SAFE_OUT         4    /* if set, disables f$out (all output to stdout) */
#define MPC_SAFE_FILE        8    /* if set, disables f$file_info */
#define MPC_SAFE_DIR        16    /* if set, disables f$dir_list */
#define MPC_HANDLE_NORMAL    0    /* concerns substitutions during macro records */
#define MPC_HANDLE_RECORD    1
#define MPC_SMUGGLE_STATUS   666666 
                                  /* signal to use smuggled status, rather than supplied, for setjmp */

#define  CFR_B_LETTERS        0x0001  /* Bitmaps for "set" based find clauses */
#define  CFR_B_NUMERIC        0x0002
#define  CFR_B_ALPHANUMERIC   0x0200
#define  CFR_B_VISIBLE        0x0004
#define  CFR_B_CONTROL        0x0008
#define  CFR_B_WHITESPACE     0x0010
#define  CFR_B_PUNCTUATION    0x0020
#define  CFR_B_UPPER          0x0040
#define  CFR_B_LOWER          0x0080
#define  CFR_B_HEX            0x0100

/* all enum definitions */

enum isspecial {IS_SPECIAL,IS_ORDINARY};                      /* special variables can change type, ordinary cannot */
enum istype {STRINGVAR,INTVAR,DBLVAR,MACRO,RPNOPER,ANYVAR};   /* variable types */
enum macrostate {EMPTY,RECORDING,DONE,PLAYING};               /* macro states */
enum ismortal   {MORTAL,IMMORTAL};                            /* may variable be deleted */
enum iftype  {IFLABEL,IFMACRO,IFIN};                          /* used for if/else processing */
enum ifstate {IF,ELSEIF,ELSE,ENDIF};                          /* includes the NOT forms */
enum ifdoneit  {YES,NO};                                      /* whether the if has been performed */
enum fromtype {INNOMORE,INFROMMACRO,INFROMFILE,INFROMCOMMAND};
                                                              /* source of commands for interpreter */
enum vexist {MUST,MUSTARRAY,MAY};                             /* when looking up a variable, whether it MUST or MAY exist
                                                                 and whether it MUST be an ARRAY */
enum iskeepquote {KEEPIT,EATIT};                              /* return or strip quote/double quote around text */
/* all #typedef  and struct statements */

typedef struct isdeath      DEATHNODE;     /* put in typedef's so structures may contain forward references */
typedef struct isfirstdeath FIRSTDEATHNODE;
typedef struct macrovar     MVARIABLE;
typedef struct isavar       VARIABLE;
typedef struct isa_rpn_var  RPN_VARIABLE;
typedef struct isanif       IFNODE;
typedef struct isanop       OPNODE;
typedef struct isanarray    ARRAYNEXUS;
typedef struct sstruct      STRINGNODE;
typedef struct tsstruct     TEMPSTRINGNODE;
typedef struct isaconstant  CONSTNODE;

struct isanarray {
   int dim[MPC_TUPLE];    /* dimensions of the array, size A,B,C, all >= 1 if real, or 0 if unspecified */
   int active[MPC_TUPLE]; /* active cell in the array */
   int offset;            /* offset in cells (not bytes!) to the active cell */
   int subdim[MPC_TUPLE]; /* used for addition and multiplication only.  
                             Specifies that the array is a collection of 
                             smaller arrays, each of which is to be operated on */
   void *data;            /* link to the actual data */
   };

struct tsstruct {
   STRINGNODE *sdata;
   TEMPSTRINGNODE *next;
   };

struct sstruct {
   char * string;
   int ssize;
   };

struct isaconstant {
   char * name;
   int    val;
   };

struct isfirstdeath {                 /* head entry of a deathnode list */
   DEATHNODE *head;                   /* first DEATHNODE in chain, NULL if none */
   DEATHNODE *tail;                   /* last DEATHNODE in chain, NULL if none */
   };

struct isdeath {                      /* a deathlist node */
   DEATHNODE *next;                   /* next DEATHNODE in chain, NULL if this is last */
   DEATHNODE *last;                   /* last DEATHNODE in chain, NULL if this is first */
   FIRSTDEATHNODE *list;              /* pointer to deathlist "FIRST" record */
   VARIABLE  *target;                 /* VARIABLE which will die */
   };

struct macrovar {                     /* an MVARIABLE node */
   char * macro;                      /* pointer used when playing back macros.  It
                                         Starts at the beginning of string and 
                                         increments out through the storage area.
                                         Also, macros are not recursive, so if macro
                                         is not NULL an attempt to play that macro is
                                         an error. */
   char * body;                       /* alternative start, defined if f$macro_body
                                         has been executed.  Used on 2nd ->Nth 
                                         iteration. */
   char * altprefix;                  /* altprefix within this macro */
   FIRSTDEATHNODE * deathlist;        /* pointer to the top of the deathlist for this macro */
   int msize;                         /* of all of the strings in a macro, in bytes 
                                         up to and including the final \0 */
   enum macrostate mstate;            /* State of the macro.  Macros most be DONE
                                         to play, and once DONE, they may not be 
                                         extended.*/
   int  mc[MPC_TUPLE];                /* macro loop indices, used when looping */
   int  mcmax[MPC_TUPLE];             /* macro loop index limits, used when looping */
   };

struct isavar {                       /* a VARIABLE node */
   enum istype type;                  /* STRINGVAR,INTVAR,DBLVAR,MACRO */
   char * name;                       /* pointer to the name of the variable */
   VARIABLE *next;                    /* next VARIABLE in chain, NULL if this is last */
   VARIABLE *last;                    /* last VARIABLE in chain, NULL if this is first */
   DEATHNODE *assassin;               /* pointer to this variable's entry on a death list*/
   int    ival;                       /* integer, if integer */
   double dval;                       /* double, if double */
   char * string;                     /* string pointed to, if string or macro*/
   int ssize;                         /* size of the string/macro storage area, when it
                                         was created with malloc or realloc, in bytes */
   ARRAYNEXUS *array;                 /* pointer to an array, if this is an array variable */
   enum isspecial special;            /* ordinary or special, ordinary has type locked */
   MVARIABLE *mfields;                /* pointer to macro fields */
   };

struct isa_rpn_var {                  /* RPN stack node, simpler than regular variable! */
   enum istype type;                  /* STRINGVAR,INTVAR,DBLVAR,MACRO */
   int    ival;                       /* integer, if integer */
   double dval;                       /* double, if double */
   char * string;                     /* string pointed to, if string or macro */
   int ssize;                         /* size of the string/macro storage area, when it
                                         was created with malloc or realloc, in bytes */
   };

struct isanif {                       /* an if stack node */
   enum iftype type;                  /* IFLABEL,IFMACRO,IFIN  */
   enum ifstate state;                /* IF,IFNOT,ELSEIF,ELSEIFNOT,ELSE,ENDIF */
   enum ifdoneit doneit;              /* YES, NO , passed a test in this particular logic block */
   enum ifdoneit invert;              /* YES, NO, whether or not to invert the logic */
   char string[MPC_MAXLABELLEN];
   char altprefix[MPC_MAXALTPREFIX];      /* altprefix which was in effect on f$in or macro call */
};

struct isanop {                       /* an operator node for mpcf_do_evaluate */
   char *name;                        /* operator name*/
   int opnum;                         /* used in a switch statement */
   int defoperands;                   /* default number of operands, usually == minoperands */
   int minoperands;                   /* smallest number of operands */
   int maxoperands;                   /* largest number of operands */
   enum istype operandtype;           /* data type of operands */ 
   int argnumber;                     /* number of arguments for function */
   enum istype argtype[MPC_RPNARGS];  /* argument types */
};

/* Global storage.  Modules included in this file will access these,
   but others should interact with miniproc through findvar() and miniproc()
   only.  The former for reading out results, the latter for generating 
   them.  
   "Dummy" here means "actual storage for", as opposed to
   pointer to */

#ifndef _EMBEDDED_MINIPROC_

int MPC_ISMAX_RPN=0;                  /* holds size of RPN array, after sort */
int ddosubnum=1;                      /* dummy SUBNUM variable */
int dmacrosubnum=0;                   /* dummy MACROSUBNUM variable */
int dconvertwidth=32;                 /* dummy CONVERT_WIDTH variable */
int dtrace=0;                         /* dummy TRACE variable, needed just until user 
                                         visible variables are initialized */
int dsafety=0;                        /* dummy safety variable, needed just until user 
                                         visible variables are initialized */
int ddeprecate=0;                     /* dummy deprecate variable, needed just until user 
                                         visible variables are initialized */
FIRSTDEATHNODE  mdnode,mpnode;        /* storage for deathnode lists */     
RPN_VARIABLE rpn_stack[MPC_MAXSTACK]; /* RPN stack, operands and operators */
char *actionstack[MPC_MAXSTACK];      /* maximum number of tokens in the stack */
int stackptr=-1;                      /* index, begins as invalid */
IFNODE ifstack[MPC_MAXIFDEPTH];       /* if stack with nodes of this type */
int ifptr=-1;                         /* pointer into stack, initially invalid */
enum ifdoneit ifscan=NO;              /* start with if scanning off */
VARIABLE *head[MPC_HASHSPACE];        /* pointers to FIRST in the variable hash chains */
VARIABLE *tail[MPC_HASHSPACE];        /* pointers to LAST in the chains */
unsigned int hashed;                  /* hash value from last hash operation*/
VARIABLE *rpn_result=NULL;            /* pointer to "result" */
VARIABLE *recmacro=NULL;              /* pointer to a macro being recorded */
VARIABLE *altprefix;                  /* altprefix variable */

VARIABLE *mcshort[MPC_TUPLE];         /* MC* and MC*MAX are referenced constantly */
VARIABLE *mcmaxshort[MPC_TUPLE];      /* so provide pointers, rather than constant findvar() calls */

enum fromtype fromsource =INFROMCOMMAND;
                                      /* where input comes from */
enum fromtype nextfromsource = INFROMCOMMAND;
                                      /* where input comes from NEXT */

int howtohandle = MPC_HANDLE_NORMAL;

VARIABLE *activemacro[MPC_MAXMACRODEPTH];
                                      /* macro calling stack */
int isactivemacro=-1;                 /* index into macro calling stack, here, none active */

OPNODE all_optypes[]={
   {"power"     ,  0, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"modulo"    ,  1, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"add"       ,  2, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"+"         ,  2, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"subtract"  ,  3, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"-"         ,  3, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"multiply"  ,  4, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"*"         ,  4, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"divide"    ,  5, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"/"         ,  5, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"eq"        ,  6, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"ne"        ,  7, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"ge"        ,  8, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"gt"        ,  9, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"lt"        , 10, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"le"        , 11, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"xor"       , 12, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"not"       , 13, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"and"       , 14, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"or"        , 15, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"nand"      , 16, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"nor"       , 17, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"head"      , 18, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"tail"      , 19, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"segment"   , 20, 1, 1, MPC_MAXSTACK, STRINGVAR,2, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"d->s"      , 21, 1, 1, MPC_MAXSTACK, DBLVAR,   1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"i->s"      , 22, 1, 1, MPC_MAXSTACK, DBLVAR,   1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"s->d"      , 23, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"s->i"      , 24, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"append"    , 25, 2, 2, MPC_MAXSTACK, STRINGVAR,1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"uppercase" , 26, 1, 1, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"lowercase" , 27, 1, 1, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"shortest"  , 28, 1, 1, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"longest"   , 29, 1, 1, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"eliminate" , 30, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"retain"    , 31, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"element"   , 32, 1, 1, MPC_MAXSTACK, STRINGVAR,2, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"locate"    , 33, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"compare"   , 34, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"ccompare"  , 35, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"length"    , 36, 1, 1, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"lexhigh"   , 37, 2, 2, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"lexlow"    , 38, 2, 2, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"stringdel" , 39, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"resize"    , 40, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"getenv"    , 41, 1, 1, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"swap"      , 42, 1, 1, MPC_MAXSTACK, ANYVAR,   1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"duplicate" , 43, 1, 1, MPC_MAXSTACK, ANYVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"sin"       , 44, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"cos"       , 45, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"tan"       , 46, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"asin"      , 47, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"acos"      , 48, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"atan"      , 49, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"expe"      , 50, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"exp10"     , 51, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"loge"      , 52, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"log10"     , 53, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"deg->rad"  , 54, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"rad->deg"  , 55, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"showstack" , 56, 1, 0, MPC_MAXSTACK, ANYVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"storage"   , 57, 1, 1, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"rot->up"   , 58, 2, 2, MPC_MAXSTACK, ANYVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"rot->down" , 59, 2, 2, MPC_MAXSTACK, ANYVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"delete"    , 60, 1, 1, MPC_MAXSTACK, ANYVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"elements"  , 61, 1, 1, MPC_MAXSTACK, STRINGVAR,2, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"load"      , 62, 1, 1, MPC_MAXSTACK, ANYVAR,   0, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"store"     , 63, 1, 1, MPC_MAXSTACK, ANYVAR,   1, {STRINGVAR, DBLVAR,    DBLVAR} },
   {"scale"     , 64, 1, 1, MPC_MAXSTACK, DBLVAR,   1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"offset"    , 65, 1, 1, MPC_MAXSTACK, DBLVAR,   1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"substitute", 66, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"array"     , 67, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"lifetime"  , 68, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"free"      , 69, 1, 1, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"stacksize" , 70, 1, 1, MPC_MAXSTACK, ANYVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"edit"      , 71, 1, 1, MPC_MAXSTACK, STRINGVAR,2, {STRINGVAR, STRINGVAR, DBLVAR} },
   {"showscope" , 72, 0, 0, MPC_MAXSTACK, ANYVAR,   0, {STRINGVAR, STRINGVAR, DBLVAR} },
   {"setdepth"  , 73, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {DBLVAR,    STRINGVAR, DBLVAR} },
   {"debug"     , 74, 1, 1, MPC_MAXSTACK, ANYVAR,   0, {STRINGVAR, STRINGVAR, DBLVAR} },
   {"b-xor"     , 75, 1, 1, MPC_MAXSTACK, DBLVAR,   1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"b-not"     , 76, 1, 1, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"b-and"     , 77, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"b-or"      , 78, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"(dim)"     , 79, 1, 1, MPC_MAXSTACK, STRINGVAR,3, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"()"        , 80, 1, 1, MPC_MAXSTACK, STRINGVAR,3, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"(showdim)" , 81, 1, 1, 1,            STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"(showcell)", 82, 1, 1, 1,            STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"(scale)"   , 83, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"(offset)"  , 84, 1, 1, MPC_MAXSTACK, STRINGVAR,1, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"(subdim)"  , 85, 1, 1, MPC_MAXSTACK, STRINGVAR,3, {DBLVAR,    DBLVAR,    DBLVAR} },
   {"constant"  , 86, 1, 1, MPC_MAXSTACK, STRINGVAR,0, {DBLVAR,    DBLVAR,    DBLVAR} },
   {""          , 87, 2, 2, MPC_MAXSTACK, DBLVAR,   0, {DBLVAR,    DBLVAR,    DBLVAR} },
   };

/* the following list must be in alphabetical order by first value !!! */
CONSTNODE exposed_constants[]={
   {"ADDVAR_TRACE"    ,MPC_TRACE_ADDVAR},
   {"ALPHANUMERIC"    ,CFR_B_ALPHANUMERIC},
   {"COMMAND_TRACE"   ,MPC_TRACE_COMMAND},
   {"CONTROL"         ,CFR_B_CONTROL},
   {"DELVAR_TRACE"    ,MPC_TRACE_DELVAR},
   {"FULLNAME_TRACE"  ,MPC_TRACE_FULLNAME},
   {"FUNCTION_TRACE"  ,MPC_TRACE_FUNC},
   {"INPUT_TRACE"     ,MPC_TRACE_INPUT},
   {"HEX"             ,CFR_B_HEX},
   {"LETTERS"         ,CFR_B_LETTERS},
   {"LOWER"           ,CFR_B_LOWER},
   {"MACRO_TRACE"     ,MPC_TRACE_MACRO},
   {"MAXINLINE"       ,MPC_MAXINLINE},
   {"NUMERIC"         ,CFR_B_NUMERIC},
   {"OUTPUT_TRACE"    ,MPC_TRACE_OUTPUT},
   {"PUNCTUATION"     ,CFR_B_PUNCTUATION},
   {"SETVAR_TRACE"    ,MPC_TRACE_SETVAR},
   {"SUBS_TRACE"      ,MPC_TRACE_SUBS},
   {"UPPER"	      ,CFR_B_UPPER},
   {"VISIBLE"         ,CFR_B_VISIBLE},
   {"WHITESPACE"      ,CFR_B_WHITESPACE},
   {""                ,0}
   };
FILE *fin[MPC_STREAMS];                        /* space for MPC_STREAMS input stream files (all at filenumber MPC_FILE_ILOW) */
FIRSTDEATHNODE  *fin_death[MPC_STREAMS];       /* space for MPC_STREAMS pointers to MPC_STREAMS death lists */
FILE *finr[MPC_FILE_ISIZE-1];                  /* space for the remaining regular input files (filenumber 
                                                  MPC_FILE_ILOW+1 through MPC_FILE_IHIGH), all offset by
                                                 -(MPC_FILE_ILOW +1) */
char finname[MPC_FILE_ISIZE][MPC_MAXFNAME];       /* names of currently open input files */
int finc=-1;                          /* array pointer for fin, initially invalid */
FILE *fout[]= {NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL};
                                      /* output files filenumber 0-9*/

char cdeck[MPC_MAXVARLEN];            /* holds deck characters */
char cfullname[MPC_MAXVARLEN+MPC_MAXVARLEN];
                                      /* for expanded local names */
char clastfullname[MPC_MAXVARLEN+MPC_MAXVARLEN]="\0"; 
                                      /* for expanded local names */

jmp_buf  mpc_env;                     /* used by setjmp/longjmp */
char mpc_dbg_buffer[MPC_MAXDBGLEN];   /* static buffer used to pass messages to debug output routine */
int  mpc_smuggle_exit_status;         /* longjmp cannot return status 0, so  when setjmp returns with
                                         MPC_SMUGGLE_STATUS read the real exit status from here */




int *dosubnum=&ddosubnum;             /* substitute to one level */
int *domacrosubnum=&dmacrosubnum;     /* substitute to zero levels */
int *convertwidth=&dconvertwidth;     /* controls d->s,i->s output sizes */
int *trace=&dtrace;                   /* trace disabled */
int *instatus=&dtrace;                /* internal status variable, at init will be remapped to STATUS */
int *safety=&dsafety;                 /* safety variables, set on command line ONLY to 
                                         restrict actions taken by possibly hostile input files */
int *deprecate=&ddeprecate;           /* deprecate variable, warns about 
                                         deprecated usage */
FIRSTDEATHNODE *master_death=&mdnode; /* master death list for miniproc */
FIRSTDEATHNODE *program_death=&mpnode;/* master death list for program */
char *deck=&cdeck[0];                 /* pointer to cdeck storage */
char *fullname=&cfullname[0];         /* pointer to fullname storage */
char *lastfullname=&clastfullname[0]; /* pointer to lastfullname storage */
char *mpc_dbg_out=&mpc_dbg_buffer[0]; /* pointer to miniproc debug buffer */


#else   /* _EMBEDDED_MINIPROC_ */
extern char *mpc_dbg_out;             /* pointer to miniproc debug buffer */
#endif  /* _EMBEDDED_MINIPROC_ */

                        
/* ----------------------------------------------------------------------
   prototypes used by miniproc below, declarations above
   all EXCEPT miniproc() have an mpcf_ prefix.
   ---------------------------------------------------------------------- */

int  miniproc(int margc, char *margv[]);

void mpcf_add_to_current_deathlist(VARIABLE * newvar);
void mpcf_add_node_to_deathlist(FIRSTDEATHNODE **thelist, DEATHNODE *thednode);
void mpcf_array_offset(VARIABLE *somevar, double dval);
void mpcf_array_scale(VARIABLE *somevar, double dval);
void mpcf_change_lifetime(char *newscope, int tomove);
void mpcf_clear_deathlist(FIRSTDEATHNODE *thelist);
void mpcf_copy_rpn_stack(int from, int to, int rpnstackptr);
void mpcf_delvar(char *name);
void mpcf_delvar_byvar(VARIABLE *old);
void mpcf_array_active(VARIABLE *somevar, int tdim[MPC_TUPLE]);
void mpcf_array_dim(VARIABLE *somevar, int itemp, int tdim[MPC_TUPLE]);
void mpcf_array_subdim(VARIABLE *somevar, int tdim[MPC_TUPLE]);
void mpcf_deprecated_code(char *message);
void mpcf_do_date_time(int first);
void mpcf_do_edit(char *string, char *editop, char *editchars );
void mpcf_do_evaluate(void);
void mpcf_do_function(void);
void mpcf_do_ifelse(enum ifstate instate, enum ifdoneit invert);
void mpcf_do_macro(void);
void mpcf_do_rpn_evaluate(void);
void mpcf_do_subs(char *line);
void mpcf_dump_deathlist(FIRSTDEATHNODE **thelist, FILE *towhere);
void mpcf_emasculate_variable(VARIABLE* thevar);
void mpcf_enlarge_string(char **string,int *ssize, int newsize,char *oops);
void mpcf_ensure_double(int rpnptr,int numops,char * message);
void mpcf_ensure_string(int rpnptr,int numops,char * message);
void mpcf_fixorder(char *string,int rot, int ssize);
void mpcf_init_variables(void);
void mpcf_load_on_stack(char * name, int tptr);
int  mpcf_lookup_constant(char *cname, int* ival);
void mpcf_macro_make_playable(void);
int  mpcf_macro_gets(char *vinline);
void mpcf_macro_record(char *name, char *sval);
void mpcf_macro_repeat(char *name, VARIABLE *array);
void mpcf_macro_return(int mode);
VARIABLE *mpcf_findvar(enum vexist varexist, char *name);
void mpcf_makefullname(char *name); /* returns "hashed" value */
void mpcf_parse_and_run(char *line);
void mpcf_parse_rpn_operator(char *string,int *ival, double *dval);
void mpcf_pathsafe(char *string);
void mpcf_print_out(char *line);
void mpcf_reset_mpc(void);
void mpcf_resolveright(char *string, char **sval, int *ival, double *dval, enum istype *type, int mode);
void mpcf_rpn_stack_overflow(void);
void mpcf_setaltprefix(char *newaltprefix);
void mpcf_setcounters(VARIABLE *play,int setmc, int setvismc);
void mpcf_shiftactive(int left);
void mpcf_show_deathlists(FILE *towhere);
void mpcf_showstack(FILE *towhere, int startat, int rpnstackptr);
void mpcf_sort_rpn_operators(void);
int  mpcf_strip_trailing_comments(char * string,int endstring, int lowerlimit);
char * mpcf_strtok(char *instring,const char *delim, enum iskeepquote keepquote);
void mpcf_store_from_stack(char * name, int tptr);
int  mpcf_strccmp(char *first, char *second);
void mpcf_stuffaltprefix(void);
void mpcf_test(char *sval, int ival, double dval, enum istype type, enum ifdoneit invert);
void mpcf_trim_command_line(char *line);
void mpcf_wipe_tsstruct(TEMPSTRINGNODE *tshead);
void mpcf_yankaltprefix(void);

/*
#__SLICEDICE "embed.c"
*/

/* ----------------------------------------------------------------------
   functions used by miniproc below, prototypes above 
   ---------------------------------------------------------------------- */

/* mpcf_lookup_constant(), looks up a string "cname" in the list of known 
constants.  If it finds an exact match, it sets the corresponding integer value 
and returns 1, if it fails, it just returns 0 and ival is undefined.*/

int mpcf_lookup_constant(char *cname, int* ival){
int i,j;
  for(i=0; *(exposed_constants[i].name) != '\0'; i++){
    for(j=0; ;j++){
       if( toupper(cname[j]) != 
           toupper((exposed_constants[i].name)[j]) 
         )break;
       if(cname[j] == '\0'){
          *ival = exposed_constants[i].val;
          return 1;
       }
    }
  }
  return 0;
}


/* mpcf_strtok(), modified version of strtok which considers a single quote 
   to be a special kind of character.  Tokens beginning with a single
   quote search for the next instance of single quote followed by delimiter
   or end of line, and return that as the next token. Also "&" as the first
   nondilimiter character on a line means "pass the entire rest of the 
   line" back as a token, with & still on it". */

char * mpcf_strtok(char *instring,const char *delim, enum iskeepquote keepquote){
static char *holdstring;
static int  ftoken, etoken, dqetoken, endstring, firstcase;
int cpnt;
enum sstates {NEXTDELIM,NEXTDELIMQ,NEXTQUOTE,NEXTDQUOTE,NEXTDELIMDQ,NEXTREGULAR,ENDSEARCH};
enum sstates searchstate;
enum vdquote {ISVALID,ISINVALID};
enum vdquote validdquote;

  if(instring != NULL){  /* new string */
     endstring = strlen(instring) - 1;
     holdstring = instring; /* write on the actual input string, not a copy */
     ftoken=0;
     etoken=0;
     firstcase=1;
  }
  else {  /* look for next token in existing string */
     etoken++;     /* last call would have left this on '\0' */
     firstcase=0;
  }

  /* this is the part that does the search
     an exit path is left from this loop to a return=NULL outside so that
     some compilers (DECC) won't complain about there not being a final
     return statement */

  for( searchstate = NEXTREGULAR; searchstate != ENDSEARCH ; etoken++){
    if(searchstate != NEXTREGULAR){firstcase=0;}
    if(etoken > endstring){  /* always check first to see if search has extended past string end */
       switch (searchstate){
          case NEXTREGULAR: /* hit end of string, no tokens found */
             searchstate=ENDSEARCH;
             break;
          case NEXTDELIM:   /* end of token is end of string */
          case NEXTDELIMQ:  /* end of token immediately follows final quote in string */
             if(keepquote == EATIT){  /* nip off ends of double/single quotes */
               ftoken++;
               holdstring[endstring]='\0';
             }
             return &holdstring[ftoken];
          case NEXTDELIMDQ:   /* handle double quote, the potential double 
                                 quoted string MUST be followed by a delimiter or an EOL */
             etoken=dqetoken;
             etoken++;
             validdquote=ISINVALID;
             if(etoken == endstring + 1){ /* EOL case */
               validdquote=ISVALID;
             }
             else {
               for(cpnt=0; delim[cpnt] != '\0';cpnt++){
                 if(holdstring[etoken] == delim[cpnt]){  /* NON-EOL case */
                   validdquote=ISVALID;
                   break;
                 }
               }
             }
             if(validdquote == ISVALID){
               if(keepquote == KEEPIT){
                 holdstring[etoken]='\0';
               }
               else {
                 holdstring[etoken]=' ';    /* put a space (delimiter) in over the quote/double quote */
                 etoken--;
                 holdstring[etoken]='\0';   /* delimit string */
                 etoken++;                  /* restore etoken, there may be another call */   
                 ftoken++;                  /* nip off leading quote/double quote */
               }
               return &holdstring[ftoken];
             }
             else {
               (void) sprintf(mpc_dbg_out,"miniproc, fatal error, unmatched external (double) quotes");
               MPC_WRITE_AND_BAILOUT(4203);
             }
          case NEXTQUOTE:   /* unmatched quotes - bad */
             (void) sprintf(mpc_dbg_out,"miniproc, fatal error, unmatched internal (single) quotes");
             MPC_WRITE_AND_BAILOUT(4202);
          case NEXTDQUOTE:   /* unmatched double quotes - bad */
             (void) sprintf(mpc_dbg_out,"miniproc, fatal error, unmatched external (double) quotes");
             MPC_WRITE_AND_BAILOUT(4203);
       }
    }
    else {
       switch (searchstate){
          case NEXTREGULAR:    /* is the current character NOT a delimiter? */
            switch(holdstring[etoken]){
              case '&':            /* possible start of an & line */
                 if(firstcase == 1){ /* only consider it if no other states have been tried */
                   ftoken = etoken;
                   etoken = endstring + 1;
                   return &holdstring[ftoken];
                 }
                 ftoken = etoken;            /* it is a regular token, not a delimiter */
                 searchstate=NEXTDELIM;
                 break;
              case '\'':           /* start of a single quote delimited token */
                 searchstate = NEXTQUOTE;
                 ftoken = etoken;  /* single quotes are part of the token */
                 break;
              case '"':            /* start of a double quote delimited token */
                 searchstate = NEXTDQUOTE;
                 ftoken = etoken;  /* double quotes are part of the token */
                 break;
              default:
                for(cpnt=0; delim[cpnt] != '\0';cpnt++){
                  if(holdstring[etoken] != delim[cpnt]){
                    ftoken = etoken;
                    searchstate=NEXTDELIM;
                  }
                }
            }
            break;
          case NEXTDELIM: /* see if the current character is a delimiter */
            for(cpnt=0; delim[cpnt] != '\0';cpnt++){
              if(holdstring[etoken] == delim[cpnt]){  /* a token has been found */
                 holdstring[etoken] = '\0';     /* terminate it */
                 return &holdstring[ftoken];
              }
            }
            break;
          case NEXTDELIMQ: /* this character MUST be a delimiter or another 
                              ', as '' are reduced to one.  Special case 
                              handling for ''', as first triggers possible 
                              end of qoute, second is eaten, and third shifts
                              back underneath second, requiring some 
                              jiggering of etoken to avoid skipping it*/
           
            if(holdstring[etoken] == '\''){ /* eat this character, shift all left, state returns to NEXTQUOTE */
              for(cpnt=etoken; holdstring[cpnt] != '\0'; cpnt++){
                holdstring[cpnt]=holdstring[cpnt+1];
              }
              etoken--;    /* move etoken back too, as this position must 
                              be examined again as it is a NEW character */
              endstring--; /* adjust line size */
              searchstate=NEXTQUOTE;
              break;
            }
            else {
              for(cpnt=0; delim[cpnt] != '\0';cpnt++){
                if(holdstring[etoken] == delim[cpnt]){  /* a token has been found */
                   holdstring[etoken] = '\0';     /* terminate it */
                   return &holdstring[ftoken];
                }
              }
            }
            (void) sprintf(mpc_dbg_out,"miniproc, fatal error, internal (single) quotes not followed by delimiter or EOL");
            MPC_WRITE_AND_BAILOUT(4204);
            break;
          case NEXTQUOTE: /* see if the current character is a quote */
            if(holdstring[etoken] == '\''){   /* a token has been found */
               searchstate = NEXTDELIMQ;      /* next char better be delimiter or EOL*/
            }
            break;
          case NEXTDELIMDQ: /* see if the current character is a quote */
          case NEXTDQUOTE:  /* see if the current character is a quote */
            if(holdstring[etoken] == '"'){    /* a valid token has been found, may not be final one though */
               searchstate = NEXTDELIMDQ;     /* indicates that at least one valid token was found */
               dqetoken = etoken;
            }
            break;
       }
    } 
  } 
  return NULL;
}

/* mpcf_strccmp(), for systems which might not have strccmp().  Returns 1 if
   the two strings are identical, 0 otherwise */

int mpcf_strccmp(char *first, char *second){
char * from;
char * to;
int    toreturn;

  for(toreturn=1,to=first,from=second; ;from++,to++){
    if(toupper(*from)!=toupper(*to)){
      toreturn=0;
      break; 
    }
    if(*to=='\0')break; /* since *from=*to, both are '\0' */
  }
  return toreturn;
}


/* mpcf_array_scale.  Multiply every element in a DBLVAR array by the 
value provided.  Somevar will always exist  */

void mpcf_array_scale(VARIABLE *somevar, double dval){
int itemp,ival;
  if(somevar->type != DBLVAR){
    (void) sprintf(mpc_dbg_out,"miniproc, cannot (scale), array %.4000s is not of type double (float)",somevar->name);
    MPC_WRITE_AND_BAILOUT(4301);
  }
  for(ival=1,itemp=0; itemp< MPC_TUPLE; itemp++){
    if(somevar->array->dim[itemp] > 0){                /* only for nonzero indices! */
      ival = ival * somevar->array->dim[itemp];
    }
  }
  for(itemp=0; itemp < ival; itemp++){
    ((double *)somevar->array->data)[itemp] = dval * ((double *)somevar->array->data)[itemp];
  }  
}

/* mpcf_array_offset.  Offset every element in a DBLVAR array by the 
value provided.  Somevar will always exist  */

void mpcf_array_offset(VARIABLE *somevar, double dval){
int itemp,ival;
  if(somevar->type != DBLVAR){
    (void) sprintf(mpc_dbg_out,"miniproc, cannot (scale), array %.4000s is not of type double (float)",somevar->name);
    MPC_WRITE_AND_BAILOUT(4401);
  }
  for(ival=1,itemp=0; itemp< MPC_TUPLE; itemp++){
    if(somevar->array->dim[itemp] > 0){                /* only for nonzero indices! */
      ival = ival * somevar->array->dim[itemp];
    }
  }
  for(itemp=0; itemp < ival; itemp++){
    ((double *)somevar->array->data)[itemp] = dval + ((double *)somevar->array->data)[itemp];
  }  
}

void mpcf_deprecated_code(char *message){
  switch (*deprecate) {
    case 0:
      break;
    case 1:
      (void) sprintf(mpc_dbg_out,"miniproc, deprecated code warning: %.4000s",message);
      WRITE_DEBUG();
      break;
    case 2:
      (void) sprintf(mpc_dbg_out,"miniproc, deprecated code warning: %.4000s",message);
      MPC_WRITE_AND_BAILOUT(101);
    default:
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, deprecate variable out range [0-2], is %d",*deprecate);
      MPC_WRITE_AND_BAILOUT(102);
  }
}


/* ------------------------------
  mpcf_makefullname().  Expands local variable or macronames and also calculates
  the hash value.  Local variables (form=":var") expand based on fromsource.
  For instance, if :var is in myfile.mpc, the variable is stored as
  "^myfile.mpc^var", and if it is in a local macro ":amacro" running from that file 
  it becomes "^^myfile.mpc^amacro^var".  Kind of ugly, but the point is to 
  keep the local variables out of the global namespace, and this pretty 
  much does it unless you go to great lengths to force an overlap.
*/
void mpcf_makefullname(char *name){
unsigned int i,last;
char *from;
VARIABLE *play;

  if(*name==':'){ /* local variables must have their names expanded */
    (void)strcpy(fullname,"^");
    switch (fromsource){
      case  INFROMMACRO:
        play=activemacro[isactivemacro];
        if(play==NULL){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, infrommacro in mpcf_makefullname");
          MPC_WRITE_AND_BAILOUT(201);
        }
        (void)strcat(fullname,play->name);
        break;
      case  INFROMFILE:
        (void)strcat(fullname,&finname[finc][0]); /* names of currently open input file */
        break;
      case  INFROMCOMMAND:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, local variable [%200s]used on command line",name);
        MPC_WRITE_AND_BAILOUT(202);
    }
    (void)strcat(fullname,"^");
    from=name;
    from++;
    (void)strcat(fullname,from);
    (void)strcat(fullname,"\0");
  }
  else {
    (void) strcpy(fullname,name);
  }

  /* now hash it if it is not exactly the same name as the preceding one, 
  that is, if it will have a different hash value */

  if(strcmp(lastfullname,fullname)!=0){
    (void)strcpy(lastfullname,fullname); /* for the next time */
    for(i=1, last=1, hashed=0, from=fullname ; *from!='\0' ; from++,i++){
      hashed=(i * last) * (unsigned int)(*from) + hashed;
      last=*from;
    }
    last = hashed/MPC_HASHSPACE;
    hashed = hashed - last*MPC_HASHSPACE;
  }
  if((*trace & MPC_TRACE_FULLNAME)==MPC_TRACE_FULLNAME){
    (void) sprintf(mpc_dbg_out,"  name [%.2000s] -> hash [%d], fullname [%.2000s]",name,hashed,fullname);
    WRITE_DEBUG();
  }
}

/*  mpcf_add_node_to_deathlist(), add a node to the specified deathlist.  If the deathlist
is NULL then the dnode is deleted, and the assassin reference to it removed. */

void  mpcf_add_node_to_deathlist(FIRSTDEATHNODE **thelist, DEATHNODE *thednode){
   if(thednode==NULL){
       (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, attempt to move a nonexistant death node");
       MPC_WRITE_AND_BAILOUT(301);
   }
   if(thelist == NULL){       /* CAREFUL!!! master->tail, master->head, and lastnode->next must be
                                 dealt with elsewhere!!! */
     thednode->target->assassin=NULL;       /* no assassin for this variable anymore */
     free(thednode);                        /* free the dnode's space */
     return;
   }
   thednode->list = (*thelist);                /* link to which list it is in */
   if((*thelist)->tail == NULL){
       (*thelist)->head  = thednode;        /* head and tail are both this node */
       (*thelist)->tail  = thednode;
       thednode->last = NULL;               /* no back link for first node */
       thednode->next = NULL;               /* no forward link for tail of chain */
    }
    else {
       (*thelist)->tail->next  = thednode;  /* old end points to this new node */
       thednode->last = (*thelist)->tail;   /* this node back links to old end */
       thednode->next = NULL;               /* no forward link for tail of chain */
       (*thelist)->tail  = thednode;        /* this node becomes new end */
    }
}

void mpcf_dump_deathlist(FIRSTDEATHNODE **thelist, FILE *towhere){
DEATHNODE *thednode;

   thednode=(*thelist)->head;
   if(thednode==NULL){
      (void) fprintf(towhere,"  none\n");
   }
   else {
     for( ; thednode != NULL; thednode=thednode->next){
       (void) fprintf(towhere,"  %.4000s\n",thednode->target->name);
     }
   }
}

/*---------------------------------------------------------
   mpcf_show_deathlists() debugging tool, shows
   the contents (briefly) of all 3 current sets of deathlists
*/
void mpcf_show_deathlists(FILE *towhere){
FIRSTDEATHNODE **thelist;
      (void) fprintf(towhere,"variables listed by scope levels\n");
   /* macro scope */      
      if(isactivemacro >= 0){
        (void) fprintf(towhere," MACRO scope, Macro = %.4000s\n",activemacro[isactivemacro]->name);
        thelist=&((activemacro[isactivemacro])->mfields->deathlist);
        mpcf_dump_deathlist(thelist, towhere);
      }
      else {
        (void) fprintf(towhere," MACRO scope, inactive\n");
      }
   /* file scope */      
      thelist=&(fin_death[finc]);   /* names of currently open input file */
      (void) fprintf(towhere," FILE scope, File = %.4000s\n",&finname[finc][0]);
      mpcf_dump_deathlist(thelist, towhere);
   /* miniproc scope */      
      (void) fprintf(towhere," MINIPROC scope\n");
      thelist=&master_death;
      mpcf_dump_deathlist(thelist, towhere);
   /* program scope */ 
      (void) fprintf(towhere," PROGRAM scope\n");
      thelist=&program_death;
      mpcf_dump_deathlist(thelist, towhere);
}

/*---------------------------------------------------------
   mpcf_add_to_current_deathlist(), create, and return a pointer to, the assassin 
   node for the VARIABLE which is the sole argument.
*/
void mpcf_add_to_current_deathlist(VARIABLE * newvar){
FIRSTDEATHNODE **thelist;
DEATHNODE *thednode;
    switch (fromsource){
      case  INFROMMACRO:  /* attach to this macro's deathlist */
        thelist=&((activemacro[isactivemacro])->mfields->deathlist);
        break;
      case  INFROMFILE:
        thelist=&(fin_death[finc]);   /* names of currently open input file */
        break;
      case  INFROMCOMMAND: /* attach to the main miniproc deathlist */
        thelist=&master_death;
        break;
    }

    /* make a deathnode for this variable */

    thednode=malloc(sizeof(DEATHNODE));
    if(thednode == NULL){
       (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not allocate enough memory");
       MPC_WRITE_AND_BAILOUT(401);
    }
    newvar->assassin = thednode;
    thednode->target = newvar;
    thednode->next = NULL;
    thednode->list = (*thelist);

    /* put the deathnode on the this deathlist */

    mpcf_add_node_to_deathlist(thelist, thednode);
}


/*---------------------------------------------------------
   mpcf_change_lifetime(), move N entries from the current deathlist up
   in lifetime.  Scopes are PROGRAM{ MINIPROC{ FILE{ MACRO{}}}}.
   Valid transitions are MACRO -> FILE, MINIPROC, PROGRAM
                         FILE  -> MINIPROC, PROGRAM
                         MINIPROC -> PROGRAM
   Transition to a FILE lifetime moves it to the CURRENT file.
   TOPFILE moves it to the first open file (finc=0)
   UPFILE  moves it to the file above this one (finc--)
   Neither of these fails OR generates a warning:
     Transitions of more entries than exist in the list just move everything.
     Invalid scope changes (FILE->MACRO) are IGNORED.
   Note, when this routine is called there will ALWAYS be a file
     active, so it is always possible to promote variables to the FILE
     deathlist.
   Trying to change the scope of a local variable (:varname) generates
     a fatal error.

*/

void mpcf_change_lifetime(char *newscope, int tomove){
FIRSTDEATHNODE **thelist;
FIRSTDEATHNODE **thenewlist;
DEATHNODE *thednode;
int levelin,levelout;

    switch (fromsource){
      case  INFROMMACRO:  /* attach to this macro's deathlist */
        thelist=&((activemacro[isactivemacro])->mfields->deathlist);
        levelin=0;
        break;
      case  INFROMFILE:
        thelist=&(fin_death[finc]);   /* names of currently open input file */
        levelin=1;
        break;
      case  INFROMCOMMAND: /* attach to the main miniproc deathlist */
        thelist=&master_death;
        levelin=2;
        break;
    }

    thenewlist=NULL;
    levelout=0;
    if(mpcf_strccmp(newscope,"file")){
        if(finc < 0){  /* this can happen if running in SLAVE_MACRO, no file is open*/
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, SLAVE_MACRO objects cannot be raised to FILE lifetime");
           MPC_WRITE_AND_BAILOUT(502);
        }
        levelout=1;
        thenewlist=&(fin_death[finc]);
    }
    if(mpcf_strccmp(newscope,"topfile")){
        if(finc < 0){  /* this can happen if running in SLAVE_MACRO, no file is open*/
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, SLAVE_MACRO objects cannot be raised to FILE lifetime");
           MPC_WRITE_AND_BAILOUT(502);
        }
        levelout=1;
        thenewlist=&(fin_death[0]);
    }
    if(mpcf_strccmp(newscope,"upfile")){
        if(finc < 0){  /* this can happen if running in SLAVE_MACRO, no file is open*/
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, SLAVE_MACRO objects cannot be raised to FILE lifetime");
           MPC_WRITE_AND_BAILOUT(502);
        }
        levelout=1;
        thenewlist=&(fin_death[mpc_max(finc - 1, 0 )]);
    }
    if(mpcf_strccmp(newscope,"miniproc")){
        levelout=2;
        thenewlist=&master_death;
    }
    if(mpcf_strccmp(newscope,"program")){
        levelout=3;
        thenewlist=&program_death;
    }

    if(levelout <= levelin)return;    /* do nothing */
    for( thednode = (*thelist)->tail; (thednode != NULL && tomove > 0); tomove--){
       if(*(thednode->target->name) == '^'){
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to change scope of local variable %.4000s",thednode->target->name);
         MPC_WRITE_AND_BAILOUT(501);
       }
       if(thednode->last != NULL)thednode->last->next = NULL;  /* adjust link in list */
       (*thelist)->tail = thednode->last;  /* adjust tail, may make it NULL */
       if((*thelist)->tail == NULL)(*thelist)->head=NULL; /* adjust head, if list is empty */
       mpcf_add_node_to_deathlist(thenewlist, thednode); /* will delete node if list is NULL */
       thednode=(*thelist)->tail;  /* get ready for the next cycle */
    }
}

/*---------------------------------------------------------
   mpcf_findvar(), return the pointer to a variable based on its name, NULL
   if there is no match 
*/

VARIABLE *mpcf_findvar(enum vexist varexist, char *name){
VARIABLE *slide, *current;

  if(*name=='\0')return NULL; /* no way to match a NULL string */
  mpcf_makefullname(name);  /* this sets the value of the global "hashed" */

  current = slide = head[hashed];
  while (slide != NULL){
    current = slide;
    slide = (VARIABLE *)(current->next);
    if(strcmp(current->name,fullname)==0){
      if(current->array == NULL && varexist == MUSTARRAY){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to access regular variable %.4000s as an array ",name);
        MPC_WRITE_AND_BAILOUT(3001);
      }
      return current;
    }
  }
  if(varexist == MUST){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to access or modify nonexistant variable or macro %.4000s",name);
    MPC_WRITE_AND_BAILOUT(3000);
  }
  return NULL;
}


/*---------------------------------------------------------
   mpcf_addvar(), creates another variable based on the name supplied,
   Issues a warning and exits on duplicate names or lack of space.
*/

void mpcf_addvar(char *name, char *sval, int ival, double dval, enum istype type,
  enum isspecial special, enum ismortal mortality){

  int slen;
  VARIABLE *newvar=NULL;

  if(sval!=NULL)slen=1 + strlen(sval);
  if((*trace & MPC_TRACE_ADDVAR)==MPC_TRACE_ADDVAR){
    switch (type) {
    case INTVAR:
      (void) sprintf(mpc_dbg_out,"  add integer variable >%.4000s<",name);
      WRITE_DEBUG();
      (void) sprintf(mpc_dbg_out,"  new value            [%d]",ival);
      WRITE_DEBUG();
      break;
    case DBLVAR:
      (void) sprintf(mpc_dbg_out,"  add double variable  >%.4000s<",name);
      WRITE_DEBUG();
      (void) sprintf(mpc_dbg_out,"  new value            [%e]",dval);
      WRITE_DEBUG();
      break;
    case STRINGVAR:
      (void) sprintf(mpc_dbg_out,"  add string variable  >%.4000s<",name);
      WRITE_DEBUG();
      (void) sprintf(mpc_dbg_out,"  new value            [%.4000s]",sval);
      WRITE_DEBUG();
      break;
    case RPNOPER:
      (void) sprintf(mpc_dbg_out,"  add RPN operator     >%.4000s<",name);
      WRITE_DEBUG();
      (void) sprintf(mpc_dbg_out,"  string value         [%.4000s]",sval);
      WRITE_DEBUG();
      (void) sprintf(mpc_dbg_out,"  index value          [%d]",ival);
      WRITE_DEBUG();
      (void) sprintf(mpc_dbg_out,"  operands value       [%e]",dval);
      WRITE_DEBUG();
      break;
    case MACRO:
      (void) sprintf(mpc_dbg_out,"  adding macro         >%.4000s<",name);
      WRITE_DEBUG();
      break;
    }
  }

/* make sure the name is new */

  if(mpcf_findvar(MAY,name) != NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, duplicate name %.4000s",name);
    MPC_WRITE_AND_BAILOUT(601);
  }

/* create the variable */

  newvar = malloc(sizeof(VARIABLE));
  if(newvar == NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not allocate space for %.4000s",name);
    MPC_WRITE_AND_BAILOUT(602);
  }

/* put the name on */

  newvar->name = malloc((1 + strlen(fullname))*sizeof(char)); /* leave space for the final \0 !! */
  if(newvar->name == NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not allocate space for %.4000s",name);
    MPC_WRITE_AND_BAILOUT(603);
  }
  (void) strcpy(newvar->name,fullname);

/* set all fields, then overwrite those that are particular to certain 
   types*/

  newvar->string=NULL;
  newvar->mfields=NULL;
  newvar->next=NULL;           /* no link forward yet from the new one */
  newvar->last=NULL;           /* no link backward yet from the new one */
  newvar->array=NULL;          /* at creation, all variables are scalar */
  newvar->ssize=0;
  newvar->ival=0;
  newvar->dval=MPC_ZEROD;
  if(mortality == MORTAL){   /* this variable can die */
    mpcf_add_to_current_deathlist(newvar);  /* newvar->assassin set in here */
  }

  switch (type) {
    case STRINGVAR:
      newvar->type=STRINGVAR;
      if(sval !=NULL){
        newvar->string=malloc(slen*sizeof(char)); /* leave space for the final \0 !!*/
        if(newvar->string== NULL){
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not allocate space for %.4000s",name);
           MPC_WRITE_AND_BAILOUT(604); 
           }
        (void) strcpy(newvar->string,sval);
        newvar->ssize=slen;
      }
      newvar->special=special;
      break;
    case MACRO:
      newvar->type=MACRO;
      newvar->mfields=malloc(sizeof(MVARIABLE));
      if(newvar->mfields == NULL){
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not allocate space for %.4000s",name);
         MPC_WRITE_AND_BAILOUT(605);
      } 
     else {
        newvar->mfields->macro=NULL;
        newvar->mfields->body=NULL;
        newvar->mfields->altprefix=NULL;
        newvar->mfields->deathlist=malloc(sizeof(FIRSTDEATHNODE));
        if(newvar->mfields->deathlist == NULL){
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, out of memory");
           MPC_WRITE_AND_BAILOUT(606);
        }
        newvar->mfields->deathlist->head = NULL;
        newvar->mfields->deathlist->tail = NULL;
        newvar->mfields->msize=0;
        newvar->mfields->mc[0]=1;    /* active loop counters, default to 1 pass*/
        newvar->mfields->mc[1]=0;
        newvar->mfields->mc[2]=0;
        newvar->mfields->mcmax[0]=1; /* max loop size counters, default is 1 pass */
        newvar->mfields->mcmax[1]=0;
        newvar->mfields->mcmax[2]=0;
     }

      newvar->string=malloc(MPC_DEFMACRO*sizeof(char)); /* initial allocation for a macro */
      if(newvar->string== NULL){
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not allocate space for %.4000s",name);
         MPC_WRITE_AND_BAILOUT(607);
     }
      newvar->mfields->mstate=EMPTY;
      newvar->mfields->macro=newvar->string;
      newvar->ssize=MPC_DEFMACRO;
      newvar->mfields->msize=0;
      newvar->special=IS_ORDINARY;  /* macros cannot be reassigned, ever */
      newvar->mfields->altprefix=malloc((1+strlen(altprefix->string))*sizeof(char)); /* space to hold altprefix */
      if(newvar->mfields->altprefix== NULL){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not allocate space for %.4000s",name);
        MPC_WRITE_AND_BAILOUT(608);
      }
      (void) strcpy(newvar->mfields->altprefix,altprefix->string); /* a string of some sort */
      recmacro=newvar;             /*the newest macro is always the one recording */
      break;
    case INTVAR:
      newvar->type=INTVAR;
      newvar->special=special;
      newvar->ival=ival;
      break;
    case DBLVAR:
      newvar->type=DBLVAR;
      newvar->special=special;
      newvar->dval=dval;
      break;
    case RPNOPER:
      if(sval !=NULL){
        newvar->string=malloc(slen*sizeof(char)); /* leave space for the final \0 !!*/
        if(newvar->string== NULL){
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not allocate space for %.4000s",name);
           MPC_WRITE_AND_BAILOUT(609);
           }
        (void) strcpy(newvar->string,sval);
        newvar->ssize=slen;
      }
      newvar->type=RPNOPER;
      newvar->special=special;
      newvar->ival=ival;
      newvar->dval=dval;
      break;
    default:
      (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, call to mpcf_addvar() with illegal type");
      MPC_WRITE_AND_BAILOUT(610);
  }

/* put it into head/tail strings. Maintaining the tail string is the ONLY 
way short of following all of the NEXT links again to add a variable to the
end, since mpcf_findvar(MAY,) returns NULL for no match, not the address of the last good one.*/

  if(head[hashed]==NULL){
    head[hashed] = newvar;       /* this is the first variable, so assign head/tail */
    tail[hashed] = newvar;       /* this is the first variable, so assign head/tail */
  }
  else {                         /* already some, so fix tail */
    tail[hashed]->next=newvar;   /* stick in the link from the old tail */
    newvar->last=tail[hashed];   /* stick in the link to the preceding variable */
    tail[hashed] = newvar;       /* update tail */
  }

  return;
}

/*---------------------------------------------------------
   mpcf_setvar(), modifies the value of an existing variable
   Issues a warning and exits on lack of space or an attempt
   to set a nonexistant variable.
   Use mpcf_macro_record to modify (set) macros. */

void mpcf_setvar(char *name, char *sval, int ival, double dval, enum istype type,
  enum isspecial special){
  VARIABLE *old;
  char **tstring;  /* need these pointers because of arrays */
  int   *tssize;

  if((*trace & MPC_TRACE_SETVAR)==MPC_TRACE_SETVAR){
    (void) sprintf(mpc_dbg_out,"setting variable: %.4000s",name);
    WRITE_DEBUG();
    switch (type) {
    case INTVAR:
      (void) sprintf(mpc_dbg_out,"  new value =[%d]",ival);
      WRITE_DEBUG();
      break;
    case DBLVAR:
      (void) sprintf(mpc_dbg_out,"  new value =[%e]",dval);
      WRITE_DEBUG();
      break;
    case RPNOPER:
      (void) sprintf(mpc_dbg_out,"  new string value         [%.4000s]",sval);
      WRITE_DEBUG();
      (void) sprintf(mpc_dbg_out,"  new index value          [%d]",ival);
      WRITE_DEBUG();
      (void) sprintf(mpc_dbg_out,"  new operands value       [%e]",dval);
      WRITE_DEBUG();
      break;
    case STRINGVAR:
      (void) sprintf(mpc_dbg_out,"  new value =[%.4000s]",sval);
      WRITE_DEBUG();
      break;
    }
  }

  /* special case.  safety may only be set from the command line OR from 
     inside the macro SLAVE_MACRO.   */

  if(  fromsource != INFROMCOMMAND){  /* fastest comparison first */
    if(  (strcmp("safety",name)==0)){ /* only do this for safety variable */
      if(fromsource == INFROMMACRO ){
        if(! strcmp(activemacro[isactivemacro]->name,"SLAVE_MACRO")){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal safety error, attempt to set safety variable within macro");
          MPC_WRITE_AND_BAILOUT(701);
        }
      }
      else {
        (void) sprintf(mpc_dbg_out,"miniproc, fatal safety error, attempt to set safety variable within file");
        MPC_WRITE_AND_BAILOUT(707);
      }
    }
  }

  /* make sure the variable exists */

  old=mpcf_findvar(MUST,name);

  /* make sure the types match OR this is a special variable */

  if (type != old->type && IS_ORDINARY == old->special){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, type mismatch when setting variable %.4000s",name);
    MPC_WRITE_AND_BAILOUT(703);
  }

  /* make sure this is not a Macro, those can be recorded, but not set */

  if (old->type == MACRO){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to set macro= %.4000s",name);
    MPC_WRITE_AND_BAILOUT(704);
  }

  /* everything is fine, so SET it.  Only set those bits that change - don't 
   waste time freeing strings for a special variable that goes stringvar->
   intvar since that string will be freed later if need be, and once type goes to
   intvar nothing will be able to get to the old string.   */

  switch (type) {  /* will only be stringvar or intvar */
    case STRINGVAR:
      if(old->array  == NULL){
        tstring = &(old->string);
        tssize  = &(old->ssize);
      }
      else {
        tstring = &((STRINGNODE *)old->array->data)[old->array->offset].string;
        tssize  = &((STRINGNODE *)old->array->data)[old->array->offset].ssize;
      }

      if(old->string != NULL){
        free(*tstring);     /* lose old data */
        *tstring=NULL;
      }

      if(sval !=NULL){
        *tstring = malloc((1 + strlen(sval))*sizeof(char)); /* leave space for the final \0 !!*/
        if(*tstring == NULL){
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not allocate space for %.4000s",name);
           MPC_WRITE_AND_BAILOUT(705);
        }
        (void) strcpy(*tstring , sval);
        *tssize = 1 + strlen(sval);
      }
      else {
        *tstring = NULL;
        *tssize  = 0;
      }
      break;
    case INTVAR:
      if(old->array  == NULL){
        old->ival=ival;
      }
      else {
        ((int *) old->array->data)[old->array->offset] = ival;
      }
      break;
    case DBLVAR:
      if(old->array  == NULL){
        old->dval=dval;
      }
      else {
        ((double *) old->array->data)[old->array->offset] = dval;
      }
      old->dval=dval;
      break;
    case RPNOPER:
      if(old->string != NULL)
        free(old->string);     /* loose old data */
        old->string=NULL;
      if(sval !=NULL){
        old->string=malloc((1 + strlen(sval))*sizeof(char)); /* leave space for the final \0 !!*/
        if(old->string== NULL){
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not allocate space for %.4000s",name);
           MPC_WRITE_AND_BAILOUT(706);
           }
        (void) strcpy(old->string,sval);
        old->ssize=1+strlen(sval);
      }
      old->ival=ival;
      old->dval=dval;
      break;

  } /* end of switch on type */

  old->type=type;
  old->special=special;


} /* end of mpcf_setvar() */

/* mpcf_clear_deathlist(deathlist).  Deletes all variables/macros in the
deathlist (recursively).  When it returns, the deathlist is empty.
It calls mpcf_delvar_byvar, which can call mpcf_clear_deathlist again, and so
move recursively down through a complex nested deathlist.
*/

void mpcf_clear_deathlist(FIRSTDEATHNODE *thelist){
DEATHNODE  *dnode;
     for(dnode=thelist->tail; dnode != NULL; dnode=thelist->tail){
        mpcf_delvar_byvar(dnode->target); 
        dnode->target=NULL;     /* clear the pointer to the variable too */
     }
}


/* mpcf_delvar_byvar().  Called by delvar, does the real work, also calls itself
   recursively if it needs to to clean up deathlists attached to macros.
   Should not have any states which cause it to try to exit - as it may be
   called from the exit handler (see mpcf_reset_mpc() ).
   Note that all pointers are set to NULL after they are freed.  This helps
   in detecting bugs on some platforms. */

void mpcf_delvar_byvar(VARIABLE *old){
DEATHNODE  *dnext;
DEATHNODE  *dlast;
VARIABLE *nextnode;
VARIABLE *lastnode;

  if((*trace & MPC_TRACE_DELVAR)==MPC_TRACE_DELVAR){
    (void) sprintf(mpc_dbg_out,"     deleting variable >%.4000s< by name or deathlist",old->name);
    WRITE_DEBUG();
  }
  if(old->assassin == NULL){  /* if it isn't on a deathlist, it is immortal, and may not be removed */
     (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to delete immortal variable/macro %.4000s",old->name);
     MPC_WRITE_AND_BAILOUT(3801);
  }
  if(old->type == MACRO){ /* it may have its own deathlist, clean that up first */
     mpcf_clear_deathlist((*(*old).mfields).deathlist);

     /* remove the macro specific variable size memory structures here */

     free(old->mfields->deathlist);  /* the deathlist */
     old->mfields->deathlist = NULL;
     free(old->string);              /* the macro contents */
     old->string = NULL;             /* in a macro ->string points to the macro strings*/
     old->mfields->macro = NULL;     /* ->macro is a pointer within these fields, 
                                        when ->string is freed, ->macro dangles  */
     if(old->mfields->altprefix != NULL){
          free(old->mfields->altprefix);  /* altprefix, if one is defined */
          old->mfields->altprefix = NULL;
        }
     free(old->mfields);             /* now the full mfields structure */
     old->mfields = NULL;
  }

  /* it is now safe to remove this variable, as its own deathlist, if any, has 
     been removed.  But first adjust all of the deathlist links around it
     in the deathlist which contains it.  */

  dnext = old->assassin->next;  /* next deathnode, if any */
  dlast = old->assassin->last;  /* last deathnode, if any */
  if(dnext != NULL){
    if(dlast != NULL){  /* middle of a list */
      dlast->next = dnext;
      dnext->last = dlast;
    }
    else {             /* front of a list */
      old->assassin->list->head = dnext;
      dnext->last = NULL;
    }
  }
  else {
    if(dlast != NULL){  /* end of list */
      old->assassin->list->tail = dlast;
      dlast->next = NULL;
    }
    else {             /* sole member of a list*/
      old->assassin->list->head = NULL;
      old->assassin->list->tail = NULL;
    }
  }

  /* now clean up the links in the hash list which contains it,
     if any. [hashed] needs to be recreated on the fly */

  mpcf_makefullname(old->name); /* sets global "hashed" */
  nextnode = old->next;  /* next hash node, if any */
  lastnode = old->last;  /* last hash node, if any */
  if(nextnode != NULL){
    if(lastnode != NULL){  /* middle of a list, head/tail unaffected */
      lastnode->next = nextnode;
      nextnode->last = lastnode;
    }
    else {             /* front of a list, head affected */
      head[hashed] = nextnode;
      nextnode->last = NULL;
    }
  }
  else {
    if(lastnode != NULL){  /* end of list, tail affected */
      tail[hashed]   = lastnode;
      lastnode->next = NULL;
    }
    else {             /* sole member of a list, both head/tail affected */
      head[hashed] = NULL;
      tail[hashed] = NULL;
    }
  }
 
  /* now remove the remaining bits and pieces of this variable from memory */

  free(old->assassin);       /* remove the deathnode structure */
  free(old->name);           /* remove the name string */
  old->assassin = NULL;
  old->name     = NULL;
  if(old->string != NULL){   /* careful about Macro, else dangling pointer blows up here/ */
      free(old->string);     /* remove the string, if there is one */
      old->string = NULL;
  }
  if(old->array != NULL)mpcf_emasculate_variable(old);
  free(old);                 /* remove all of the rest of it */
}

/* mpcf_emasculate_variable() strips the ARRAY part off of a variable  */

void mpcf_emasculate_variable(VARIABLE* thevar){
int k,i;

  if(thevar->array != NULL){    /* clear out array contents */
    if(thevar->type == STRINGVAR){ /* array of strings, free their memory before freeing the array */
      for( k=1,i=0 ; i< MPC_TUPLE; i++){  /* figure out how many free's to do */
         k = k * thevar->array->dim[i];
      }
      for(i=0 ; i < k; i++){ /* now remove all the strings' memory, one by one */
         /* it's possible this routine might be called during a failure in 
            an array operation, where some strings do NOT have storage 
            associated with them, so check the pointer  */
         if(((STRINGNODE *)thevar->array->data)[i].string != NULL){
              free( ((STRINGNODE *)thevar->array->data)[i].string  );
         }
/* not required as next line will erase it anyway
         ((STRINGNODE *)thevar->array->data)[i].string = NULL;
*/
      }
    }
    free( thevar->array->data );  /* free the array itself */
    free( thevar->array);         /* free the ARRAYNEXUS */
    thevar->array=NULL;
  }
}


/* mpcf_delvar().  Deletes a variable, frees its allocated space, removes
   any associated deathnodes.  If the variable is a macro, it will also
   remove all data associated with that (everything in its deathlist
   and deathlists below it */

void mpcf_delvar(char *name){
VARIABLE *old;

   if((*trace & MPC_TRACE_DELVAR)==MPC_TRACE_DELVAR){
      (void) sprintf(mpc_dbg_out,"  .free. variable >%.4000s< explicitly by name",name);
      WRITE_DEBUG();
   }
   old=mpcf_findvar(MUST,name);

   /* Remove it.  mpcf_delvar_byvar will verify that it is on a deathlist (not immortal)*/
   mpcf_delvar_byvar(old);   /* no need to clear this pointer, it isn't global */
}

/* mpcf_setaltprefix, this is done often enough that it is worth a separate 
   routine to skip all of the hash lookups */

void mpcf_setaltprefix(char *newaltprefix){
  free(altprefix->string); /* safe - it cannot be NULL*/
  altprefix->string = malloc((strlen(newaltprefix) + 1 )*sizeof(char));
  if(altprefix->string == NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, out of memory");
    MPC_WRITE_AND_BAILOUT(901);
  }
  (void) strcpy(altprefix->string,newaltprefix);
}
/* -----------------------------------
   mpcf_stuffaltprefix() sets the altprefix field in an ifstack entry.
   This is called by f$in and mpcf_do_macro
*/
void mpcf_stuffaltprefix(void){
  if(strlen(altprefix->string) > MPC_MAXALTPREFIX-1){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, altprefix [%.4000s] is too long",altprefix->string);
    MPC_WRITE_AND_BAILOUT(1001);
  }
  *(ifstack[ifptr].altprefix)='\0';
  (void) strncat(ifstack[ifptr].altprefix,altprefix->string,MPC_MAXALTPREFIX-1);
}

/* -----------------------------------
   mpcf_yankaltprefix() sets the altprefix field from the value in an ifstack entry.
   This is called by f$exit, f$break, f$macro_break, f$macro_return.
*/

void mpcf_yankaltprefix(void){
  mpcf_setaltprefix(ifstack[ifptr].altprefix);
}


/* -----------------------------------
   mpcf_strip_trailing_comments.  Removes !comment! from the ends of lines.
   The integer value returned is the position of the first "!" in the 
   comment.  That site is also set to '\0'.
*/
int  mpcf_strip_trailing_comments(char * string, int endstring, int lowerlimit){
int holdend;

   /* This function is only called if the last character on the line is a
      '!', so predecrement endstring before starting the loop */ 
 
   for(holdend=endstring, endstring--; endstring >= lowerlimit ; endstring--){
      if(string[endstring]=='!'){
        string[endstring]='\0';  /* it may be used outside in a strcpy */
        return endstring;
       }
   }
   return holdend;
}


/* -----------------------------------
   mpcf_pathsafe().  Modify a file descriptor string to
   return only the string to the right of any /\]>:
   that might be present, and so restrict access to
   the current directory (hopefully on any OS that this might run on!)
*/
void mpcf_pathsafe(char *string){
char *from;
char *to;
  for(to=string,from=string; *from!='\0'; from++){
    switch (*from){
      case '/':
      case '\\':  /* escaped \ character */
      case ']':
      case '>':
      case ':':
        to=string;
        break;
      default:
        *to=*from;
        to++;
        break;
    }
  }
  *to='\0'; /* terminate string */
} /* end of mpcf_pathsafe */

void mpcf_setcounters(VARIABLE *play,int setmc, int setvismc){
int i;

  if(setmc !=0 ){
    for (i=0 ; i<=2 ; i++){
      if (play->mfields->mcmax[i] == 0)
        play->mfields->mc[i] = 0;
      else
        play->mfields->mc[i] = 1;
    }
  }

  if(setvismc !=0 ){  /* copy mc, mcmax into the visible variables */
    for (i=0 ; i<=2 ; i++){
      mcshort[i]->ival = play->mfields->mc[i];
      mcmaxshort[i]->ival = play->mfields->mcmax[i];
    }
  }

} /*end of mpcf_setcounters */

/* --------------------------------------------------------
    mpcf_sort_rpn_operators().  Called once (more is pointless).
    It sorts the rpn operator records into ascending order by
    operator name.  When completed, it sets the global variable
    MPC_ISMAX_RPN to the size of the array, not counting the empty
    operator at the end.  This does a comb sort,
    fast enough for something that only needs to be done once.
*/
void mpcf_sort_rpn_operators(void){
OPNODE swap_optype;
int i;
int jumpsize;
int swapped;
#define shrink_top    10
#define shrink_bottom 13

/* first find out how many RPN operators there are, last one has an empty
   string for a name.  At the same type, check for any bad argnumber bits
   the programmer may have set. */

  for(i=0; *(all_optypes[i].name) != '\0' ;i++){
    if(all_optypes[i].argnumber > MPC_RPNARGS || 
       all_optypes[i].argnumber < 0){
         (void) sprintf(mpc_dbg_out,
         "miniproc, fatal programming error, invalid argnumber for RPN operator %.4000s ",all_optypes[i].name);
         MPC_WRITE_AND_BAILOUT(3901);
    }
  };
  MPC_ISMAX_RPN=i-1;

/* now combsort it */

  for(jumpsize=MPC_ISMAX_RPN; jumpsize >= 1;){
    for(i=0, swapped=0;  i+jumpsize <= MPC_ISMAX_RPN; i++){
       if(strcmp(all_optypes[i].name, all_optypes[i+jumpsize].name) > 0){
	 swapped++;
         swap_optype = all_optypes[i];
         all_optypes[i] = all_optypes[i+jumpsize];
         all_optypes[i+jumpsize] = swap_optype;
       }
    }
    if(jumpsize == 1 && swapped == 0)break;               /* sort is completed */
    jumpsize = (jumpsize * shrink_top) / shrink_bottom ;  /* divide by 1.3 */
    if(jumpsize < 1)jumpsize=1; /* in case it happens to truncate to zero */
  }
}

/* --------------------------------------------------------
    mpcf_parse_rpn_operator(), is passed a string like
    .operator_#. and tries to parse out the name
    of the operator (case doesn't matter?), and also
    the optional _# part (number of operands).  Operator ARRAY
    POSITION (NOT operator number!!!!!!!!)
    comes back in ival, operands in dval (default
    is 2).
  -------------------------------------------------------- */
void mpcf_parse_rpn_operator(char *string,int *ival, double *dval){
char *underscore;
char *hold_underscore;
char *lastdot;
int  looking, lowrange, highrange;
char *schar;
char *ochar;
int i;

  /* is there an optional operand count? */
  string++; /* nick off the leading "." */
  *dval=1.0; /* indicate operands from table */
  for (hold_underscore=NULL,underscore = string, lastdot=NULL; *underscore!='\0'; underscore++){
    if(*underscore == '_'){
      *underscore = '\0';  /* delimit the operator */
      *dval = (double) MPC_MAXSTACK;  /* indicate operands from _, not from table, 
                            operands must be integer and >= 0 */
      hold_underscore=underscore;
      break;
    }
    if(*underscore=='.')lastdot=underscore;
  }
  if(lastdot!=NULL)*lastdot='\0'; /* for forms with no _ in them */

  /* which operator is it?  This finds the ARRAY position, which isn't the 
     same as the Operator number!!!!*/

/* 3.01 and earlier used linear search
struct isanop *opptr;
  for(i=0,opptr=all_optypes ; strlen(opptr->name)!=0 ; i++,opptr++){
    if(strcmp(opptr->name,string)==0)break;
  }
  if (strlen(opptr->name) == 0){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal RPN operator error, unknown operator %.4000s ",string);
    MPC_WRITE_AND_BAILOUT(1101);
  }
*/

/* 3.03 and later use binary search on the sorted list of rpn operators.  
This search method typically only checks a few letters of each operator 
before determining the next step in the binary search */

  for(looking=1, i=MPC_ISMAX_RPN/2, lowrange=0, highrange=MPC_ISMAX_RPN; looking != 0;){
    for(schar=string, ochar=all_optypes[i].name;  ; schar++,ochar++){
      if(*schar == *ochar){
        if(*schar == '\0'){  /* this is a match on an RPN operator*/
          looking = 0;
          break;
        }
      }
      else { /* mismatch on a character, next step or give up */
        if(*schar > *ochar)lowrange  = i+1;
        if(*schar < *ochar)highrange = i-1;
        if(lowrange > highrange){  /* this happens when last test was lowrange==highrange */
          (void) sprintf(mpc_dbg_out,"miniproc, fatal RPN operator error, unknown operator %.4000s ",string);
          MPC_WRITE_AND_BAILOUT(1101);
        }
	else {
	  i = (highrange + lowrange)/2 ;
	}
        break;
      } /* end of mismatch character condition */
    }
  }

  if(*dval == (double) MPC_MAXSTACK){ /* number of operands from _ suffix */
    underscore++;
    if(*underscore != '.'){     /* arg count present, *dval will not be 0 */
      errno=0;
      *dval = strtod(underscore, NULL);
      if(errno){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal RPN operator syntax error, invalid operand count %.4000s ",string);
        MPC_WRITE_AND_BAILOUT(1102);
      }
    }
  }
  else { /* number of operands from table = defoperands, unless that is
            zero, then specify at least one be done */
    *dval = (double) mpc_max(1,all_optypes[i].defoperands);
  }

  *ival = i; /* pointer into all_optypes, but NOT the operator number */
  /* restore string to its previous state, so that it can be stored in a 
     variable, if needed */
  if(hold_underscore!=NULL)*hold_underscore='_';
  if(lastdot!=NULL)*lastdot='.';
}

/*---------------------------------------------------------
   mpcf_resolveright(), interprets the right hand side of an =
   statement.  Handles, *,&," operators.  Determines the type
   of the variable.  This is how it maps out:
   name        contents of variable "name"
   &name       immediate string value
   "name sdf sdf jlsdf "  immediate string value
   [-,0-9]     immediate integer value
   ***name     indirect value from variable
   .whatever. 
   .whatever_.
   .whatever_#.  immediate RPN operators, returns the NUMBER of operator
                 in ivar, and the optional # (number of operands) is
                 returned in dvar.

   mode 0 rejects resolving a "plain" macro name
   mode 1 resolves a "plain" macro name to itself (so no &name
       or "name" is required.)  The resolved name comes back in sval
       and the type is set to STRINGVAR
   mode 2 resolves a macro name, will return a MACRO type.  It will also
          return a string that is not stored as an immediate string.  Used
          to pass f$evaluate back to mpcf_do_ifelse
   mode 3 Used by do_subs. Similar to mode 2, except that it assumes it is
          looking at the name of a variable, so skips looking for -,& and so forth.  
          If it does not find the variable, it returns the search string 
          with type STRINGVAR.  Only INTVAR, DBLVAR, RPNOPER, and STRINGVAR are valid
          types in this mode.  A real Macro, or unmatched variable sets the 
          MACRO type, which is ignored in do_subs.
*/

void mpcf_resolveright(char *string, char **sval, int *ival, double *dval, enum istype *type, int mode){
  char first;
  int  redirect;
  VARIABLE *ourvar;
  int itemp;

  if(mode != 3){  /* only possible variables are considered in this mode  */
  first=*string;
  if(first == '&'){ /* immediate string value */
    *type=STRINGVAR;
    *sval=string;
    (*sval)++; /* point to string, not to & that precedes it */
    *ival=0;
    *dval=MPC_ZEROD;
    return;
  }
  if( (first == '[' || first == ']') && strlen(string)==1 ){ /* delimited [ or ] , special immediate string values */
    *type=STRINGVAR;
    *sval=string;
    *ival=0;
    *dval=MPC_ZEROD;
    return;
  }
  if(first == '"'){ /* immediate externally quoted string value */
    *type=STRINGVAR;
    *sval=strrchr(string,'"');
    if(*sval == NULL){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal syntax error, unmatched double quotes");
      MPC_WRITE_AND_BAILOUT(12001);
    }
    **sval='\0';
    (*sval)++;   /* there should not be anything after this except maybe spaces or tabs */
    if(strcspn(*sval,"	 ")){ /* tab and space inside the match string */
      (void) sprintf(mpc_dbg_out,"miniproc, fatal syntax error, remainder following double quote string");
      MPC_WRITE_AND_BAILOUT(12009);
    }
    *sval=string;
    (*sval)++; /* point to string, not to " that precedes it */
    ival=0;
    *dval=MPC_ZEROD;
    return;
  }
  if(first == '\''){ /* immediate internally quoted string value */
    *type=STRINGVAR;    /* mpcf_strtok will have insured that it begins and 
                           ends with "'", but there may be more internally */
    string++;  /* point to string, not to ' that precedes it */
    for(*sval = string; **sval != '\0'; (*sval)++){} /* find the end */
    (*sval)--; /* final "'" character */
    if(**sval != '\''){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, did not detect syntax error, unmatched single quotes");
      MPC_WRITE_AND_BAILOUT(12006);
    }
    **sval='\0'; /* terminate the string */
    (*sval)++;   /* there should not be anything after this except maybe spaces or tabs */
    if(strcspn(*sval,"	 ")){ /* tab and space inside the match string */
      (void) sprintf(mpc_dbg_out,"miniproc, fatal syntax error, remainder following single quote string");
      MPC_WRITE_AND_BAILOUT(12008);
    }
    *sval=string; /* put pass variable back at beginning */
    ival=0;
    *dval=MPC_ZEROD;
    return;
  }
  itemp=strlen(string)-1;  /* .something. can be an RPN operator */
  if( (first == '.') && itemp>=2 && (string[itemp] == '.')  ){ /* immediate RPN operator */
     mpcf_parse_rpn_operator(string,ival,dval);
     *type=RPNOPER;
     *sval=string;
     return;
  }
  if(first == '-' || first == '.' && string[itemp] != '.' || (first >= '0' && first <= '9')){ /* immediate value,
              integer or double, eliminate '.' and '.anything.' from consideration */
    if(strcspn(string,".eE") == strlen(string)){ /* must be an integer*/    
      *type=INTVAR;
      if(sscanf(string,"%d",ival)==EOF){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal syntax error, invalid integer %.4000s ",string);
        MPC_WRITE_AND_BAILOUT(1202);
      }
      *sval=string;  /* have to set it to something... */
      *dval=MPC_ZEROD;
      return;
    }
    else { /* a double */
      *type=DBLVAR;
      errno=0;
      *dval = strtod(string, NULL);
      if(errno){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal syntax error, invalid double %.4000s ",string);
        MPC_WRITE_AND_BAILOUT(1203);
      }
      *sval=string;  /* have to set it to something... */
      *ival=0;
      return;
    }
  } /* immediate value */
  } /* mode != 3 */

/* find out how many levels of variable redirection have been
   requested, ie *, **, ***, etc. It could be NONE */

   for (*sval = string, redirect=0;
        redirect < strlen(string);
        redirect++,(*sval)++){
      if(**sval!='*')break;
   }

/* next error condition happens if the string "****" is encountered - a 
series of redirects, but no symbol at the end */

   if(redirect==strlen(string)){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal syntax error, rvalue may not be %.4000s ",string);
      MPC_WRITE_AND_BAILOUT(1204);
   }

/* now extract the value from the variable, or die trying.  In modes 2,3, if
   the value does NOT map to a preexisting variable, then that string will be
   returned verbatim.  Do as many redirects as required. */

   for (ourvar=mpcf_findvar(MAY,*sval);redirect>0;redirect--){
     if(ourvar!=NULL){
       if(ourvar->array  != NULL){ /* redirect through string arrays are allowed */
         if( ((STRINGNODE *)ourvar->array->data)[ourvar->array->offset].string !=NULL) {
           ourvar=mpcf_findvar(MAY,((STRINGNODE *)ourvar->array->data)[ourvar->array->offset].string);
         }
         else {ourvar=NULL;}
       }
       else {
         if(ourvar->string !=NULL) {ourvar=mpcf_findvar(MAY,ourvar->string);}
         else {ourvar=NULL;}
       }
     }
     else {
       break;
     }
   }

   if(ourvar==NULL){  /* not a variable, return the query string */
      switch (mode){
        case 2: /* goes various places, means use string verbatim */
         *type=STRINGVAR;
         *sval=string;
         *ival=0;
         *dval=MPC_ZEROD;
         return;
        case 3: /* goes to mpcf_do_subs, tells it to ignore subsitution attempt */
         *type=MACRO;
         return;
        default: /* not a variable, fatal error in other modes */
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not resolve value of \"%.4000s\" ",string);
          MPC_WRITE_AND_BAILOUT(1205);
      }
   }

/* if we have not bailed out yet, then ourvar hold the REAL value that we 
   want */

   switch (ourvar->type){
      case INTVAR:
         *sval=ourvar->string;
         *dval=ourvar->dval;
         *type=ourvar->type;
         if(ourvar->array == NULL){*ival=ourvar->ival;}
         else {*ival=((int *)ourvar->array->data)[ourvar->array->offset];}
         break;
      case DBLVAR:
         *sval=ourvar->string;
         *ival=ourvar->ival;
         *type=ourvar->type;
         if(ourvar->array == NULL){*dval=ourvar->dval;}
         else {*dval=((double *)ourvar->array->data)[ourvar->array->offset];}
         break;
      case STRINGVAR:
         *dval=ourvar->dval;
         *type=ourvar->type;
         if(ourvar->array == NULL){
           *sval=ourvar->string;
           *ival=ourvar->ssize; /* pass out the reserved size */
         }
         else {
           *sval=((STRINGNODE *)ourvar->array->data)[ourvar->array->offset].string;
           *ival=((STRINGNODE *)ourvar->array->data)[ourvar->array->offset].ssize;
         }
         break;
      case RPNOPER:
         *sval=ourvar->string;
         *ival=ourvar->ival; /* pass out the operator array position */
         *dval=ourvar->dval;
         *type=ourvar->type;
         break;
      default:
        switch (mode){
          case 0:
            (void) sprintf(mpc_dbg_out,"miniproc, fatal error, right value references macro, not variable %.4000s ",string);
            MPC_WRITE_AND_BAILOUT(1207);
          case 1: /* used to coerce the use of macro names as plain strings*/
            *sval=ourvar->name;
            *ival=ourvar->ival; /* just be sure something is in there */
            *dval=ourvar->dval; /* just be sure something is in there */
            *type=STRINGVAR;
             break;
          case 2: /* used to coerce a macro type out, with name in sval*/
            *sval=ourvar->name;
            *ival=ourvar->ival; /* just be sure something is in there */
            *dval=ourvar->dval; /* just be sure something is in there */
            *type=MACRO;
             break;
        }
   }
   return;
}

void mpcf_trim_command_line(char *line){
char *p;
char *from;
char *to;
int i,j;

  if(strlen(line)<1){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, command line has zero length");
    MPC_WRITE_AND_BAILOUT(1301);
  }

  /* remove any preceding tabs or spaces, if any */

  for (from=line,i=1,j=1; i<=strlen(line); i++,from++){
    switch (*from){
       case ' ':
       case '\t':
         break;
       default:
         j=i;
         from--; /* because for will increment it again */
         i=strlen(line)+10; /* force exit on for loop */
         break;
    }
  }
  if( j != 1 ){  /*drag it all left*/
    for (i=j, to=line; i<=strlen(line); *to=*from,i++,from++,to++){}
    *to='\0';  /* terminate the string at the new end */
  }

  p = strrchr(line,' '); /* find final space */
  if (p == NULL) return; /* no spaces */
  p++;                   /* see which character is next */
  if( *p != '\0')return; /* last space is not a trailing space */
   for (p--;p!=line;p--){ /* starting with the space, trim backwards */
    if(*p!=' ')
      return;
    else
      *p='\0';
  }
/* only get here if the line has (maybe) one character, then all spaces
   which isn't a legal line*/

  (void) sprintf(mpc_dbg_out,"miniproc, fatal error, command line has nothing on ");
  MPC_WRITE_AND_BAILOUT(1302);
}



/* mpcf_do_subs makes the determined number of passes through the line
   replacing <<variable>> constructs with the variable's contents. Have
   to be careful, as something like <<fo<foobar>> is legal text, it
   just should not be substituted as fo<foobar is an illegal variable name.
   However, issue warnings for things like this!  The maximum length of
   the expanded string is MPC_MAXINLINE -1 (for the terminator character)
   didsubs keeps track of the number of changes on a line, if dosubnum
   is large it stops doing substitutions after the first pass that did
   none.  */

void mpcf_do_subs(char *line){
int count,maybefront,maybeback,fragfront;
char cinline[MPC_MAXINLINE];
char coutline[MPC_MAXINLINE];
char cmaybevar[MPC_MAXVARLEN];
char csintvar[32]; /* will be used for expanding integer variables */
char *vinline=&cinline[0];
char *outline=&coutline[0];
char *maybevar=&cmaybevar[0];
char *sintvar=&csintvar[0];
char *runner;
char *mrunner;
char *rl1;
char choldfront;
int didsubs;
int subsruns=0;
char *lastfrag;

/* next group are needed for resolveright */

char *sval;
char **tsval=&sval;
int ival;
double dval;
enum istype type;



  if(strlen(line) > MPC_MAXINLINE){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, line is longer than the maximum allowed [%d]",MPC_MAXINLINE);
    MPC_WRITE_AND_BAILOUT(1401);
  }
  (void) strcpy(vinline,line);
  switch (howtohandle){
    case MPC_HANDLE_NORMAL:
      subsruns=*dosubnum;
      break;
    case MPC_HANDLE_RECORD:
      subsruns=*domacrosubnum;
      break;
  }
  
  for (didsubs=1  ; (didsubs>0 && subsruns>0) ; subsruns--){
    for (vinline=&cinline[0],
         runner=vinline,
         count=1,
         didsubs=0,
         maybefront=strlen(vinline)+1,
         maybeback=strlen(vinline)+1,
         lastfrag=vinline,
         outline=&coutline[0],
         *outline='\0',
         fragfront=0,rl1=NULL
         ;
         count<=strlen(&cinline[0])
         ;
         rl1=runner,
         runner++,
         count++){
      if(count >= 2){
        switch (*runner){
          case '>':
            if(*rl1=='>' && choldfront=='<') /* << must match with >> */
              maybeback=count-2;
            else
              maybeback=0;
            break;
          case '}':
            if(*rl1=='}' && choldfront=='{') /* {{ must match with }} */
              maybeback=count-2;
            else
              maybeback=0;
            break;
          case '<':
            if(*rl1=='<'){
              choldfront='<';
              maybefront=count+1;
	      mrunner=runner;
              mrunner++;
              }
            else
              maybefront=0;
            break;
          case '{':
            if(*rl1=='{'){
              choldfront='{';
              maybefront=count+1;
	      mrunner=runner;
              mrunner++;
              }
            else
              maybefront=0;
            break;
          default:
          break;
        }

/* test if maybefront and maybeback are defined */
        if(maybefront!=0 && maybeback!=0 && 
          maybefront<=maybeback && maybeback <= strlen(vinline)){
  
          if( maybeback-maybefront+1 > 0){
            *maybevar='\0';
            (void) strncat(maybevar,mrunner,maybeback-maybefront+1);
          }

          mpcf_resolveright(maybevar,tsval,&ival,&dval,&type,3);
  
/* if the variable has been found, it will come back with a type OTHER than
MACRO.  If the variable was a MACRO, then we ignore it anyway. */
  
          if(type != MACRO){
            didsubs++;
            (void) strncat(outline,lastfrag,maybefront-fragfront-3);
            fragfront=maybeback+2;

            switch (type){
              case STRINGVAR:
              case RPNOPER:
                if(strlen(sval)+strlen(outline) > MPC_MAXINLINE){
                  (void) sprintf(mpc_dbg_out,"miniproc, fatal error, substituted line is longer than the maximum allowed [%d]",MPC_MAXINLINE);
                  MPC_WRITE_AND_BAILOUT(1402);
                }
                (void) strcat(outline,sval);
                break;
              case INTVAR:
                (void) sprintf(sintvar,"%d",ival);
                (void) strcat(outline,sintvar);
                break;
              case DBLVAR:
                (void) sprintf(sintvar,"%e",dval);
                (void) strcat(outline,sintvar);
                break;
              default:
                (void) sprintf(mpc_dbg_out,"miniproc, fatal error, programming error in do_subs");
                MPC_WRITE_AND_BAILOUT(1404);
                break;
            }
            lastfrag = runner;
            lastfrag++; /* next piece to be copied over, maybe */
            maybefront=0;
            maybeback=0;
          }
        }
      } /* end of test on count */
    } /* end of inner for loop */

/* there may be some remaining, if so, pass it through */
   if(count - fragfront >= 2){
     if( ( strlen(outline) + count - fragfront) > MPC_MAXINLINE){
       (void) sprintf(mpc_dbg_out,"miniproc, fatal error, substituted line is longer than the maximum allowed [%d]",MPC_MAXINLINE);
       MPC_WRITE_AND_BAILOUT(1403);
     }
     (void) strncat(outline,lastfrag,count - fragfront);
   }
   (void) strcpy(vinline,outline); /* copy expanded string back to vinline */
   if( (didsubs !=0) && ((*trace & MPC_TRACE_SUBS)==MPC_TRACE_SUBS)){
      (void) sprintf(mpc_dbg_out,"  substitute >%.4000s",vinline);
      WRITE_DEBUG();
   }
  } /* end of outer for loop */
  (void) strcpy(line,vinline); /* copy final string back to input area */
}

/*---------------------------------------------------------
   mpcf_macro_record().  If the macro is in an appropriate state, it
   appends lines to the macro buffer.  These are stored as 
   string\0string\0string\0 and the total length of this string
   is kept in msize (including the final null).  The total number
   of available bytes for storage is kept in ssize.
*/

void mpcf_macro_record(char *name, char *sval){
  int i;
  int split;

  if(sval == NULL){ /* creating a new macro */
    if(recmacro!=NULL){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, trying to simultaneously record a second macro %.4000s",name);
      MPC_WRITE_AND_BAILOUT(1501);
    }
    mpcf_addvar(name, NULL, 1, MPC_ZEROD, MACRO, IS_ORDINARY, MORTAL); /* create the macro, side effect, set "recmacro" */
    howtohandle = MPC_HANDLE_RECORD;
    return;
  }

  if (recmacro == NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, attempt to record nonexistant macro %.4000s",name);
    MPC_WRITE_AND_BAILOUT(1502);
  }

  /* check the state of the macro, it must not be DONE */

  switch (recmacro->mfields->mstate){
     case EMPTY:
       recmacro->mfields->mstate=RECORDING;
     case RECORDING:
       break;
     case DONE:
     case PLAYING:
       (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to rerecord macro %.4000s",name);
       MPC_WRITE_AND_BAILOUT(1503);
       break;
  }

  /* find out if there is room for the next string, if not realloc */

  if(strlen(sval) + recmacro->mfields->msize + 1 > recmacro->ssize){
    recmacro->string=realloc(recmacro->string,(recmacro->ssize+MPC_DEFMACRO)*sizeof(char));
    if(recmacro->string == NULL){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, ran out of space recording macro %.4000s",name);
      MPC_WRITE_AND_BAILOUT(1504);
    }
    recmacro->ssize=recmacro->ssize+MPC_DEFMACRO;

   /* fix the recmacro->macro pointer, it may point to freed memory, note
      that the pointer must be positioned immediately after the last
      defined character (which is \0) */

    for (recmacro->mfields->macro=recmacro->string , i=0   ;
        i < recmacro->mfields->msize                       ;
        recmacro->mfields->macro++,i++){}

  }

  /* append the string to the macro, leaving pointer AFTER the terminating NULL */

  for(split=0 ; !split ; (recmacro->mfields->macro)++, (recmacro->mfields->msize)++, sval++){
    *(recmacro->mfields->macro)=*sval;;
    if(*sval == '\0')split=1;
  }

/* add the string witha \0 on the end 
  (void) strcpy(recmacro->mfields->macro,sval); 
  for(i=1;i<=1+strlen(sval);recmacro->mfields->macro++,i++,recmacro->mfields->msize++){}
*/

}


/*---------------------------------------------------------
   mpcf_macro_repeat().  Sets the repeat counter buffers in the 
   named macro.  Repeat counts must all be >=0, and the first
   must be like 1,1,0, not 1,0,1 or 0,1,1  (that is, loops
   from left, and cannot have a null loop "inside").
   If a VARIABLE is supplied, then it must be an array, and
   the repeat counts are mapped from the DIM values.
*/

void mpcf_macro_repeat(char *name,VARIABLE *mapfrom){
char *sval;
char **tsval=&sval;
int ival;
double dval;
enum istype type;
VARIABLE *play;
int i;
int last;

  play=mpcf_findvar(MUST,name);

  if(play->mfields->mstate != DONE){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_repeat setting counters on active macro ");
    MPC_WRITE_AND_BAILOUT(1602);
  }

  /* the next part of the loop will only set those values that were 
  supplied.  So here set first to 1 and all others to zero, which is
  the default*/

  for (i=0; i<MPC_TUPLE;i++){play->mfields->mcmax[i]=0;}  
  play->mfields->mcmax[0]=1;

  if(mapfrom == NULL){
    for ( last=1 , i=0 ; (i <= stackptr-2 && i<MPC_TUPLE); i++){
      mpcf_resolveright(actionstack[i+2],tsval,&ival,&dval,&type,0);
      if(type != INTVAR){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_repeat had noninteger parameter");
        MPC_WRITE_AND_BAILOUT(1603);
      }
  
      if(ival<0){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_repeat parameters must be >0 ");
        MPC_WRITE_AND_BAILOUT(1604);
      }    
  
      if(last==0 && ival > 0 ){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_repeat must be like 1,1,0, NOT 1,0,1 or 0,0,1");
        MPC_WRITE_AND_BAILOUT(1605);
      }
  
      last = ival;
      play->mfields->mcmax[i]=ival;
    }
  }
  else {  /* map from an array */
    if(mapfrom->array==NULL){ /* oops, it isn't an array, this should never happen!!! */
      (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, f$macro_map passed a nonarray as an array");
      MPC_WRITE_AND_BAILOUT(1606);
    }
    for(last=1, i=0; i<MPC_TUPLE; i++){
      play->mfields->mcmax[i]=mapfrom->array->dim[i];
    }
  }

} /* end of mpcf_macro_repeat */

int mpcf_macro_gets(char *vinline){
  int i;
  VARIABLE *play;

  if(isactivemacro < 0 || isactivemacro > MPC_MAXMACRODEPTH-1){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, attempting to run invalid macro");
    MPC_WRITE_AND_BAILOUT(1701);
  }

  play = activemacro[isactivemacro];

  if(play==NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, attempting to run invalid macro");
    MPC_WRITE_AND_BAILOUT(1702);
  }

  if (play->mfields->mstate != PLAYING){
       (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, cannot read from macro which is not PLAYING");
       MPC_WRITE_AND_BAILOUT(1703);
  }

  if(play->ival >= play->mfields->msize ){ /* hit the end of the macro, there should
                                     have been a f$macro_return first, so 
                                     this is a syntax error */
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, macro %.4000s does not end with f$macro_return ",play->name);
    MPC_WRITE_AND_BAILOUT(1704);
  }

  /* copy the current string to the output */

  (void) strcpy(vinline,play->mfields->macro);

  /* modify the pointers*/

  for(i=1;i<=1+strlen(vinline);play->mfields->macro++,i++,play->ival++){}

  return 0;
}

/*------------------------------------------
  mpcf_macro_return().  Indicates the end of a macro.  The
  loop counters are examined, and if appropriate incremented
  appropriately and the macro restarted.  Otherwise, the
  macro call stack is rolled up one.  If it rolls up off
  the top, redirect back to INFROMFILE.

  if mode is 0 it does the full f$macro_return clean up and checking.
  if mode is 1 it forces an immediate exit from the macro with minimal
  clean up and no syntax checking.
*/

void mpcf_macro_return(int mode){
char *sval;
char **tsval=&sval;
int ival;
double dval;
enum istype type;
VARIABLE *play;
int i,carry;

  if(fromsource != INFROMMACRO){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_return or _break outside of a macro");
    MPC_WRITE_AND_BAILOUT(1804);
  }

  switch (stackptr){
    case 0:
      ival=1; /*default status*/
      break;
    case 1:
      mpcf_resolveright(actionstack[1],tsval,&ival,&dval,&type,0);
      break;
    default: 
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_return, too many parameters");
      MPC_WRITE_AND_BAILOUT(1801);
      break;
  }
  *instatus=ival;

  play = activemacro[isactivemacro];


  /* examine the counters, increment if necessary 
     if we are in here not all max counters are zero */

  if(mode==0){
    for (carry = 1, i=0 ; i<=2 ; i++){
      if ( play->mfields->mcmax[i] == 0){
          break;
      }
      play->mfields->mc[i] =  play->mfields->mc[i] + carry;
      carry = 0;
      if ( play->mfields->mc[i] > play->mfields->mcmax[i]){
         carry = 1;
         play->mfields->mc[i]  = 1;
      }
      else
       break;
    }
  }
  else {  /* this forces a macro in any state to immediately exit */
    carry = 1;
  }

  /* reset internal bits so that it can be played again in future iterations */

  if(play->mfields->body != NULL){
    play->mfields->macro = play->mfields->body;  /* pointer to first line macro "body" */
  }
  else {
    play->mfields->macro = play->string;  /* pointer to first line of macro */
  }
  play->ival=0;                /* count of positions back to the beginning */

  if(carry == 1){              /* completely done with this macro */
    play->mfields->mstate=DONE;
    mpcf_setcounters(play,1,0); /* reset the current macros counters */
    isactivemacro--;
    if(isactivemacro < 0)
       if(finc >= 0 ){  /* first macro was started from a file */
          nextfromsource = INFROMFILE;
       }
       else { /* first macro was SLAVE_MACRO */
          nextfromsource = INNOMORE;
       }
    else {
       play = activemacro[isactivemacro];
       mpcf_setcounters(play,0,1); /* restore the visible counters for
                                 the current macro */
    }

    /* for break mode just roll up to the next ifmacro label. */

    if(mode==1){ /* roll up the ifstack until the ifmacro is found */
      for( ;( (ifstack[ifptr].type != IFMACRO) && ifptr>=0);ifptr--){}
      if(ifptr < 0){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, f$macro_break destroyed ifstack");
        MPC_WRITE_AND_BAILOUT(1802);
      }
    }

    /* clean up the ifstack before returning control.  The last tag in ifstack 
       better be MACRO. If so roll it up. 
       ifptr better be valid or there is a bug elsewhere in the program!*/

    if(ifstack[ifptr].type != IFMACRO){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, incomplete if/then/else in macro %.4000s",play->name);
      MPC_WRITE_AND_BAILOUT(1803);
    }
    else{

       mpcf_yankaltprefix(); /* restore the altprefix variable to what it was 
                           when the macro was called */

       /* Macro may have executed in some sort of if/else/elseif context. The 
       first case is for the macro internal to a logic block.  In that
       case it must NOT have been scanning, and it should not begin
       scanning until it hits another part of the if/elseif/else/endif
       structure.  The second case is when the macro is the TEST for a logic
       block.  Here if the test fails it should begin scanning, and if the
       test passes, it should not begin scanning, but rather executing. */ 
  
      ifptr--;
      ifscan = NO;
      if(ifptr >= 0 ){
        if( ifstack[ifptr].type == IFLABEL){
          if(ifstack[ifptr].doneit == YES) 
            ifscan=NO;
          else {
            if(  (ifstack[ifptr].invert==NO  && *instatus == 0) ||
                 (ifstack[ifptr].invert==YES && *instatus != 0) ){
              ifscan = YES;
            }
            else {
              ifscan = NO;
              ifstack[ifptr].doneit = YES;
            }
          }
        }
      }
    }
  }
  else { /* going to play it again */
    mpcf_setcounters(play,0,1); /* update the visible counters to match internal ones*/

    /* roll back the ifstack - there may be a bunch of if structures built 
       within this macro and they must be rebuilt at each iteration */

    for (;ifptr>=0;ifptr--){
      if( ifstack[ifptr].type == IFMACRO)
         break;
    }

    /* set altprefix to the recorded type.  Otherwise if the macro
       changed altprefix it will blow up on the next loop */

    mpcf_setaltprefix(play->mfields->altprefix);
  }

  return;
} /* end of mpcf_macro_return*/

/* mpcf_macro_make_playable(void) 
   Set the state of the currently recording macro to playable.  Brought out 
   as a separate routine so that programs calling embedded miniproc can 
   conveniently terminate recording a SLAVE_MACRO.
*/
void mpcf_macro_make_playable(void){
    /* reset internal bits so that it can be played */

    recmacro->mfields->mstate=DONE;
    recmacro->mfields->macro = recmacro->string;  /* pointer set to the first line of the macro */
    recmacro->ival=0;               /* count of positions back to the beginning */
    recmacro = NULL; /* dereference the recording macro "handle" */
}


void mpcf_enlarge_string(char **string,int *ssize, int newsize,char *oops){
  if(newsize < *ssize)return;
  /*make it 1 bigger for the terminator.  Note that it is possible that
  for special variables the string may not yet exist, so test that.*/
  if(*string==NULL){
    *string=malloc((newsize+1)*sizeof(char)); 
  }
  else {
    *string=realloc(*string,(newsize+1)*sizeof(char)); 
  }
  if(*string==NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, %.4000s",oops);
    MPC_WRITE_AND_BAILOUT(1901);
  }
  *ssize=newsize+1;
}

/* mpcf_do_edit().  Edit a string, similar to DCL f$edit command.
Takes as arguments a pointer to the string to modify, the type of action,
one of "COLLAPSE,TRIM,COMPRESS,CLASSIFY", and a pointer to the list
of characters to consider editing.  The list is case sensitive, so if
you want to consider both "a" and "A", but both in the list.  (Normally
this list only contains punctuation, so it isn't a problem.)
THe string returned is always the same size as, or smaller than the one sent in. */

void mpcf_do_edit(char *string, char *editop, char *editchars ){
char *from;
char *to;
char *fchar;
enum edittypes {EDIT_BOGUS,EDIT_COLLAPSE,EDIT_COMPRESS,EDIT_CLASSIFY,EDIT_TRIM};
enum edittypes etype;

  if(*string == '\0')return; /* can't edit an empty string */
  if(*editchars == '\0'){  /* fata error */
     (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .edit. ] character list is empty");
     MPC_WRITE_AND_BAILOUT(3701);
  }
  etype=EDIT_BOGUS;
  if(mpcf_strccmp(editop,"collapse")){etype=EDIT_COLLAPSE;}
  if(mpcf_strccmp(editop,"compress")){etype=EDIT_COMPRESS;}
  if(mpcf_strccmp(editop,"classify")){etype=EDIT_CLASSIFY;}
  if(mpcf_strccmp(editop,"trim")    ){etype=EDIT_TRIM;}

  switch (etype) {
     case EDIT_BOGUS:
       (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ editop editlist .edit. ], [%s] is an invalid editop",editop);
       MPC_WRITE_AND_BAILOUT(3702);
     case EDIT_COLLAPSE: /* these 3 use very similar processing */
     case EDIT_COMPRESS:
     case EDIT_CLASSIFY:
       for(fchar=NULL, from=to=string; ; ){
         if(*from=='\0'){
           *to='\0';
           return;
         }
         else {
           if(strchr(editchars,*from)==NULL){
             fchar=NULL;
             *to=*from;
             to++;
           }
           else {
             switch (etype){
               case EDIT_COMPRESS:
                 if(fchar==NULL){
                   *to=*from; /* first character in run replaces run */
                   fchar=to; /* anything other than NULL */
                   to++;
                 }
                 break;
               case EDIT_COLLAPSE: /* whole run is deleted */
                 break;
               case EDIT_CLASSIFY:
                 if(fchar==NULL){
                   *to=*editchars; /* first character from editchars replaces run */
                   fchar=to; /* anything other than NULL */
                   to++;
                 }
                 break;
             }
           }
           from++;
         } /* end of if *from=='\0' */
       } /* end of for */
       break;
     case EDIT_TRIM:
       for (fchar=from=to=string; *from != '\0' ; ){ /* handle front run, if any */
          if(strchr(editchars,*from)==NULL){ /* end of run */
             break;
          }
          else {
             from++;
          }
       }
       for ( ; *from!= '\0' ;from++,to++ ){ /* handle midsection , if any */
          *to=*from;
       }
       *to='\0';  /* fix up end of string indicator */
       for (to-- ; from != fchar ; to-- ){ /* tail end, if any */
          if(strchr(editchars,*to)==NULL){ /* end of run */
             break;
          }
          else {
            *to='\0';  /* trim the tail end some more */
          }
       }

       break;
  }
  return;
}

/* --------------------------------------
   mpcf_fixorder() is used by f$evaluate tail and segment.
   It takes a string which is rotated right by r characters
   and puts it back into the proper position.
*/

void mpcf_fixorder(char *string,int rot, int ssize){
char *from;
char *to;
char *temp;
int i,j;

   if(rot==1)return;
   temp=malloc(ssize*sizeof(char));
   if(temp==NULL){
     (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$evaluate out of memory");
     MPC_WRITE_AND_BAILOUT(2001);
   }
   (void) strncpy(temp,string,ssize);

   /* position the from pointer */

   for(j=1,from=temp;j<rot;j++,from++){}

   /* shuffle everything back into place */

   for(i=1,to=string;  i<=ssize ; i++,to++){
      *to=*from;
      if(j==ssize){
         from=temp;
         j=1;
      }
      else {
         from++;
         j++;
      }
   }
   *to='\0';
   free(temp);
   temp=NULL;
}

/* ----------------------------------------------------------------
   mpcf_ensure_string(), verify that number and types of operands are ok
   ---------------------------------------------------------------- */ 
void mpcf_ensure_string(int rpnptr,int numops,char * message){
int i,j;
   if(numops > rpnptr+1){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, too few operands in stack "); 
        MPC_WRITE_AND_BAILOUT(2101);
   }
   for (i=rpnptr,j=1; j<=numops; i--,j++){
     if(rpn_stack[i].type != STRINGVAR){
        (void) sprintf(mpc_dbg_out, "miniproc, fatal error, %.4000s, some or all operands not strings",message); 
        MPC_WRITE_AND_BAILOUT(2201);
     }
   }
}

/* ----------------------------------------------------------------
   mpcf_ensure_double(), verify that number and types of operands are ok
   ---------------------------------------------------------------- */ 
void mpcf_ensure_double(int rpnptr,int numops,char * message){
int i,j;
   if(numops > rpnptr+1){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, too few operands in stack "); 
        MPC_WRITE_AND_BAILOUT(2301);
   }
   for (i=rpnptr,j=1; j<=numops; i--,j++){
     if(rpn_stack[i].type != DBLVAR){
        (void) sprintf(mpc_dbg_out, "miniproc, fatal error, %.4000s, some or all operands not numeric",message); 
        MPC_WRITE_AND_BAILOUT(2302);
     }
   }
}

void mpcf_rpn_stack_overflow(void){
  (void) sprintf(mpc_dbg_out,"miniproc, fatal error, RPN stack overflow in [] expression ]");
  MPC_WRITE_AND_BAILOUT(2401);
}

void mpcf_copy_rpn_stack(int from, int to, int rpnstackptr){
  if(from <0 || from > rpnstackptr ||
     to   <0 || from > rpnstackptr) {
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to access outside of RPN stack ");
    MPC_WRITE_AND_BAILOUT(2501);
  }
  rpn_stack[to].type = rpn_stack[from].type;
  rpn_stack[to].ival = rpn_stack[from].ival;
  rpn_stack[to].dval = rpn_stack[from].dval;
  if(rpn_stack[from].string != NULL){
    mpcf_enlarge_string(&(rpn_stack[to].string),
      &(rpn_stack[to].ssize),
      strlen(rpn_stack[from].string)+1,
      "[] RPN expression failed while storing a string");
    (void) strcpy(rpn_stack[to].string,rpn_stack[from].string);
  }
}

void mpcf_showstack(FILE *towhere, int startat, int rpnstackptr){
int j;
  if(rpnstackptr < 0 ){
    (void) fprintf(towhere,"Empty Stack\n");
    return;
  }
  for(j=rpnstackptr; j>=startat;j--){
     (void) fprintf(towhere," %5d: ",j);
     switch (rpn_stack[j].type){
       case INTVAR:
         (void) fprintf(towhere,"I: %d\n",rpn_stack[j].ival);
         break;
       case DBLVAR:
         (void) fprintf(towhere,"D: %e\n",rpn_stack[j].dval);
         break;
       case STRINGVAR:
         if(rpn_stack[j].string == NULL){
           (void) fprintf(towhere,"S: NULL string\n");
         }
         else {
	   if(strlen(rpn_stack[j].string) > 0){
              (void) fprintf(towhere,"S: %s\n",rpn_stack[j].string);
	   }
	   else {
              (void) fprintf(towhere,"S: ZERO LENGTH string\n");
           }	
         }
         break;
       case RPNOPER:
         (void) fprintf(towhere,"O: %s\n",rpn_stack[j].string);
         break;
     }
  }
}

/* ----------------------------------------------------------------
   mpcf_store_from_stack(), place the value of a given stack location into the named variable
   ---------------------------------------------------------------- */ 

void mpcf_store_from_stack(char * name, int tptr){
VARIABLE *somevar=NULL;
  somevar=mpcf_findvar(MAY,name);
  if(somevar==NULL){ /* create it and store results into it */
    mpcf_addvar(name,               /* name */
      rpn_stack[tptr].string,  /* string, if string */
      rpn_stack[tptr].ival,    /* int value */
      rpn_stack[tptr].dval,    /* double value */
      rpn_stack[tptr].type,    /* type */
      IS_ORDINARY, MORTAL);
  }
  else {
    switch (somevar->type){
      case MACRO:
      case RPNOPER:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [] var, var is a MACRO or RPN operator");
        MPC_WRITE_AND_BAILOUT(2601);
      case INTVAR:
        switch (rpn_stack[tptr].type){ /* conversions etc. */
          case INTVAR:
            somevar->ival=rpn_stack[tptr].ival;
            break;
          case DBLVAR:
            somevar->ival=(int) rpn_stack[tptr].dval;
            break;
          case STRINGVAR:
          default:
            (void) sprintf(mpc_dbg_out,"miniproc, fatal error, RPN stack STRING -> INTEGER variable");
            MPC_WRITE_AND_BAILOUT(2602);
        }
        break;
      case DBLVAR:
        switch (rpn_stack[tptr].type){ /* conversions etc. */
          case INTVAR:
            somevar->dval=(double) rpn_stack[tptr].ival;
            break;
          case DBLVAR:
            somevar->dval=rpn_stack[tptr].dval;
            break;
          case STRINGVAR:
          default:
            (void) sprintf(mpc_dbg_out,"miniproc, fatal error, RPN stack STRING -> Double variable");
            MPC_WRITE_AND_BAILOUT(2603);
        }
        break;
      case STRINGVAR:
        switch (rpn_stack[tptr].type){ /* conversions etc. */
          case STRINGVAR:
            mpcf_enlarge_string(
               &(somevar->string),
               &(somevar->ssize),
               strlen(rpn_stack[tptr].string) + 1,
              "[ ] failed while enlarging result string");
            (void) strcpy(somevar->string,rpn_stack[tptr].string);
            break;
          default:
            (void) sprintf(mpc_dbg_out,"miniproc, fatal error, RPN stack nonSTRING -> STRING variable");
            MPC_WRITE_AND_BAILOUT(2604);
        }
        break;
    } /* end of switch on type of existing output variable */
  } /* end of if, does output variable exist */
}

/* ----------------------------------------------------------------
   mpcf_load_on_stack(), place the value of a given named variable onto the stack
   ---------------------------------------------------------------- */ 

void mpcf_load_on_stack(char * name, int tptr){
VARIABLE *somevar=NULL;
  somevar=mpcf_findvar(MUST,name);
  switch (somevar->type){
    case MACRO:
    case RPNOPER:
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ var .load. ], var is a MACRO or RPN operator");
      MPC_WRITE_AND_BAILOUT(2702);
    case INTVAR:
      rpn_stack[tptr].dval = somevar->ival;
      break;
    case DBLVAR:
      rpn_stack[tptr].dval = somevar->dval;
      break;
    case STRINGVAR:
      mpcf_enlarge_string(&(rpn_stack[tptr].string),
                     &(rpn_stack[tptr].ssize),
                     strlen(somevar->string)+1,
                     "[ var .load. ] RPN expression failed while storing a string");
      (void) strcpy(rpn_stack[tptr].string,somevar->string);
      break;
  } /* end of switch on type of existing output variable */
  rpn_stack[tptr].type = somevar->type;
}


/* ----------------------------------------------------------------
  mpcf_array_dim().  Attempt to dimension a variable as an array.
*/
void mpcf_array_dim(VARIABLE *somevar, int ncells, int tdim[MPC_TUPLE]){
size_t tsize;
int i;
int restzero;
char *sptr;

  if(somevar == NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to .dim. a nonexistant variable array");
    MPC_WRITE_AND_BAILOUT(4001);
  }
  if(somevar->array != NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to .dim. an existing array");
    MPC_WRITE_AND_BAILOUT(4002);
  }
  switch (somevar->type) {
     case INTVAR:
        tsize=sizeof(int);
        break;
     case DBLVAR:
        tsize=sizeof(double);
        break;
     case STRINGVAR:
        tsize=sizeof(STRINGNODE);
        break;
     default: /* no other type has an array */
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, only integer, double, and string variables may be arrays");
        MPC_WRITE_AND_BAILOUT(4003);
  }
  somevar->array = (ARRAYNEXUS *) malloc(sizeof(ARRAYNEXUS));
  if(somevar->array == NULL){
     (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not dimension array, insufficient memory");
     MPC_WRITE_AND_BAILOUT(4004);
  }

  /* for int, double, and string all values are allocated zero filled.  
  That means integer and double arrays full of zero values, and STRINGNODE
  arrays full of 0 = NULL pointer and 0 ssize, ie, zero size, which is 
  consistent.  No further data initialization should be required.  Howerver,
  this does mean that strings must be resized for the first use. */

  somevar->array->data = (void *) calloc(ncells,tsize);
  if(somevar->array->data == NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not dimension array, insufficient memory");
    MPC_WRITE_AND_BAILOUT(4005);
  }
  for(restzero=0, i=0 ; i < MPC_TUPLE; i++){
     somevar->array->dim[i]=tdim[i]; /* store the dimensions */
     if(tdim[i] > 0){
       if(restzero == 1){
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, arrays cannot have a nonzero dimension following a zero dimension");
         MPC_WRITE_AND_BAILOUT(4006);
       }
       somevar->array->active[i]=1;    /* set active to all ones = first cell */
     }
     else {
       restzero=1;
       somevar->array->active[i]=0;    /* set active to all zeros = inactive */
     }
     somevar->array->subdim[i]=0;      /* set subdim area to all zero */
  }
  somevar->array->offset = 0;        /* arrays start out with lowest element "active" */
  if(somevar->type == STRINGVAR){    /* allocate a bunch of empty strings (containing just '\0')*/
    for(i=0;i<ncells;i++){
      sptr=calloc(1,sizeof(char));
      if(sptr == NULL){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, out of memory while allocating space for string array");
        MPC_WRITE_AND_BAILOUT(4007);
      }
      else {
        ((STRINGNODE *)somevar->array->data)[i].string=sptr;
        ((STRINGNODE *)somevar->array->data)[i].ssize=1;
      }
    }
  }
}

/* ----------------------------------------------------------------
  mpcf_array_subdim().  set the subdim values for an array
*/
void mpcf_array_subdim(VARIABLE *somevar, int tdim[MPC_TUPLE]){
int i;

  if(somevar == NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to .(subdim). a nonexistant variable array");
    MPC_WRITE_AND_BAILOUT(4501);
  }
  if(somevar->array == NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to .(subdim). a nonarray variable");
    MPC_WRITE_AND_BAILOUT(4502);
  }
  for(i=0 ; i < MPC_TUPLE; i++){
    if(tdim[i] > 0){
      if(tdim[i] > somevar->array->dim[i]){  /* subdims must be smaller than dims */
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(subdim). indices larger than .(dim). indices");
        MPC_WRITE_AND_BAILOUT(4503);
      }
      if(i == MPC_TUPLE-1 || somevar->array->dim[i+1] == 0){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(subdim). too many dimensions - no free indices");
        MPC_WRITE_AND_BAILOUT(4504);
      }
      somevar->array->subdim[i]=tdim[i]; /* store the dimensions */
    }
    else {
      somevar->array->subdim[i]=0;       /* zero the rest */
    }
  }
}

/* ----------------------------------------------------------------
  mpcf_array_transmult().  Multiplies after first transposing the right
  array.  Results are stored in the right array.  If subdim values are
  defined all subdims in the array are processed.  Names of the variables
  are passed.
*/
void mpcf_array_transmult(char *leftvar, char *rightvar){
VARIABLE *leftarray;
VARIABLE *rightarray;
int *tdim;
double *resultant;
int R,L;
int rdim,ldim;
int dorepeat;
int i,iskip;

   /* convert names to arrays variables.  Must exist and be a double array */

   leftarray  = mpcf_findvar(MUSTARRAY,leftvar);
   if(leftarray->type != DBLVAR){
     (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(transmult)., %.4000s is not a floating point array",leftvar);
     MPC_WRITE_AND_BAILOUT(4601);
   }
   rightarray = mpcf_findvar(MUSTARRAY,rightvar);
   if(leftarray->type != DBLVAR){
     (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(transmult)., %.4000s is not a floating point array",rightvar);
     MPC_WRITE_AND_BAILOUT(4602);
   }
  
   /* make sure the sizes work out so that the transposed multiplied result 
      can go back into the rightmost array, or it's subdims.  Subdims are 
      ignored on the left, only dims are used */

   if(rightarray->array->subdim[0] != 0){  
      tdim = rightarray->array->subdim;
      dorepeat = 1;
   }
   else {
      tdim = rightarray->array->dim; 
      dorepeat = 0;
   }
   for ( L=0,ldim=0 ; leftarray->array->dim[L] != 0 && L<MPC_TUPLE ; L++,ldim++){}
   for ( R=0,rdim=0 ; tdim[R] != 0 && R<MPC_TUPLE                ; R++,rdim++){}
   if(ldim > 2 || rdim >2){
     (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(transmult)., (sub)arrays may not have more than 2 dimensions");
     MPC_WRITE_AND_BAILOUT(4603);
   }
   for (R=0,L=ldim-1; R<rdim; R++,L--){
     if(tdim[R] != leftarray->array->dim[L]){
       (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(transmult)., incompatible array sizes");
       MPC_WRITE_AND_BAILOUT(4604);
     }
   }  
   if(leftarray->array->dim[0] != tdim[0] ||
      tdim[R]                != tdim[rdim-1]){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(transmult)., resultant array has different size from right (result) array");
      MPC_WRITE_AND_BAILOUT(4605);
   }

   /* safe to do the multiplication, get enough space for it */

   resultant = malloc(sizeof(DBLVAR)*rdim*ldim);
   if(resultant == NULL){ /* oops */
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(transmult)., could not allocate temporary memory to hold result");
      MPC_WRITE_AND_BAILOUT(4606);
   }

   /* figure out how to skip through a subdim'd array */

   if(dorepeat == 1){ 
      rightarray->array->offset = 0;   /* always starts at first cell */
      for (R=0,iskip=1; R<rdim;R++){iskip = iskip * rightarray->array->dim[R];}
      dorepeat = rightarray->array->dim[rdim-1];  /* number of subarrays present */
   }
   while(dorepeat >= 0){
     for(i=0;i<rdim*ldim;i++){resultant[i]=0.0;} /* clear the result */
     
/*
 WORKING HERE!!!!!!!!!!!!!!!!!!!!  The actual multiplication part
*/

   }

}


/* ----------------------------------------------------------------
  mpcf_array_active().  Set the active position.
*/
void mpcf_array_active(VARIABLE *somevar, int tdim[MPC_TUPLE]){
int itemp,jtemp;
int i,j;

  if(somevar == NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(). only acts on existant variables");
    MPC_WRITE_AND_BAILOUT(4101);
  }
  if(somevar->array == NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(). only acts on arrays");
    MPC_WRITE_AND_BAILOUT(4102);
  }
  for(jtemp=0,itemp=1, i=0; i< MPC_TUPLE; i++){
    j = tdim[i];
    j--;  /* offsets run 0 to N-1, not 1-N like array indices */
    if(somevar->array->dim[i] > 0){  /* index in nonzero dimension */
      if( j >= 0  &&  j < somevar->array->dim[i]){ /* must be in dimensioned range for index*/
        jtemp = jtemp + j * itemp;                 /* the accumulating offset */
        itemp = itemp * somevar->array->dim[i];    /* the accumulating step size */
      }
      else {
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(). specifies invalid array cell or wildcard");
        MPC_WRITE_AND_BAILOUT(4103);
      }
    }
    else {
       if(tdim[i] != 0){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, .(). specifies a nonzero index for a zeroed dimension");
          MPC_WRITE_AND_BAILOUT(4104);
       }
       j = 0;
    }
    somevar->array->active[i] = j;
  }
  somevar->array->offset    = jtemp; 
}

/* ----------------------------------------------------------------
  mpcf_double_to_uint().  Convert double values to unsigned int. Truncate
  into range as necessary.  That is, -3.0 -> zero, 2^40> MAXINT.
*/
unsigned int mpcf_double_to_uint(double dval){
unsigned int itemp;
  if(dval < 0.0 )return 0;
  if(dval > (double) UINT_MAX)return UINT_MAX;
  itemp = (int) dval;
  return itemp;
}

/* ----------------------------------------------------------------
  mpcf_do_rpn_evaluate().  Perform verious operations on operands and return the
  result.  Uses RPN notation (sort of).
*/
void mpcf_do_rpn_evaluate(void){
int i,j,k,m,r;
int all1,all0;
int whichop;
char *sval;
char **tsval=&sval;
char cscratch[MPC_MAXVARLEN];
char cscr2[MPC_MAXVARLEN];
char cinline[MPC_MAXINLINE];  /* used for substitutions */
int ival;
double dval;
enum istype type;
VARIABLE *somevar=NULL;
int ifirst;
int fsize,startnum;
int isok;
char *holdsval;
char *from;
char *to;
char *temp;
int dofixorder;
int sure;
int bothends;
int rpnstackptr;
int numops;
int localstatus;
int holdsubs;
double dtemp;
unsigned int  itemp,jtemp;
int tdim[MPC_TUPLE]; /* used for matrix operations */

  /* initialize the RPN stack, MPC_MAXSTACK reserved for swapping */
  
  bothends = 0;
  rpnstackptr= -1;
  localstatus = 1;  /* assume that result will be true, move to STATUS at 
                       end so that STATUS may be an operand on stack */
  for (i=0; i<=stackptr; i++){
    if(*actionstack[i] == ']'){
       bothends = 1;
       i++;
       break; 
    }
    mpcf_resolveright(actionstack[i],tsval,&ival,&dval,&type,0);
    rpnstackptr++;
    if(rpnstackptr > MPC_MAXSTACK - 2)mpcf_rpn_stack_overflow();
    switch (type){
      case MACRO:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ macro ] is illegal"); 
        MPC_WRITE_AND_BAILOUT(2801);
      case INTVAR:
        rpn_stack[rpnstackptr].type = DBLVAR;
        rpn_stack[rpnstackptr].dval = ival;
        break;
      case DBLVAR:
        rpn_stack[rpnstackptr].dval = dval;
        rpn_stack[rpnstackptr].type = DBLVAR;
        break;
      case STRINGVAR:
        mpcf_enlarge_string(&(rpn_stack[rpnstackptr].string),
                       &(rpn_stack[rpnstackptr].ssize),
                       strlen(sval)+1,
                       "[] RPN expression failed while storing a string");
        (void) strcpy(rpn_stack[rpnstackptr].string,sval);
        rpn_stack[rpnstackptr].type = STRINGVAR;
        break;
      case RPNOPER:
        rpnstackptr--;
        whichop= all_optypes[ival].opnum;

        /* validate arguments, if any, note how rpnstackptr moves!!! */

        if(all_optypes[ival].argnumber != 0){
          /* in the next loop r will not exceed MPC_RPNARGS because the 
             argnumber fields are checked at program startup, so even if 
             there is a programming error with optypes, it won't get to here */
          for(r=0; r<all_optypes[ival].argnumber; r++){ 
             if(rpn_stack[rpnstackptr].type != all_optypes[ival].argtype[r]){
               (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [.%.4000s.] argument mismatch",
                  all_optypes[ival].name); 
               MPC_WRITE_AND_BAILOUT(2802);
             }
             rpnstackptr--;
             if(rpnstackptr < 0){
               (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [.%.4000s.] too few arguments",
                  all_optypes[ival].name);
               MPC_WRITE_AND_BAILOUT(2803);
             }
          }
        }


        /* validate operand types, if any, note how rpnstackptr DOES NOT move!!! */

        if(dval == 0.0 ){ /* 0.0 and MPC_MAXSTACK are both signals to do the whole stack,
                             but only 0.0 needs special handling */
          numops=rpnstackptr+1;  /* do everything on stack */
        }
        else {
           numops= (int) dval;
        }
        if(numops > all_optypes[ival].maxoperands){
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [.%.4000s.] too many operands",all_optypes[ival].name); 
           MPC_WRITE_AND_BAILOUT(2840);
        }
        if(numops < all_optypes[ival].minoperands){
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [.%.4000s.] too few operands",all_optypes[ival].name); 
           MPC_WRITE_AND_BAILOUT(2841);
        }
        k=rpnstackptr-numops+1;
        k = mpc_max(k,0);                   /* rpn operator might have size too big, truncate to full stack */
        if(all_optypes[ival].operandtype != ANYVAR){ /* ANYVAR means type irrelevant */
          for(j=k ; j<= rpnstackptr ; j++){
            if(rpn_stack[j].type != all_optypes[ival].operandtype){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [.%.4000s.] operand type mismatch",
                 all_optypes[ival].name); 
              MPC_WRITE_AND_BAILOUT(2804);
            }
          }
        }

        switch (whichop){
          case  0:  /* power */
            for(j=rpnstackptr;j>k;j--){
              rpn_stack[j-1].dval = 
              pow(rpn_stack[j].dval,rpn_stack[j-1].dval);
            }
            rpnstackptr=k;
            break;
          case  1:  /* modulo */
            for(j=rpnstackptr;j>k;j--){
              rpn_stack[j-1].dval = 
              fmod(rpn_stack[j].dval,rpn_stack[j-1].dval);
            }
            rpnstackptr=k;
            break;
          case  2:  /* add */
            for(j=k+1;j<=rpnstackptr;j++){
               rpn_stack[k].dval= rpn_stack[k].dval + rpn_stack[j].dval;
            }
            rpnstackptr=k;
            break;
          case  3:  /* subtract */
            for(j=k+1;j<=rpnstackptr;j++){
               rpn_stack[k].dval= rpn_stack[k].dval - rpn_stack[j].dval;
            }
            rpnstackptr=k;
            break;
          case  4:  /* multiply */
            for(j=k+1;j<=rpnstackptr;j++){
               rpn_stack[k].dval= rpn_stack[k].dval * rpn_stack[j].dval;
            }
            rpnstackptr=k;
            break;
          case  5:  /* divide */
            for(j=k+1;j<=rpnstackptr;j++){
               if(rpn_stack[j].dval == 0){
                  (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [] attempt to divide by zero"); 
                  MPC_WRITE_AND_BAILOUT(2805);
               }
               rpn_stack[k].dval= rpn_stack[k].dval / rpn_stack[j].dval;
            }
            rpnstackptr=k;
            break;
          case  6:  /* eq */
            dtemp = 1.0; /* true */
            for(j=k+1;j<=rpnstackptr;j++){
               if(!(rpn_stack[k].dval == rpn_stack[j].dval)){
                 dtemp=0.0; /* false */
                 break;
               }
            }
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case  7:  /* neq */
            dtemp = 1.0; /* true */
            for(j=k+1;j<=rpnstackptr;j++){
               if(!(rpn_stack[k].dval != rpn_stack[j].dval)){
                 dtemp=0.0; /* false */
                 break;
               }
            }
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case  8:  /* ge */
            dtemp = 1.0; /* true */
            for(j=k+1;j<=rpnstackptr;j++){
               if(!(rpn_stack[k].dval >= rpn_stack[j].dval)){
                 dtemp=0.0; /* false */
                 break;
               }
            }
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case  9:  /* gt */
            dtemp = 1.0; /* true */
            for(j=k+1;j<=rpnstackptr;j++){
               if(!(rpn_stack[k].dval > rpn_stack[j].dval)){
                 dtemp=0.0; /* false */
                 break;
               }
            }
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case 10:  /* lt */
            dtemp = 1.0; /* true */
            for(j=k+1;j<=rpnstackptr;j++){
               if(!(rpn_stack[k].dval < rpn_stack[j].dval)){
                 dtemp=0.0; /* false */
                 break;
               }
            }
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case 11:  /* le */
            dtemp = 1.0; /* true */
            for(j=k+1;j<=rpnstackptr;j++){
               if(!(rpn_stack[k].dval <= rpn_stack[j].dval)){
                 dtemp=0.0; /* false */
                 break;
               }
            }
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case 12:  /* xor (for more than 2, means "not all 1 AND not all 0" */
            dtemp = 1.0; /* true */
            all1=1;
            all0=1;
            for(j=k;j<=rpnstackptr;j++){
               if( rpn_stack[j].dval == 0.0)all1=0; 
               if( rpn_stack[j].dval != 0.0)all0=0; 
            }
            dtemp=1.0;
            if( all1 || all0)dtemp=0.0;
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case 13:  /* not */ 
            dtemp = 1.0; /* true */
            for(j=k;j<=rpnstackptr;j++){
               if(rpn_stack[j].dval == 0.0){
                 rpn_stack[j].dval = 1.0;
               }
               else {
                 rpn_stack[j].dval = 0.0;
               }
            }
            break;
          case 14:  /* and */ 
            dtemp = 1.0; /* true */
            for(j=k;j<=rpnstackptr;j++){
               if(rpn_stack[j].dval == 0.0){
                 dtemp=0.0; /* false */
                 break;
               }
            }
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case 15:  /* or */  
            dtemp = 0.0; /* false */
            for(j=k;j<=rpnstackptr;j++){
               if(rpn_stack[j].dval != 0.0 ){
                 dtemp=1.0; /* true */
                 break;
               }
            }
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case 16:  /* nand */
            dtemp = 0.0; /* false */
            for(j=k;j<=rpnstackptr;j++){
               if(rpn_stack[j].dval == 0.0){
                 dtemp=1.0; /* true */
                 break;
               }
            }
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case 17:  /* nor */ 
            dtemp = 1.0; /* true */
            for(j=k;j<=rpnstackptr;j++){
               if(rpn_stack[k].dval != 0.0 ){
                 dtemp=0.0; /* false */
                 break;
               }
            }
            rpnstackptr=k;
            rpn_stack[rpnstackptr].dval = dtemp;
            break;
          case 18:  /* head */
            fsize= (int) rpn_stack[rpnstackptr+1].dval;  /* number of characters for "head" */
            if(fsize<0){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ N .head.], N < 0 "); 
               MPC_WRITE_AND_BAILOUT(2806);
            }
            mpcf_enlarge_string(
              &(rpn_stack[k].string),
              &(rpn_stack[k].ssize),
              fsize+1,
              "[ .head.] failed while enlarging result string");
            for(from=to=rpn_stack[k].string, j=1,m=k; j<=fsize; j++,from++,to++){
              if(*from=='\0'){
                m++;
                if(m>rpnstackptr)break;
                from=rpn_stack[m].string;
              }
              *to=*from;
            }
            *to='\0';
            rpnstackptr=k;
            break;
          case 19:  /* tail */
           /* this one is a bit odd.  Simply write characters into the
              modulo style (wrapping) into the output space.  Then at the end slide 
              everything around into the right position.  This saves
              having to count everything. */
            fsize= (int) rpn_stack[rpnstackptr+1].dval;  /* number of characters for "tail" */
            if(fsize<0){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ N .tail.], N < 0 "); 
               MPC_WRITE_AND_BAILOUT(2807);
            }
            if(fsize==0){
               *rpn_stack[k].string='\0';
               rpnstackptr=k;
               break;
            }
            mpcf_enlarge_string(
              &(rpn_stack[k].string),
              &(rpn_stack[k].ssize),
              fsize+1,
              "[ .tail.] failed while enlarging result string");
            for (to=rpn_stack[k].string, j=1, dofixorder=0,m=k,from=rpn_stack[m].string;
                 m<=rpnstackptr;
                 from++){
              if(*from=='\0'){
                m++;
                if(m>rpnstackptr)break;
                from=rpn_stack[m].string;
              }
              *to=*from;
              if(j==fsize){
                j=1;
                to=rpn_stack[k].string;
                dofixorder=1;
              }
              else {
                j++;
                to++;
              }
            }
            /* at this point, there are fsize characters stored, but maybe in 
                the wrong order.  j is the COUNT where the front is */
            if(dofixorder){
              for (m=j;m<=fsize;m++,to++){}
              *to='\0';  /*terminate string */
              mpcf_fixorder(rpn_stack[k].string,j,fsize);
            }
            else {
              to='\0';
            }
            rpnstackptr=k;
            break;
          case 20:  /* segment */
            /* this one is also a bit odd.  Since the arguments pretty much have to 
               come in in the wrong order, just write the results modulo style
               (wrapping) into the output space. After first skipping
               the required number of characters.   Then at the end slide 
               everything around into the right position */
            fsize=(int) rpn_stack[rpnstackptr+2].dval;  /* number of characters for "segment" */
            if(fsize<0){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ K N .segment.], N < 0 "); 
               MPC_WRITE_AND_BAILOUT(2808);
            }
            startnum= (int) rpn_stack[rpnstackptr+1].dval;  /* start character for "segment" */
            if(startnum <=  0){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ K N .segment.], K <= 0 "); 
               MPC_WRITE_AND_BAILOUT(2809);
            }
            if(fsize==0){
               *rpn_stack[k].string='\0';
               rpnstackptr=k;
               break;
            }
            mpcf_enlarge_string(
              &(rpn_stack[k].string),
              &(rpn_stack[k].ssize),
              fsize+1,
              "[ .segment.] failed while enlarging result string");
            for (to=rpn_stack[k].string, j= 1, m=k,from=rpn_stack[m].string;
                 j<=fsize+startnum-1;
                 from++,j++){
              if(*from=='\0'){
                m++;
                if(m>rpnstackptr)break;
                from=rpn_stack[m].string;
              }
              if(j >= startnum){
                *to=*from;
                to++;
              }
            }
            *to='\0';
            rpnstackptr=k;
            break;
          case 21:  /* d->s, NOTE, if user specifies more than one format in a string
                       the result will be that it reads from who knows 
                       where in memory. */
            for(j=k; j<=rpnstackptr; j++){
              mpcf_enlarge_string(
                &(rpn_stack[j].string),
                &(rpn_stack[j].ssize),
                *convertwidth,"[ .d->s. ] failed while enlarging result string"
                );
              if(sprintf(
                  rpn_stack[j].string,  /* where result goes */
                  rpn_stack[rpnstackptr+1].string,  /* format string, user supplies */
                  rpn_stack[j].dval       /* double to convert*/
                ) <= 0 ){
                (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .d->s.] ");
                MPC_WRITE_AND_BAILOUT(2810);
              }
              rpn_stack[j].type=STRINGVAR;  /* change type */
            }
            break;
          case 22:  /* i->s, see d->s for a MAJOR restriction on use */
            mpcf_enlarge_string(
              &(rpn_stack[rpnstackptr-1].string),
              &(rpn_stack[rpnstackptr-1].ssize),
              *convertwidth,"[ .i->s. ] failed while enlarging result string"
              );
            for(j=k; j<=rpnstackptr ; j++){
              rpn_stack[j].ival = (int) rpn_stack[j].dval;
              if(sprintf(
                  rpn_stack[j].string,  /* where result goes */
                  rpn_stack[rpnstackptr+1].string,    /* format string, user supplies*/
                  rpn_stack[j].ival     /* integer to convert*/
                ) <= 0 ){
                (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .i->s.] ");
                MPC_WRITE_AND_BAILOUT(2811);
              }
              rpn_stack[j].type=STRINGVAR;  /* change type */
            }
            break;
          case 23:  /* s->d, string to double */
            for(j=k; j<=rpnstackptr ; j++){
              if(sscanf(
                rpn_stack[j].string,    /* read double from here */
                rpn_stack[rpnstackptr+1].string,      /* format string, user supplies*/
                &(rpn_stack[j].dval)    /* store double here */
                )==EOF){
                (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .s->d. ] ");
                MPC_WRITE_AND_BAILOUT(2812);
              }
              rpn_stack[j].type=DBLVAR;  /* change type */
            }
            break;
          case 24:  /* s->i */
            for(j=k; j<=rpnstackptr ; j++){
              if(sscanf(
                rpn_stack[j].string,    /* read integer from here */
                rpn_stack[rpnstackptr+1].string,      /* format string, user supplies*/
                &(rpn_stack[j].ival)    /* store integer here */
                )==EOF){
                (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .s->i. ] ");
                MPC_WRITE_AND_BAILOUT(2813);
              }
              rpn_stack[j].type=DBLVAR;  /* change type */
              rpn_stack[j].dval=(double) rpn_stack[j].ival;  /* convert to double */
            }
            break;
          case 25:  /* append */ 
            for(fsize=0,j=k;j<=rpnstackptr;j++){
               fsize=fsize+strlen(rpn_stack[j].string);
            }
            mpcf_enlarge_string(
              &(rpn_stack[k].string),
              &(rpn_stack[k].ssize),
              fsize + 1 + (rpnstackptr-k)*strlen(rpn_stack[rpnstackptr+1].string),
              "[ .append.] failed while enlarging result string");
            for(m=k+1; m<=rpnstackptr; m++){
              (void) strcat(rpn_stack[k].string,rpn_stack[rpnstackptr+1].string);
              (void) strcat(rpn_stack[k].string,rpn_stack[m].string);
            }
            rpnstackptr=k;
            break;
          case 26:  /* uppercase, acts in situ on N strings */
            for(from=rpn_stack[k].string, m=k; ;){
              if(*from=='\0'){
                m++;
                if(m>rpnstackptr)break;
                from=rpn_stack[m].string;
              }
              *from=toupper(*from);
              from++;
            }
            break;
          case 27:  /* lowercase */
            for(from=rpn_stack[k].string, m=k; ; ){
              if(*from=='\0'){
                m++;
                if(m>rpnstackptr)break;
                from=rpn_stack[m].string;
              }
              *from=tolower(*from);
              from++;
            }
            break;
          case 28:  /* shortest, assumes no string longer than 2billion */
            for(j=k,fsize=2000000000,m=k; m<=rpnstackptr ; m++){
              if(strlen(rpn_stack[m].string) < fsize){
                j=m;
                fsize=strlen(rpn_stack[m].string);
              }
            }
            mpcf_enlarge_string(
              &(rpn_stack[k].string),
              &(rpn_stack[k].ssize),
              fsize+1,
              "[ .shortest.] failed while enlarging result string");
            (void) strcpy(rpn_stack[k].string,rpn_stack[j].string);
            rpnstackptr=k;
            break;
          case 29:  /* longest */
            for(j=k,fsize=0,m=k; m<=rpnstackptr ; m++){
              if(strlen(rpn_stack[m].string) > fsize){
                j=m;
                fsize=strlen(rpn_stack[m].string);
              }
            }
            mpcf_enlarge_string(
              &(rpn_stack[k].string),
              &(rpn_stack[k].ssize),
              fsize+1,
              "[ .shortest.] failed while enlarging result string");
            (void) strcpy(rpn_stack[k].string,rpn_stack[j].string);
            rpnstackptr=k;
            break;
          case 30:  /* eliminate */
            for(m=k, from=to=rpn_stack[m].string; ; ){
              if(*from=='\0'){
                *to='\0';
                m++;
                if(m>rpnstackptr)break;
                from=to=rpn_stack[m].string;
              }
              else {
                if(strchr(rpn_stack[rpnstackptr+1].string,*from)==NULL){
                  *to=*from;
                   to++;
                }
                else {
                  localstatus=2; /* signal that some were eliminated */
                }
                from++;
              }
            }
            *to='\0';
            break;
          case 31:  /* retain */ 
            for(m=k, from=to=rpn_stack[m].string; ; ){
              if(*from=='\0'){
                *to='\0';
                m++;
                if(m>rpnstackptr)break;
                from=to=rpn_stack[m].string;
              }
              else {
                if(strchr(rpn_stack[rpnstackptr+1].string,*from)!=NULL){
                  *to=*from;
                   to++;
                }
                else {
                  localstatus=2; /* signal that some were kept */
                }
                from++;
              }
            }
            *to='\0';
            break;
          case 32:  /* element */  
            ifirst= (int) rpn_stack[rpnstackptr+1].dval;
            if(ifirst<1){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .element.], negative index value");
              MPC_WRITE_AND_BAILOUT(2814);
            }
            if(strlen(rpn_stack[rpnstackptr+2].string)<1){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .element.], empty delimiter string");
              MPC_WRITE_AND_BAILOUT(2815);
            }
            for (m=k; m<=rpnstackptr; m++){            
              to=strtok(rpn_stack[m].string,rpn_stack[rpnstackptr+2].string);
	      if(ifirst != 1){
	        for(j=1; (to!=NULL && j<ifirst) ; j++){
                  to=strtok(NULL,rpn_stack[rpnstackptr+2].string);
                }
	      }
              if(to==NULL){  /* asked for too many, an error, but not fatal */
                *rpn_stack[m].string='\0';  /*result is a null string if ask for too many */
                localstatus = 0;
              }
              else {  /* got one */
                (void) strcpy(rpn_stack[m].string,to);
              }
            }
            break;
          case 33:  /* locate */ 
            for(m=k; m<=rpnstackptr; m++){
              temp=strstr(rpn_stack[m].string,
                          rpn_stack[rpnstackptr+1].string);
              if(temp==NULL){
                rpn_stack[m].dval = 0;
              }
              else{
                for(j=1,from=rpn_stack[m].string;from!=temp;from++,j++){}
                rpn_stack[m].dval = j;
              }
              rpn_stack[m].type = DBLVAR;
            }
            break;
          case 34:  /* compare */
            for (m=k; m<=rpnstackptr; m++){
              rpn_stack[m].dval=1.0;
              rpn_stack[m].type=DBLVAR;
              for(to=rpn_stack[m].string,from=rpn_stack[rpnstackptr+1].string; ;from++,to++){
                if(*from!=*to){
                  rpn_stack[m].dval=0.0;
                  break; 
                }
                if(*to=='\0')break; /* since *from=*to, both are '\0' */
              }
            }
            break;
          case 35:  /* ccompare */
            for (m=k; m<=rpnstackptr; m++){
              rpn_stack[m].dval=0.0;
              rpn_stack[m].type=DBLVAR;
              if(mpcf_strccmp(rpn_stack[m].string,rpn_stack[rpnstackptr+1].string)){
                rpn_stack[m].dval=1.0;
              }
            }
            break;
          case 36:  /* length */
            for (m=k; m<=rpnstackptr; m++){
              rpn_stack[m].type=DBLVAR;
              fsize=strlen(rpn_stack[m].string);
              rpn_stack[m].dval=fsize;
            }
            break;
          case 37: /* lexhigh */
            for (m=k,j=k+1; j<=rpnstackptr; j++){
               from=rpn_stack[m].string;  /* m is going to point to lexhigh */
               to=rpn_stack[j].string;
               for (sure=0; sure==0 ;from++,to++){
                 if(*from < *to){ /* new is bigger */
                   m=j;
                   sure=1;
                 }
                 if(*from > *to){sure=1;} /* new one is smaller */
                 if(*from=='\0'){sure=1;} /*from=to=0, so strings are identical*/
               }
            }
            fsize=strlen(rpn_stack[m].string);
            mpcf_enlarge_string(
              &(rpn_stack[k].string),
              &(rpn_stack[k].ssize),
              fsize+1,
              "[ .lexhigh.] failed while enlarging result string");
            (void) strcpy(rpn_stack[k].string,rpn_stack[m].string);
            rpnstackptr=k;
            break;
          case 38: /* lexlow, final operation at end of loop */
            for (m=k,j=k+1; j<=rpnstackptr; j++){
               from=rpn_stack[m].string;  /* m will point to lexlow */
               to=rpn_stack[j].string;
               for (sure=0; sure==0 ;from++,to++){
                 if(*from > *to){ /* new is smaller */
                   m=j;
                   sure=1;
                 }
                 if(*from < *to){sure=1;} /* new one is NOT smaller */
                 if(*from=='\0'){sure=1;} /*from=to=0, so strings are identical*/
               }
            }
            fsize=strlen(rpn_stack[m].string);
            mpcf_enlarge_string(
              &(rpn_stack[k].string),
              &(rpn_stack[k].ssize),
              fsize+1,
              "[ .lexlow.] failed while enlarging result string");
            (void) strcpy(rpn_stack[k].string,rpn_stack[m].string);
            rpnstackptr=k;
            break;
          case 39: /* string del */
            temp=rpn_stack[rpnstackptr+1].string;      
            fsize=strlen(temp);
            if(fsize==0){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .stringdel.], pattern has zero length");
              MPC_WRITE_AND_BAILOUT(2816);
            }
            for (m=k; m<=rpnstackptr; m++){
               sval=rpn_stack[m].string;
               to=strstr(sval,temp);  /*find the first time pattern appears */
               if(to!=NULL){  /* remove the match */
                 localstatus=2; /* signal that a change was made */
                 for(j=1,from=to;*from!='\0';j++,from++){           
                    if(j>fsize){
                       *to=*from;
                       to++;
                    }
                 }
                 *to='\0';
               }
            }
            break;
          case 40: /* resize (string).  Each operand is the NAME of a string variable 
                      to resize.  Example:
                      string="abc"
                      ["string" 1000 .resize]
                      Result, "string" ends up with storage space for 1000 
                      characters, but still holds "abc\0" 
                      If ANY truncate, status is set to FALSE */
            ival=(int) rpn_stack[rpnstackptr+1].dval;
            if(ival <=0){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ \"strvar\" size .resize.], size is invalid");
              MPC_WRITE_AND_BAILOUT(2817);
            }
            for (m=k;m<=rpnstackptr;m++){
              somevar=mpcf_findvar(MUST,rpn_stack[m].string);
              if(somevar->type != STRINGVAR){
                (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ \"rvar\" size .resize.], var is not a string variable");
                MPC_WRITE_AND_BAILOUT(2819);
              }
              fsize=strlen(somevar->string); /* original size */
              somevar->string=realloc(somevar->string,ival*sizeof(char));
              if(somevar->string==NULL){
                (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .resize.] could not obtain or release memory");
                MPC_WRITE_AND_BAILOUT(2820);
              }
              somevar->ssize=ival;
              if(fsize>=ival){ /*it truncated, so \0 is lost and must be restored */
                localstatus=0;
                for (to=somevar->string, j=1; j<ival; j++,to++){}
                *to='\0';
              }
            }
            break;
          case 41: /* getenv (May not work on some operating systems */
            for (m=k;m<=rpnstackptr;m++){
              holdsval = getenv(rpn_stack[m].string);
              if(holdsval == NULL){
                localstatus=2;  /* status is true, but = 2 to show it wasn't there */  
                *(rpn_stack[m].string)='\0';
              }
              else{
                localstatus=1;  /* status is true, and 1, to show that it is there */
                fsize=strlen(holdsval)+1;
                mpcf_enlarge_string(
                  &(rpn_stack[m].string),
                  &(rpn_stack[m].ssize),
                  fsize,
                  "[ .getenv.] failed while enlarging result string");
                rpn_stack[m].ssize=fsize;
                (void) strcpy(rpn_stack[m].string,holdsval);
/*
                free(holdsval);    

Do not free the memory at the location returned.  On DU, for instance, 
this generates a unaligned access warnings.  Just hope that NOT freeing it
will not generate a memory leak.
*/

                holdsval=NULL;
              }
            }
            break;
          case 42: /* swap, can also invert entire stack */
            fsize= (int) rpn_stack[rpnstackptr+1].dval;
            if(2*fsize > 1+rpnstackptr-k)fsize=(1+rpnstackptr-k)/2; /* max number of swaps */
            if(fsize > 0){
              for (m=k, r=rpnstackptr, j=1; j<=fsize; j++,m++,r--){
                mpcf_copy_rpn_stack(r,MPC_MAXSTACK-1,MPC_MAXSTACK-1);
                mpcf_copy_rpn_stack(m,r,MPC_MAXSTACK-1);
                mpcf_copy_rpn_stack(MPC_MAXSTACK-1,m,MPC_MAXSTACK-1);
              }
            }
            break;
          case 43: /* duplicate */
            fsize=rpnstackptr-k+1;
            if(rpnstackptr+fsize > MPC_MAXSTACK-2){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .duplicate.] overflows RPN stack");
              MPC_WRITE_AND_BAILOUT(2821);
            }
	    for (m=k,j=rpnstackptr+1 ; m <= rpnstackptr ; m++,j++){
              mpcf_copy_rpn_stack(m,j,MPC_MAXSTACK-1);
            }
	    rpnstackptr=j-1;
            break;
          case 44: /* sin */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=sin(rpn_stack[m].dval);
            }
            break;
          case 45: /* cos */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=cos(rpn_stack[m].dval);
            }
            break;
          case 46: /* tan */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=tan(rpn_stack[m].dval);
            }
            break;
          case 47: /* asin */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=asin(rpn_stack[m].dval);
            }
            break;
          case 48: /* acos */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=acos(rpn_stack[m].dval);
            }
            break;
          case 49: /* atan */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=atan(rpn_stack[m].dval);
            }
            break;
          case 50: /* expe */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=exp(rpn_stack[m].dval);
            }
            break;
          case 51: /* exp10 */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=pow( (double) 10.0,rpn_stack[m].dval);
            }
            break;
          case 52: /* loge */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=log(rpn_stack[m].dval);
            }
            break;
          case 53: /* log10 */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=log10(rpn_stack[m].dval);
            }
            break;
          case 54: /* deg2rad */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=(rpn_stack[m].dval * MPC_PI)/180.0;
            }
            break;
          case 55: /* rad2deg */
	    for (m=k; m<=rpnstackptr ; m++){
              rpn_stack[m].dval=(rpn_stack[m].dval * 180.0)/MPC_PI;
            }
            break;
          case 56: /* showstack */
            mpcf_showstack(fout[0],k,rpnstackptr); /* always to default output stream */
            break;
          case 57: /* storage, show space allocated (not used) for strings */
            for (m=k;m<=rpnstackptr;m++){
              somevar=mpcf_findvar(MUST,rpn_stack[m].string);
              if(somevar->type != STRINGVAR){
                (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ \"strvar\" .resize.], var is not a string variable");
                MPC_WRITE_AND_BAILOUT(2823);
              }
              fsize=somevar->ssize; /* amount of storage allocated */
              rpn_stack[m].dval = (double) somevar->ssize;
              rpn_stack[m].type = DBLVAR;
            }
            break;
          case 58: /* ->up, rotate, one step, UP */
            mpcf_copy_rpn_stack(k,MPC_MAXSTACK-1,MPC_MAXSTACK-1);
	    for (m=k+1; m<=rpnstackptr ; m++){
              mpcf_copy_rpn_stack(m,m-1,MPC_MAXSTACK-1);
            }
            mpcf_copy_rpn_stack(MPC_MAXSTACK-1,rpnstackptr,MPC_MAXSTACK-1);
            break;
          case 59: /* ->down, rotate, one step, DOWN */
            mpcf_copy_rpn_stack(rpnstackptr,MPC_MAXSTACK-1,MPC_MAXSTACK-1);
	    for (m=rpnstackptr-1; m>=k ; m--){
              mpcf_copy_rpn_stack(m,m+1,MPC_MAXSTACK-1);
            }
            mpcf_copy_rpn_stack(MPC_MAXSTACK-1,k,MPC_MAXSTACK-1);
            break;
          case 60: /* delete */
            rpnstackptr=k-1;
            if(rpnstackptr < -1)rpnstackptr=-1;
            break;
	  case  61:  /* elements (name separator number) */
/* [ ... &, 5 .elements_. ]
  extracts the first 5 elements of the preceding stack entry, and places
  them on the stack.  If the stack held 4 strings, it would
  put 4 on the stack, and put an empty string on for the last one.
*/
            ifirst= (int) rpn_stack[rpnstackptr+1].dval;
	    if(rpnstackptr + ifirst - 1 > MPC_MAXSTACK - 2)mpcf_rpn_stack_overflow();
            if(ifirst<1){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .elements.], zero or negative index value");
              MPC_WRITE_AND_BAILOUT(2824);
            }
            if(strlen(rpn_stack[rpnstackptr+2].string)<1){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .elements.], empty delimiter string");
              MPC_WRITE_AND_BAILOUT(2825);
            }
	    (void) strncpy(cscr2,rpn_stack[rpnstackptr+2].string,MPC_MAXVARLEN-1); /* a VERY long delimiter string */
	    to=strtok(rpn_stack[rpnstackptr].string,cscr2);
	    for(j=0; (to!=NULL && j<ifirst) ; j++){
               if(j==0){
	          if(to==NULL){  /* first element not found, replace with an empty string */
                     localstatus = 0;
	             *rpn_stack[rpnstackptr].string='\0';
	          }
	       }
               else {
                  to=strtok(NULL,cscr2);
	          rpnstackptr++;
	          rpn_stack[rpnstackptr].type=STRINGVAR;
	          if(to==NULL){
                     localstatus = 0;
	             *rpn_stack[rpnstackptr].string='\0';
	          }
	          else {
                     mpcf_enlarge_string(
                     &(rpn_stack[rpnstackptr].string),
                     &(rpn_stack[rpnstackptr].ssize),
                     strlen(to)+1,
                     "[ .elements.] failed while enlarging result string");
                     (void) strcpy(rpn_stack[rpnstackptr].string,to);
	          }
	       }
            }
	    if(to==NULL && j < ifirst-1){  /* as many empty strings as are needed */
               localstatus = 0;
	       for(j++; j<ifirst ; j++){
	          rpnstackptr++;
	          rpn_stack[rpnstackptr].type=STRINGVAR;
	          *rpn_stack[rpnstackptr].string='\0';
               }
            }
	    break;
	  case  62:  /* load  (name) */
/* Load the VALUE of the NAMED variable(s) onto the stack. 
   [ ... name .load. ] -> [ ... value_of_name ]

   Load DOES NOT do the inverse of store on arrays!!!!
   To do that, use name[5] name[4] ... name[1] .load_5.

*/
	    for (m=k; m<=rpnstackptr ; m++){
              if(rpn_stack[m].type == STRINGVAR){
	        mpcf_load_on_stack(rpn_stack[m].string, m);
	      }
	      else {
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ operand .load. ], nonstring operand, is not variable name");
              MPC_WRITE_AND_BAILOUT(2826);
              }     
            }
            break;
	  case  63:  /* store (name) */
/* note that [ ... name .store_5. ] will create a sort of array, a series 
of variables named name[1] through name[5] which will hold the values of 
the 5 preceding stack items, but [ ... name .store. ] will just create
a variable named "name" which holds the preceding value. Store is 
NONdestructive, use a separate command to clear the stack space. */

	    m = rpnstackptr + 1;
            if(rpn_stack[m].type != STRINGVAR){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ operand .store. ], nonstring operand, is not variable name");
              MPC_WRITE_AND_BAILOUT(2827);
	    }
	    if(k == rpnstackptr){
              mpcf_store_from_stack(rpn_stack[m].string,k);
	    }
	    else{
              for(j=rpnstackptr;j>=k;j--){
	         (void) sprintf(cscratch,"%s[%d]",rpn_stack[m].string,1+rpnstackptr-j);
                 mpcf_store_from_stack(cscratch,j);
              }
	    }
	    break;
	  case  64:  /* scale (value) */
	    m = rpnstackptr + 1;
            for(j=k;j<=rpnstackptr;j++){
               rpn_stack[j].dval= rpn_stack[j].dval * rpn_stack[m].dval;
            }
	    break;
          case  65:  /* offset (value) */
	    m = rpnstackptr + 1;
            for(j=k;j<=rpnstackptr;j++){
               rpn_stack[j].dval= rpn_stack[j].dval + rpn_stack[m].dval;
            }
            break;
          case  66:  /* substitute: do <<>> replacements in a string*/
            holdsubs = *dosubnum; /* temporarily replace dosubnum */
            *dosubnum = (int) rpn_stack[rpnstackptr+1].dval;
            if (*dosubnum < 0 ){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .substitute. ], subs level %d is illegal",*dosubnum);
              MPC_WRITE_AND_BAILOUT(2828);
            }
            for(j=k;j<=rpnstackptr;j++){
             fsize=strlen(rpn_stack[j].string);
             if(fsize >= MPC_MAXINLINE){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .substitute. ], %d character string is larger than substitution buffer",fsize);
              MPC_WRITE_AND_BAILOUT(2833);
             }
	     (void) strcpy(cinline,rpn_stack[j].string);
             mpcf_do_subs(&cinline[0]);
             mpcf_enlarge_string(
               &(rpn_stack[j].string),
               &(rpn_stack[j].ssize),
               strlen(cinline)+1,
              "[ .substitute.] failed while enlarging result string");
              (void) strcpy(rpn_stack[j].string,cinline);
            }
            *dosubnum = holdsubs; 
            break;
          case  67:  /* array (value) load a set of array names onto the stack */
            ifirst= (int) rpn_stack[rpnstackptr+1].dval;
	    if(rpnstackptr + ifirst - 1 > MPC_MAXSTACK - 2)mpcf_rpn_stack_overflow();
            if(ifirst<1){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .array.], zero or negative array size value");
              MPC_WRITE_AND_BAILOUT(2829);
            }
	    isok=1;
            if(rpn_stack[rpnstackptr].type != STRINGVAR)isok=0;
            if(strlen(rpn_stack[rpnstackptr].string)<1)isok=0;
	    if(isok != 1){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .array.], name is empty or not a string");
              MPC_WRITE_AND_BAILOUT(2830);
	    }
	    else {
	      (void) strncpy(cscr2,rpn_stack[rpnstackptr].string,MPC_MAXVARLEN-10); /* leave space for count */
            }
	    m = rpnstackptr;
	    rpnstackptr = rpnstackptr + ifirst -1 ;
            for(j=m;j<=rpnstackptr;j++){
	         rpn_stack[j].type=STRINGVAR;
	         (void) sprintf(cscratch,"%s[%d]",cscr2,1+rpnstackptr-j);
	         mpcf_enlarge_string(
	           &(rpn_stack[j].string),
	           &(rpn_stack[j].ssize),
	           strlen(cscratch)+1,
	           "[ .array.] failed while enlarging result string");
                 (void) strcpy(rpn_stack[j].string,cscratch);
            }
            break;
          case  68:  /* lifetime (promotes last  N declared variables to a higher scope) */
            ifirst= (int) rpn_stack[rpnstackptr+1].dval;
            if(ifirst<1){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ .lifetime.], zero or negative number specified");
              MPC_WRITE_AND_BAILOUT(2831);
            }
            mpcf_change_lifetime(rpn_stack[rpnstackptr].string,ifirst);
            rpnstackptr--;
            break;
          case  69:  /* free (deletes variables and macros ) */
            for(j=k;j<=rpnstackptr;j++){
              mpcf_delvar(rpn_stack[j].string);
            }
            rpnstackptr=k-1;
            break;
          case  70:  /* stacksize - put size of rpn stack onto stack */
	    if(rpnstackptr + 1 > MPC_MAXSTACK - 1)mpcf_rpn_stack_overflow();
            rpnstackptr++;
            rpn_stack[rpnstackptr].dval=(double) rpnstackptr;
	    rpn_stack[rpnstackptr].type=DBLVAR;
            break;
          case  71:  /* EDIT, similar to DCL's F$EDIT */
            for(j=k;j<=rpnstackptr;j++){
              mpcf_do_edit(rpn_stack[j].string, 
                      rpn_stack[rpnstackptr+1].string,
                      rpn_stack[rpnstackptr+2].string);
            }
            break;
          case  72:  /* showscope */
            mpcf_show_deathlists(fout[0]);
            break;
          case  73:  /* setdepth, sets the depth on RPN operator variable(s), by name */
            dtemp = rpn_stack[rpnstackptr+1].dval;
            if(dtemp < 0){
              (void) sprintf(mpc_dbg_out,"miniproc, [ N .setdepth.], N must >= zero");
              MPC_WRITE_AND_BAILOUT(2833);
            }
            for(j=k;j<=rpnstackptr;j++){
              somevar=mpcf_findvar(MUST,rpn_stack[j].string);
              if(somevar->type != RPNOPER){
                (void) sprintf(mpc_dbg_out,"miniproc, fatal error, %.4000s is not the name of an RPN operator variable",somevar->name);
                MPC_WRITE_AND_BAILOUT(2835);
              }
              somevar->dval = dtemp;  /* this is where the depth is actually set */
            }
            rpnstackptr=k-1; /* remove these operands from the stack */
            break;
          case  74:  /* debug, write string(s) to the debug device */
            for(j=k;j<=rpnstackptr;j++){
              (void) sprintf(mpc_dbg_out,"%.4000s",rpn_stack[j].string);
              WRITE_DEBUG();
            }
            rpnstackptr=k-1;
            break;
          case 75:  /* b-xor (a b c N -> a^N b^N c^N */
            itemp = mpcf_double_to_uint(rpn_stack[rpnstackptr+1].dval);
            for(j=k;j<=rpnstackptr;j++){
              jtemp = mpcf_double_to_uint(rpn_stack[j].dval);
              jtemp = jtemp ^ itemp;
              rpn_stack[j].dval = (double) jtemp;
            }
            break;
          case 76:  /* b-not */ 
            for(j=k;j<=rpnstackptr;j++){
               itemp = mpcf_double_to_uint(rpn_stack[j].dval);
               itemp = ~itemp;
               rpn_stack[j].dval = (double) itemp;
            }
            break;
          case 77:  /* b-and  A and B and C and ... */ 
            itemp = mpcf_double_to_uint(rpn_stack[rpnstackptr].dval);
            for(j=k;j<=rpnstackptr-1;j++){
              jtemp = mpcf_double_to_uint(rpn_stack[j].dval);
              itemp = jtemp & itemp;
            }
            rpnstackptr = k;
            rpn_stack[rpnstackptr].dval = (double) itemp;
            break;
          case 78:  /* b-or */  
            itemp = mpcf_double_to_uint(rpn_stack[rpnstackptr].dval);
            for(j=k;j<=rpnstackptr-1;j++){
              jtemp = mpcf_double_to_uint(rpn_stack[j].dval);
              itemp = jtemp | itemp;
            }
            rpnstackptr = k;
            rpn_stack[rpnstackptr].dval = (double) itemp;
            break;
          case 79:  /* (dim) */  
          case 85:  /* (subdim) */  
            for (ival=1, itemp=0, j=rpnstackptr + 1; j <= rpnstackptr + MPC_TUPLE; j++,itemp++){
               if(rpn_stack[j].dval < 0 || rpn_stack[j].dval > (float) INT_MAX ){
                 (void) sprintf(mpc_dbg_out,"miniproc, fatal error, one or more dimensions out of range");
                 MPC_WRITE_AND_BAILOUT(2836);
               }
	       else {  /* all dimension arguments are in range */
                 tdim[itemp] = (int) rpn_stack[j].dval;
                 if(tdim[itemp] > 0){                /* only for nonzero indices! */
                   if( INT_MAX/tdim[itemp] < ival){  /* predict and head off integer overflow */
                     (void) sprintf(mpc_dbg_out,"miniproc, fatal error, product of dimensions is too large");
                     MPC_WRITE_AND_BAILOUT(2839);
                   }
	           ival = ival * tdim[itemp];
                 }
               }
            }
            for(j=k;j<=rpnstackptr;j++){
              somevar=mpcf_findvar(MUST,rpn_stack[j].string);
	      switch (whichop){
	        case 79:
                  mpcf_array_dim(somevar,ival,tdim); /* NULL variables handled here */
                  break;
	        default: /* 85, for now */
                  mpcf_array_dim(somevar,ival,tdim); /* NULL variables handled here */
                  break;
              }
            }
            break;
          case 80:  /* (), set the active field and offset */  
            for (itemp=0, j=rpnstackptr + 1; j <= rpnstackptr + MPC_TUPLE; j++,itemp++){
               tdim[itemp] = (int) rpn_stack[j].dval;
            }
            for(j=k;j<=rpnstackptr;j++){
              somevar=mpcf_findvar(MUST,rpn_stack[j].string);
              mpcf_array_active(somevar,tdim); /* NULL variables handled here */
            }
            break;
          case 81:  /* (showdim), has only one operand */  
            somevar=mpcf_findvar(MUST,rpn_stack[rpnstackptr].string);
            if(somevar->array == NULL){
              (void) sprintf(mpc_dbg_out,
                 "miniproc, fatal programming error, [ .showdim.] operand %.4000s is not an array",
                 rpn_stack[rpnstackptr].string);
              MPC_WRITE_AND_BAILOUT(2842);
            }
            for (itemp=0, j=rpnstackptr; (itemp < MPC_TUPLE) && (somevar->array->dim[itemp] != 0); j++,itemp++){
               rpn_stack[j].dval = (double) somevar->array->dim[itemp];
               rpn_stack[j].type = DBLVAR;
            }
            j--;
            rpnstackptr=j;
            break;
          case 82:  /* (showcell) show the active cell for one array */  
            somevar=mpcf_findvar(MUST,rpn_stack[rpnstackptr].string);
            if(somevar->array == NULL){
              (void) sprintf(mpc_dbg_out,
                 "miniproc, fatal programming error, [ .showdim.] operand %.4000s is not an array",
                 rpn_stack[rpnstackptr].string);
              MPC_WRITE_AND_BAILOUT(2843);
            }
            for (itemp=0, j=rpnstackptr; (itemp < MPC_TUPLE) && (somevar->array->dim[itemp] != 0); j++,itemp++){
               rpn_stack[j].dval = (double) somevar->array->active[itemp];
               rpn_stack[j].type = DBLVAR;
            }
            j--;
            rpnstackptr=j;
            break;
	  case  83:  /* (scale) multiply all matrix elements by value */
	    m = rpnstackptr + 1;
            for(j=k;j<=rpnstackptr;j++){
               somevar=mpcf_findvar(MUST,rpn_stack[j].string);
               mpcf_array_scale(somevar,rpn_stack[m].dval);
            }
            rpnstackptr=k-1; /* remove these operands from the stack */
	    break;
	  case  84:  /* (offset) add value to all matrix elements */
	    m = rpnstackptr + 1;
            for(j=k;j<=rpnstackptr;j++){
               somevar=mpcf_findvar(MUST,rpn_stack[j].string);
               mpcf_array_offset(somevar,rpn_stack[m].dval);
            }
            rpnstackptr=k-1; /* remove these operands from the stack */
	    break;
	  case  86:  /* .constant. look up one or more constants and leave values on the stack */
            for(j=k;j<=rpnstackptr;j++){
               if(!mpcf_lookup_constant(rpn_stack[j].string, (int *)&itemp)){
                  (void) sprintf(mpc_dbg_out,"miniproc, fatal error, unknown constant >%.4000s< ",rpn_stack[j].string);
                  MPC_WRITE_AND_BAILOUT(2844);
               }
               rpn_stack[j].dval = (double) itemp;
               rpn_stack[j].type = DBLVAR;
            }
            break;
          default:
            (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, [ .operator.], unknown operator");
            MPC_WRITE_AND_BAILOUT(2832);
       } /* end of switch  on operator */
    } /* end of switch on type for incoming stack arguments */
  } /* end of incoming stack for loop */
  if(!bothends){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [] expression lacks terminal ]");
    MPC_WRITE_AND_BAILOUT(2896);
  }

  /* at this point need to offload the data from the stack into the 
  user provided variables, if any.  First one always also goes into
  RESULT, as either an int or a string.  Check that there IS a first
  one!!! If there isn't, then RESULT will not be set, or rather, it
  will stay what it was.  */

  if(rpnstackptr >=0 ){ 
    rpn_result->type=rpn_stack[rpnstackptr].type;
    switch (rpn_result->type){
       case DBLVAR:
          rpn_result->dval =  rpn_stack[rpnstackptr].dval;
          break;
       case STRINGVAR:
          mpcf_enlarge_string(
            &(rpn_result->string),
            &(rpn_result->ssize),
            strlen(rpn_stack[rpnstackptr].string) + 1,
            "[ ] failed while enlarging result string");
           (void) strcpy(rpn_result->string,rpn_stack[rpnstackptr].string);
    }
  }
  /* now do any named variables which the user may have supplied */
  while (i<=stackptr && rpnstackptr >=0){
    mpcf_store_from_stack(actionstack[i],rpnstackptr);
    i++;    
    rpnstackptr--;
  } /* end of while */
  if(i != stackptr + 1){  /* more syntax checking, all output variables used? */
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, [ ] followed by unused output variables");
    MPC_WRITE_AND_BAILOUT(2898);
  }

  *instatus = localstatus;

} /* end of do_rpn_function */


/* ----------------------------------------------------------------
  mpcf_do_evaluate().  Perform verious operations on operands and return the
  result.  DEPRECATED, use mpcf_do_rpn_evaluate instead.
*/

void mpcf_do_evaluate(void){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, use [] instead of f$evaluate or f$<-");
    MPC_WRITE_AND_BAILOUT(2901);
}


/*------------------------------------
shift the actionstack left "left" positions.
Used to lop off tokens like "if label" or
"elseif label".  It can also shift right, but that
leaves the now emptied positions in actionstack at NULL.
*/

void mpcf_shiftactive(int left){
int i;
  if(left > 0){
    for(i=0 ; i <= stackptr-left ;  actionstack[i]=actionstack[i+left], i++){}
    stackptr = stackptr - left;
    return;
  }
  if(left < 0){
    if(stackptr - left > MPC_MAXSTACK -1){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, command has too many operands");
      MPC_WRITE_AND_BAILOUT(3001);
    }
    stackptr = stackptr - left;
    for(i=stackptr; i >= -left ;  actionstack[i]=actionstack[i+left], i--){}
    return;
  }
} /* end of mpcf_shiftactive */

/* set the day/date/time variables */
void mpcf_do_date_time(int first){
#define LOCALMAXBUF 16
  struct  tm  *time_structure;
  time_t time_val;
  char cbuffer[LOCALMAXBUF];
  char *string=&cbuffer[0];

  (void) time(&time_val);
  time_structure = localtime(&time_val);


  if(first){
      mpcf_addvar("unixtime", NULL, (int) time_val, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
      (void) strftime(string, LOCALMAXBUF, "%a", time_structure); /* Sun - Sat */
      mpcf_addvar("day", string, 0, MPC_ZEROD, STRINGVAR, IS_ORDINARY, IMMORTAL);
      (void) strftime(string, LOCALMAXBUF, "%b", time_structure); /* Jan - Dec */
      mpcf_addvar("month", string, 0, MPC_ZEROD, STRINGVAR, IS_ORDINARY, IMMORTAL);
      mpcf_addvar("wday", NULL, 1+time_structure->tm_wday, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL); /* weekday 1-7 */
      mpcf_addvar("yday", NULL, 1+time_structure->tm_yday, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL); /* yearday 1-365 */
      mpcf_addvar("dd", NULL, time_structure->tm_mday, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL); /* date 1-31 */
      mpcf_addvar("mm", NULL, 1+time_structure->tm_mon, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL); /* month 1-12*/
      mpcf_addvar("yyyy", NULL, 1900+time_structure->tm_year, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL); /* year */
      mpcf_addvar("hour", NULL, time_structure->tm_hour, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL); /* hour 0-23 */
      mpcf_addvar("minute", NULL, time_structure->tm_min, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL); /* minute 0-59 */
      mpcf_addvar("second", NULL, time_structure->tm_sec, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL); /* minute 0-59 */
  }
  else {
      mpcf_setvar("unixtime", NULL, (int) time_val, MPC_ZEROD, INTVAR, IS_ORDINARY);
      (void) strftime(string, LOCALMAXBUF, "%a", time_structure); /* Sun - Sat */
      mpcf_setvar("day", string, 0, MPC_ZEROD, STRINGVAR, IS_ORDINARY);
      (void) strftime(string, LOCALMAXBUF, "%b", time_structure); /* Jan - Dec */
      mpcf_setvar("month", string, 0, MPC_ZEROD, STRINGVAR, IS_ORDINARY);
      mpcf_setvar("wday", NULL, 1+time_structure->tm_wday, MPC_ZEROD, INTVAR, IS_ORDINARY); /* weekday 1-7 */
      mpcf_setvar("yday", NULL, 1+time_structure->tm_yday, MPC_ZEROD, INTVAR, IS_ORDINARY); /* yearday 1-365 */
      mpcf_setvar("dd", NULL, time_structure->tm_mday, MPC_ZEROD, INTVAR, IS_ORDINARY); /* date 1-31 */
      mpcf_setvar("mm", NULL, 1+time_structure->tm_mon, MPC_ZEROD, INTVAR, IS_ORDINARY); /* month 1-12*/
      mpcf_setvar("yyyy", NULL, 1900+time_structure->tm_year, MPC_ZEROD, INTVAR, IS_ORDINARY); /* year */
      mpcf_setvar("hour", NULL, time_structure->tm_hour, MPC_ZEROD, INTVAR, IS_ORDINARY); /* hour 0-23 */
      mpcf_setvar("minute", NULL, time_structure->tm_min, MPC_ZEROD, INTVAR, IS_ORDINARY); /* minute 0-59 */
      mpcf_setvar("second", NULL, time_structure->tm_sec, MPC_ZEROD, INTVAR, IS_ORDINARY); /* minute 0-59 */
  }
#undef LOCALMAXBUF
}

/* mpcf_do_directory().  Returns an array of strings corresponding
to the file names in the directory.  If the variable passed in
does not correspond to a directory which may be opened, status=0
(failure), if there are files status=1, if it is a directory,
but it is empty, status =2. */

void mpcf_do_directory(VARIABLE *thevar, char *name){
DIR *dir_pointer;
struct dirent  *dp;
int i,count;
TEMPSTRINGNODE *tshead;
TEMPSTRINGNODE *tsptr;
TEMPSTRINGNODE *tsptr_prev;
STRINGNODE *snptr;
char *sptr;

  tshead = NULL;
  tsptr_prev = NULL;
  dir_pointer = opendir(name);
  if(dir_pointer == NULL){
    *instatus = 0;
    return;
  }
  else {  
   *instatus = 1;
    count = 0;
    for (dp = readdir(dir_pointer); dp != NULL; dp = readdir(dir_pointer)){
      count++;
      sptr= (char *) malloc(1 + strlen(dp->d_name)); /* store the string */
      if(sptr == NULL){
        mpcf_wipe_tsstruct(tshead);
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, out of memory during directory listing");
        MPC_WRITE_AND_BAILOUT(4701);
      }
      (void) strcpy(sptr,dp->d_name);
      snptr = (STRINGNODE *) malloc(sizeof(STRINGNODE)); /* put together the sstruct */
      if(snptr == NULL){
        free(sptr);
        mpcf_wipe_tsstruct(tshead);
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, out of memory during directory listing");
        MPC_WRITE_AND_BAILOUT(4702);
      }
      snptr->string=sptr;
      snptr->ssize =strlen(dp->d_name);
      tsptr = (TEMPSTRINGNODE *) malloc(sizeof(TEMPSTRINGNODE)); /* put together the tsstruct */
      if(tsptr == NULL){
        free(sptr);
        free(snptr);
        mpcf_wipe_tsstruct(tshead);
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, out of memory during directory listing");
        MPC_WRITE_AND_BAILOUT(4703);
      }
      tsptr->sdata = snptr;
      tsptr->next  = NULL;
      if(tsptr_prev != NULL){ tsptr_prev->next = tsptr; }
      if(tshead == NULL){ 
         tshead = tsptr; /* this one stays fixed at the head of the list */
         tsptr_prev = tsptr; /* this one lags one behind tsptr */
      }
      tsptr_prev = tsptr;
    }
    (void) closedir(dir_pointer);

    /* at this point the TEMPSTRINGNODE linked list has been built.
    Strip the string variable down to nothing (wipe out any existing
    strings and the array block) then build it all back up again
    from the data in the linked list */

    if(count == 0){
      *instatus = 2;
      return;
    }
    if(thevar->array != NULL)mpcf_emasculate_variable(thevar);
    thevar->array = (ARRAYNEXUS *) malloc(sizeof(ARRAYNEXUS));
    if(thevar->array == NULL){
       mpcf_wipe_tsstruct(tshead);
       (void) sprintf(mpc_dbg_out,"miniproc, fatal error, out of memory during directory listing");
       MPC_WRITE_AND_BAILOUT(4704);
    }
    thevar->array->data = (void *) calloc(count,sizeof(STRINGNODE));
    if(thevar->array->data == NULL){
      mpcf_wipe_tsstruct(tshead);
      free(thevar->array); 
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not dimension array, insufficient memory");
      MPC_WRITE_AND_BAILOUT(4705);
    }
    for(i=0 ; i < MPC_TUPLE; i++){
       thevar->array->dim[i]=0;
       thevar->array->subdim[i]=0;
    }
    thevar->array->dim[0]=count;
    thevar->array->offset = 0;

    /* wasn't that fun?  Now finally we can move the pointers from the 
    linked list to the array, munching the linked list as we go */

    for(i=0,tsptr=tshead; ;i++){
      ((STRINGNODE *)thevar->array->data)[i].string=tsptr->sdata->string;
      ((STRINGNODE *)thevar->array->data)[i].ssize =tsptr->sdata->ssize;
      tsptr_prev=tsptr;
      tsptr=tsptr->next;
      free(tsptr_prev);
      if(tsptr==NULL)break;
    }
  }
}

/* mpcf_wipe_tsstruct() wipe out a tempstringnode linked list and it's data.
This is in here to prevent memory leaks */

void mpcf_wipe_tsstruct(TEMPSTRINGNODE *tshead){
TEMPSTRINGNODE *tsptr;
TEMPSTRINGNODE *tsptr2;
  for(tsptr=tshead;tsptr!=NULL;){
    free(tsptr->sdata->string);     /* the string pointed to by the STRINGNODE */
    free(tsptr->sdata);             /* the STRINGNODE itself */
    tsptr2=tsptr;                   /* hang onto the TEMPSTRINGNODE pointer */
    tsptr =tsptr->next;             /* step along the list by one */
    free(tsptr2);                   /* ok, NOW wipe out the "current" node */
  }
  tshead=NULL;
}

/* --------------------------------------
   mpcf_do_function - routine where all of the f$whatever
   functions live (or at least, a stub that calls out
   to them is located.)
*/


void mpcf_do_function(void){
char *sval;
char **tsval=&sval;
int ival;
double dval;
enum istype type;
char disposition[3];
static struct stat *statbuffer=NULL;
VARIABLE *tempvar;
char *newline;
int filenum;
FILE *ffrom;
int isabreak;
int i;


  if((*trace & MPC_TRACE_FUNC)==MPC_TRACE_FUNC){
     (void) sprintf(mpc_dbg_out," function    >%.4000s< ",actionstack[0]);
     WRITE_DEBUG();
  }

/* f$break implementation, this intentionally falls through into f$exit! */

  if(strcmp(actionstack[0],"f$break")==0){
    if(ifstack[ifptr].type == IFIN){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$break executed outside of if/then/else structure");
      MPC_WRITE_AND_BAILOUT(3101);
    }
    (void) strcpy(actionstack[0],"f$exit");
    isabreak=1;  /* only used in f$exit, which follows immediately */
  }
  else {
    isabreak=0;
  }

/* f$exit implementation, do not separate from f$break! */

  if(strcmp(actionstack[0],"f$exit")==0){
    ival=EXIT_SUCCESS;
    switch (stackptr){
      case 0:
        break;
      case 1:
      case 2: /* doesn't matter what 2nd param is, just so long as it exists */
        mpcf_resolveright(actionstack[1],tsval,&ival,&dval,&type,0);
        if(type != INTVAR){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$exit had noninteger parameter");
          MPC_WRITE_AND_BAILOUT(3102);
        }
        break;
      default:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$exit, too many parameters");
        MPC_WRITE_AND_BAILOUT(3103);
    }

    if(stackptr==2){ /* Program triggered an immediate exit, pass ival to OS */
        mpc_smuggle_exit_status=ival;  /*smuggle this all the way up to main, it may be zero! */
        MPC_BAILOUT(MPC_SMUGGLE_STATUS);
    }

    if(fromsource!=INFROMFILE){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$exit or f$break used to exit a macro");
      MPC_WRITE_AND_BAILOUT(3104);
    }

    if(finc >= 0){ /* close the existing file */
      if(fclose(fin[finc]) != 0){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$exit, could not close current input file ");
        MPC_WRITE_AND_BAILOUT(3105);
      }

      mpcf_clear_deathlist(fin_death[finc]); /* clean out the deathlist for this file */
      finc--;                           /* decrement the file indicator */

    }

    if(finc >=0 ){    /* there are still files open above the one we just closed */
      /* if this is an f$exit verify that the last tag in the ifstack is an
         IN, then roll it up. If it is an f$break, we don't do this test.
         ifptr better be valid or there is a bug elsewhere in the program!*/

      if(ifstack[ifptr].type != IFIN){
        if(isabreak == 0){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, at f$exit incomplete if/then/else structure detected");
          MPC_WRITE_AND_BAILOUT(3106);
        }
        else { /* roll up the if stack to the next file mark */
          for( ; (ifstack[ifptr].type!=IFIN && ifptr>=0) ;ifptr-- ){}
        }
      }

      mpcf_yankaltprefix(); /* restore altprefix which was in force when the 
                          f$in was invoked */

      ifptr--;
      ifscan = NO;
/* on return from an f$exit we should NEVER be scanning, since we are 
   absolutely in a block of executing code - subsequent elseif or else
   may trigger a scan though.

      if(ifptr >= 0 ){
        if( (ifstack[ifptr].type == IFLABEL) && (ifstack[ifptr].doneit == YES))
        ifscan=YES;
      }
*/
      *instatus=1;
      return;
    }
    else {           /* all files closed, exit program */
      if(ifptr >= 0){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, at program exit incomplete if/then/else structure detected");
        MPC_WRITE_AND_BAILOUT(3107);
      }
      /* exit(ival);  */
      mpc_smuggle_exit_status=ival;  /*smuggle this all the way up to main, it may be zero! */
      MPC_BAILOUT(MPC_SMUGGLE_STATUS);
    }
    *instatus=1;
    return;

  } /* end of f$exit */

/* f$in implementation */

  if(strcmp(actionstack[0],"f$in")==0){

    if( (*safety & MPC_SAFE_IN) == MPC_SAFE_IN){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal safety error, f$in attempted");
      MPC_WRITE_AND_BAILOUT(3108);
    }
    switch (stackptr) {
       case 0:
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$in, not enough parameters");
         MPC_WRITE_AND_BAILOUT(3109);
       case 2:
         mpcf_resolveright(actionstack[2],tsval,&ival,&dval,&type,0);
         if(type != INTVAR){
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$in parameter 2 must be a filenumber");
           MPC_WRITE_AND_BAILOUT(3110);
         }
         if(ival < MPC_FILE_ILOW || ival > MPC_FILE_IHIGH ){
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$in filenumber is illegal");
           MPC_WRITE_AND_BAILOUT(3111);
         }
         filenum=ival;
         break;
       case 1:
         filenum=10;
         break;
       default:
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$in, too many parameters");
         MPC_WRITE_AND_BAILOUT(3112);
    }

    mpcf_resolveright(actionstack[1],tsval,&ival,&dval,&type,0);
    if(type != STRINGVAR){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$in parameter must be a filename");
      MPC_WRITE_AND_BAILOUT(3113);
    }

    switch (filenum){
      case 10: /* the command stream = special case */
        if (finc < 9){   /* space to open a new file */
          finc++;
          if( (*safety & MPC_SAFE_PATH) == MPC_SAFE_PATH)mpcf_pathsafe(sval);
          if(strlen(sval) > MPC_MAXFNAME - 1 ){
            (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$in, input filename too long");
            MPC_WRITE_AND_BAILOUT(3114);
          }
          fin[finc]=fopen(sval,"r");
          if(fin[finc]==NULL){
            (void) sprintf(mpc_dbg_out,"miniproc, couldn't open %.4000s ",sval);
            MPC_WRITE_AND_BAILOUT(3115);
          }
          (void) strcpy(&finname[finc][0],sval);

  
          /* drop a tag into the ifstack so that label searches don't go up
             past this point */
  
          ifptr++;
          if(ifptr < MPC_MAXIFDEPTH){
            ifstack[ifptr].type = IFIN;
            mpcf_stuffaltprefix();
          }
          else{
            (void) sprintf(mpc_dbg_out,"miniproc, fatal error, if stack overflow when macro %.4000s was invoked ",actionstack[0]);
            MPC_WRITE_AND_BAILOUT(3116);
          }
        }
        else {           /* maximum number of files already open */
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$in %.4000s, max. # of files already open ",sval);
          MPC_WRITE_AND_BAILOUT(3117);
        }
        break;
      default: /*files MPC_FILE_INORMAL -> MPC_FILE_IHIGH */
        if(finr[filenum - MPC_FILE_INORMAL]!=NULL){
          if(fclose(finr[filenum-MPC_FILE_INORMAL])!=0){
            (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$out, could not close current output file");
            MPC_WRITE_AND_BAILOUT(3118);
          }
          finr[filenum-MPC_FILE_INORMAL]=NULL;
        }

        *instatus=1;
        if(strlen(sval)==0)return;  /* empty filename closes the file */

        if( (*safety & MPC_SAFE_PATH) == MPC_SAFE_PATH)mpcf_pathsafe(sval);
        finr[filenum-MPC_FILE_INORMAL]=fopen(sval,"r");
        if(finr[filenum-MPC_FILE_INORMAL] == NULL){           /* oops, bad input file */
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$in, could not open %.4000s ",sval);
          MPC_WRITE_AND_BAILOUT(3119);
        }
    }

    *instatus=1;
    return;

  } /* end of f$in */

/* f$out implementation */

  if(strcmp(actionstack[0],"f$out")==0){

    if( (*safety & MPC_SAFE_OUT) == MPC_SAFE_OUT){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal safety error, f$out attempted");
      MPC_WRITE_AND_BAILOUT(3120);
    }
    (void) strcpy(disposition,"w");  /* default is "w", for all modes */
    switch (stackptr){
      case 0:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$out, no filename supplied");
        MPC_WRITE_AND_BAILOUT(3121);
      case 1:
        filenum=0; /*default to first filenum*/
        break;
      case 3: /* case 3 intentionally falls through into case 2 !!!
                 Use mpcf_resolveright so that "append" or append can be
                 the third argument.  Former form is preferred. */
        mpcf_resolveright(actionstack[3],tsval,&ival,&dval,&type,2);
        (void) strcpy(disposition,"X");
        if(strcmp(sval,"append")==0)
          (void) strcpy(disposition,"a");
        if(strcmp(sval,"new")==0)
          (void) strcpy(disposition,"w");
        if(strcmp(disposition,"X")==0){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$out disposition must be append or new");
          MPC_WRITE_AND_BAILOUT(3122);
        }
      case 2:
        mpcf_resolveright(actionstack[2],tsval,&ival,&dval,&type,0);
        if(type != INTVAR){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$out filenumber is invalid");
          MPC_WRITE_AND_BAILOUT(3123);
        }
        else {
          if(ival <0 || ival >9){
            (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$out filenumber is invalid");
            MPC_WRITE_AND_BAILOUT(3124);
          }
          else {
            filenum=ival;
          }
        }
        break;
      default:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$out, too many parameters");
        MPC_WRITE_AND_BAILOUT(3125);
    }

    mpcf_resolveright(actionstack[1],tsval,&ival,&dval,&type,0);
    if(type != STRINGVAR){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$out parameter must be a filename");
      MPC_WRITE_AND_BAILOUT(3126);
    }

    if(fout[filenum]!=NULL){
       if(fclose(fout[filenum])!=0){
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$out, could not close current output file");
         MPC_WRITE_AND_BAILOUT(3127);
       }
       fout[filenum]=NULL;
    }

    *instatus=1;
    if(strlen(sval)==0)return;  /* empty filename closes the file */

    if( (*safety & MPC_SAFE_PATH) == MPC_SAFE_PATH)mpcf_pathsafe(sval);
    fout[filenum]=fopen(sval,&disposition[0]);
    if(fout[filenum] == NULL){           /* oops, bad output file */
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$out, could not open %.4000s ",sval);
      MPC_WRITE_AND_BAILOUT(3128);
    }
    *instatus=1;
    return;

  } /* end of f$out */

/* f$read implementation */

  if(strcmp(actionstack[0],"f$read")==0){

    switch (stackptr){
      case 2:
        break;
      default:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$read, wrong number of parameters");
        MPC_WRITE_AND_BAILOUT(3129);
    }

    tempvar=mpcf_findvar(MUST,actionstack[1]);
    if(tempvar->type != STRINGVAR){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$read, variable is not a string");
      MPC_WRITE_AND_BAILOUT(3131);
    }

    mpcf_resolveright(actionstack[2],tsval,&ival,&dval,&type,0);
    if(type != INTVAR){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$read 2nd parameter must be a filenumber");
      MPC_WRITE_AND_BAILOUT(3132);
    }

    if( ival == MPC_FILE_ILOW)
        ffrom=fin[finc];
    else {
        if(ival >= MPC_FILE_INORMAL && ival <= MPC_FILE_IHIGH){
          ffrom=finr[ival-MPC_FILE_INORMAL];
        }
        else {
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$read filenumber is invalid");
          MPC_WRITE_AND_BAILOUT(3133);
        }
    }

    if(fgets(tempvar->string,tempvar->ssize,ffrom) == NULL){
      *instatus=0;
      return;
    }

    /* if the whole line fit, there will be a \n on the end, if not, string 
       was too small to hold it */

    newline=strstr(tempvar->string,"\n");
    if(newline == NULL){ /* bad news, string truncated */
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$read, %.4000s too small to hold string",actionstack[1]);
      MPC_WRITE_AND_BAILOUT(3134);
    }
    *newline='\0';
    *instatus=1;
    return;

  } /* end of f$read */

/* f$write implementation */

  if(strcmp(actionstack[0],"f$write")==0){

    switch (stackptr){
      case 2:
        break;
      default:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$write, wrong number of parameters");
        MPC_WRITE_AND_BAILOUT(3135);
    }


    mpcf_resolveright(actionstack[2],tsval,&ival,&dval,&type,0);
    if(type != INTVAR){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$read 2nd parameter must be a filenumber");
      MPC_WRITE_AND_BAILOUT(3136);
    }
    if(ival >= MPC_FILE_OLOW && ival <= MPC_FILE_OHIGH){
      filenum=ival;
    }
    else {
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$write filenumber is invalid");
      MPC_WRITE_AND_BAILOUT(3137);
    }

    mpcf_resolveright(actionstack[1],tsval,&ival,&dval,&type,0);
    if(type != STRINGVAR){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$write, variable is not a string");
      MPC_WRITE_AND_BAILOUT(3138);
    }

    if(fprintf(fout[filenum],"%s\n",sval) <= 0){
      *instatus=0;
    }
    else {
      *instatus=1;
    }
    return;

  } /* end of f$write */

/* f$<- implementation. Was based on f$evaluate, and has been deprecated */

  if(strcmp(actionstack[0],"f$<-")==0){
    mpcf_do_evaluate();
    return;
  }

/* f$evaluate implementation */

  if(strcmp(actionstack[0],"f$evaluate")==0){
    mpcf_do_evaluate();
    return;
  }

/* f$date implementation */

  if(strcmp(actionstack[0],"f$date")==0){
    mpcf_do_date_time(0);
    *instatus=1;
    return;
  }

/* f$type implementation */
  if(strcmp(actionstack[0],"f$type")==0){
    switch (stackptr){
      case 0:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$type, no variable name supplied");
        MPC_WRITE_AND_BAILOUT(3139);
      case 1:
        break;
      default:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$type, too many parameters");
        MPC_WRITE_AND_BAILOUT(3140);
    }

    tempvar=mpcf_findvar(MAY,actionstack[1]);
    if(tempvar==NULL){
      *instatus=0;
      return;
    }
    else {
      switch (tempvar->type){
         case INTVAR:
           *instatus=1;
           return;
         case STRINGVAR:
           *instatus=2;
           if(tempvar->string != NULL){
             if(*(tempvar->string) == '\0')*instatus=4;
           }
           return;
         case MACRO:
           *instatus=3;
           return;
         case DBLVAR:
           *instatus=6;
           return;
         case RPNOPER:
           *instatus=5;
           return;
         default:
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$type encountered unimplemented variable type");
           MPC_WRITE_AND_BAILOUT(3141);
      }
    }
  } /* end of f$type */

/* f$file_info implementation */

  if(strcmp(actionstack[0],"f$file_info")==0){
    if( (*safety & MPC_SAFE_FILE) == MPC_SAFE_FILE){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal safety error, f$file_info attempted");
      MPC_WRITE_AND_BAILOUT(3142);
    }
    switch (stackptr){
      case 0:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$file_info, no filename supplied");
        MPC_WRITE_AND_BAILOUT(3143);
      case 1:
        break;
      default:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$file_info, too many parameters");
        MPC_WRITE_AND_BAILOUT(3144);
    }

    if( (*safety & MPC_SAFE_PATH) == MPC_SAFE_PATH)mpcf_pathsafe(actionstack[1]);
    if(statbuffer==NULL){
      statbuffer=malloc(sizeof(struct stat));
      if(statbuffer==NULL){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$file_info, memory allocation error");
        MPC_WRITE_AND_BAILOUT(3145);
      }
      mpcf_addvar("file_exists", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
      mpcf_addvar("file_size", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
      mpcf_addvar("file_modified", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
      mpcf_addvar("file_isdir", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
    }

    if(stat(actionstack[1],statbuffer) == 0)
      *instatus=1;
    else
      *instatus=0;

    mpcf_setvar("file_exists", NULL, *instatus, MPC_ZEROD, INTVAR, IS_ORDINARY);
    mpcf_setvar("file_size", NULL, statbuffer->st_size, MPC_ZEROD, INTVAR, IS_ORDINARY);
    mpcf_setvar("file_modified", NULL, statbuffer->st_mtime, MPC_ZEROD, INTVAR, IS_ORDINARY);
    if((int) statbuffer->st_mode & (int) S_IFDIR){mpcf_setvar("file_isdir", NULL, 1, MPC_ZEROD, INTVAR, IS_ORDINARY);}
    else {mpcf_setvar("file_isdir", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY);}

    *instatus=1;
    return;

  } /* end of f$file_info */

/* f$dir_list implementation */

  if(strcmp(actionstack[0],"f$dir_list")==0){
    if(*safety & MPC_SAFE_DIR){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal safety error, f$dir_list attempted");
      MPC_WRITE_AND_BAILOUT(3171);
    }
    switch (stackptr){
      case 0:
      case 1:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$dir_list needs to arguments: filename and variable name");
        MPC_WRITE_AND_BAILOUT(3172);
      case 2:
        break;
      default:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$dir_list, too many parameters");
        MPC_WRITE_AND_BAILOUT(3173);
    }

    if( (*safety & MPC_SAFE_PATH) == MPC_SAFE_PATH)mpcf_pathsafe(actionstack[1]);

    /* make sure there's a string variable to work on */

    mpcf_resolveright(actionstack[1],tsval,&ival,&dval,&type,2); /* allow dir, 'dir' or a variable containing dir */
    if(type != STRINGVAR){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$out parameter must be a filename");
      MPC_WRITE_AND_BAILOUT(3175);
    }
    tempvar = mpcf_findvar(MAY,actionstack[2]);
    if(tempvar==NULL || tempvar->type != STRINGVAR){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$dir_list requires an existing STRING variable");
      MPC_WRITE_AND_BAILOUT(3174);
    }

    mpcf_do_directory(tempvar,sval);

    return;

  } /* end of f$dir_list */

/* f$macro_body implementation */

  if(strcmp(actionstack[0],"f$macro_body")==0){
    switch (stackptr){
      case 0:
        if(fromsource != INFROMMACRO){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_body called outside of a macro");
          MPC_WRITE_AND_BAILOUT(3156);
        }
        (activemacro[isactivemacro])->mfields->body = (activemacro[isactivemacro])->mfields->macro;
        break;
      default:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_body does not take parameters");
        MPC_WRITE_AND_BAILOUT(3157);
    }
    *instatus=1;
    return;

  } /* end of f$macro_body */

/* f$macro_map implementation */

  if(strcmp(actionstack[0],"f$macro_map")==0){
    switch (stackptr){
      case 0:
      case 1:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_map needs two or more parameters");
        MPC_WRITE_AND_BAILOUT(3158);
      default: /* first parameter is the name of an array of ANY type, 2nd -> N are macros */
        tempvar= mpcf_findvar(MUST,actionstack[1]);
        if(tempvar->array == NULL){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_map parameter is not an array");
          MPC_WRITE_AND_BAILOUT(3161);
        }
        for(i=2; i<=stackptr;i++){
           mpcf_resolveright(actionstack[i],tsval,&ival,&dval,&type,1); /* note mode = 1 !*/
           if(type != STRINGVAR){
             (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_map invalid macro name");
             MPC_WRITE_AND_BAILOUT(3162);
           }
           mpcf_macro_repeat(sval,tempvar); /* map the repeat counts on this one macro (in sval)*/
        }
        break;
    }
    *instatus=1;
    return;

  } /* end of f$macro_map */

/* f$macro_record implementation */

  if(strcmp(actionstack[0],"f$macro_record")==0){
    switch (stackptr){
      case 0:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_record needs a macro name");
        MPC_WRITE_AND_BAILOUT(3146);
      case 1:
        (void) strcpy(deck,"f$macro_end");
        break;
      case 2:
        (void) strcpy(deck,actionstack[2]);
        break;
      default:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_record, too many parameters");
        MPC_WRITE_AND_BAILOUT(3147);
    }

    mpcf_macro_record(actionstack[1], NULL); /* create a new macro and set it to record */
    *instatus=1;
    return;

  } /* end of f$macro_record */

/* f$macro_create implementation */

  if(strcmp(actionstack[0],"f$macro_create")==0){
    switch (stackptr){
      case 0:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_create needs a macro name");
        MPC_WRITE_AND_BAILOUT(3154);
      case 1:
        mpcf_macro_record(actionstack[1], NULL); /* create a new macro and set it to record */
        howtohandle = MPC_HANDLE_NORMAL;         /* recording, but keep processing normally */
        break;
      default:
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_record, too many parameters");
        MPC_WRITE_AND_BAILOUT(3155);
    }

    *instatus=1;
    return;

  } /* end of f$macro_create */

/* f$macro_repeat implementation */

  if(strcmp(actionstack[0],"f$macro_repeat")==0){
    switch (stackptr){
      case 1:
      case 2:
      case 3:
      case 4:
        mpcf_resolveright(actionstack[1],tsval,&ival,&dval,&type,1); /* note mode = 1 !*/
        if(type != STRINGVAR){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_repeat invalid macro name");
          MPC_WRITE_AND_BAILOUT(3148);
        }
        break;
      default:
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, f$macro_repeat illegal number of parameters");
         MPC_WRITE_AND_BAILOUT(3149);
    }

    mpcf_macro_repeat(sval,NULL); /* set the repeat counts on this one macro */
    *instatus=1;
    return;

  } /* end of f$macro_repeat */

/* deck/f$macro_end implementation */

  if(strcmp(actionstack[0],"f$macro_end")==0){
 
    if(stackptr >= 2){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, deck/f$macro_end, too many parameters");
      MPC_WRITE_AND_BAILOUT(3150);
    }

    if (recmacro == NULL){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, cannot end recording a macro before starting");
      MPC_WRITE_AND_BAILOUT(3151);
    }

    /* check the state of the macro, it must not be RECORDING */

    if(recmacro->mfields->mstate != RECORDING){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, nothing recorded in macro %.4000s",recmacro->name);
      MPC_WRITE_AND_BAILOUT(3152);
    }

    mpcf_macro_make_playable();

    /* reset normal line processing */

    howtohandle = MPC_HANDLE_NORMAL;
    *instatus=1;
    return;

  } /* end of f$macro_end */

/* f$macro_return implementation */

  if(strcmp(actionstack[0],"f$macro_return")==0){
    mpcf_macro_return(0);    
    return;
  } /* end of f$macro_return */

/* f$macro_break implementation */

  if(strcmp(actionstack[0],"f$macro_break")==0){
    mpcf_macro_return(1);    
    return;
  } /* end of f$macro_return */

  (void) sprintf(mpc_dbg_out,"miniproc, fatal error, no such function as %.4000s",actionstack[0]);
  MPC_WRITE_AND_BAILOUT(3153);

} /* end of mpcf_do_function */

/* ---------------------------------------------------------
   mpcf_do_macro(), start playing the defined macro.
*/

void mpcf_do_macro(void){
  int i;
  char cvarname[MPC_MAXVARLEN];
  char *varname=&cvarname[0];
  char *sval;
  char **tsval=&sval;
  int ival;
  double dval;
  enum istype type;
  VARIABLE *play;
  int doit;
  int havepassed;

  if(stackptr > 10){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, more than 9 parameters passed to macro %.4000s ",actionstack[0]);
    MPC_WRITE_AND_BAILOUT(3201);
  }

  if((*trace & MPC_TRACE_MACRO)==MPC_TRACE_MACRO){
     (void) sprintf(mpc_dbg_out," macro       >%.4000s<",actionstack[0]);
     WRITE_DEBUG();
  }
  isactivemacro++;
  if(isactivemacro > MPC_MAXMACRODEPTH - 1){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, call stack exhausted at macro %.4000s ",actionstack[0]);
    MPC_WRITE_AND_BAILOUT(3202);
  }

  activemacro[isactivemacro]=mpcf_findvar(MUST,actionstack[0]);
  play=activemacro[isactivemacro];

  if(play->type != MACRO){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, syntax error, [%.4000s] cannot be invoked as a Macro ",actionstack[0]);
    MPC_WRITE_AND_BAILOUT(3204);
  }

  if(play->mfields->mstate != DONE){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, attempt to recursively execute macro [%.4000s] ",actionstack[0]);
    MPC_WRITE_AND_BAILOUT(3205);
  }

  /* make sure that all max counters aren't zero, if they are, this macro 
     has been set to loop zero times, so roll up and return*/

  for (doit=0, i=0 ; i<=2 ; doit=doit+play->mfields->mcmax[i],i++){}
  if(doit==0){   /* all indices are zero, so skip the macro */
    play=NULL;
    isactivemacro--;
    *instatus=0;  /*return a status of 1 for this case*/
    return;
  }

  /* Looks like the macro is ok to execute. 
     Set up the internal loop counters and P and MC parameters */

  play->mfields->mstate=PLAYING;
  mpcf_setcounters(play,1,1);  /* init counters and visible counters */

  /* local variables must be for CALLING macro, but at this point
     we have already set isactivemacro to indicated CALLED macro.
     This will cause local variables to resolve to the CALLED
     macro, rather than to the CALLING macro.  Kick the isactivemacro
     back to the CALLING macro temporarily, then set it back
     again to the called. */

  isactivemacro--;
  for (havepassed=0,i=1;i<=MPC_PNUM_MAX ;i++){
    (void) sprintf(varname,"P%d",i);
    if(i <= stackptr){
      mpcf_resolveright(actionstack[i],tsval,&ival,&dval,&type,0);
      mpcf_setvar(varname, sval , ival, MPC_ZEROD, type, IS_SPECIAL);
      havepassed++;
   }
    else {
      mpcf_setvar(varname, "" , ival, MPC_ZEROD, STRINGVAR, IS_SPECIAL);
    }
  }
  (void) strcpy(varname,"P0"); /* set the argument count  */
  mpcf_setvar(varname, NULL , havepassed, MPC_ZEROD, INTVAR, IS_ORDINARY);
  isactivemacro++;

  nextfromsource=INFROMMACRO;

  /* drop a tag into the ifstack so that label searches don't go up
     past this point */

  ifptr++;
  if(ifptr < MPC_MAXIFDEPTH){
    ifstack[ifptr].type=IFMACRO;
    mpcf_stuffaltprefix();
  }
  else{
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, if stack overflow when macro %.4000s was invoked ",actionstack[0]);
    MPC_WRITE_AND_BAILOUT(3206);
  }

  /* reset user definable variable altprefix to match what was in force
     when the macro was recorded */

  mpcf_setaltprefix(play->mfields->altprefix);

  /* verify that the deathlist for this macro is empty (or it is a 
     programming error) */

/* UNCOMMENT THIS ONCE deathlist deletions are in place!
  if(play->mfields->deathlist->head != NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal programming error, deathlist NOT empty at macro run");
    MPC_WRITE_AND_BAILOUT(3207);
  }
*/

/* set the macro pointer back to "string", it may have been left at "body" */
  play->mfields->macro = play->string;

} /*end of mpcf_do_macro */

/* --------------------------------------------------
   mpcf_test().  If the test *fails* then start scanning.
   If it passes, then don't start scanning, but set the doneit
   flag.  If invert is YES, then invert the logic of the test.
*/

void mpcf_test(char *sval, int ival, double dval, enum istype type, enum ifdoneit invert){
  switch (type) {
    case INTVAR:
      if(  (invert==NO  && ival == 0) || 
           (invert==YES && ival != 0) ){
        ifscan = YES;
      }
      else {
        ifscan = NO;
        ifstack[ifptr].doneit = YES;
      }
      break;
    case DBLVAR:
      if(  (invert==NO  && dval == MPC_ZEROD) || 
           (invert==YES && dval != MPC_ZEROD) ){
        ifscan = YES;
      }
      else {
        ifscan = NO;
        ifstack[ifptr].doneit = YES;
      }
      break;
    case STRINGVAR:
      if(  (invert==NO  && (sval == NULL || strlen(sval) == 0))  ||
           (invert==YES && (sval != NULL && strlen(sval) != 0))  ){
        ifscan = YES;
      }
      else {
        ifscan = NO;
        ifstack[ifptr].doneit = YES;
      }
      break;
    case MACRO:
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, TEST resolves to macro name, not integer or string variable ");
      MPC_WRITE_AND_BAILOUT(3301);
  }
}

/* ----------------------------------------
  mpcf_do_ifelse() has most of the logic for if/else/elseif/endif,
  the rest being in mpcf_macro_return.  If invert is YES it means
  ifnot or elseifnot. This is stored in the ifstack and 
  used to invert the logic when the final test completes */

void mpcf_do_ifelse(enum ifstate instate, enum ifdoneit invert){
char slabel[MPC_MAXLABELLEN];
char *sval;
char **tsval=&sval;
int ival;
double dval;
enum istype type;
int i;

  switch (instate){
     case IF:
     case ELSEIF:
       if(stackptr == 1){
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, if or elseif requires missing TEST");
         MPC_WRITE_AND_BAILOUT(3401);
       }
       if(stackptr == 0){
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, if or elseif missing mandatory LABEL");
         MPC_WRITE_AND_BAILOUT(3402);
       }
       break;
     case ELSE:
     case ENDIF:
     default:
       if(stackptr > 1){
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, else or endif with extra parameters");
         MPC_WRITE_AND_BAILOUT(3403);
       }
       if(stackptr == 0){
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, else or endif missing mandatory LABEL");
         MPC_WRITE_AND_BAILOUT(3404);
       }
       break;    
  }

  /* number of parameters are ok , The second one is ALWAYS an immediate label,
     move it into sval */

  if(strlen(actionstack[1]) >= MPC_MAXLABELLEN){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, LABEL is too long: %.4000s",sval);
    MPC_WRITE_AND_BAILOUT(3405);
  }

  sval=&slabel[0];
  (void) strcpy(sval,actionstack[1]);

  /* very abbreviated checking when scanning.  Looking for else/elseif/
     endif lines with matching labels, if they don't match, just return.
     Structure within scanned regions may be incorrect, since logic
     checking only occurs when we pass through a region in normal mode */

  if(ifscan == YES){
    if(ifptr < 0 || ifptr > MPC_MAXIFDEPTH -1){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, invalid if/elseif/else/endif structure, label= %.4000s",sval);
      MPC_WRITE_AND_BAILOUT(3406);
    }
    if(ifstack[ifptr].state != IFLABEL ){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, invalid if/elseif/else/endif structure, label= %.4000s",sval);
      MPC_WRITE_AND_BAILOUT(3408);
    }
    if( strcmp(ifstack[ifptr].string,sval) !=0 ) /* label doesn't match, so do nothing */
      return;
  }


  /* verify that we can use this label in this position in this manner.  
     If this is not IF, it must be the last thing on the stack,
     if it is IF, it must not be elsewhere on the stack up to the last 
     MACRO or IN tag. */

  if(instate == IF){
    for(i=ifptr ; i>=0 ; i--){
      switch (ifstack[i].type){
        case IFIN:        /* don't care what labels are in other modules */
        case IFMACRO:
          i=-1; /* force exit from for loop */
          break;
        case IFLABEL:  /* is the label already in the stack for this module? */
          if(strcmp(ifstack[i].string,sval)==0){  /* it is */
              (void) sprintf(mpc_dbg_out,"miniproc, fatal error, invalid if/elseif/else/endif structure, label= %.4000s",sval);
              MPC_WRITE_AND_BAILOUT(3409);
          }
          break;
      }
    }
  }
  else {
    if ( strcmp( ifstack[ifptr].string,sval) != 0 ){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, invalid if/elseif/else/endif structure, label= %.4000s",sval);
      MPC_WRITE_AND_BAILOUT(3410);
    }
  }

  /* further verify the if/elseif/else/endif structure, but only if the 
     label matches! */

  if(ifptr >= 0){
    if(ifstack[ifptr].type==IFLABEL){  /* don't care about IFMACRO or IFIN */
      if(strcmp(sval,ifstack[ifptr].string)==0){
        if( ((ifstack[ifptr].state == IF) && (instate == IF)) ||
            ((ifstack[ifptr].state == ELSEIF) && (instate == IF)) ||
            ((ifstack[ifptr].state == ELSE) && (instate != ENDIF)) ){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, invalid if/elseif/else/endif structure, label= %.4000s",sval);
          MPC_WRITE_AND_BAILOUT(3411);
        }
    }
    }
  }

  switch (instate){
    case IF:  /* start a new if/elseif/else/endif structure, always test */
      ifptr++;
      if(ifptr >= MPC_MAXIFDEPTH){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, if structures nested too deeply");
        MPC_WRITE_AND_BAILOUT(3411);
      }
      (void) strcpy(ifstack[ifptr].string,sval);
      ifstack[ifptr].type   = IFLABEL;
      ifstack[ifptr].doneit = NO;
      ifstack[ifptr].invert = invert;
      ifscan=NO;
      break;
    case ELSEIF: /* only test if a block has not already been done */
      if( ifstack[ifptr].doneit == YES){
        ifscan=YES;
        return;
      }
      else {
        ifscan=NO;
        ifstack[ifptr].invert = invert;
      }
      break;
    case ELSE:  /*do this block if no other blocks have marked doneit,
                  otherwise start scanning.  Never test*/
      if( ifstack[ifptr].doneit == NO){
        ifscan=NO;
        ifstack[ifptr].doneit=YES;
      }
      else 
        ifscan=YES;
      return;
    case ENDIF:  /* end this if/elseif/else/endif if block, reset ifscan to 
                    match last block.*/
      ifptr--;
      ifscan = NO;
/* next encountered if,else,elseif will set ifscan as required, but there 
   may be more valid statements in this block following an endif, example:

  if   blah
  else blah
    if    blah2
    endif blah2
    statements that need to be done following the endif, so ifscan must be NO
  endif blah
*/
      return;
  }

  /* wasn't that fun?  Now the ifstack is set correctly, and we can move on
     to the heart of the matter, which is what to do about the actual tests
     for IF and ELSEIF.  The simplest case is that it is just a variable
     so try that first.  For macros and functions rotate the actionstack
     and call mpcf_do_macro, mpcf_do_function or mpcf_do_rpn_evaluate.  If the test FAILS,
     then start scanning.  mpcf_resolveright is used in a mode which will return
     a macro name in sval, and a type of MACRO, if it sees one.

     If a macro is used as the test parameter, control passes into
     that macro.  Cannot know until that macro completes what the status
     will be, and that will happen only on the final mpcf_macro_return.
     Look in mpcf_macro_return for the actions.  */

  mpcf_resolveright(actionstack[2],tsval,&ival,&dval,&type,2); /**/
  switch (type){
    case MACRO:   
      mpcf_shiftactive(2);
      mpcf_do_macro();
      break;
    case INTVAR:  /* this one is easy */
      mpcf_test(sval,ival,dval,type,invert);
      break;
    case STRINGVAR:  /* may be a function, or a real stringvar */
      if(strncmp(sval,"f$",2)==0){  /* it is a function */
        mpcf_shiftactive(2);
        mpcf_do_function();
        mpcf_test(NULL,*instatus,MPC_ZEROD,INTVAR,invert);
        return;
      }
      if(strncmp(sval,"[",1)==0){  /* it is an RPN function */
        mpcf_shiftactive(3);
        mpcf_do_rpn_evaluate();
        if(*instatus == 0){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, rpn error in if/elseif/else/endif structure ");
          MPC_WRITE_AND_BAILOUT(3413);
        }
        mpcf_test(rpn_result->string,rpn_result->ival,rpn_result->dval,rpn_result->type,invert);
        return;
      }

      /* it is a really a string */
      mpcf_test(sval,ival,dval,type,invert);
      break;
  }
}

void mpcf_print_out(char *line){
  if(fout[0]==NULL){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, write to output with no output file defined ");
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, at: %.4000s ",line);
    MPC_WRITE_AND_BAILOUT(3501);
  }
  if((*trace & MPC_TRACE_OUTPUT)==MPC_TRACE_OUTPUT){
      (void) sprintf(mpc_dbg_out,"   output    >%.4000s<",line);
      WRITE_DEBUG();
  }

/*
 if f$macro_create has been set then all OUTPUT lines are redirected to
   the recording macro until an f$macro_end is encountered.
   This differs from f$macro_record in that not only
   does recmacro exist, but howtohandle is NORMAL instead of MACRO.
   That is, normal processing takes place.
*/

  if( recmacro != NULL && howtohandle == MPC_HANDLE_NORMAL){
    mpcf_macro_record(NULL,line);
  }
  else {
    (void) fprintf(fout[0],"%s\n",line);
  }
}


void mpcf_parse_and_run(char *line){
char cvarleft[MPC_MAXVARLEN];
char *varleft=&cvarleft[0];
char *temp;
VARIABLE *ourvar;
int ival;
double dval;
char *sval;
char **tsval=&sval;
enum istype type;
int i,bail;
char *from;
char *to;
int equalcount;

  if((*trace & MPC_TRACE_COMMAND)==MPC_TRACE_COMMAND){
     (void) sprintf(mpc_dbg_out,"command line >%.4000s",line);
     WRITE_DEBUG();
  }

  /* First parsing phase.  See if it fits pattern:
     #__"whatever"  or #__&whatever.  If so, output "whatever"
     and return.  This is shorthand for either of these other
     operations:
         #__a="#__some long string"
         <<a>>
             or
         #__write a 
  */
  
  switch (*line){
    case '&':
      if (ifscan == YES)return; /* none of these operations when scanning */
      from=line;
      from++;
      mpcf_print_out(from);
      return;
    case '"':  /* external quotes, handled outside of mpcf_strtok for speed */
    case '\'':  /* internal quotes */
      if (ifscan == YES)return; /* none of these operations when scanning */
      from = mpcf_strtok(line," 	",EATIT); /* space and tab are delimiters */
      to   = mpcf_strtok(NULL," 	",EATIT); /* space and tab are delimiters */
      if(to != NULL){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal syntax error, command line arguments following a passthrough line");
        MPC_WRITE_AND_BAILOUT(3608);
      }
      mpcf_print_out(from);
      return;
    default:  /* do nothing */
      break;
  }

  /* Second parsing phase.  See if it fits pattern:
     name = whatever
     The key point is that there must be an "=" between the first two 
     tokens, maybe with extra space.  If so, it is a simple variable 
     addition or reassignment */

  for( from=line , i=1  ;
       ((*from != '=') && (*from != ' ') && (*from != '\t') && i<=strlen(line))  ;
       i++,from++){}
  if(i>MPC_MAXVARLEN-1){
    (void) sprintf(mpc_dbg_out,"miniproc, fatal error, variable name is too long %.4000s",line);
    MPC_WRITE_AND_BAILOUT(3602);
  }
  temp=from;
  for(bail=0, equalcount=0; i<=strlen(line)  ; i++,from++){
    switch (*from){
        case '=':
          equalcount++;
        case '\t':
        case ' ':
          break;
        default:
          bail=1;
          break;
    }
    if(bail==1)break;
  }

  switch (equalcount) {
     case 0:  /* function, macro, if, or something else*/
       break;
     case 1:  /* looks good, exactly one = in this region */
       if (ifscan == YES)return; /* no variable operations when scanning */
       *temp='\0'; /* terminate first token */
       (void) strcpy(varleft,line);
       ourvar= mpcf_findvar(MAY,varleft);
       from = mpcf_strtok(from," 	",KEEPIT); /* next token on line, space and tab are delimiters */
       to   = mpcf_strtok(NULL," 	",KEEPIT); /* next token on line, should not be one! */
       if(to != NULL){
         (void) sprintf(mpc_dbg_out,"miniproc, fatal error, too many arguments right of = in command line");
         MPC_WRITE_AND_BAILOUT(3609);
       }
       mpcf_resolveright(from,tsval,&ival,&dval,&type,0);
       if(ourvar==NULL)   /* make a new variable with this name */
         mpcf_addvar(varleft, sval, ival, dval, type, IS_ORDINARY, MORTAL);
       else               /* reset an existing variable with this name */
         mpcf_setvar(varleft, sval, ival, dval, type, IS_ORDINARY);
       return;
     default: /*2 or more =, not valid syntax */
       (void) sprintf(mpc_dbg_out,"miniproc, fatal error, too many = in command line: %.4000s",line);
       MPC_WRITE_AND_BAILOUT(3603);
  }

  /* Third parsing phase - the command is either a function,
     a reverse polish notation function,
     a macro,
     or an if,elseif,else,endif structure.
     Break it up into tokens and stuff them into the action stack. In
     this mode " " has no special meaning - they are just regular characters!!!
     (Translate the tokens only when the stack executes) */

    stackptr=-1;  /* next token goes into the first position in the stack */
    sval = mpcf_strtok(line," ",KEEPIT); 
    for (;sval!=NULL;){
         stackptr++;
         if(stackptr > MPC_MAXSTACK){
           (void) sprintf(mpc_dbg_out,"miniproc, fatal error, more than %d tokens in a function or macro line ",MPC_MAXSTACK);
           MPC_WRITE_AND_BAILOUT(3604);
         }
         actionstack[stackptr]=sval;
         sval=mpcf_strtok(NULL," ",KEEPIT);
    }

  /* Fourth parsing phase - decide if it is a macro or a function, and act 
     appropriately */


    if(strcmp(line,"if")==0){      /* it is an if */
      mpcf_do_ifelse(IF,NO);
      return;
    }
    if(strcmp(line,"ifnot")==0){      /* it is an ifnot */
      mpcf_do_ifelse(IF,YES);
      return;
    }
    if(strcmp(line,"elseif")==0){      /* it is an elseif */
      mpcf_do_ifelse(ELSEIF,NO);
      return;
    }
    if(strcmp(line,"elseifnot")==0){      /* it is an elseifnot */
      mpcf_do_ifelse(ELSEIF,YES);
      return;
    }
    if(strcmp(line,"else")==0){      /* it is an else */
      mpcf_do_ifelse(ELSE,NO);
      return;
    }
    if(strcmp(line,"endif")==0){      /* it is an endif */
      mpcf_do_ifelse(ENDIF,NO);
      return;
    }

    if( ifscan == YES)return;          /* scanning for part of if/elseif/else/endif */

    if(*actionstack[0] == '['){  /* is it an RPN function?*/
        mpcf_shiftactive(1);
        mpcf_do_rpn_evaluate();
        return;
    }

    if(strncmp(line,"f$",2)==0){      /* it is a built in function */
      mpcf_do_function();
      return;
    }
    /* all that is left is that it is a macro */

    if(fromsource == INFROMCOMMAND){
      (void) sprintf(mpc_dbg_out,"miniproc, fatal error, macros may not run from the command line ");
      MPC_WRITE_AND_BAILOUT(3605);
    }
    mpcf_do_macro();

}

/* mpcf_reset_mpc().  cleans up miniproc, closes all output files, sequentially 
closes all input files, and removes everything on the associated deathlists.
Then it removes everything on the miniproc death list and returns.  Any 
fatal error results in an uncontrolled program exit */

void mpcf_reset_mpc(void){
int i;

 /* close any open output files */

 for (i=MPC_FILE_OLOW; i<=MPC_FILE_OHIGH ; i++){
   if(fout[i] != NULL){
     if (fclose(fout[i]) != 0){ /* bad news, unrecoverable error */
       (void) sprintf(mpc_dbg_out,"Miniproc unrecoverable error, could not close open output file");
       WRITE_UNRECOVERABLE();
       exit(EXIT_FAILURE);
     }
     fout[i]=NULL;
   }
 }

  /* The next line isn't obvious.  It has to be like this so that 
     variables are removed from deathlists from the top down, which 
     will eventually take out all macros, and all variables which they
     have created (unless promoted onto the program death list, which we
     don't touch.)  Leave it at any other setting and  variable removal
     from deathlists will either blow up or miss some entries. */ 

 fromsource = INFROMFILE;  
 
 /* close any open input files and destroy everything on their deathlists */

 for (i=finc;  i>=0 ; i--){
   if(fin[i] != NULL){
     if (fclose(fin[i]) != 0){ /* bad news, unrecoverable error */
       (void) sprintf(mpc_dbg_out,"Miniproc unrecoverable error, could not close open input file");
       WRITE_UNRECOVERABLE();
       exit(EXIT_FAILURE);
     }

     fin[i]=NULL;
     finname[i][0]='\0';  /* storage is statically allocated, remove contents */

     /* wipe out the associated deathlist.  Each time mpcf_delvar_byvar is 
        called it destroys the variable named AND fixes up the deathlist
        which contained that variable.  So by resetting dnode to the tail
        of the list (head would work too) each time, and then calling 
        mpcf_delvar_byvar, the list will eventually empty */

     mpcf_clear_deathlist(fin_death[i]);
   }
 }

 /* empty the "miniproc" deathlist now too */

     mpcf_clear_deathlist(master_death);

 /* variables with program scope will survive into the next miniproc call
    if information is being passed out to the calling program, it must go
    through variables at that level.  */

  return;
}

/* mpcf_init_variables().  create the special variables P1-P9, then all
   of the standard variables, such as, MC*,MX*MAX, dates, and so forth.
   This function is reentrant - on the second and later calls it resets 
   these variables.
*/ 

void mpcf_init_variables(void){
  char cvarname[MPC_MAXVARLEN];
  char *varname=&cvarname[0];
  int i;
  int ival;
  VARIABLE *ourvar;
  static int firstpass=1;

 /* (re)initialize an assortment of global variables */

  ddosubnum=1;                      /* dummy SUBNUM variable */
  dmacrosubnum=0;                   /* dummy MACROSUBNUM variable */
  dconvertwidth=32;                 /* dummy CONVERT_WIDTH variable */
  dtrace=0;                         /* dummy TRACE variable, needed just until user 
                                       visible variables are initialized */
  dsafety=0;                        /* dummy safety variable, needed just until user 
                                        visible variables are initialized */
  ddeprecate=0;                     /* dummy deprecate variable, needed just until user 
                                       visible variables are initialized */
  stackptr=-1;                      /* index, begins as invalid */
  ifptr=-1;                         /* pointer into stack, initially invalid */
  ifscan=NO;                        /* start with if scanning off */
  fromsource =INFROMCOMMAND;        /* where input comes from */
  nextfromsource = INFROMCOMMAND;   /* where input comes from NEXT */
  howtohandle = MPC_HANDLE_NORMAL;  /* substitution for recording macros */
  isactivemacro=-1;                 /* index into macro calling stack, here, none active */
  finc=-1;                          /* array pointer for fin, initially invalid */
  clastfullname[0]='\0';            /* for expanded local names */

  /* initialize master and file death lists, start as empty */

  master_death->head=NULL;      /* empty master deathlist, statically allocated */
  master_death->tail=NULL;
  program_death->head=NULL;     /* empty program deathlist, statically allocated */
  program_death->tail=NULL;
  for(i=0;i<MPC_FILE_ISIZE;i++){
    if(firstpass){
    fin_death[i]=malloc(sizeof(FIRSTDEATHNODE));
      if(fin_death[i] == NULL){
        (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not initialize file deathlists, insufficient memory");
        MPC_WRITE_AND_BAILOUT(3606);
      }
    }
    (fin_death[i])->head=NULL;  /* empty deathlist for this file */
    (fin_death[i])->tail=NULL;  
  }

  /* initialize the head variables on first pass only - later passes 
     depend on deathlists/garbage collection */

  if(firstpass){
    for(i=0; i < MPC_HASHSPACE; i++){
      head[i]=NULL;
      tail[i]=NULL;
    }
  }

  /* initialize finname to be all null strings - all passes */

  for(i=0;i<MPC_FILE_ISIZE;i++){
    finname[i][0]='\0';
  }
  /* put something in deck */

  if(firstpass){(void) strcpy(deck,"f$macro_end");}

 /* any internal variables linked to user visible variables must be done 
    first, otherwise the pointers are undefined on first use! */

  if(firstpass){
    mpcf_addvar("trace", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
    ourvar= mpcf_findvar(MUST,"trace");
    trace=&(ourvar->ival);       /* internal trace variable points to
                                  variable's integer value */
    mpcf_addvar("subs", NULL, 1, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
    ourvar= mpcf_findvar(MUST,"subs");
    dosubnum=&(ourvar->ival);       /* internal subs variable points to
                                  variable's integer value */
    mpcf_addvar("macrosubs", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
    ourvar= mpcf_findvar(MUST,"macrosubs");
    domacrosubnum=&(ourvar->ival);       /* internal subs variable points to
                                  variable's integer value */
    mpcf_addvar("safety", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
    ourvar= mpcf_findvar(MUST,"safety");
    safety=&(ourvar->ival);       /* internal subs variable points to
                                  variable's integer value */
    mpcf_addvar("altprefix", "#__", 0, MPC_ZEROD, STRINGVAR, IS_ORDINARY, IMMORTAL);
    ourvar= mpcf_findvar(MUST,"altprefix");
    altprefix=ourvar;    /* internal altprefix variable points to
                            variable with same name, NOT string, which may
                            change */
    mpcf_addvar("convertwidth", NULL, 32, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
    ourvar= mpcf_findvar(MUST,"convertwidth");
    convertwidth=&(ourvar->ival);    /* point to internal variable */

    mpcf_addvar("deprecate", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
    ourvar= mpcf_findvar(MUST,"deprecate");
    deprecate=&(ourvar->ival);       /* points to internal variable */

    mpcf_addvar("STATUS", NULL, 1, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
    ourvar= mpcf_findvar(MUST,"STATUS");
    instatus=&(ourvar->ival);       /* internal status variable points to
                                  variable's integer value */

    mpcf_addvar("RESULT", NULL, 1, MPC_ZEROD, STRINGVAR, IS_SPECIAL, IMMORTAL); /* used by f$<- */
    ourvar= mpcf_findvar(MUST,"RESULT");
    rpn_result=ourvar;    /* internal rpn_result points to RESULT */

    (void) strcpy(varname,"P0");
    mpcf_addvar(varname, NULL , 0, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
    for (i=1;i<=MPC_PNUM_MAX;i++){
      (void) sprintf(varname,"P%d",i);
      mpcf_addvar(varname, NULL , 0, MPC_ZEROD, STRINGVAR, IS_SPECIAL, IMMORTAL);
    }

 /* MC variables and their shortcuts */

    for (i=1;i<=MPC_TUPLE;i++){
      if(i==1) {ival=1;}
      else     {ival=0;}
      (void) sprintf(varname,"MC%d",i);
      mpcf_addvar(varname, NULL , ival, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
      mcshort[i-1]=mpcf_findvar(MUST,varname);
      (void) sprintf(varname,"MC%dMAX",i);
      mpcf_addvar(varname, NULL , ival, MPC_ZEROD, INTVAR, IS_ORDINARY, IMMORTAL);
      mcmaxshort[i-1]=mpcf_findvar(MUST,varname);
    }

    mpcf_do_date_time(1); /* 1 creates the variables*/

    mpcf_sort_rpn_operators();  /* sort the RPN operators and set MPC_ISMAX_RPN */

  }
  else {  /* ! firstpass */
    mpcf_setvar("trace", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY);
    mpcf_setvar("subs", NULL, 1, MPC_ZEROD, INTVAR, IS_ORDINARY);
    mpcf_setvar("macrosubs", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY);
    mpcf_setvar("safety", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY);
    mpcf_setvar("altprefix", "#__", 0, MPC_ZEROD, STRINGVAR, IS_ORDINARY);
    mpcf_setvar("convertwidth", NULL, 32, MPC_ZEROD, INTVAR, IS_ORDINARY);
    mpcf_setvar("deprecate", NULL, 0, MPC_ZEROD, INTVAR, IS_ORDINARY);
    mpcf_setvar("STATUS", NULL, 1, MPC_ZEROD, INTVAR, IS_ORDINARY);
    mpcf_setvar("RESULT", NULL, 1, MPC_ZEROD, STRINGVAR, IS_SPECIAL); /* used by f$<- */
    (void) strcpy(varname,"P0");
    mpcf_setvar(varname, NULL , 0, MPC_ZEROD, INTVAR, IS_ORDINARY);
    for (i=1;i<=MPC_PNUM_MAX;i++){
      (void) sprintf(varname,"P%d",i);
      mpcf_setvar(varname, NULL , 0, MPC_ZEROD, STRINGVAR, IS_SPECIAL);
    }


    for (i=1;i<=MPC_TUPLE;i++){
      if(i==1) {ival=1;}
      else     {ival=0;}
      (void) sprintf(varname,"MC%d",i);
      mpcf_setvar(varname, NULL, ival, MPC_ZEROD, INTVAR, IS_ORDINARY);
      (void) sprintf(varname,"MC%dMAX",i);
      mpcf_setvar(varname, NULL, ival, MPC_ZEROD, INTVAR, IS_ORDINARY);
    }

    mpcf_do_date_time(0); /* 0 resets the variables*/

  /* cleanout the detritus on the RPN stack so it can reinit properly  */

    for (i=0; i<MPC_MAXSTACK; i++){
      free(rpn_stack[i].string);
    }

  }


  /* initialize the RPN stack
         first 20 are preset to have storage for 256 bytes of string, rest 
         for none (NULL).  All are set to type MACRO, which will generate
         a fatal error if used before they are reset.  These preset sizes 
         are just an estimate for some type of "standard" usage - customize
         these if your typical stacks are bigger or smaller.
  */

   for (i=0; i<MPC_MAXSTACK; i++){
      rpn_stack[i].type = MACRO;
      rpn_stack[i].dval = 0.0;
      rpn_stack[i].ival = 0;
      if(i <20){
        varname = malloc(256*sizeof(char));
        rpn_stack[i].ssize  = 256;
        if(varname == NULL){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, could not initialize RPN stack, insufficient memory");
          MPC_WRITE_AND_BAILOUT(3607);
        }
      }
      else { /* all others long enough to hold an empty string */
        varname = malloc(sizeof(char));
        rpn_stack[i].ssize  = 1;
      }
      *varname='\0';
      rpn_stack[i].string = varname;
    }

  firstpass = 0;  /* subsequent passes act differently than first */

}


int miniproc(int margc, char *margv[]){
char cfname[MPC_MAXFNAME];
char cmacroname[MPC_MAXVARLEN];
char cinline[MPC_MAXINLINE];
char csumline[MPC_MAXINLINE];
char *fname=&cfname[0];
char *macroname=&cmacroname[0];
char *vinline=&cinline[0];
char *sumline=&csumline[0];
char *temp;
int prefix;
int endstring;
int ok=1;
int argptr;
int commandmatch;
int status_jmp;
enum continuetypes {IN_MORE,IN_NEW} linestatus=IN_NEW;

/* set up the point to return to on fatal errors.  If embedded, it tries to 
clean up and return, if not embedded, it goes bye-bye */

  status_jmp = setjmp(mpc_env);
  if(status_jmp != 0){
     mpcf_reset_mpc();  /* clean up open files, deathlists, and so forth */
     if(status_jmp == MPC_SMUGGLE_STATUS){ /* this is an intended exit, no debug lines to print */
        return mpc_smuggle_exit_status;
     }
     else {
        (void) sprintf(mpc_dbg_out,"Exit error code: %d",status_jmp);
        WRITE_DEBUG();
        return EXIT_FAILURE;
     }
  }

  mpcf_init_variables();
  
  switch (margc){
    case 0:  /* run from the SLAVE_MACRO macro, used by embedded applications */
      (void) strcpy(sumline,"SLAVE_MACRO");  /* force in name of macro and run it */
      actionstack[0]=sumline;
      mpcf_do_macro();
      nextfromsource=INFROMMACRO;
      break;
    case 1:
      (void) sprintf(mpc_dbg_out,"Enter the name of the file to process: ");
      WRITE_DEBUG();
      if(scanf("%s",fname) != 1){
        (void) sprintf(mpc_dbg_out,"miniproc, aborted processing, no input file");
        WRITE_DEBUG();
        return EXIT_FAILURE;
      }
      nextfromsource=INFROMFILE;
      break;

    case 2:
      if(strlen(margv[1]) > MPC_MAXFNAME - 1 ){
        (void) sprintf(mpc_dbg_out,"miniproc, aborted processing, input filename too long");
        WRITE_DEBUG();
        return EXIT_FAILURE;
      }
      (void) strcpy(fname,margv[1]);  /* set the input file name */
      nextfromsource=INFROMFILE;
      break;

    default:
      if(strlen(margv[1]) > MPC_MAXFNAME - 1 ){
        (void) sprintf(mpc_dbg_out,"miniproc, aborted processing, input filename too long");
        WRITE_DEBUG();
        return EXIT_FAILURE;
      }
      (void) strcpy(fname,margv[1]);  /* set the input file name */
      nextfromsource=INFROMCOMMAND;
      argptr=2;
      break;
  }

  /* open the input file that is in fname */
  
 if(  (strcmp(fname,"HELP") == 0) ||
      (strcmp(fname,"help") == 0) ||
      (strcmp(fname,"-help") == 0) ||
      (strcmp(fname,"-HELP") == 0) ||
      (strcmp(fname,"-h") == 0) ||
      (strcmp(fname,"-H") == 0) ||
      (strcmp(fname,"?") == 0)){
    (void) sprintf(mpc_dbg_out,"%s\nFor more information read the file miniproc.doc",MPC_VERSION_INFO);
    WRITE_DEBUG();
    return EXIT_SUCCESS;
  }

  if(nextfromsource != INFROMMACRO){
    finc=0;
    fin[finc]=fopen(fname,"r");
    if(fin[finc]==NULL){
      (void) sprintf(mpc_dbg_out,"miniproc, couldn't open %.4000s ",fname);
      WRITE_DEBUG();
      return EXIT_FAILURE;
    }
    (void) strcpy(&finname[finc][0],fname);
  }

  /* enter the main loop, which just keeps reading, processing, reading,
     processing.  However, it may either be reading from a file or out of
     a macro */
  
  csumline[0]='\0'; /* clear the line accumulator */
  while (ok) {

    if(nextfromsource == INNOMORE){  /* exit mechanism for SLAVE_MACRO */
      mpc_smuggle_exit_status=1;  /*smuggle this all the way up to main, it may be zero! */
      MPC_BAILOUT(MPC_SMUGGLE_STATUS);
    }
    fromsource = nextfromsource;  /* redirects take effect here */ 
 
    switch (fromsource){
      case INFROMFILE:
 
        if(fgets(vinline,MPC_MAXINLINE,fin[finc]) == NULL){
          (void) sprintf(mpc_dbg_out,"miniproc, fatal error, encountered end of input file %.4000s "
             ,&finname[finc][0]);
          WRITE_DEBUG();
          return EXIT_FAILURE;
        }

        /* put a series of \0 on top of the new line character, if any */

        prefix=strcspn(vinline,"\n\r");
        if(prefix < strlen(vinline)){
           cinline[prefix]='\0';
           endstring=prefix-1;
        }
        else
          endstring=strlen(vinline)-1; /* endstring is an array index, 
                                          arrays start at 0 */

        break;

      case INFROMMACRO:
        if(mpcf_macro_gets(vinline) != 0){
          (void) sprintf(mpc_dbg_out,"miniproc, could not read macro %.4000s ",macroname);
          WRITE_DEBUG();
          return EXIT_FAILURE;
        }
        endstring=strlen(vinline)-1; /* crud on line ends was removed at first reading*/
        break;

      case INFROMCOMMAND:
        (void) strcpy(vinline,"#__");  /* make it look like a command from a file or macro*/
        (void) strcpy(&cinline[3],margv[argptr]);
        argptr++;
        if(argptr>=margc)nextfromsource=INFROMFILE;
        endstring=strlen(vinline)-1;
        break;  

    } /* end of switch of fromsource */
  
    /* test to see if this line is a command, the tests are, in order:
       1.  does it have the standard prefix?
       2.  is altprefix an empty string?  (= all lines are commands)
       3.  does the altprefix match?
    */

    if( strncmp(vinline,"#__",3)==0){  /* command prefix exists and is size 3 */
      commandmatch=3;
    }
    else {    
      commandmatch=strlen(altprefix->string);
      if(commandmatch!=0){             /* test for no prefix required */
        if(strncmp(vinline,altprefix->string,commandmatch) != 0){ /* test altprefix, if it matches, commandmatch already holds length */
           commandmatch = -1;    /* no command prefix on line */
        }        
      }
    }
        
    /* either recording a macro, in which case the lines are stored verbatim, or 
       we are doing normal processing */

    switch (howtohandle){ 
      case MPC_HANDLE_RECORD: /* scanning for deck/f$macro_end command line */
        if((*trace & MPC_TRACE_INPUT)==MPC_TRACE_INPUT){
           (void) sprintf(mpc_dbg_out,"input line   >%.4000s",vinline);
           WRITE_DEBUG();
        }
        /* check for deck/f$macro_end, which would end macro recording */

        if(commandmatch >= 0){  /* is a command */
          temp=strchr(&cinline[commandmatch],cdeck[0]);     /* first place first character appears */
          if(temp!=NULL){                /* if it appears */ 
            if(strcmp(temp,deck)==0){ /* is it deck/f$macro_end? */
              (void) strcpy(sumline,&cinline[commandmatch]);
              mpcf_trim_command_line(sumline);
              /* whatever the deck string was, always pass in f$macro_end */
              if(strcmp(sumline,deck)==0)(void) strcpy(sumline,"f$macro_end"); 
              mpcf_parse_and_run(sumline);
              break;
            }
          }
        }

        /* must not have been a deck/f$macro_end, so record it */

        mpcf_do_subs(vinline);  /* controlled by domacrosubnum */
        mpcf_macro_record(NULL, vinline); /* add this line to the currently 
                                   recording macro */
        break;
      case MPC_HANDLE_NORMAL:

        /* see if it is a command line, if not, process it and output it */

        if( commandmatch < 0){  /* not a command line */
          if(ifscan==NO){
            if(linestatus == IN_MORE){
              (void) sprintf(mpc_dbg_out,"miniproc, fatal syntax error, unterminated line continuation");
              WRITE_DEBUG();
              return EXIT_FAILURE;
            }
            if((*trace & MPC_TRACE_INPUT)==MPC_TRACE_INPUT){
               (void) sprintf(mpc_dbg_out,"input line   >%.4000s",vinline);
               WRITE_DEBUG();
            }
            mpcf_do_subs(vinline);
            mpcf_print_out(vinline);
          }
        }

        else {   /* is a command line, commandmatch holds command prefix length */

        /* take care of continuation.  command lines ending with "-" are continued
         onto subsequent command lines. NONcommand lines are NOT continued 
         (as they are text, and the program is embedded in them) */

          if(cinline[commandmatch] != '!'){ /* ignore comment lines, wherever they appear */
            if(cinline[endstring]=='-'){
              if(cinline[endstring-1]=='!'){endstring = 
                 mpcf_strip_trailing_comments(&cinline[0],endstring-1,commandmatch);}
              if(linestatus==IN_NEW){  /* first line of continuation */
                (void) strncpy(sumline,&cinline[commandmatch],endstring-commandmatch);
                csumline[endstring-commandmatch]='\0';  /* need a new 0 on end!!!! */              
                linestatus=IN_MORE;
              }
              else {/*already have some lines put together here */
                if(endstring+strlen(sumline) >= MPC_MAXINLINE){
                  (void) sprintf(mpc_dbg_out,"miniproc, fatal error, continued line of exceeds %d characters",MPC_MAXINLINE);
                  WRITE_DEBUG();
                  return EXIT_FAILURE;
                }
                (void) strncat(sumline,&cinline[commandmatch],endstring-commandmatch); /* minus #__ and - */
              }  
            }
            else { 
              if(cinline[endstring]=='!'){endstring = 
                  mpcf_strip_trailing_comments(&cinline[0],endstring,commandmatch);}
              if(linestatus==IN_NEW) /*this isn't part of a continued line*/
                (void) strcpy(sumline,&cinline[commandmatch]);
              else { /*this part ends a continued line*/
                if(endstring+strlen(sumline) >= MPC_MAXINLINE){
                  (void) sprintf(mpc_dbg_out,"miniproc, fatal error, continued line of exceeds %d characters",MPC_MAXINLINE);
                  WRITE_DEBUG();
                  return EXIT_FAILURE;
                }
                (void) strcat(sumline,&cinline[commandmatch]);
                linestatus=IN_NEW;
              }
            }
      
            /* If the whole command has been read in, do final preprocessing on it,
            then parse and run it */
      
            if(linestatus == IN_NEW){
              mpcf_trim_command_line(sumline);
              mpcf_do_subs(sumline);  /* do the <<>> replacements*/
              mpcf_parse_and_run(sumline);
              csumline[0]='\0'; /* clear the line accumulator before next loop */
            }  
          }  /* test for comment lines */
  
         /* that's it for the main loop, go back for more input.  Note that
            as a result of changes made by the preceding function the next
            line may be read out of a different file, or from a macro */
      
        } /* processing for #__ lines*/
        break;

    } /* switch on howtohandle */

  } /* outer read loop */

  return EXIT_SUCCESS;
} /* end of miniproc() */

/*
#__SLICEDICE "tinymain.c"
*/

#ifndef _EMBEDDED_MINIPROC_
int main(int argc, char *argv[]){
int outstatus;

/* This is an example of how to configure a minimal SLAVE_MACRO and run it 
(note the use of argc as zero to cause it to run).  Have to use mpcf_addvar here,
and not mpcf_macro_record, because slave_macro must be immortal.  Since the 
normal "recording" method is completely bypassed, the two are equivalent.
The calls to mpcf_macro_record here just append strings, no substitution is
available!  

  mpcf_init_variables();
  mpcf_addvar("SLAVE_MACRO", NULL, 1, MPC_ZEROD, MACRO, IS_ORDINARY, IMMORTAL);
  mpcf_macro_record(NULL,"#__ altprefix = &_");
  mpcf_macro_record(NULL,"_ setbyslave1 = 1");
  mpcf_macro_record(NULL,"_ [ &program 1 .lifetime. ]");
  mpcf_macro_record(NULL,"_ f$macro_return");
  mpcf_macro_make_playable();
  outstatus=miniproc(0, argv);
*/

/* now do a regular one */

  outstatus=miniproc(argc, argv);

/* test for memory leaks */

  exit(outstatus);
}
#endif
