/*
 * ASNCodingException.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.asn1;

import java.io.*;
import java.util.Vector;

/**
 * This class implements the base ASN.1 constructed object.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class ASNConstruct extends ASNObject
{
	protected ASNObject[] objs;

	/**
	 * Creates an ASNConstruct object.
	 * @param tag the tag (type) of the object
	 * @param objs the objects from which to make the ASN.1 construct
	 * @see ASNObject
	 */
	public ASNConstruct(int tag, ASNObject[] objs)
	{
		this.tag = tag;
		this.objs = objs;
	}

	/**
	 * Creates an ASNConstruct object.
	 * @param tag the tag (type) of the object
	 * @param vect the Vector containing objects from which to make the construct
	 * @see ASNObject
	 * @see java.util.Vector
	 */
	public ASNConstruct(int type, Vector vect)
	{
		this.tag = tag;
		this.objs = new ASNObject[vect.size()];

		for (int i = 0; i < objs.length; i++)
		{
			Object obj = vect.elementAt(i);
			if (obj instanceof ASNObject)
				objs[i] = (ASNObject) obj;
			else
				throw new IllegalArgumentException("element " + i + " of Vector is not an ASNObject");
		}
	}

	/**
	 * Returns the number of objects in this construct.
	 */
	public int size()
	{
		return objs.length;
	}

	/**
	 * Returns an ASNObject contained in the construct.
	 * @param index the index of the requested object
	 */	 
	public ASNObject getObject(int index)
	{
		return objs[index];
	}

	/**
	 * Returns the ASN.1 DER encoded representation of the construct.
	 * @return the ASN.1 DER encoded representation of the construct
	 */
	public byte[] getContent() throws IOException
	{
		if (objs == null)
			return null;

		ByteArrayOutputStream bs = new ByteArrayOutputStream();
		DEROutputStream ds = new DEROutputStream(bs);

		for (int i = 0; i < objs.length; i++)
			ds.writeObject(objs[i]);

		ds.close();
		bs.close();

		return bs.toByteArray();
	}

	/**
	 * Returns the string representation of the construct
	 * @return the string representation of the construct
	 */
	public String toString()
	{
		return toString("");
	}

	/*
	 * Returns the string representation of the construct
	 * @param prefix the string to insert before every line of the representation
	 * @return the string representation of the construct
	 */
	protected String toString(String prefix)
	{
		String indent = prefix + " ";

		StringBuffer buf = new StringBuffer();

		buf.append(prefix);
		buf.append("CONSTRUCTED TAG " + tag + "\n");

		for (int i = 0; i < objs.length; i++)
		{
			buf.append(objs[i].toString(indent));
			if (i < objs.length-1)
				buf.append("\n");
		}
		return buf.toString();
	}
}
