/*
 * ASNInteger.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.asn1;

import java.io.IOException;
import java.math.BigInteger;

/**
 * This class implements an ASN.1 INTEGER object
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class ASNInteger extends ASNObject
{
	private BigInteger value;

	/**
	 * Creates an ASNInteger object from a long integer.
	 * @param l the long value to represent
	 */
	public ASNInteger(long l)
	{
		tag = DER.ASN_INTEGER;
		value = BigInteger.valueOf(l);
	}

	/**
	 * Creates an ASNInteger object from a BigInteger.
	 * @param b the BigInteger value to represent
	 */
	public ASNInteger(BigInteger b)
	{
		tag = DER.ASN_INTEGER;
		value = b;
	}

	/**
	 * Creates an ASNInteger object from a byte encoding.
	 * @param b the byte encoding of the integer value
	 */
	public ASNInteger(byte[] b)
	{
		tag = DER.ASN_INTEGER;
		value = new BigInteger(b);
	}

	/**
	 * Returns the content of the object.
	 * @return the content of the object
	 */
	public byte[] getContent()
	{
		if (value == null)
			return null;

		if (content == null)
			content = value.toByteArray();

		return content;
	}

	/**
	 * Returns the BigInteger representation of the object
	 * @return the BigInteger representation of the object
	 */
	public BigInteger getValue()
	{
		return value;
	}

	/**
	 * Returns the int value of the object
	 * @return the int value of the object
	 */
	public int intValue()
	{
		return value.intValue();
	}

	/**
	 * Returns the long value of the object
	 * @return the long value of the object
	 */
	public long longValue()
	{
		return value.longValue();
	}

	/**
	 * Returns the string representation of the object
	 * @return the string representation of the object
	 */
	public String toString()
	{
		return "INTEGER value " + value.toString(16);
	}
}
