/*
 * DERInputStream.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.asn1;

import java.io.*;
import java.util.Vector;

/**
 * This class implements a FilterInputStream that can read ASN.1 DER encoded objects
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class DERInputStream extends FilterInputStream
{
	/**
	 * Creates a DERInputStream from an InputStream.
	 * @param in the underlying InputStream
	 */
	public DERInputStream(InputStream in)
	{
		super(in);
	}

	/**
	 * Returns the next ASNObject decoded from the input stream.
	 * @return the ASNObject decoded from the input stream
	 */
	public ASNObject readObject() throws IOException
	{
		int type, tmp;

		if ((tmp = read()) < 0)
			return null;

		type = tmp;

		if ((tmp = read()) < 0)
			throw new ASNCodingException("");

		int contentLength = 0;

		if (tmp <= 0x7f)
		{
			contentLength = tmp;
		}
		else if (tmp >= 0x81 && tmp <= 0x84)
		{
			int count = tmp & 0x3;

			for (int i = 0; i < count; i++)
			{
				if ((tmp = read()) < 0)
					throw new ASNCodingException("");

				contentLength <<= 8;
				contentLength += tmp;
			}

			if (contentLength < 0)
				throw new ASNCodingException("");
		}
		else
			throw new ASNCodingException("object length is too long, " + tmp);

		// read content bytes
		byte[] content = new byte[contentLength];
		
		int offset = 0;
		int length = contentLength;

		while (length > 0)
		{
			if ((tmp = read(content, offset, length)) < 0)
				throw new ASNCodingException("");
			offset += tmp;
			length -= tmp;
		}

		if ((type & DER.ASN_CONSTRUCTED) == 0)
		{
			switch (type & 0x1f)
			{
			case DER.ASN_INTEGER:
				return new ASNInteger(content);
			case DER.ASN_BITSTRING:
				return new ASNBitString(content);
			case DER.ASN_OCTETSTRING:
				return new ASNOctetString(content);
			case DER.ASN_NULL:
				return new ASNNull();
			case DER.ASN_OID:
				return new ASNObjectID(content);
			case DER.ASN_PRINTABLESTRING:
				return new ASNPrintableString(content);
			default:
				return new ASNObject(type, content);
			}
		}
		else
		{
			Vector objs = new Vector();

			ByteArrayInputStream bis = new ByteArrayInputStream(content);
			DERInputStream dis = new DERInputStream(bis);

			while (true)
			{
				ASNObject obj = dis.readObject();

				if (obj == null)
				{
					dis.close();
					bis.close();
					break;
				}
				
				objs.addElement(obj);
			}

			switch (type & 0x1f)
			{
			case DER.ASN_SEQUENCE:
				return new ASNSequence(objs);
			case DER.ASN_SET:
				return new ASNSet(objs);
			default:
				return new ASNConstruct((byte) type, objs);
			}
		}
	}
}
