/*
 * DEROutputStream.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.asn1;

import java.io.*;

/**
 * This class implements a FilterOutputStream that can write ASN.1 DER encoded objects
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class DEROutputStream extends FilterOutputStream
{
	/**
	 * Creates a DEROutputStream from an OutputStream.
	 * @param out the underlying OutputStream
	 */
	public DEROutputStream(OutputStream out)
	{
		super(out);
	}

	/**
	 * Writes the DER encoding for the length of the content followed by the content itself to
	 * the output stream.
	 * @param content the DER content to write
	 */
	private void writeLengthContents(byte[] content) throws IOException
	{
		int len = content.length;
		
		if (len < 0x80)
		{
			write(len);
		}
		else if (len < 0x100)
		{
			write(0x81);
			write(len);
		}
		else if (len < 0x10000)
		{
			write(0x82);
			write(len >> 8);
			write(len & 0xff);
		}
		else if (len < 0x1000000)
		{
			write(0x83);
			write(len >> 16);
			write((len >> 8) & 0xff);
			write(len & 0xff);
		}
		else
		{
			write(0x84);
			write(len >> 24);
			write((len >> 16) & 0xff);
			write((len >> 8) & 0xff);
			write(len & 0xff);
		}
		write(content);
	}

	/**
	 * Writes an ASNObject to the OutputStream
	 * @param obj the ASNObject to write
	 */
	public void writeObject(ASNObject obj) throws IOException
	{
		byte[] contents = obj.getContent();

		write(obj.tag);
		writeLengthContents(contents);
	}
}
