/*
 * BlockCipher.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto;

import java.util.Arrays;
import java.security.*;
import java.security.spec.*;
import javax.crypto.*;
import javax.crypto.spec.*;
import com.beeyond.crypto.spec.*;

/**
 * This abstract class implements a generic blockcipher.
 * <p>
 * Subclasses will only need to implement setup, encrypt and decrypt
 * methods.
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public abstract class BlockCipher extends CipherSpi
{
	/**
	 * Definitions for the feedback mode.
	 */
	public final static int
		MODE_ECB = 0,
		MODE_CBC = 1;

	/**
	 * Definitions for the feedback mode names.
	 */
	public final static String[]
		MODES = { "ECB", "CBC" };

	/**
	 * Definitions for the block padding.
	 */
	public final static int
		PADDING_NOPADDING = 0,
		PADDING_PKCS5 = 1;

	/**
	 * Definitions for the block padding names.
	 */
	private final static String[]
		PADDINGS = { "NoPadding", "PKCS5Padding" };

	private String algorithm;

	protected int state = 0;
	protected int mode;
	protected int padding;
	protected int blockSize;

	/**
	 * Holds unprocessed bytes.
	 */
	protected byte[] buffer = null;

	/**
	 * Holds the number of unprocessed bytes in buffer.
	 */
	protected int bufferOffset = 0;

	/**
	 * Holds the cipher's IV.
	 */
	protected byte[] iv = null;

	/**
	 * Constructs a blockcipher; to be called by subclasses.
	 * @param algorithm the cipher algorithm
	 * @param blockSize the block size in bytes of the blockcipher
	 */
	protected BlockCipher(String algorithm, int blockSize)
	{
		super();
		this.algorithm = algorithm;
		this.blockSize = blockSize;
		/* buffer one block */
		this.buffer = new byte[blockSize];
	}

	/**
	 * For security reasons, we cannot allow the cloning of block ciphers.
	 * @return never returns, always throws exception
	 * @throws CloneNotSupportedException ensures that BlockCipher objects are not cloned
	 */
	public final Object clone() throws CloneNotSupportedException
	{
	    throw new CloneNotSupportedException("BlockCipher cloning not allowed");
	}

	/**
	 * Setup method to be implemented by subclasses.
	 * @param opmode the operation mode of this cipher (either ENCRYPT_MODE or DECRYPT_MODE)
	 * @param key the encryption/decryption key
	 * @throws InvalidKeyException
	 */
	protected abstract void setup(int opmode, Key key) throws InvalidKeyException;

	protected abstract void setIV(byte[] iv);

	protected abstract void encryptECB(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks);

	protected abstract void decryptECB(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks);

	protected abstract void encryptCBC(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks);

	protected abstract void decryptCBC(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks);

	/**
	 * Encrypts or decrypts data in a single-part operation, or finishes a multiple-part operation.
	 * The data is encrypted or decrypted, depending on how this cipher was initialized. 
	 * @param input the input buffer
	 * @param inputOffset the offset in input where the input starts
	 * @param inputLen the input length
	 * @return the new buffer with the result
	 * @throws IllegalBlockSizeException if this cipher is a block cipher, no padding has been requested (only in encryption mode), and the total input length of the data processed by this cipher is not a multiple of block size
	 * @throws BadPaddingException if this cipher is in decryption mode, and (un)padding has been requested, but the decrypted data is not bounded by the appropriate padding bytes
	 * @see javax.crypto.CipherSpi#engineDoFinal(byte[], int, int)
	 */
	protected final byte[] engineDoFinal(byte[] input, int inputOffset, int inputLen) throws IllegalBlockSizeException, BadPaddingException
	{
		int outputSize = engineGetOutputSize(inputLen);

		if (outputSize == 0)
			return null;

		try
		{
			byte[] tmp = new byte[outputSize];
			int length = engineDoFinal(input, inputOffset, inputLen, tmp, 0);
			/* check for unpadding */
			if (length < tmp.length)
			{
				/* make a new array for the unpadded data */
				byte[] unpad = new byte[length];
				System.arraycopy(tmp, 0, unpad, 0, length);
				return unpad;
			}
			return tmp;
		}
		catch (ShortBufferException e)
		{
			throw new ProviderException(e.getMessage());
		}
	}

	/**
	 * Encrypts or decrypts data in a single-part operation, or finishes a multiple-part operation.
	 * The data is encrypted or decrypted, depending on how this cipher was initialized. 
	 * @param input the input buffer
	 * @param inputOffset the offset in input where the input starts
	 * @param inputLen the input length
	 * @param output the buffer for the result
	 * @param outputOffset the offset in output where the result is stored
	 * @return the number of bytes stored in output
	 * @throws ShortBufferException if the given output buffer is too small to hold the result
	 * @throws IllegalBlockSizeException if this cipher is a block cipher, no padding has been requested (only in encryption mode), and the total input length of the data processed by this cipher is not a multiple of block size
	 * @throws BadPaddingException if this cipher is in decryption mode, and (un)padding has been requested, but the decrypted data is not bounded by the appropriate padding bytes
	 * @see javax.crypto.CipherSpi#engineDoFinal(byte[], int, int, byte[], int)
	 */
	protected final int engineDoFinal(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset) throws ShortBufferException, IllegalBlockSizeException, BadPaddingException
	{
		switch (padding)
		{
		case PADDING_NOPADDING:
			return paddingNone(input, inputOffset, inputLen, output, outputOffset);
		case PADDING_PKCS5:
			return paddingPKCS5(input, inputOffset, inputLen, output, outputOffset);
		}
		return 0;
	}

	protected final int engineGetBlockSize()
	{
		return blockSize;
	}

	protected final byte[] engineGetIV()
	{
		return iv;
	}

	protected final int engineGetOutputSize(int inputLen)
	{
		switch (padding)
		{
		case PADDING_NOPADDING:
			return bufferOffset + inputLen;
		case PADDING_PKCS5:
			switch (state)
			{
			case Cipher.ENCRYPT_MODE:
				int padvalue = blockSize - ((bufferOffset + inputLen) % blockSize);
				return bufferOffset + inputLen + padvalue;
			case Cipher.DECRYPT_MODE:
				return bufferOffset + inputLen;
			}
		}
		return 0;
	}

	protected AlgorithmParameters engineGetParameters()
	{
		AlgorithmParameters params = null;
		try
		{
			params = AlgorithmParameters.getInstance(algorithm);
			params.init(new BlockCipherParameterSpec(algorithm + "/" + MODES[mode] + "/" + PADDINGS[padding], iv));
		}
		catch (NoSuchAlgorithmException nsae)
		{
		}
		catch (InvalidParameterSpecException ipe)
		{
		}
		return params;
	}

	protected final void engineInit(int opmode, Key key, SecureRandom random) throws InvalidKeyException
	{
		state = 0;

		if (key == null)
			throw new InvalidKeyException("key is null");

		if (!key.getFormat().equalsIgnoreCase("RAW"))
			throw new InvalidKeyException("key has wrong format");

		switch (opmode)
		{
		case Cipher.ENCRYPT_MODE:
		case Cipher.DECRYPT_MODE:
			state = opmode;
			break;
		default:
			throw new IllegalArgumentException("opmode " + opmode + " not supported");
		}

		try
		{	
			setup(opmode, key);
		}
		catch (InvalidKeyException e)
		{
			state = 0;
			throw e;
		}
	}

	protected final void engineInit(int opmode, Key key, AlgorithmParameterSpec params, SecureRandom random) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		state = 0;

		if (this instanceof PBECipher)
		{
			if (params instanceof PBEParameterSpec)
			{
				if (key instanceof PBEKey)
				{
					PBECipher cipher = (PBECipher) this;
					
					/* do the transformation of the PBEKey and PBEParameterSpec into IV and SecretKey */
					key = keyTransform(cipher.getDigest(), (PBEKey) key, (PBEParameterSpec) params);
				}
				else
					throw new InvalidKeyException("PBEKey expected here");
			}
			else
				throw new InvalidAlgorithmParameterException(params.getClass().getName() + " not supported here");
		}
		else
		{
			if (params instanceof IvParameterSpec)
			{
				byte[] tmp = ((IvParameterSpec) params).getIV();
				if (tmp.length == blockSize)
				{
					setIV(tmp);
				}
				else
					throw new InvalidAlgorithmParameterException("IV length must be " + blockSize);
			}
			else
				throw new InvalidAlgorithmParameterException(params.getClass().getName() + " not supported here");
		}

		switch (opmode)
		{
		case Cipher.ENCRYPT_MODE:
		case Cipher.DECRYPT_MODE:
			state = opmode;
			break;
		default:
			throw new IllegalArgumentException("opmode " + opmode + " not supported");
		}

		try
		{	
			setup(opmode, key);
		}
		catch (InvalidKeyException e)
		{
			state = 0;
			throw e;
		}
	}

	protected final void engineInit(int opmode, Key key, AlgorithmParameters params, SecureRandom random) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		state = 0;
		try
		{
			AlgorithmParameterSpec paramSpec = params.getParameterSpec(IvParameterSpec.class);
			this.engineInit(opmode, key, paramSpec, random);
		}
		catch (InvalidParameterSpecException e)
		{
			/* just copy the message here */
			throw new InvalidAlgorithmParameterException(e.getMessage());
		}
	}

	/**
	 * This method sets the feedback mode for the blockcipher
	 * <p>
	 * The supported feedback modes are:
	 * <ul>
	 * <li>ECB
	 * <li>CBC
	 * </ul>
	 * <p>
	 */
	protected final void engineSetMode(String mode) throws NoSuchAlgorithmException
	{
		if (mode.length() == 0)
		{
			this.mode = MODE_ECB;
		}
		else if (mode.equalsIgnoreCase("ECB"))
		{
			this.mode = MODE_ECB;
		}
		else if (mode.equalsIgnoreCase("CBC"))
		{
			this.mode = MODE_CBC;
		}
		else
			throw new NoSuchAlgorithmException("Feedback mode " + mode + " not supported");
	}

	/**
	 * This method sets the feedback mode for the blockcipher
	 * <p>
	 * The supported padding algorithms are:
	 * <ul>
	 * <li>NoPadding
	 * <li>PKCS5Padding
	 * </ul>
	 * <p>
	 */
    protected final void engineSetPadding(String padding) throws NoSuchPaddingException
	{
		if (padding.length() == 0)
			this.padding = PADDING_NOPADDING;
		else if (padding.equalsIgnoreCase("NoPadding"))
			this.padding = PADDING_NOPADDING;
		else if (padding.equalsIgnoreCase("PKCS5Padding"))
			this.padding = PADDING_PKCS5;
		else
			throw new NoSuchPaddingException("Padding " + padding + " not supported");
	}

	protected final byte[] engineUpdate(byte[] input, int inputOffset, int inputLen)
	{
		/* if we can, just buffer it */
		if (bufferOffset + inputLen < buffer.length)
		{
			System.arraycopy(input, inputOffset, buffer, bufferOffset, inputLen);
			bufferOffset += inputLen;
			return null;
		}

		int outputSize = engineGetOutputSize(inputLen);

		try
		{
			byte[] tmp = new byte[outputSize];
			int length = engineUpdate(input, inputOffset, inputLen, tmp, 0);
			/* check for unpadding of the data */
			if (length < tmp.length)
			{
				/* make a new array for the unpadded data */
				byte[] unpad = new byte[length];
				System.arraycopy(tmp, 0, unpad, 0, length);
				return unpad;
			}
			return tmp;
		}
		catch (ShortBufferException e)
		{
			throw new ProviderException(e.getMessage());
		}
	}

	protected final int engineUpdate(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset) throws ShortBufferException
	{
		/* if we can, just buffer it */
		if (bufferOffset + inputLen < buffer.length)
		{
			System.arraycopy(input, inputOffset, buffer, bufferOffset, inputLen);
			bufferOffset += inputLen;
			return 0;
		}

		int outputSize = bufferOffset + inputLen;

		if (outputSize > output.length - outputOffset)
			throw new ShortBufferException("output too short");

		return update(input, inputOffset, inputLen, output, outputOffset);
	}

	private Key keyTransform(MessageDigest md, PBEKey key, PBEParameterSpec paramSpec)
	{
		byte[] salt = paramSpec.getSalt();
		byte[] password = key.getEncoded();
		byte[] digest;
		int iterationCount = paramSpec.getIterationCount();
		int mod;

		md.reset();
		/* the initial step: need to update salt and password up to a multiple of 64 bytes */
		md.update(salt);
		mod = salt.length & 0x3f;
		while (mod != 0)
		{
			if (mod + salt.length < 64)
			{
				md.update(salt);
				mod += salt.length;
			}
			else
			{
				md.update(salt, 0, 64 - mod);
				break;
			}
		}
		md.update(password);
		mod = password.length & 0x3f;
		while (mod != 0)
		{
			if (mod + password.length < 64)
			{
				md.update(password);
				mod += password.length;
			}
			else
			{
				md.update(password, 0, 64 - mod);
				break;
			}
		}
		digest = md.digest();

		/* now feed hash itself up to iterationCount */
		while (--iterationCount > 0)
			digest = md.digest(digest);

		/* set iv */
		iv = new byte[blockSize];
		System.arraycopy(digest, 0, iv, 0, iv.length);

		byte[] secret = new byte[md.getDigestLength() - blockSize];
		System.arraycopy(digest, blockSize, secret, 0, secret.length);

		/* return the appropriate SecretKey */
		return new SecretKeySpec(secret, algorithm);
	}

	/**
	 * This method is called in case no padding was selected.
	 * <p>
	 * @throws IllegalBlockSizeException
	 *		in case no padding is specified, and the input is not a whole number of blocks.
	 */
	private final int paddingNone(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset) throws ShortBufferException, IllegalBlockSizeException
	{
		if (((bufferOffset + inputLen) % blockSize) != 0)
			throw new IllegalBlockSizeException("not a multiple of " + blockSize + " bytes");

		return update(input, inputOffset, inputLen, output, outputOffset);
	}

	/**
	 * This method performs (extended) PKCS#5 padding; it will also work for block ciphers with
	 * a block size of more than 8 bytes, up to 255 bytes (for example the AES candidates).
	 */
	private final int paddingPKCS5(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset) throws ShortBufferException, IllegalBlockSizeException, BadPaddingException
	{
		int padvalue, updated;

		switch (state)
		{
		case Cipher.ENCRYPT_MODE:
			padvalue = blockSize - ((bufferOffset + inputLen) % blockSize);

			byte[] padding = new byte[padvalue];
			for (byte b = 0; b < padvalue; padding[b++] = (byte) padvalue);

			updated = update(input, inputOffset, inputLen, output, outputOffset);

			return updated + update(padding, 0, padvalue, output, outputOffset + updated);

		case Cipher.DECRYPT_MODE:
			if (((bufferOffset + inputLen) % blockSize) == 0)
			{
				updated = update(input, inputOffset, inputLen, output, outputOffset);
				padvalue = output[outputOffset+updated-1];

				return updated - padvalue;
			}
			throw new IllegalBlockSizeException("not a multiple of " + blockSize + " bytes");
		}
		return 0;
	}

	/**
	 * Consume whole buffers.
	 */
	private final int update(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset)
	{
		int updated, total = 0;

		/* instead of copying everything to the buffer, we only do partial copying, to get a whole number of blocks */
		/* then we count the number of blocks we can do directly from input, and do those */
		/* finally, we copy the remaining bytes, if any, to the buffer */

		if (bufferOffset != 0)
		{
			int bufferFill = blockSize - bufferOffset;

			System.arraycopy(input, inputOffset, buffer, bufferOffset, bufferFill);

			switch (state)
			{
			case Cipher.ENCRYPT_MODE:
				switch (mode)
				{
				case MODE_ECB:
					encryptECB(buffer, 0, output, outputOffset, 1);
					break;
				case MODE_CBC:
					encryptCBC(buffer, 0, output, outputOffset, 1);
					break;
				default:
					throw new ProviderException("invalid mode");
				}
				updated = blockSize;
				break;
			case Cipher.DECRYPT_MODE:
				switch (mode)
				{
				case MODE_ECB:
					decryptECB(buffer, 0, output, outputOffset, 1);
					break;
				case MODE_CBC:
					decryptCBC(buffer, 0, output, outputOffset, 1);
					break;
				default:
					throw new ProviderException("invalid mode");
				}

				updated = blockSize;
				break;
			default:
				throw new ProviderException("invalid state");
			}

			inputOffset += bufferFill;
			inputLen -= bufferFill;

			total += updated;
		}

		int blocks = inputLen / blockSize;

		if (blocks > 0)
		{
			switch (state)
			{
			case Cipher.ENCRYPT_MODE:
				switch (mode)
				{
				case MODE_ECB:
					encryptECB(input, inputOffset, output, outputOffset, blocks);
					break;
				case MODE_CBC:
					encryptCBC(input, inputOffset, output, outputOffset, blocks);
					break;
				default:
					throw new ProviderException("invalid mode");
				}
				updated = blockSize * blocks;
				break;
			case Cipher.DECRYPT_MODE:
				switch (mode)
				{
				case MODE_ECB:
					decryptECB(input, inputOffset, output, outputOffset, blocks);
					break;
				case MODE_CBC:
					decryptCBC(input, inputOffset, output, outputOffset, blocks);
					break;
				default:
					throw new ProviderException("invalid mode");
				}
				updated = blockSize * blocks;
				break;
			default:
				throw new ProviderException("invalid state");
			}

			inputLen -= updated;
			inputOffset += updated;
			
			total += updated;
		}

		/* cache the rest */
		if (inputLen > 0)
		{
			System.arraycopy(input, inputOffset, buffer, 0, bufferOffset = inputLen);
		}

		return total;
	}
}
