/*
 * BlockCipherKeyGenerator.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto;

import javax.crypto.*;
import javax.crypto.spec.*;
import java.security.*;
import java.security.spec.*;

/**
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
abstract class BlockCipherKeyGenerator extends KeyGeneratorSpi
{
	private String algorithm;
	private int defaultKeysize;
	private int keysize = 0;

	private SecureRandom random = null;

	protected BlockCipherKeyGenerator(String algorithm, int defaultKeysize)
	{
		this.algorithm = algorithm;
		this.defaultKeysize = defaultKeysize;
	}

	abstract boolean isValidKeysize(int keysize);

	abstract boolean isWeak(byte[] keyBytes);

	protected byte[] adjust(byte[] keyBytes)
	{
		return keyBytes;
	}

	protected int keysizeToBytes(int keysize)
	{
		return (keysize + 7) >> 3;
	}

	protected void engineInit(SecureRandom random)
	{
		this.keysize = defaultKeysize;
		this.random = random;
	}

	protected void engineInit(AlgorithmParameterSpec params, SecureRandom random) throws InvalidAlgorithmParameterException
	{
		throw new InvalidAlgorithmParameterException("not supported");
	}

	protected void engineInit(int keysize, SecureRandom random)
	{
		if (!isValidKeysize(keysize))
			throw new IllegalArgumentException("keysize " + keysize + " not supported");

		this.keysize = keysize;
		this.random = random;
	}

	protected SecretKey engineGenerateKey()
	{
		if (random == null)
			random = new SecureRandom();

		if (keysize == 0)
			keysize = defaultKeysize;

		byte[] keyBytes = new byte[keysizeToBytes(keysize)];

		do
		{
			random.nextBytes(keyBytes);
		} while (isWeak(keyBytes));

		return new SecretKeySpec(adjust(keyBytes), algorithm);
	}
}
