/*
 * BlockCipherParameters.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto;

import java.io.IOException;
import java.security.*;
import java.security.spec.*;
import javax.crypto.spec.IvParameterSpec;

import com.beeyond.asn1.*;

/**
 * This class implements BlockCipher parameters.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public abstract class BlockCipherParameters extends AlgorithmParametersSpi
{
	private int blockSize;
	private IvParameterSpec paramSpec;

	protected BlockCipherParameters(int blockSize)
	{
		this.blockSize = blockSize;
		this.paramSpec = null;
	}

	protected byte[] engineGetEncoded() throws IOException
	{
		if (paramSpec == null)
			return null;

		return DER.encode(new ASNOctetString(paramSpec.getIV()));
	}

	protected byte[] engineGetEncoded(String format) throws IOException
	{
		if (format.equalsIgnoreCase(DER.ASNFORMAT))
			return engineGetEncoded();
		else
			throw new IllegalArgumentException("format " + format + " not supported");
	}

	protected AlgorithmParameterSpec engineGetParameterSpec(Class paramSpec) throws InvalidParameterSpecException
	{
		if (paramSpec.isAssignableFrom(IvParameterSpec.class))
			return this.paramSpec;
		else
			throw new InvalidParameterSpecException("not assignable from IvParameterSpec");
	}

	protected void engineInit(AlgorithmParameterSpec paramSpec) throws InvalidParameterSpecException
	{
		if (paramSpec instanceof IvParameterSpec)
			this.paramSpec = (IvParameterSpec) paramSpec;
		else
			new InvalidParameterSpecException("init with " + paramSpec.getClass().getName() + " not supported");
	}

	protected void engineInit(byte[] params) throws IOException
	{
		ASNObject tmp = DER.decode(params);

		if (!(tmp instanceof ASNOctetString))
			throw new ASNCodingException("not an octet string");

		byte[] iv = tmp.getContent();

		if (iv.length < blockSize)
			paramSpec = new IvParameterSpec(iv);
		else
			throw new IllegalArgumentException("iv length must be at least " + blockSize + " bytes");
	}

	protected void engineInit(byte[] params, String format) throws IOException
	{
		if (format.equalsIgnoreCase(DER.ASNFORMAT))
			engineInit(params);
		else
			throw new IllegalArgumentException("format " + format + " not supported");
	}

	protected String engineToString()
	{
		return null;
	}
}
