/*
 * DHKeyAgreement.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto;

import java.math.BigInteger;
import java.security.*;
import java.security.spec.*;
import javax.crypto.*;
import javax.crypto.interfaces.*;
import javax.crypto.spec.*;

/**
 * This class implements Diffie-Hellman key agreement as a JCE1.2 Service Provider Interface
 * <p>
 * <b>References:</b><br>
 * <ul>
 * <li>Alfred J. Menezes,
 *     "Section 12.6.1: Diffie-Hellman and related key agreement protocols",
 *     Handbook of Applied Cryptography,
 *     CRC Press, 1997.
 * </ul>
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class DHKeyAgreement extends KeyAgreementSpi
{
	protected final static InvalidKeyException
		NOTDHPUBLICKEY = new InvalidKeyException("Not a DHPublicKey"),
		NOTDHPRIVATEKEY = new InvalidKeyException("Not a DHPrivateKey");

	protected final static IllegalStateException
		ILLEGALSTATE = new IllegalStateException();

	protected final static int
		STATE_UNINITIALIZED = 0,
		STATE_INITIALIZED = 1,
		STATE_SHARED = 2;

	protected int state = STATE_UNINITIALIZED;

	protected SecureRandom random = null;

	protected BigInteger p;
	protected BigInteger g;
	protected BigInteger x;
	protected BigInteger y;
	protected BigInteger s;

	protected Key engineDoPhase(Key key, boolean lastPhase) throws InvalidKeyException, IllegalStateException
	{
		if (lastPhase)
		{
			if (key instanceof DHPublicKey)
			{
				state = STATE_SHARED;
				y = ((DHPublicKey)key).getY();
				s = y.modPow(x, p);
				return null;
			}
			else
				throw NOTDHPUBLICKEY;
		}
		else
			throw ILLEGALSTATE;
	}

	protected byte[] engineGenerateSecret() throws IllegalStateException
	{
		if (state == STATE_SHARED)
		{
			state = STATE_INITIALIZED;
			return s.toByteArray();
		}
		else
			throw ILLEGALSTATE;
	}

	protected int engineGenerateSecret(byte[] sharedSecret, int offset) throws IllegalStateException, ShortBufferException
	{
		if (state == STATE_SHARED)
		{
			byte[] tmp = s.toByteArray();

			if (tmp.length > sharedSecret.length + offset)
				throw new ShortBufferException();

			System.arraycopy(tmp, 0, sharedSecret, offset, tmp.length);
			state = STATE_INITIALIZED;
			return tmp.length;
		}
		else
			throw ILLEGALSTATE;
	}

	protected SecretKey engineGenerateSecret(String algorithm) throws IllegalStateException, NoSuchAlgorithmException, InvalidKeyException
	{
		if (state == STATE_SHARED)
		{
			try
			{
				state = STATE_INITIALIZED;
				SecretKeyFactory skf = SecretKeyFactory.getInstance(algorithm);
				SecretKeySpec sks = new SecretKeySpec(s.toByteArray(), algorithm);
				return skf.generateSecret(sks);
			}
			catch (InvalidKeySpecException e)
			{
				throw new InvalidKeyException(e.getMessage());
			}
		}
		else
			throw ILLEGALSTATE;
	}

	protected void engineInit(Key key, SecureRandom random) throws InvalidKeyException
	{
		if (key instanceof DHPrivateKey)
		{
			state = STATE_INITIALIZED;
			DHPrivateKey dhk = (DHPrivateKey) key;
			p = dhk.getParams().getP();
			g = dhk.getParams().getG();
			x = dhk.getX();
			this.random = random;
		}
		else
			throw NOTDHPRIVATEKEY;
	}

	protected void engineInit(Key key, AlgorithmParameterSpec params, SecureRandom random) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		if (key instanceof DHPrivateKey)
		{
			state = STATE_INITIALIZED;
			DHPrivateKey dhk = (DHPrivateKey) key;
			p = dhk.getParams().getP();
			g = dhk.getParams().getG();
			x = dhk.getX();
			this.random = random;
		}
		else
			throw NOTDHPRIVATEKEY;
	}
}
