/*
 * DHKeyPairGenerator.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package com.beeyond.crypto;

import java.math.BigInteger;
import java.security.*;
import java.security.spec.*;
import javax.crypto.spec.*;

/**
 * This class implements Diffie-Hellman keypair generation as a JCE1.2 Service Provider Interface
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class DHKeyPairGenerator extends KeyPairGeneratorSpi
{
	private static final String
		ALGORITHM = "DH",
		PROVIDER = "BeeCrypt";

	private static final InvalidAlgorithmParameterException NOELG = new InvalidAlgorithmParameterException("Not a DHParameterSpec");

	protected DHParameterSpec params = null;
	protected SecureRandom random = null;

	protected BigInteger x, y;

	protected void generateXY()
	{
		BigInteger p, n, g;
		int l;

		// get a secret key value x with 1 < x < p-1
		p = params.getP();
		g = params.getG();
		n = p.subtract(BigInteger.ONE);
		l = params.getL();
		if (l == 0)
			l = n.bitLength();

		do
		{
			x = new BigInteger(l, random);
			while (x.compareTo(n) >= 0)
				x = x.subtract(n);
		} while (BigInteger.ONE.compareTo(x) >= 0);
		y = g.modPow(x, p);
	}

	public KeyPair generateKeyPair()
	{
		if (params == null)
			return null;

		generateXY();
		return new KeyPair(new DHPublicKey(params, y),
			new DHPrivateKey(params, x));
	}

	public void initialize(AlgorithmParameterSpec params, SecureRandom random) throws InvalidAlgorithmParameterException
	{
		if (params instanceof DHParameterSpec)
		{
			this.params = (DHParameterSpec) params;
			this.random = random;
		}
		else
			throw NOELG;
	}

	public void initialize(int keysize, SecureRandom random)
	{
		try
		{
			AlgorithmParameterGenerator apg = AlgorithmParameterGenerator.getInstance(ALGORITHM, PROVIDER);
			apg.init(keysize, random);
			this.params = (DHParameterSpec) apg.generateParameters().getParameterSpec(DHParameterSpec.class);
			this.random = random;
		}
		catch (Exception e)
		{
		}
	}
}
