/*
 * DHParameterGenerator.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package com.beeyond.crypto;

import java.math.BigInteger;
import java.security.*;
import java.security.spec.AlgorithmParameterSpec;
import javax.crypto.spec.*;
import com.beeyond.security.spec.*;

/**
 * This class implements the generation of Diffie-Hellman parameters.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class DHParameterGenerator extends java.security.AlgorithmParameterGeneratorSpi
{
	protected static final String
		PROVIDER = "BeeCrypt";
		
	private static final String
		ALGORITHM = "DH";

	private DHGenParameterSpec genParamSpec = null;

	/***/
	protected SecureRandom random = null;

	/**
	 * Inter-method storage of the prime p
	 */
	protected BigInteger p;
	/**
	 * Inter-method storage of the prime factor q of (p-1)
	 */
	protected BigInteger q;
	/**
	 * Inter-method storage of the cofactor r of (p-1)
	 */
	protected BigInteger r;
	/**
	 * Inter-method storage of the base generator g
	 */
	protected BigInteger g;

	/**
	 * This method generates a prime modulus p and either a base generator g of order (p-1) or (q) for group <bold>Z</bold><sub>p</sub><sup>*</sup>.
	 * @see com.beeyond.security.PrimeParameterGenerator
	 */
	protected void generatePQG(int primeSize, int factorSize, boolean generateSubgroup)
	{
		/*
		 * Generate a prime with the provider in com.beeyond.security
		 */
		AlgorithmParameterGenerator apg;
		AlgorithmParameters ap;
		PrimeParameterSpec primeSpec;

		try
		{
			apg = AlgorithmParameterGenerator.getInstance("Prime", PROVIDER);
			apg.init(new PrimeGenParameterSpec(primeSize, factorSize, generateSubgroup ? 1 : 2), random);
			ap = apg.generateParameters();
			primeSpec = (PrimeParameterSpec) ap.getParameterSpec(PrimeParameterSpec.class);
		}
		catch (Exception e)
		{
			throw new ProviderException(e.getMessage());
		}

		p = primeSpec.getP();
		q = primeSpec.getQ();
		r = primeSpec.getR();

		if (generateSubgroup)
		{
			/**
			 * Pick a random g with 1 < g < p-1
			 */
			do
			{
				g = new BigInteger(primeSize, random).modPow(r, p);
			} while (BigInteger.ONE.compareTo(g) >= 0);
		}
		else
		{
			/*
			 * Now that the prime decomposition of the order of the group is known
			 * (p-1=qr, r=2s, hence p-1=2qs with q and s prime), we have
			 * three prime factors which are needed for algorithm 4.40
			 */
			BigInteger n = p.subtract(BigInteger.ONE);
		//	BigInteger ndivq = r;
			BigInteger ndivr = q.shiftLeft(1);
			BigInteger ndivtwo = n.shiftRight(1);
	
			for (;;)
			{
				/**
				 * Pick a random g with 1 < g < p-1
				 */
				do
				{
					g = new BigInteger(primeSize, random);
					while (g.compareTo(n) >= 0)
						g = g.subtract(n);
				} while (BigInteger.ONE.compareTo(g) >= 0);
	
				/*
				 * Apply the tests of algorithm 4.40 of Handbook of Applied Cryptography
				 */
				if (BigInteger.ONE.equals(g.modPow(r, p))) // ndivq
					continue;
				if (BigInteger.ONE.equals(g.modPow(ndivr, p)))
					continue;
				if (BigInteger.ONE.equals(g.modPow(ndivtwo, p)))
					continue;

				return;
			}
		}
	}

	protected AlgorithmParameters engineGenerateParameters()
	{
		int primeSize = genParamSpec.getPrimeSize();

		generatePQG(primeSize, primeSize >> 1, true);

		AlgorithmParameters ap = null;

		try
		{
			ap = AlgorithmParameters.getInstance(ALGORITHM, PROVIDER);
			ap.init(new DHParameterSpec(p, g, genParamSpec.getExponentSize()));
		}
		catch (Exception e)
		{
			throw new ProviderException();
		}

		return ap;
	}

	protected void engineInit(AlgorithmParameterSpec genParamSpec, SecureRandom random) throws InvalidAlgorithmParameterException
	{
		if (genParamSpec instanceof DHGenParameterSpec)
		{
			this.genParamSpec = (DHGenParameterSpec) genParamSpec;
			this.random = random;
		}
		else
			throw new InvalidAlgorithmParameterException("not a DHGenParameterSpec");
	}

	protected void engineInit(int size, SecureRandom random)
	{
		this.genParamSpec = new DHGenParameterSpec(size, size);
		this.random = random;
	}
}
