/*
 * DHParameters.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto;

import java.io.IOException;
import java.math.BigInteger;
import java.security.*;
import java.security.spec.*;
import javax.crypto.spec.DHParameterSpec;
import com.beeyond.asn1.*;

/**
 * This class implements Diffie-Hellman parameters.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class DHParameters extends AlgorithmParametersSpi
{
	protected DHParameterSpec paramSpec = null;

	protected byte[] engineGetEncoded() throws IOException
	{
		if (paramSpec == null)
			return null;

		int l = paramSpec.getL();

		ASNObject[] objs;

		if (l == 0)
			objs = new ASNObject[] { new ASNInteger(paramSpec.getP()), new ASNInteger(paramSpec.getG()) };
		else
			objs = new ASNObject[] { new ASNInteger(paramSpec.getP()), new ASNInteger(paramSpec.getG()), new ASNInteger(l) };

		return DER.encode(new ASNSequence(objs));
	}

	protected byte[] engineGetEncoded(String format) throws IOException
	{
		if (format.equalsIgnoreCase(DER.ASNFORMAT))
			return engineGetEncoded();
		else
			throw new IllegalArgumentException("format " + format + " not supported");
	}

	protected AlgorithmParameterSpec engineGetParameterSpec(Class paramSpec) throws InvalidParameterSpecException
	{
		if (paramSpec.isAssignableFrom(DHParameterSpec.class))
			return this.paramSpec;
		else
			throw new InvalidParameterSpecException("not assignable from DHParameterSpec");
	}

	protected void engineInit(AlgorithmParameterSpec paramSpec) throws InvalidParameterSpecException
	{
		if (paramSpec instanceof DHParameterSpec)
			this.paramSpec = (DHParameterSpec) paramSpec;
		else
			new InvalidParameterSpecException("init with " + paramSpec.getClass().getName() + " not supported");
	}

	protected void engineInit(byte[] params) throws IOException
	{
		ASNObject tmp = DER.decode(params);

		if (!(tmp instanceof ASNSequence))
			throw new ASNCodingException("not a sequence");

		ASNSequence seq = (ASNSequence) tmp;
		
		int seqSize = seq.size();

		if (seqSize < 2 || seqSize > 3)
			throw new ASNCodingException("invalid number of sequence fields");

		tmp = seq.getObject(0);
		if (!(tmp instanceof ASNInteger))
			throw new ASNCodingException("first sequence field is not an integer");

		ASNInteger p = (ASNInteger) tmp;

		tmp = seq.getObject(1);
		if (!(tmp instanceof ASNInteger))
			throw new ASNCodingException("second sequence field is not an integer");

		ASNInteger g = (ASNInteger) tmp;

		if (seqSize == 2)
		{
			paramSpec = new DHParameterSpec(p.getValue(), g.getValue());
		}
		else
		{
			tmp = seq.getObject(2);

			if (!(tmp instanceof ASNInteger))
				throw new ASNCodingException("third sequence field is not an integer");

			ASNInteger l = (ASNInteger) tmp;

			paramSpec = new DHParameterSpec(p.getValue(), g.getValue(), l.intValue());
		}
	}

	protected void engineInit(byte[] params, String format) throws IOException
	{
		if (format.equalsIgnoreCase(DER.ASNFORMAT))
			engineInit(params);
		else
			throw new IllegalArgumentException("format " + format + " not supported");
	}

	protected String engineToString()
	{
		return (paramSpec == null) ? null : "P: " + paramSpec.getP().toString(16) + "\nG: " + paramSpec.getG().toString(16);
	}
}
