/*
 * ElGamalKeyFactory.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto;

import com.beeyond.crypto.spec.*;
import com.beeyond.security.*;

import java.io.*;
import java.math.BigInteger;
import java.security.*;
import java.security.interfaces.*;
import java.security.spec.*;
import javax.crypto.spec.*;

public class ElGamalKeyFactory extends KeyFactorySpi
{
	protected PublicKey engineGeneratePublic(KeySpec keySpec) throws InvalidKeySpecException
	{
		if (!(keySpec instanceof EncodedKeySpec))
			throw new InvalidKeySpecException("not an EncodedKeySpec");

		byte[] encoded = ((EncodedKeySpec) keySpec).getEncoded();

		if (keySpec instanceof BeeEncodedKeySpec)
		{
			ByteArrayInputStream ba = new ByteArrayInputStream(encoded);
			BeeInputStream b = new BeeInputStream(ba);

			try
			{
				BigInteger p = b.readBigInteger();
				BigInteger q = b.readBigInteger();
				BigInteger g = b.readBigInteger();

				DLParameterSpec spec = new DLParameterSpec(p, q, g);

				BigInteger y = b.readBigInteger();
				
				b.close();
				ba.close();
				
				return new ElGamalPublicKey(spec, y);
			}
			catch (IOException e)
			{
				throw new InvalidKeySpecException("couldn't decode public key");
			}
		}
		else
			throw new InvalidKeySpecException("format " + ((EncodedKeySpec) keySpec).getFormat() + " not supported");
	}

	protected PrivateKey engineGeneratePrivate(KeySpec keySpec) throws InvalidKeySpecException
	{
		if (!(keySpec instanceof EncodedKeySpec))
			throw new InvalidKeySpecException("not an EncodedKeySpec");

		byte[] encoded = ((EncodedKeySpec) keySpec).getEncoded();

		if (keySpec instanceof BeeEncodedKeySpec)
		{
			ByteArrayInputStream ba = new ByteArrayInputStream(encoded);
			BeeInputStream b = new BeeInputStream(ba);

			try
			{
				BigInteger p = b.readBigInteger();
				BigInteger q = b.readBigInteger();
				BigInteger g = b.readBigInteger();

				DLParameterSpec spec = new DLParameterSpec(p, q, g);

				BigInteger x = b.readBigInteger();
				
				b.close();
				ba.close();
				
				return new ElGamalPrivateKey(spec, x);
			}
			catch (IOException e)
			{
				throw new InvalidKeySpecException("couldn't decode private key");
			}
		}
		else
			throw new InvalidKeySpecException("format " + ((EncodedKeySpec) keySpec).getFormat() + " not supported");
	}

	protected KeySpec engineGetKeySpec(Key key, Class keySpec) throws InvalidKeySpecException
	{
		if (!(key instanceof com.beeyond.crypto.interfaces.ElGamalKey))
			throw new IllegalArgumentException("not an ElGamalKey");

		if (BeeEncodedKeySpec.class.isAssignableFrom(keySpec))
		{
			ByteArrayOutputStream ba = new ByteArrayOutputStream();
			BeeOutputStream b = new BeeOutputStream(ba);

			DLParameterSpec params = (DLParameterSpec) ((ElGamalKey)key).getParams();

			try
			{
				b.writeBigInteger(params.getP());
				b.writeBigInteger(params.getQ());
				b.writeBigInteger(params.getG());

				if (key instanceof com.beeyond.crypto.interfaces.ElGamalPublicKey)
					b.writeBigInteger(((ElGamalPublicKey)key).getY());
				else if (key instanceof com.beeyond.crypto.interfaces.ElGamalPrivateKey)
					b.writeBigInteger(((ElGamalPrivateKey)key).getX());
				else
					throw new IllegalArgumentException("key is neither ElGamalPublicKey nor ElGamalPrivateKey");

				b.close();
				ba.close();
			}
			catch (IOException e)
			{
				throw new ProviderException(e.getMessage());
			}
			
			return new BeeEncodedKeySpec(ba.toByteArray());
		}
		else
			throw new InvalidKeySpecException("KeySpec " + keySpec.getName() + " not implemented");
	}

	protected Key engineTranslateKey(Key key) throws InvalidKeyException
	{
		throw new ProviderException("key translation not implemented");
	}
}
