/*
 * HMac.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto;

import java.security.*;
import java.security.spec.*;
import javax.crypto.*;

/**
 * This abstract class implements hash-based MACs as a JCE1.2 SPI.
 * <p>
 * The hash function (MessageDigest) will be provided by the subclasses.
 * <p>
 * <b>References:</b><br>
 * <ul>
 * <li>RFC 2104 HMAC: Keyed-Hashing for Message Authentication,<br>
 *      H. Krawczyk, M. Bellare, R. Canetti,<br>
 *      February 1997.
 * <li>Handbook of Applied Cryptography,<br>
 *      Section 9.5.2: Constructing MACs from MDCs",<br>
 *      A. Menezes, P. van Oorschot, S. Vanstone,<br>
 *      CRC Press, 1997.
 * </ul>
 * <p>
 * @author: Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public abstract class HMac extends MacSpi
{
	private static final byte
		IPAD = (byte) 0x36,
		OPAD = (byte) 0x5C;

	private static final InvalidKeyException NOTSECRETKEY = new InvalidKeyException("Not a SecretKey");
	private static final InvalidKeyException KEYTOOLONG = new InvalidKeyException("Key is too long");

	private MessageDigest md = null;
	private byte[] kxi = new byte[64];
	private byte[] kxo = new byte[64];

	/**
	 * Creates an HMac object.
	 * @param algorithm the MessageDigest algorithm to be used
	 */
	protected HMac(String algorithm)
	{
		try
		{
			md = MessageDigest.getInstance(algorithm);
		}
		catch (NoSuchAlgorithmException nsae)
		{
			md = null;
			throw new ProviderException(nsae.getMessage());
		}
	}

	/**
	 * Returns the length (in bytes) of the message authentication code.
	 * @return the length (in bytes) of the message authentication code
	 */
	protected int engineGetMacLength()
	{
		return md.getDigestLength();
	}

	/**
	 * Initializes the HMac with the given key.
	 * @param key the HMac key
	 * @param params params are ignored
	 */
	protected void engineInit(Key key, AlgorithmParameterSpec params) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		if (key instanceof SecretKey)
		{
			byte[] k = key.getEncoded();

			if (k.length > 64)
				throw KEYTOOLONG;

			for (int i = 0; i < k.length; i++)
			{
				kxi[i] = (byte)(k[i] ^ IPAD);
				kxo[i] = (byte)(k[i] ^ OPAD);
			}
			for (int i = k.length; i < 64; i++)
			{
				kxi[i] = IPAD;
				kxo[i] = OPAD;
			}
			md.reset();
			md.update(kxi);
		}
		else
			throw NOTSECRETKEY;
	}

	/***/
	protected void engineUpdate(byte input)
	{
		md.update(input);
	}

	/***/
	protected void engineUpdate(byte[] input, int offset, int len)
	{
		md.update(input, offset, len);
	}

	/***/
	protected byte[] engineDoFinal()
	{
		byte[] tmp = md.digest();
		md.reset();
		md.update(kxo);
		md.update(tmp);
		return md.digest();
	}

	/***/
	protected void engineReset()
	{
		md.reset();
		md.update(kxi);
	}
}
