/*
 * NativeBlockCipher.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto;

import java.security.*;
import javax.crypto.*;

/**
 * This class implements the glue to native block ciphers in the
 * BeeCrypt Cryptography Library.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class NativeBlockCipher extends BlockCipher
{
	private long ciph;
	
	private long param;

	private long cache;

	private int cached;

	private static native long find(String algorithm);

	private static native long allocParam(long ciph);

	private static native void freeParam(long param);

	private static native int  getBlockSize(long ciph);
	
	private static native void setup(long ciph, long param, int opmode, byte[] key);

	private static native void setIV(long ciph, long param, byte[] iv);

	private static native void encryptECB(long ciph, long param, byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks);

	private static native void decryptECB(long ciph, long param, byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks);

	private static native void encryptCBC(long ciph, long param, byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks);

	private static native void decryptCBC(long ciph, long param, byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks);

	static
	{
		System.loadLibrary("beecrypt");
	}

	protected NativeBlockCipher(String algorithm, int blockSize)
	{
		super(algorithm, blockSize);

		ciph = find(algorithm);
		param = allocParam(ciph);
	}

	protected void setup(int opmode, Key key)
	{
		setup(ciph, param, opmode, key.getEncoded());
	}

	protected void setIV(byte[] iv)
	{
		setIV(ciph, param, iv);
	}

	public void finalize()
	{
		freeParam(param);
	}

	protected void encryptECB(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks)
	{
		encryptECB(ciph, param, input, inputOffset, output, outputOffset, blocks);
	}
	
	protected void decryptECB(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks)
	{
		decryptECB(ciph, param, input, inputOffset, output, outputOffset, blocks);
	}

	protected void encryptCBC(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks)
	{
		encryptCBC(ciph, param, input, inputOffset, output, outputOffset, blocks);
	}
	
	protected void decryptCBC(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks)
	{
		decryptCBC(ciph, param, input, inputOffset, output, outputOffset, blocks);
	}
}
