/*
 * PBEKey.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto;

import java.io.*;
import java.util.Arrays;
import java.security.ProviderException;
import javax.crypto.SecretKey;
import javax.crypto.spec.PBEKeySpec;

/**
 * This class wraps a PBEKeySpec char[] password into a SecretKey
 * as described in PKCS#12 appendix B.1.
 * <p>
 * <b>See also:</b>
 * <li><a href="http://www.rsalabs.com/pkcs/pkcs-12/index.html">PKCS#12 - Personal Information Exchange Syntax Standard</a>
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 * @see javax.crypto.spec.PBEKeySpec
 */
public class PBEKey implements SecretKey
{
	private byte[] key;

	/**
	 * Creates a PBEKey from a PBEKeySpec
	 * @param keySpec the keySpec to use for this key
	 */
	public PBEKey(PBEKeySpec keySpec)
	{
		if (keySpec == null)
			throw new IllegalArgumentException("keySpec is null");

		ByteArrayOutputStream bs = new ByteArrayOutputStream();

		try
		{
			DataOutputStream ds = new DataOutputStream(bs);
			/* first the chars */
			ds.writeChars(new String(keySpec.getPassword()));
			/* then two zero bytes */
			ds.writeShort(0);
			ds.close();
			bs.close();
		}
		catch (IOException e)
		{
			throw new ProviderException(e.getMessage());
		}

		key = bs.toByteArray();
	}

	/**
	 * Ensures that the key gets wiped from memory.
	 */	
	protected void finalize()
	{
		if (key != null)
			Arrays.fill(key, (byte) 0);
	}

	/***/
	public String getAlgorithm()
	{
		return "PBE";
	}

	/***/
	public byte[] getEncoded()
	{
		return (byte[]) key.clone();
	}

	/***/
	public String getFormat()
	{
		return "RAW";
	}

	/***/
	public int hashCode()
	{
	    int hash = 0;
	    for (int i = 1; i < key.length; i++)
            hash += key[i] * i;

        return hash ^= getAlgorithm().toLowerCase().hashCode();
    }
}
