/*
 * PureBlockCipher.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto;

import java.util.*;

/**
 * This abstract class defines a pure java BlockCipher.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public abstract class PureBlockCipher extends BlockCipher
{
	protected byte[] fdback = null;

	protected PureBlockCipher(String algorithm, int blockSize)
	{
		super(algorithm, blockSize);
		fdback = new byte[blockSize];
		Arrays.fill(fdback, (byte) 0);
	}

	protected abstract void encrypt(byte[] input, int inputOffset, byte[] output, int outputOffset);

	protected abstract void decrypt(byte[] input, int inputOffset, byte[] output, int outputOffset);

	protected void setIV(byte[] iv)
	{
		this.iv = iv;
		if (iv.length >= fdback.length)
		{
			System.arraycopy(iv, 0, fdback, 0, fdback.length);
		}
		else
		{	
			System.arraycopy(iv, 0, fdback, 0, iv.length);
			Arrays.fill(fdback, iv.length, fdback.length - iv.length, (byte) 0);
		}	
	}
	protected void encryptECB(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks)
	{
		int offset = 0;

		for (int i = 0; i < blocks; i++)
		{
			encrypt(input, inputOffset+offset, output, outputOffset+offset);
			offset += blockSize;
		}
	}

	protected void decryptECB(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks)
	{
		int offset = 0;

		for (int i = 0; i < blocks; i++)
		{
			decrypt(input, inputOffset+offset, output, outputOffset+offset);
			offset += blockSize;
		}
	}

	protected void encryptCBC(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks)
	{
		int offset = 0;

		for (int i = 0; i < blocks; i++)
		{
			// xor input into fdback so that we preserve input
			for (byte b = 0; b < blockSize; b++)
				fdback[b] ^= input[inputOffset+offset+b];

			// then encrypt fdback xor input
			encrypt(fdback, 0, output, outputOffset+offset);

			// copy the output block back to fdback
			System.arraycopy(output, outputOffset+offset, fdback, 0, blockSize);
			offset += blockSize;
		}
	}

	protected void decryptCBC(byte[] input, int inputOffset, byte[] output, int outputOffset, int blocks)
	{
		int offset = 0;

		for (int i = 0; i < blocks; i++)
		{
			decrypt(input, inputOffset+offset, output, outputOffset+offset);

			// then xor fdback into output
			for (byte b = 0; b < blockSize; b++)
				output[outputOffset+offset+b] ^= fdback[b];

			// copy the input block to fdback
			System.arraycopy(input, inputOffset+offset, fdback, 0, blockSize);
			offset += blockSize;
		}
	}	
}
