/*
 * DLGenParameterSpec.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto.spec;

import javax.crypto.spec.DHGenParameterSpec;

/**
 * This class specifies the set of parameters for generating discrete logarithm domain parameters
 * for use in algorithms such as ElGamal, DSA, etc.
 * <p>
 * This is typically done by a central authority.
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 * <p>
 * @see DLParameterSpec
 */
public class DLGenParameterSpec extends DHGenParameterSpec
{
	/**
	 * Determines whether the generator should generate the whole group,
	 * or a subgroup of order q
	 */
	private final boolean generateSubgroup;

	/**
	 * Determines the size of the prime factor q of (p-1)
	 */
	private final int factorSize;

	/**
	 * Constructs a parameter set for the generation of discrete logarithm domain parameters.
	 * <p>
	 * This constructor will produce a generator that generates a subgroup of order q, where
	 * q is a prime factor of (p-1).
	 * <p>
	 * @param primeSize the size (in bits) of the prime modulus
	 * @param factorSize the size (in bits) of the prime factor of (modulus-1) 
	 * @param exponentSize the size (in bits) of the random exponent
	 */
	public DLGenParameterSpec(int primeSize, int factorSize, int exponentSize)
	{
		super(primeSize, exponentSize);
		this.factorSize = factorSize;
		this.generateSubgroup = true;
	}

	/**
	 * Constructs a parameter set for the generation of discrete logarithm domain parameters.
	 * <p>
	 * @param primeSize the size (in bits) of the prime modulus
	 * @param factorSize the size (in bits) of the prime factor of (modulus-1) 
	 * @param exponentSize the size (in bits) of the random exponent
	 * @param generateSubgroup determines whether the generator will generate a the whole group or a subgroup.
	 */
	public DLGenParameterSpec(int primeSize, int factorSize, int exponentSize, boolean generateSubgroup)
	{
		super(primeSize, exponentSize);
		this.factorSize = factorSize;
		this.generateSubgroup = generateSubgroup;
	}

	/**
	 * Returns the size (in bits) of the prime factor of (modulus-1)
	 * @return the size (in bits) of the prime factor of (modulus-1)
	 */
	public int getFactorSize()
	{
		return factorSize;
	}

	/**
	 * Returns whether the parameters will produce a generator which generates the whole group or a subsgroup.
	 * @returns true if the parameters will produce a generator which generates a subgroup, false otherwise.
	 */
	public boolean getGenerateSubgroup()
	{
		return generateSubgroup;
	}
}
