/*
 * DLParameterSpec.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.crypto.spec;

import java.math.BigInteger;
import javax.crypto.spec.DHParameterSpec;

/**
 * This class specifies the set of discrete logarithm domain parameters as specified in:<br>
 * IEEE P1363 - Chapter 6: Primitives Based on the Discrete Logarithm Problem.
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class DLParameterSpec extends DHParameterSpec
{
	private final BigInteger q;

	/**
	 * Constructs a discrete logarithm domain parameter set, using a prime modulus p,
	 * a prime factor q of (p-1) and a base generator g.
	 * @param p the prime modulus
	 * @param q the prime factor of (modulus-1)
	 * @param g the base generator
	 */
	public DLParameterSpec(BigInteger p, BigInteger q, BigInteger g)
	{
		super(p, g);
		this.q = q;
	}

	/**
	 * Constructs a discrete logarithm domain parameter set, using a prime modulus p,
	 * a prime factor q of (p-1), a base generator g and the size in bits, l, of
	 * the random exponent (private value).
	 * @param p the prime modulus
	 * @param q the prime factor of (modulus-1)
	 * @param g the base generator
	 * @param l the size in bits of the random exponent (private value)
	 */
	public DLParameterSpec(BigInteger p, BigInteger q, BigInteger g, int l)
	{
		super(p, g, l);
		this.q = q;
	}

	/**
	 * Returns the prime factor q of (modulus-1).
	 * @return the prime factor q of (modulus-1)
	 */
	public BigInteger getQ()
	{
		return q;
	}

	/**
	 * Checks whether the domain parameters are valid, according to IEEE P1363 A.16.2.
	 * @return true if the domain parameters are valid, false otherwise
	 */
	public boolean validate()
	{
		/* not fully done */
		BigInteger p = getP();

		if (!p.testBit(p.bitLength() - 1))
			return false;

		if (p.compareTo(BigInteger.ONE) <= 0)
			return false;

		if (!q.testBit(q.bitLength() - 1))
			return false;

		if (q.compareTo(BigInteger.ONE) <= 0)
			return false;

		BigInteger g = getG();

		if (g.compareTo(BigInteger.ONE) <= 0)
			return false;

		if (g.compareTo(p) >= 0)
			return false;

		/* now do the more expensive tests */
		/* does q divide (p-1)? */

		/* are p and q probable primes? */
		if (!p.isProbablePrime(50))
			return false;

		if (!q.isProbablePrime(50))
			return false;

		/* also need to check the generator properties
		if (g.modPow(q, p).compareTo(BigInteger.ONE) != 0)
		{
			// generator doesn't have order Q, check if it has order 	
		}
		*/
		return true;
	}
}
