/*
 * BeeInputStream.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security;

import com.beeyond.asn1.*;
import com.beeyond.crypto.spec.*;

import java.io.*;
import java.math.*;
import java.security.*;
import java.security.cert.*;
import java.security.spec.*;

/**
 * This class implements an InputStream that can read cryptographic objects encoded in the "Bee" format.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public final class BeeInputStream extends DataInputStream
{
	private final static String
		PROVIDER = "BeeCrypt";

	/**
	 * Creates a BeeInputStream from an InputStream.
	 * @param in the underlying InputStream
	 */
	public BeeInputStream(InputStream in)
	{
		super(in);
	}

	/**
	 * Returns a string read from the input stream.
	 * @return the string read from the input stream
	 * @throw IOException
	 */
	public String readString() throws IOException
	{
		String s = null;
		int length = readInt();
		if (length > 0)
		{
			try
			{
				char[] tmp = new char[length];
				for (int i = 0; i < length; tmp[i++] = readChar());
				s = new String(tmp);
				tmp = null;
			}
			catch (Exception e)
			{
				System.out.println("Error allocating a char array of " + length + " length");
			}
		}
		else if (length == 0)
		{
			s = new String();
		}
		else
		{
			System.out.println("Error reading string, length is negative");
		}
		return s;
	}

	/***/
	public BigInteger readBigInteger() throws IOException
	{
		int length = readInt();
		byte[] tmp = new byte[length];
		readFully(tmp);
		return new BigInteger(tmp);
	}

	/***/
	public byte[] readSHAHash() throws IOException
	{
		byte[] hash = new byte[20];
		readFully(hash);
		return hash;
	}

	/***/
	public java.security.PrivateKey readPrivateKey() throws IOException
	{
		String algorithm = readString();
		try
		{
			KeyFactory kf = KeyFactory.getInstance(algorithm, PROVIDER);
			byte[] enc = new byte[readInt()];
			readFully(enc);
			return kf.generatePrivate(new BeeEncodedKeySpec(enc));
		}
		catch (InvalidKeySpecException e)
		{
			throw new ProviderException(e.getMessage());
		}
		catch (NoSuchAlgorithmException e)
		{
			throw new ProviderException(e.getMessage());
		}
		catch (NoSuchProviderException e)
		{
			throw new ProviderException(e.getMessage());
		}
	}

	/***/
	public java.security.PublicKey readPublicKey() throws IOException
	{
		String algorithm = readString();
		try
		{
			KeyFactory kf = KeyFactory.getInstance(algorithm, PROVIDER);
			byte[] enc = new byte[readInt()];
			readFully(enc);
			return kf.generatePublic(new BeeEncodedKeySpec(enc));
		}
		catch (InvalidKeySpecException e)
		{
			throw new ProviderException(e.getMessage());
		}
		catch (NoSuchAlgorithmException e)
		{
			throw new ProviderException(e.getMessage());
		}
		catch (NoSuchProviderException e)
		{
			throw new ProviderException(e.getMessage());
		}
	}

	/***/
	public byte[] readSignature() throws IOException
	{
		byte[] tmp = new byte[readInt()];
		readFully(tmp);
		
		ByteArrayInputStream bais = new ByteArrayInputStream(tmp);
		BeeInputStream bis = new BeeInputStream(bais);

		ASNObject[] objs = new ASNObject[] { new ASNInteger(bis.readBigInteger()), new ASNInteger(bis.readBigInteger()) };

		bis.close();
		bais.close();
		
		return DER.encode(new ASNSequence(objs));
	}

	/***/
	public java.security.cert.Certificate readCertificate() throws IOException, CertificateException, NoSuchProviderException
	{
		CertificateFactory cf = CertificateFactory.getInstance(PROVIDER, PROVIDER);
		return cf.generateCertificate(this);
	}

	/***/
	public byte[] readDigest() throws IOException
	{
		byte[] dig = new byte[readInt()];
		readFully(dig);
		return dig;
	}
}
