/*
 * BeeInputStream.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security;

import java.io.*;
import java.util.*;
import java.security.*;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.spec.*;
import javax.crypto.*;
import javax.crypto.spec.*;
import com.beeyond.crypto.*;

/**
 * This class implements a non-segregation KeyStore (try storing a
 * non-X509 certificate in the JKS KeyStore, and you'll know what I
 * mean).
 * <p>
 * It uses PBEWithSHAAndBlowfish encryption to protect the data (that is,
 * the encryption key and iv are derived from the password in the method
 * described in PCKS#12 appendix B.1;
 * <p>
 * Since SHA-1 produces a 160 bit result, and Blowfish needs a 64 bit iv,
 * we get a 96 bit encryption key.
 * <p>
 * Since we store the used algorithms in the KeyStore stream, we will even
 * be able to read keystores implemented with other algorithms, provided
 * the proper algorithms have been installed.
 * <p>
 * At the moment SHA-2, or whatever the successor to SHA-1 will be called
 * is available, we'll be able to provide a stronger protection of the
 * BeeKeyStore. For now we feel this will do much better than Sun's
 * standard KeyStore, which uses MDC/SHA-1 encryption.
 * <p>
 * Since we plan this upgrade, we should make sure that the encryption
 * algorithm used is identified in the stored file BEFORE the encrypted
 * data. This way we can read out the algorithm, and a description of the
 * algorithm parameters, and reconstruct the decryption cipher.
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public abstract class BeeKeyStore extends KeyStoreSpi
{
	private static final String
		IDENTIFIER = "BeeKeyStore",
		CIPHER_ALGORITHM = "PBEWithSHAAndBlowfish",
		PARAMETER_ALGORITHM = "PBE";

	private Hashtable entries;

	class TrustedCertificateEntry implements Serializable
	{
		Date created;
		Certificate cert;
	}

	class KeyEntry implements Serializable
	{
		Date created;
		SealedObject encryptedKey;
		Certificate[] chain;
	}

	protected BeeKeyStore(Cipher c)
	{
		entries = new Hashtable();
	}

	public Enumeration engineAliases()
	{
		return entries.keys();
	}

	public boolean engineContainsAlias(String alias)
	{
		return entries.containsKey(alias);
	}

	public void engineDeleteEntry(String alias)
	{
		synchronized (entries)
		{
			entries.remove(alias);
		}
	}

	public Certificate engineGetCertificate(String alias)
	{
		Object obj = entries.get(alias);
		
		if (obj != null)
		{
			if (obj instanceof TrustedCertificateEntry)
				return ((TrustedCertificateEntry) obj).cert;
			else if (obj instanceof KeyEntry)
			{
				KeyEntry ke = (KeyEntry) obj;
				if (ke.chain != null)
					return ke.chain[0];
			}
		}
		return null;
	}

	public String engineGetCertificateAlias(Certificate cert)
	{
		synchronized (entries)
		{
			for (Enumeration e = entries.keys(); e.hasMoreElements();)
			{
				String alias = (String) e.nextElement();
				Object obj = entries.get(alias);
				if (obj != null)
				{
					if (obj instanceof TrustedCertificateEntry)
					{
						if (cert.equals(((TrustedCertificateEntry) obj).cert))
							return alias;
					}
					else if (obj instanceof KeyEntry)
					{
						KeyEntry ke = (KeyEntry) obj;
						if (ke.chain != null)
							if (cert.equals(ke.chain[0]))
								return alias;
					}
				}
			}
		}
		return null;
	}

	public Certificate[] engineGetCertificateChain(String alias)
	{
		Object obj = entries.get(alias);
		
		if ((obj != null) && (obj instanceof KeyEntry))
		{
			KeyEntry ke = (KeyEntry) obj;

			if (ke.chain != null)
				return (Certificate[]) ke.chain.clone();
		}
		return null;
	}

	public Date engineGetCreationDate(String alias)
	{
		Object obj = entries.get(alias);		
		if (obj != null)
		{
			if (obj instanceof TrustedCertificateEntry)
				return new Date(((TrustedCertificateEntry) obj).created.getTime());
			else if (obj instanceof KeyEntry)
				return new Date(((KeyEntry) obj).created.getTime());
		}
		return null;
	}

	public Key engineGetKey(String alias, char[] password) throws NoSuchAlgorithmException, UnrecoverableKeyException
	{
		Object obj = entries.get(alias);
		
		if (obj != null)
		{
			if (obj instanceof KeyEntry)
			{
				/* complete this */
			}
		}
		return null;
	}
	
	public boolean engineIsCertificateEntry(String alias)
	{
		Object obj = entries.get(alias);
		
		return ((obj != null) && (obj instanceof TrustedCertificateEntry));
	}

	public boolean engineIsKeyEntry(String alias)
	{
		Object obj = entries.get(alias);
		
		return ((obj != null) && (obj instanceof KeyEntry));
	}

	public void engineLoad(InputStream stream, char[] password) throws IOException, NoSuchAlgorithmException, CertificateException
	{
		try
		{
			DataInputStream dis = new DataInputStream(stream);
			String tmp;

			tmp = dis.readUTF();
			if (!tmp.equals(IDENTIFIER))
				throw new IOException("Not a Bee KeyStore");

			tmp = dis.readUTF();
			Cipher c = Cipher.getInstance(tmp);
			tmp = dis.readUTF();
			AlgorithmParameters params = AlgorithmParameters.getInstance(tmp);
			short length = dis.readShort();
			byte[] encodedParams = new byte[length];
			dis.readFully(encodedParams);
			dis.close();
			params.init(encodedParams);
			c.init(Cipher.DECRYPT_MODE, null, params, null);
			CipherInputStream cis = new CipherInputStream(stream, c);
			ObjectInputStream ois = new ObjectInputStream(cis);
			entries = (Hashtable) ois.readObject();
			ois.close();
			cis.close();
		}
		catch (NoSuchPaddingException nspe)
		{
			throw new NoSuchAlgorithmException(nspe.getMessage());
		}
		catch (InvalidAlgorithmParameterException iape)
		{
			throw new IOException(iape.getMessage());
		}
		catch (InvalidKeyException ike)
		{
			throw new IOException(ike.getMessage());
		}
		catch (ClassNotFoundException cnfe)
		{
			throw new IOException(cnfe.getMessage());
		}
	}
	
	public void engineStore(OutputStream stream, char[] password) throws IOException, NoSuchAlgorithmException
	{
		try
		{
			Cipher c = Cipher.getInstance(CIPHER_ALGORITHM);
			/* get new salt equal in length to the cipher's blocksize */
			byte[] salt = SecureRandom.getSeed(c.getBlockSize());
			/* make PBEParamSpec from salt and iterationCount = 1024 */
			PBEParameterSpec paramSpec = new PBEParameterSpec(salt, 1024);
			AlgorithmParameters params = AlgorithmParameters.getInstance("PBE");
			params.init(paramSpec);
			/* store through an ObjectOutputStream piggybacked on a CipherOutputStream */
			DataOutputStream dos = new DataOutputStream(stream);
			dos.writeUTF(IDENTIFIER);
			dos.writeUTF(CIPHER_ALGORITHM);
			dos.writeUTF(PARAMETER_ALGORITHM);
			byte[] encodedParams = params.getEncoded();
			dos.writeInt(encodedParams.length);
			dos.write(encodedParams);
			dos.close();
			c.init(Cipher.ENCRYPT_MODE, null, params, null);
			CipherOutputStream cos = new CipherOutputStream(stream, c);
			ObjectOutputStream oos = new ObjectOutputStream(cos);
			oos.writeObject(entries);
			oos.close();
			cos.close();
			dos.close();
		}
		catch (NoSuchPaddingException nspe)
		{
			throw new NoSuchAlgorithmException(nspe.getMessage());
		}
		catch (InvalidParameterSpecException ipse)
		{
			throw new IOException(ipse.getMessage());
		}
		catch (InvalidAlgorithmParameterException iape)
		{
			throw new IOException(iape.getMessage());
		}
		catch (InvalidKeyException ike)
		{
			throw new IOException(ike.getMessage());
		}
	}
}