/*
 * BeeOutputStream.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security;

import java.io.*;
import java.math.*;
import java.security.*;
import java.security.interfaces.*;
import java.security.spec.*;

import com.beeyond.asn1.*;
import com.beeyond.crypto.spec.*;

/**
 * This class implements an OutputStream that can write cryptographic objects in the "Bee" encoding format.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public final class BeeOutputStream extends DataOutputStream
{
	private static final ASNCodingException
		SIGFORMAT = new ASNCodingException("signature format invalid");

	/**
	 * Creates a BeeOutputStream from an OutputStream
	 * @param out the underlying OutputStream
	 */
	public BeeOutputStream(OutputStream out)
	{
		super(out);
	}

	/***/
	public void writeString(String s) throws IOException
	{
		if (s == null)
		{
			writeInt(0);
		}
		else
		{
			writeInt(s.length());
			writeChars(s);
		}
	}

	/***/
	public void writeBigInteger(BigInteger b) throws IOException
	{
		byte[] tmp = b.toByteArray();
		writeInt(tmp.length);
		write(tmp);
	}

	/***/
	public void writeKey(Key key) throws IOException
	{
		String algorithm = key.getAlgorithm();

		try
		{
			KeyFactory kf = KeyFactory.getInstance(algorithm);
			BeeEncodedKeySpec spec = (BeeEncodedKeySpec) kf.getKeySpec(key, BeeEncodedKeySpec.class);
	
			writeString(algorithm);
		
			byte[] enc = spec.getEncoded();
	
			writeInt(enc.length);
			write(enc);
		}
		catch (InvalidKeySpecException e)
		{
			throw new ProviderException(e.getMessage());
		}
		catch (NoSuchAlgorithmException e)
		{
			throw new ProviderException(e.getMessage());
		}
	}

	/***/
	public void writePublicKey(PublicKey pub) throws IOException
	{
		writeKey(pub);
	}

	/***/
	public void writeSignature(byte[] sig) throws IOException
	{
		ASNInteger r, s;
		ASNObject tmp = DER.decode(sig);

		if (!(tmp instanceof ASNSequence))
			throw SIGFORMAT;
	
		ASNSequence seq = (ASNSequence) tmp;

		if (seq.size() != 2)
			throw SIGFORMAT;

		tmp = seq.getObject(0);
			
		if (!(tmp instanceof ASNInteger))
			throw SIGFORMAT;

		r = (ASNInteger) tmp;

		tmp = seq.getObject(1);
			
		if (!(tmp instanceof ASNInteger))
			throw SIGFORMAT;

		s = (ASNInteger) tmp;

		byte[] rtmp = r.getContent(), stmp = s.getContent();

		writeInt(8 + rtmp.length + stmp.length);
		writeInt(rtmp.length);
		write(rtmp);
		writeInt(stmp.length);
		write(stmp);
	}

	/***/
	public void writeDigest(byte[] d) throws IOException
	{
		writeInt(d.length);
		write(d);
	}
}
