/*
 * ElGamalSignature.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security;

import com.beeyond.crypto.*;
import java.io.*;
import java.math.BigInteger;
import java.security.*;
import java.security.spec.*;

/**
 * This abstract class implements the base for ElGamal signature variants.
 * <p>
 * <b>References:</b><br>
 * <ul>
 * <li>Alfred J. Menezes,
 *     "Section 11.5.2: The ElGamal Signature Scheme",
 *     Handbook of Applied Cryptography,
 *     CRC Press, 1997.
 * </ul>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public abstract class ElGamalSignature extends RSSignature
{
	/**
	 * The default digest algorithm for this type of signature.
	 */
	private static final String
		DEFAULTDIGEST = "SHA";

	private static final InvalidKeyException
		NOTELGPUBKEY = new InvalidKeyException("not an ElGamalPublicKey"),
		NOTELGPRIKEY = new InvalidKeyException("not an ElGamalPrivateKey");

	protected ElGamalPrivateKey privateKey;
	protected ElGamalPublicKey publicKey;
	protected SecureRandom random = null;

	/**
	 * Creates a ElGamalSignature object.
	 * @param variant the variant of the ElGamal algorithm
	 */
	public ElGamalSignature(String variant) throws NoSuchAlgorithmException
	{
		super(variant);

		int slash = variant.indexOf('/');

		md = MessageDigest.getInstance(slash < 0 ? DEFAULTDIGEST : variant.substring(0, slash));
	}

	/***/
	protected void engineInitVerify(PublicKey publicKey) throws InvalidKeyException
	{
		if (publicKey instanceof ElGamalPublicKey)
		{
			state = VERIFY;
			md.reset();
			this.publicKey = (ElGamalPublicKey) publicKey;
		}
		else
			throw NOTELGPUBKEY;
	}

	/***/
	protected void engineInitSign(PrivateKey privateKey) throws InvalidKeyException
	{
		if (privateKey instanceof ElGamalPrivateKey)
		{
			state = SIGN;
			md.reset();
			this.privateKey = (ElGamalPrivateKey) privateKey;
			this.random = new SecureRandom();
		}
		else
			throw NOTELGPRIKEY;
	}

	/***/
	protected void engineInitSign(PrivateKey privateKey, SecureRandom random) throws InvalidKeyException
	{
		if (privateKey instanceof ElGamalPrivateKey)
		{
			state = SIGN;
			md.reset();
			this.privateKey = (ElGamalPrivateKey) privateKey;
			this.random = random;
		}
		else
			throw NOTELGPRIKEY;
	}
}
