/*
 * ElGamalV1Signature.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security;

import com.beeyond.crypto.*;
import java.io.*;
import java.math.BigInteger;
import java.security.*;
import java.security.spec.*;

/**
 * This class implements ElGamal variant 1 signatures
 * <p>
 * It uses the following equations:
 * <ul>
 * <li>Signing: r = g<sup>k</sup> mod p; s = k<sup>-1</sup>(h(m) - xr) mod (p-1)
 * <li>Verification: g<sup>h(m)</sup> = y<sup>r</sup>r<sup>s</sup>
 * </ul>
 * <p>
 * <b>References:</b><br>
 * <ul>
 * <li>Alfred J. Menezes,
 *     "Section 11.5.2: The ElGamal Signature Scheme",
 *     Handbook of Applied Cryptography,
 *     CRC Press, 1997.
 * </ul>
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class ElGamalV1WithSHA extends ElGamalSignature
{
	private static final String
		ALGORITHM = "SHA/ElGamal/v1";

	/**
	 * Creates an ElGamalV1WithSHA object.
	 */
	public ElGamalV1WithSHA() throws NoSuchAlgorithmException
	{
		super(ALGORITHM);
	}

	/**
	 * Implements the signing algorithm for variant 1.
	 */
	protected void RSSign()
	{
		BigInteger n = privateKey.getP().subtract(BigInteger.ONE);
		BigInteger hm = new BigInteger(1, md.digest());

		BigInteger k;

		while (true)
		{
			do
			{
				/* make sure we get an odd k, since k even implies gcd(k,n) = 2 */
				k = BigInteger.ONE.or(new BigInteger(n.bitCount(), random));
				/* in this loop we subtract even value from an odd k, so k will stay odd */
				while (k.compareTo(n) >= 0)
					k = k.subtract(n);
			} while (BigInteger.ONE.compareTo(k) >= 0);
	
			try
			{
				r = privateKey.getG().modPow(k, privateKey.getP());
				s = k.modInverse(n).multiply(hm.subtract(privateKey.getX().multiply(r)).mod(n)).mod(n);
			}
			catch (ArithmeticException ae)
			{
				/* try again */
				continue;
			}

			break;
		}
	}

	/**
	 * Implements the verifying algorithm for variant 1.
	 * @return true if the signature valid, false otherwise
	 */
	protected boolean RSVerify()
	{
		if (r.compareTo(BigInteger.ONE) < 0)
			return false;

		BigInteger n = publicKey.getP().subtract(BigInteger.ONE);

		if (r.compareTo(n) > 0)
			return false;
		
		BigInteger hm = new BigInteger(1, md.digest());
		BigInteger v1 = publicKey.getY().modPow(r, publicKey.getP()).multiply(r.modPow(s, publicKey.getP())).mod(publicKey.getP());
		BigInteger v2 = publicKey.getG().modPow(hm, publicKey.getP());
		return v1.equals(v2);
	}
}
