/*
 * ElGamalV3WithSHA.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package com.beeyond.security;

import com.beeyond.crypto.*;
import java.io.*;
import java.math.BigInteger;
import java.security.*;
import java.security.spec.*;

/**
 * This class implements ElGamal variant 3 signatures
 * <p>
 * It uses the following equations:
 * <ul>
 * <li>Signing: r = g<sup>k</sup> mod p; s = (xr + kh(m)) mod (p-1)
 * <li>Verification: g<sup>s</sup> = y<sup>r</sup>r<sup>h(m)</sup>
 * </ul>
 * <p>
 * <b>References:</b><br>
 * <ul>
 * <li>Alfred J. Menezes,
 *     "Section 11.5.2: The ElGamal Signature Scheme",
 *     Handbook of Applied Cryptography,
 *     CRC Press, 1997.
 * </ul>
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class ElGamalV3WithSHA extends ElGamalSignature
{
	private static final String
		ALGORITHM = "SHA/ElGamal/v3";

	/**
	 * Creates an ElGamalV3WithSHA object.
	 */
	public ElGamalV3WithSHA() throws NoSuchAlgorithmException
	{
		super(ALGORITHM);
	}

	/**
	 * Implements the signing algorithm for variant 3.
	 */
	protected void RSSign()
	{
		BigInteger n = privateKey.getP().subtract(BigInteger.ONE);
		BigInteger hm = new BigInteger(1, md.digest());
		BigInteger k;
		do
		{
			k = new BigInteger(n.bitCount(), random);
			while (k.compareTo(n) >= 0)
				k = k.subtract(n);
		} while (BigInteger.ONE.compareTo(k) >= 0);

		r = privateKey.getG().modPow(k, privateKey.getP());
		s = r.multiply(privateKey.getX()).add(k.multiply(hm)).mod(n);
	}

	/**
	 * Implements the verifying algorithm for variant 3.
	 * @return true if the signature was valid, false otherwise
	 */
	protected boolean RSVerify()
	{
		if (r.compareTo(BigInteger.ONE) < 0)
			return false;
			
		BigInteger n = publicKey.getP().subtract(BigInteger.ONE);

		if (r.compareTo(n) > 0)
			return false;
		
		BigInteger hm = new BigInteger(1, md.digest());
		BigInteger v1 = publicKey.getY().modPow(r, publicKey.getP()).multiply(r.modPow(hm, publicKey.getP())).mod(publicKey.getP());
		BigInteger v2 = publicKey.getG().modPow(s, publicKey.getP());
		return v1.equals(v2);
	}
}
