/*
 * NativeMessageDigest.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security;

import java.security.*;

/**
 * This class implements the glue to native message digests (hash functions)
 * in the BeeCrypt Cryptography Library.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class NativeMessageDigest extends MessageDigestSpi
{
	private long hash;
	private long param;

	private static native long find(String algorithm);

	private static native long allocParam(long hash);
	
	private static native long cloneParam(long hash, long param);

	private static native void freeParam(long param);

	private static native void reset(long hash, long param);

	private static native void update(long hash, long param, byte input);
	
	private static native void updateBlock(long hash, long param, byte[] input, int offset, int len);

	private static native byte[] digest(long hash, long param);

	private static native int digestLength(long hash);

	static
	{
		System.loadLibrary("beecrypt");
	}

	private NativeMessageDigest(long hash, long param)
	{
		this.hash = hash;
		this.param = param;
	}

	protected NativeMessageDigest(String algorithm)
	{
		hash = find(algorithm);
		param = allocParam(hash);
		reset(hash, param);
	}

	public Object clone()
	{
		return new NativeMessageDigest(hash, cloneParam(hash, param));
	}

	protected byte[] engineDigest()
	{
		return digest(hash, param);
	}
	
	protected int engineDigest(byte[] buf, int offset, int len) throws DigestException
	{
		byte[] tmp = digest(hash, param);
		if (tmp.length > len)
			throw new DigestException("buffer too small");
		System.arraycopy(tmp, 0, buf, offset, len);
		return tmp.length;
	}
	
	protected int engineGetDigestLength()
	{
		return digestLength(hash);
	}

	protected void engineReset()
	{
		reset(hash, param);
	}

	protected void engineUpdate(byte input)
	{
		update(hash, param, input);
	}
	
	protected void engineUpdate(byte[] input, int offset, int len)
	{
		updateBlock(hash, param, input, offset, len);
	}

	public void finalize()
	{
		freeParam(param);
	}
}
