/*
 * NativeSecureRandom.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security;

import java.security.*;

/**
 * This class implements the glue to native random generators
 * in the BeeCrypt Cryptography Library.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public class NativeSecureRandom extends SecureRandomSpi
{
	private long prng;

	private long param;

	private static native long find(String algorithm);

	private static native long allocParam(long hash);
	
	private static native long cloneParam(long hash, long param);

	private static native void freeParam(long param);

	private static native void setup(long hash, long param);

	private static native void setSeed(long hash, long param, byte[] seed);

	private static native void nextBytes(long hash, long param, byte[] bytes);

	private static native void generateSeed(byte[] seed);

	static
	{
		System.loadLibrary("beecrypt");
	}

	private NativeSecureRandom(long prng, long param)
	{
		this.prng = prng;
		this.param = param;
	}

	protected NativeSecureRandom(String algorithm)
	{
		prng = find(algorithm);
		param = allocParam(prng);
		setup(prng, param);
	}

	public Object clone()
	{
		return new NativeSecureRandom(prng, cloneParam(prng, param));
	}

	protected byte[] engineGenerateSeed(int numBytes)
	{
		byte[] result = new byte[numBytes];
		generateSeed(result);
		return result;
	}

	protected void engineNextBytes(byte[] bytes)
	{
		nextBytes(prng, param, bytes);
	}

	protected void engineSetSeed(byte[] seed)
	{
		setSeed(prng, param, seed);
	}

	public void finalize()
	{
		freeParam(param);
	}
}
