/*
 * PrimeParameterGenerator.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security;

import java.math.BigInteger;
import java.security.*;
import java.security.spec.*;
import com.beeyond.security.spec.*;

/**
 * This class implements the generation of secure primes with a method
 * derived and extended from IEEE P1363.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public final class PrimeParameterGenerator extends java.security.AlgorithmParameterGeneratorSpi
{
	/***/
	private static final String
		ALGORITHM = "Prime",
		PROVIDER = "BeeCrypt";

	/***/
	private PrimeGenParameterSpec genParamSpec = null;

	/***/
	private SecureRandom random = null;

	/**
	 * Returns the recommended security for testing a randomly generated prime of given size.
	 * @return the recommended security for testing a randomly generated prime of given size
	 */
	public static int getPrimeSecurity(int primeSize)
	{
		if (primeSize >= 1854)
			return 2;
		if (primeSize >= 1223)
			return 3;
		if (primeSize >= 927)
			return 4;
		if (primeSize >= 747)
			return 5;
		if (primeSize >= 627)
			return 6;
		if (primeSize >= 543)
			return 7;
		if (primeSize >= 480)
			return 8;
		if (primeSize >= 431)
			return 9;
		if (primeSize >= 393)
			return 10;
		if (primeSize >= 361)
			return 11;
		if (primeSize >= 335)
			return 12;
		if (primeSize >= 314)
			return 13;
		if (primeSize >= 295)
			return 14;
		if (primeSize >= 279)
			return 15;
		if (primeSize >= 265)
			return 16;
		if (primeSize >= 253)
			return 17;
		if (primeSize >= 242)
			return 18;
		if (primeSize >= 232)
			return 19;
		if (primeSize >= 223)
			return 20;
		if (primeSize >= 216)
			return 21;
		if (primeSize >= 209)
			return 22;
		if (primeSize >= 202)
			return 23;
		if (primeSize >= 196)
			return 24;
		if (primeSize >= 191)
			return 25;
		if (primeSize >= 186)
			return 26;
		if (primeSize >= 182)
			return 27;
		if (primeSize >= 178)
			return 28;
		if (primeSize >= 174)
			return 29;
		if (primeSize >= 170)
			return 30;
		if (primeSize >= 167)
			return 31;
		if (primeSize >= 164)
			return 32;
		if (primeSize >= 161)
			return 33;
		if (primeSize >= 160)
			return 34;
		return 35;
	}

	/**
	 * Returns a generated secure prime as AlgorithmParameters.
	 * @return a generated secure prime as AlgorithmParameters
	 */
	protected AlgorithmParameters engineGenerateParameters()
	{
		if (random == null)
			random = new SecureRandom();

		int primeSize, factorSize, cofactorType;

		if (genParamSpec == null)
		{
			primeSize = 1024;
			factorSize = 512;
			cofactorType = 1;
		}
		else
		{
			primeSize = genParamSpec.getPrimeSize();
			factorSize = genParamSpec.getFactorSize();
			cofactorType = genParamSpec.getCofactorType();
		}

		AlgorithmParameters ap = null;

		try
		{
			ap = AlgorithmParameters.getInstance(ALGORITHM, PROVIDER);
		}
		catch (NoSuchAlgorithmException e)
		{
			throw new ProviderException(e.getMessage());
		}
		catch (NoSuchProviderException e)
		{
			throw new ProviderException(e.getMessage());
		}

		/* make p such that p = n+1, and n=qr; r must be even */
		BigInteger p, n, q, r, s;

		q = new BigInteger(factorSize, getPrimeSecurity(factorSize), random);

		int pt = getPrimeSecurity(primeSize);
		int st = getPrimeSecurity(primeSize - factorSize - 1);

		p = r = BigInteger.ZERO;

		do
		{
			if (cofactorType < 3)
			{
				/* make an initial s one bit smaller than needed for r */
				s = new BigInteger(primeSize - factorSize - 1, random);
				
				if (cofactorType > 0)
				{
					/* if we need cofactor checking, make s odd */
					s = BigInteger.ONE.or(s);
				}
	
				if (cofactorType == 1)
				{
					/* if (q <= s) check if gcd(q,s) != 1 */
					if ((q.compareTo(s) <= 0) && !(q.gcd(s).equals(BigInteger.ONE)))
						continue;
				}
				else if (cofactorType == 2)
				{
					/* check if s is probable prime */
					if (!s.isProbablePrime(st))
						continue;
				}
				r = s.shiftLeft(1);
				n = q.multiply(r);
			}
			else
			{
				s = BigInteger.ONE;
				r = s.shiftLeft(1);
				n = q.shiftLeft(1);
			}

			p = n.add(BigInteger.ONE);
		} while (!p.isProbablePrime(pt));

		try
		{
			ap.init(new PrimeParameterSpec(p, q, r));
		}
		catch (InvalidParameterSpecException e)
		{
			throw new ProviderException(e.getMessage());
		}

		return ap;
	}

	/**
	 * Initializes the generator with an AlgorithmParameterSpec and a SecureRandom.
	 * @param genParamSpec the set of parameters to use for generating the secure prime
	 * @param random the random generator to use
	 */
	protected void engineInit(AlgorithmParameterSpec genParamSpec, SecureRandom random) throws InvalidAlgorithmParameterException
	{
		if (genParamSpec instanceof PrimeGenParameterSpec)
		{
			this.genParamSpec = (PrimeGenParameterSpec) genParamSpec;
			this.random = random;
		}
		else
			throw new InvalidAlgorithmParameterException("Not a DHGenParameterSpec");
	}

	/**
	 * Initializes the generator with a size (in bits) and a SecureRandom.
	 * @param size the size (in bits) of the prime to generate
	 * @param random the random generator to use
	 */
	protected void engineInit(int size, SecureRandom random)
	{
		this.genParamSpec = new PrimeGenParameterSpec(size, size >> 1, 1);
		this.random = random;
	}
}
