/*
 * PrimeParameters.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security;

import java.io.*;
import java.security.*;
import java.security.spec.*;
import com.beeyond.asn1.*;
import com.beeyond.security.spec.*;

/**
 * This class implements secure prime parameters.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public final class PrimeParameters extends AlgorithmParametersSpi
{
	private PrimeParameterSpec paramSpec = null;

	protected byte[] engineGetEncoded() throws IOException
	{
		if (paramSpec == null)
			return null;

		ASNObject[] objs = new ASNObject[] { new ASNInteger(paramSpec.getP()), new ASNInteger(paramSpec.getQ()), new ASNInteger(paramSpec.getR()) };

		return DER.encode(new ASNSequence(objs));
	}

	protected byte[] engineGetEncoded(String format) throws IOException
	{
		if (format.equalsIgnoreCase(DER.ASNFORMAT))
			return engineGetEncoded();
		else
			throw new IllegalArgumentException("format " + format + " not supported");
	}

	protected AlgorithmParameterSpec engineGetParameterSpec(Class paramSpec) throws InvalidParameterSpecException
	{
		if (paramSpec.isAssignableFrom(PrimeParameterSpec.class))
			return this.paramSpec;
		else
			throw new InvalidParameterSpecException("not assignable from PrimeParameterSpec");
	}

	protected void engineInit(AlgorithmParameterSpec paramSpec) throws InvalidParameterSpecException
	{
		if (paramSpec instanceof PrimeParameterSpec)
			this.paramSpec = (PrimeParameterSpec) paramSpec;
		else
			new InvalidParameterSpecException("init with " + paramSpec.getClass().getName() + " not supported");
	}

	protected void engineInit(byte[] params) throws IOException
	{
		ASNObject tmp = DER.decode(params);
		if (!(tmp instanceof ASNSequence))
			throw new ASNCodingException("not a sequence");

		ASNSequence seq = (ASNSequence) tmp;
		
		int seqSize = seq.size();

		if (seqSize != 3)
			throw new ASNCodingException("invalid number of sequence fields");

		tmp = seq.getObject(0);
		if (!(tmp instanceof ASNInteger))
			throw new ASNCodingException("first sequence field is not an integer");

		ASNInteger p = (ASNInteger) tmp;

		tmp = seq.getObject(1);
		if (!(tmp instanceof ASNInteger))
			throw new ASNCodingException("second sequence field is not an integer");

		ASNInteger q = (ASNInteger) tmp;

		tmp = seq.getObject(2);
		if (!(tmp instanceof ASNInteger))
			throw new ASNCodingException("third sequence field is not an integer");

		ASNInteger r = (ASNInteger) tmp;

		paramSpec = new PrimeParameterSpec(p.getValue(), q.getValue(), r.getValue());
	}

	protected void engineInit(byte[] params, String format) throws IOException
	{
		if (format.equalsIgnoreCase(DER.ASNFORMAT))
			engineInit(params);
		else
			throw new IllegalArgumentException("format " + format + " not supported");
	}

	protected String engineToString()
	{
		return null;
	}
}
