/*
 * RSSignature.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security;

import com.beeyond.asn1.*;

import java.io.*;
import java.math.BigInteger;
import java.security.*;
import java.security.spec.*;

/**
 * This class implements an abstract signature which is based
 * on two BigIntegers R and S, for example DSA and ElGamal.
 * <p>
 * It should only be accessed from within this package.
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
abstract class RSSignature extends Signature
{
	private static final SignatureException
		WRONGSTATE = new SignatureException("not in correct state"),
		SIGINVALID = new SignatureException("invalid signature encoding"),
		SIGTOOLONG = new SignatureException("signature encoding is too long for buffer");

	protected MessageDigest md;
	protected BigInteger r, s;

	protected RSSignature(String algorithm)
	{
		super(algorithm);
	}

	private byte[] encode()
	{
		try
		{
			ASNObject[] objs = new ASNObject[] { new ASNInteger(r), new ASNInteger(s) };

			return DER.encode(new ASNSequence(objs));
		}
		catch (IOException e)
		{
			throw new ProviderException("IOException in DER.encode:" + e.getMessage());
		}
	}

	/**
	 * The signing method to be implemented by subclasses.
	 */
	protected abstract void RSSign();

	/**
	 * The verifying method to be implemented by subclasses.
	 * @return true if the signature was verified, false otherwise.
	 */
	protected abstract boolean RSVerify();

	protected void engineUpdate(byte b) throws SignatureException
	{
		md.update(b);
	}

	protected void engineUpdate(byte[] b, int off, int len) throws SignatureException
	{
		md.update(b, off, len);
	}

	protected byte[] engineSign() throws SignatureException
	{
		if (state == SIGN)
		{
			RSSign();
			return encode();
		}
		else
			throw WRONGSTATE;
	}

	protected int engineSign(byte[] outBuf, int off, int len) throws SignatureException
	{
		if (state == SIGN)
		{
			RSSign();
			byte[] tmp = encode();
			if (tmp.length <= len)
			{
				System.arraycopy(outBuf, off, tmp, 0, tmp.length);
				return tmp.length;
			}
			else
				throw SIGTOOLONG;
		}
		else
			throw WRONGSTATE;
	}

	protected boolean engineVerify(byte[] sigBytes) throws SignatureException
	{
		try
		{
			ASNObject tmp = DER.decode(sigBytes);

			if (!(tmp instanceof ASNSequence))
				throw new SignatureException("signature is not an ASN.1 sequence");

			ASNSequence seq = (ASNSequence) tmp;

			if (seq.size() != 2)
				throw new SignatureException("signature ASN.1 sequence doesn't have two elements");
			tmp = seq.getObject(0);

			if (!(tmp instanceof ASNInteger))
				throw new SignatureException("sequence first element is not integer");

			r = ((ASNInteger) tmp).getValue();

			tmp = seq.getObject(1);

			if (!(tmp instanceof ASNInteger))
				throw new SignatureException("sequence second field is not integer");

			s = ((ASNInteger) tmp).getValue();
		}
		catch (IOException e)
		{
			throw new SignatureException("signature ASN.1 decoding failed:" + e.getMessage());
		}

		return RSVerify();
	}

	/**
	 * @deprecated
	 */
	protected void engineSetParameter(String param, Object value)
	{
		throw new IllegalArgumentException();
	}

	protected void engineSetParameter(AlgorithmParameterSpec params) throws InvalidAlgorithmParameterException
	{
		throw new InvalidAlgorithmParameterException();
	}

	/**
	 * @deprecated
	 */
	protected Object engineGetParameter(String param)
	{
		throw new IllegalArgumentException();
	}
}
