/*
 * BeeCertificate.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 1999-2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security.cert;

import com.beeyond.security.*;
import com.beeyond.crypto.*;

import java.io.*;
import java.security.*;
import java.security.cert.*;
import java.security.spec.*;

public class BeeCertificate extends java.security.cert.Certificate
{
	/**Type of certificate to use.*/
	private static final String TYPE = "Bee";
	private static final String ALGORITHM = "ElGamal";
	private static final String PROVIDER = "BeeCrypt";
	private static final int PUBK = 0x5055424B;
	private static final int HIVE = 0x48495645;
	private static final int ADDR = 0x41444452;
	private static final int ROAM = 0x524F414D;
	/**The entity that signs the certificate.*/
	private String issuer;
	/**Subject of the certificate.*/
	private String subject;
	/**Fixed id indication the server.*/
	private long hiveAddress;
	/**Indication when the certificate is created.*/
	private long created;
	/**Indication when the certificate expires.*/
	private long expires;
	/**The roaming flag indicates that the ip-address is fixed or not.*/
	private byte roaming = 0;
	/**Public key of the issuer.*/
	private PublicKey key = null;
	/**certificate bytes data.*/
	private byte[] certificate = null;
	/**Signed bytes data.*/
	private byte[] signature = null;
	/***/
	private byte[] encoding = null;

	/**Constructor.*/
	public BeeCertificate(KeyPair pair)
	{
		super(TYPE);
		issuer = subject = "";
		created = System.currentTimeMillis();
		expires = -1L;
		key = pair.getPublic();
		certificate = encode();
		encoding = certificate;
	}

	/**Constructor.*/
	public BeeCertificate(String self, KeyPair pair) throws SignatureException
	{
		super(TYPE);
		issuer = self;
		subject = self;
		created = System.currentTimeMillis();
		expires = -1L;
		key = pair.getPublic();
		certificate = encode();
	
		try
		{
			Signature s = Signature.getInstance(ALGORITHM);
			s.initSign((ElGamalPrivateKey) pair.getPrivate());
			s.update(certificate);
			signature = s.sign();
		}
		catch (NoSuchAlgorithmException nsae)
		{
			throw new ProviderException("algorithm " + ALGORITHM + " not present");
		}
		catch (InvalidKeyException ike)
		{
			throw new SignatureException("cannot sign with that key");
		}
	}

	/**Constructor.*/
	public BeeCertificate(String issuer, String subject, long created, long expires, PublicKey key)
	{
		super(TYPE);
		this.issuer = issuer;
		this.subject = subject;
		this.created = created;
		this.expires = expires;
		this.key = key;
	}

	/**Constructor.*/
	public BeeCertificate(byte[] cert) throws CertificateException
	{
		super(TYPE);
		decode(cert);
	}

	/***/
	private byte[] encode()
	{
		ByteArrayOutputStream bos = null;
		try
		{
			bos = new ByteArrayOutputStream();
			BeeOutputStream bee = new BeeOutputStream(bos);
			
			bee.writeString(issuer);
			bee.writeString(subject);
			bee.writeLong(created);
			bee.writeLong(expires);

//other options are possible here
			bee.writeInt(1);
			bee.writeInt(PUBK);
			bee.writeKey(key);

			bee.close();
			bos.close();
		}
		catch (IOException ioe)
		{
			throw new ProviderException(ioe.getMessage());
		}
		return bos.toByteArray();
	}

	/***/
	private void decode(byte[] cert) throws CertificateException
	{
		try
		{
			ByteArrayInputStream bis = new ByteArrayInputStream(cert);
			BeeInputStream bee = new BeeInputStream(bis);
	
			certificate = new byte[bee.readInt()];
		
			issuer = bee.readString();
			subject = bee.readString();
			created = bee.readLong();
			expires = bee.readLong();
	
			int fieldCount = bee.readInt();
			for(int i=0; i<fieldCount; i++)
			{
				int fieldType = bee.readInt();
				int fieldLength = bee.readInt();
				switch (fieldType)
				{
					case PUBK:
						key = bee.readPublicKey();
						break;
					case HIVE:
						hiveAddress = bee.readLong();
						break;
					case ADDR:
						bee.skip(fieldLength);
					case ROAM:
						roaming = 0;//(byte) bee.read();
						break;
					default:
						bee.skip(fieldLength);
				}
			}
	
			if (issuer.length() == 0 && subject.length() == 0)
				signature = null;
			else
				signature = bee.readSignature();
	
			bee.close();
			bis.close();			
	
			System.arraycopy(cert, 4, certificate, 0, certificate.length);
		}
		catch (Exception e)
		{
			throw new CertificateException(e.getMessage());
		}
	}

	/**Returns the certificate data.*/
	public byte[] getEncoded()
	{
		if (encoding == null)
		{
			try
			{
				ByteArrayOutputStream bout = new ByteArrayOutputStream();
				DataOutputStream dout = new DataOutputStream(bout);
				dout.writeInt(certificate.length);
				dout.write(certificate);
				dout.writeInt(signature.length);
				dout.write(signature);
				dout.close();
				encoding = bout.toByteArray();
				bout.close();
			}
			catch(IOException ioe)
			{
			}
		}
			
		return encoding;
	}

	/**Returns the public key of the issuer.*/
	public PublicKey getPublicKey()
	{
		return key;
	}

	/**Returns the public key of the issuer.*/
	public long getHiveAddress()
	{
		return hiveAddress;
	}

	/**Return the roaming flag.*/
	public byte getRoaming()
	{
		return roaming;
	}

	/**Verifies that this certificate was signed using the private key that corresponds to the specified public key. */
	public void verify(PublicKey publicKey)  throws CertificateException,
                            NoSuchAlgorithmException,
                            InvalidKeyException,
                            NoSuchProviderException,
                            SignatureException
	{
		Signature s = Signature.getInstance(ALGORITHM);
		s.initVerify(publicKey);
		s.update(certificate);
		if (!s.verify(signature))
			throw new SignatureException();
	}

	/**Verifies that this certificate was signed using the private key that corresponds to the specified public key.*/
	public void verify(PublicKey publicKey, String sigProvider)
	{
	}

	/**Certificate toString.*/
	public String toString()
	{
		return "Certificate type is " + TYPE +
			   " issued by " + issuer +
			   " with subject " + subject + ".\n"+
			   "The certificate was created on " + created +
			   " and expires on " + expires + ".";
	}
}
