/*
 * PrimeGenParameterSpec.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security.spec;

import java.security.*;
import java.security.spec.*;

/**
 * This class specifies the set of parameters for generating a secure prime
 * for use in cryptographic algorithms.
 * <p>
 * The method for constructing is derived from IEEE P1363.
 * <p>
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public final class PrimeGenParameterSpec implements AlgorithmParameterSpec
{
	private int primeSize;
	private int factorSize;
	private int cofactorType;

	/**
	 * Constructs a PrimeGenParameterSpec from a prime size (in bits),
	 * a prime factor size (in bits) and a cofactor type.
	 * <p>
	 * The prime p is composed as follows:<br>
	 * p=qr+1, where q is the prime factor, and r is the cofactor. 
	 * <p>
	 * @param primeSize the size (in bits) of the prime p
	 * @param factorSize the size (int bits) of the prime factor q of (p-1)
	 * @param cofactorType the type of check to do on cofactor r
	 * <ul>
	 * <li>cofactorType = 0: no extra check
	 * <li>cofactorType = 1: make sure that gcd(q,r) != 1
	 * <li>cofactorType = 2: make sure that (r/2) is probable prime
	 * <li>cofactorType = 3: make sure that r = 2 (this makes p a Sophie-German prime)
	 * </ul>
	 */
	public PrimeGenParameterSpec(int primeSize, int factorSize, int cofactorType)
	{
		if (primeSize < 0)
			throw new IllegalArgumentException("bad prime size");

		if (factorSize < 0 || factorSize >= primeSize)
			throw new IllegalArgumentException("bad factor size");

		if (cofactorType < 0 || cofactorType > 3)
			throw new IllegalArgumentException("bad cofactor type");

		if (cofactorType == 3)
			if (factorSize != primeSize-1)
				throw new IllegalArgumentException("bad factor size");

		this.primeSize = primeSize;
		this.factorSize = factorSize;
		this.cofactorType = cofactorType;
	}

	/**
	 * Returns the cofactor type.
	 * @return the cofactor type
	 */
	public int getCofactorType()
	{
		return cofactorType;
	}

	/**
	 * Returns the factor size (in bits).
	 * @return the factor size (in bits)
	 */
	public int getFactorSize()
	{
		return factorSize;
	}

	/**
	 * Returns the prime size (in bits).
	 * @return the prime size (in bits)
	 */
	public int getPrimeSize()
	{
		return primeSize;
	}
}
