/*
 * PrimeParameterSpec.java
 *
 * This class is part of our JCE 1.2 provider
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package com.beeyond.security.spec;

import java.math.BigInteger;
import java.security.*;
import java.security.spec.*;

/**
 * This class defines the parameters of secure prime p, with prime factor q of (p-1) and cofactor r, such that p = qr+1.
 * @author Bob Deblier &lt;bob@virtualunlimited.com&gt;
 */
public final class PrimeParameterSpec implements AlgorithmParameterSpec
{
	private BigInteger p;
	private BigInteger q;
	private BigInteger r;

	/**
	 * Constructs a PrimeParameterSpec from a prime, a prime factor and a cofactor.
	 * @param p the prime p
	 * @param q the prime factor q of (p-1)
	 * @param r the cofactor r of (p-1)
	 */
	public PrimeParameterSpec(BigInteger p, BigInteger q, BigInteger r)
	{
		if (!p.equals(q.multiply(r).add(BigInteger.ONE)))
			throw new IllegalArgumentException("p is not qr+1");

		this.p = p;
		this.q = q;
		this.r = r;
	}

	/**
	 * Returns the prime p.
	 * @return the prime p
	 */
	public BigInteger getP()
	{
		return p;
	}

	/**
	 * Returns the prime factor q of (p-1).
	 * @return the prime factor q of (p-1)
	 */
	public BigInteger getQ()
	{
		return q;
	}

	/**
	 * Returns the cofactor r of (p-1).
	 * @return the cofactor r of (p-1)
	 */
	public BigInteger getR()
	{
		return r;
	}
}
