
// _________________________________________________________________
//
//                          SBS CabWoman.c
//                Windows Cabinet File Manager V1.00
//                   01-20-1997 Sven B. Schreiber
//                          sbs@orgon.com
// _________________________________________________________________

#include <windows.h>
#include <guistart.h>
#include <cablib.h>
#include "CabWoman.h"

// =================================================================
// REVISION HISTORY
// =================================================================

/*

01-20-1997  V1.00  SBS  Original version.

*/

// =================================================================
// GLOBAL VARIABLES
// =================================================================

HINSTANCE hInst             = NULL;
HWND      hWndMain          = NULL;
BOOL      fMainDlgInit      = FALSE;

HMODULE   hCtl3d32          = NULL;
PROC      Ctl3dRegister     = NULL;
PROC      Ctl3dAutoSubclass = NULL;
PROC      Ctl3dUnregister   = NULL;

// =================================================================
// GLOBAL STRINGS
// =================================================================

BYTE abMainClass       [] = MAIN_CLASS;
BYTE abMainCaption     [] = MAIN_CAPTION;
BYTE abMainError       [] = MAIN_ERROR;
BYTE abMainIcon        [] = MAIN_ICON;

BYTE abVolume          [] = "Vol";
BYTE abVolumes         [] = "Vol";
BYTE abDirectory       [] = "Directory";
BYTE abDirectories     [] = "Directories";
BYTE abCabinet         [] = "Cabinet";
BYTE abCabinets        [] = "Cabinets";
BYTE abFile            [] = "File";
BYTE abFiles           [] = "Files";

BYTE abSectionDefaults [] = "Defaults";
BYTE abKeyVolume       [] = "Volume";
BYTE abKeyDestination  [] = "Destination";
BYTE abExtracting      [] = "Extracting";
BYTE abNull            [] = "";

BYTE abExeFile [MAX_PATH] = "";
BYTE abIniFile [MAX_PATH] = "";

// =================================================================
// GLOBAL TABLES
// =================================================================

BYTE UCase [] =
    {
      0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
     16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
     32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
     48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
     64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
     80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
     96, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
     80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90,123,124,125,126,127,
    128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
    144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
    160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
    176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
    192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
    208,209,210,211,212,213,214,215,216,217,218,219,220,221,254,223,
    192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
    208,209,210,211,212,213,214,247,216,217,218,219,220,221,254,159
    };

// =================================================================
// AUXILIARY ROUTINES
// =================================================================

BOOL LoadSystemDll (PHMODULE phModule,
                    PBYTE    pbModule,
                    DWORD    dProcedures,
                    ...)
    {
    PDWORD          pdArg = (&dProcedures) + 1;
    BYTE            abModule [MAX_PATH];
    HANDLE          hFindFile;
    WIN32_FIND_DATA wfd;
    DWORD           i;
    BOOL            fOK = FALSE;

    *phModule = NULL;
    for (i = 0; i < dProcedures; i++)
        {
        *((PPROC) pdArg [i+i]) = NULL;
        }
    i = GetSystemDirectory (abModule, MAX_PATH);
    if (i && (i < MAX_PATH))
        {
        if (abModule [i-1] != '\\')
            {
            abModule [i++] = '\\';
            }
        if ((i + lstrlen (pbModule)) < MAX_PATH)
            {
            lstrcpy (abModule+i, pbModule);
            hFindFile = FindFirstFile (abModule, &wfd);
            if (hFindFile != INVALID_HANDLE_VALUE)
                {
                FindClose (hFindFile);
                *phModule = LoadLibrary (abModule);
                if (*phModule != NULL)
                    {
                    fOK = TRUE;
                    for (i = 0; i < dProcedures; i++)
                        {
                        *((PPROC) pdArg [i+i]) =
                            GetProcAddress (*phModule,
                                            (PBYTE) pdArg [i+i+1]);
                        fOK = (fOK &&
                               (*((PPROC) pdArg [i+i]) != NULL));
                        }
                    }
                }
            }
        }
    return fOK;
    }

// -----------------------------------------------------------------

BOOL UnloadSystemDll (PHMODULE phModule)
    {
    BOOL fOK;

    if (fOK = (*phModule != NULL))
        {
        FreeLibrary (*phModule);
        *phModule = NULL;
        }
    return fOK;
    }

// -----------------------------------------------------------------

DWORD WINAPI GetFile (PBYTE pbSource,
                      PBYTE pbItem,
                      PBYTE pbTarget,
                      DWORD dBytes)
    {
    DWORD i, j, k, n;

    i = (pbSource [0] ? (pbSource [1] == ':' ? 2 : 0) : 0);
    k = j = i;
    do
        {
        if (pbSource [i] == '.' )     j = i;
        if (pbSource [i] == '\\') k = j = i;
        }
    while (pbSource [i++]);

    if (pbSource [j] != '.' ) j = i - 1;
    if (pbSource [k] == '\\') k++;

    i = (pbItem [0] == '.' ? j : k);
    if ((n = i + lstrlen (pbItem)) < dBytes)
        {
        lstrcpyn (pbTarget,   pbSource, i+1);
        lstrcpy  (pbTarget+i, pbItem);
        }
    else
        {
        if ((n = 0) < dBytes) pbTarget [0] = 0;
        }
    return n;
    }

// -----------------------------------------------------------------

PBYTE WINAPI GetFileName (PBYTE pbPath)
    {
    DWORD i, j;

    j = i = (pbPath [0] ? (pbPath [1] == ':' ? 2 : 0) : 0);

    while (pbPath [i++])
        {
        if (pbPath [i-1] == '\\') j = i;
        }
    return pbPath+j;
    }

// -----------------------------------------------------------------

BOOL WINAPI SetDirectory (PBYTE pbVolume,
                          PBYTE pbDirectory)
    {
    BYTE  abPath [MAX_PATH];
    DWORD dSize;
    BOOL  fOK = FALSE;

    if (SetCurrentDirectory (pbDirectory) ||
        SetCurrentDirectory (pbVolume))
        {
        dSize = GetCurrentDirectory (MAX_PATH, abPath);

        if ((dSize < MAX_PATH) && dSize)
            {
            WritePrivateProfileString (abSectionDefaults,
                                       abKeyVolume,
                                       pbVolume,
                                       GetFileName (abIniFile));

            WritePrivateProfileString (abSectionDefaults,
                                       pbVolume,
                                       abPath,
                                       GetFileName (abIniFile));

            fOK = TRUE;
            }
        }
    return fOK;
    }

// -----------------------------------------------------------------

BOOL WINAPI SetDefaultDirectory (PBYTE pbVolume)
    {
    BYTE  abPath [MAX_PATH];
    DWORD dSize;

    dSize = GetPrivateProfileString (abSectionDefaults,
                                     pbVolume,
                                     abNull,
                                     abPath,
                                     MAX_PATH,
                                     GetFileName (abIniFile));

    return SetDirectory (pbVolume, (dSize ? abPath : pbVolume));
    }

// -----------------------------------------------------------------

BYTE WINAPI GetDefaultVolume (void)
    {
    BYTE  abPath [MAX_PATH];
    DWORD dSize;

    dSize = GetPrivateProfileString (abSectionDefaults,
                                     abKeyVolume,
                                     abNull,
                                     abPath,
                                     MAX_PATH,
                                     GetFileName (abIniFile));

    if ((dSize < 2) || (abPath [1] != ':'))
        {
        dSize = GetCurrentDirectory (MAX_PATH, abPath);

        if ((dSize >= MAX_PATH) ||
            (dSize < 2) || (abPath [1] != ':'))
            {
            dSize = 0;
            }
        }
    return UCase [(dSize ? abPath [0] : 'C')];
    }

// -----------------------------------------------------------------

BOOL printfDlgItem (HWND  hWnd,
                    int   iDlgItem,
                    PBYTE pbFormat,
                    ...)
    {
    BYTE abBuffer [MAX_MESSAGE];

    wvsprintf (abBuffer, pbFormat, (PVOID) ((&pbFormat)+1));
    return SetDlgItemText (hWnd, iDlgItem, abBuffer);
    }

// -----------------------------------------------------------------

BOOL ClearDlgItem (HWND  hWnd,
                   int   iDlgItem)
    {
    return SetDlgItemText (hWnd, iDlgItem, abNull);
    }

// =================================================================
// WINDOW MANAGEMENT
// =================================================================

BOOL WINAPI CenterWindow (HWND hWnd)
    {
    RECT SysRect, WndRect;

    return ((hWnd != NULL) &&
            GetWindowRect (GetDesktopWindow (), &SysRect) &&
            GetWindowRect (hWnd, &WndRect) &&
            MoveWindow (hWnd,
                        ((SysRect.right - SysRect.left) -
                         (WndRect.right - WndRect.left)) / 2,
                        ((SysRect.bottom - SysRect.top) -
                         (WndRect.bottom - WndRect.top)) / 2,
                        WndRect.right  - WndRect.left,
                        WndRect.bottom - WndRect.top,
                        TRUE));
    }

// -----------------------------------------------------------------

LRESULT WINAPI Dispatcher (DISPATCH  Dispatch [],
                           DWORD     dEvent,
                           HWND      hWnd,
                           UINT      uMsg,
                           WPARAM    wParam,
                           LPARAM    lParam)
    {
    DWORD i;

    i = 0;
    while ((Dispatch [i].dEvent != DEFAULT_HANDLER) &&
           (Dispatch [i].dEvent != DEFAULT_RESULT)  &&
           (Dispatch [i].dEvent != dEvent)) i++;

    return (Dispatch [i].dEvent != DEFAULT_RESULT
            ? Dispatch [i].xHandler (hWnd, uMsg, wParam, lParam)
            : (LRESULT) Dispatch [i].xHandler);
    }

// =================================================================
// LIST BOX MANAGEMENT
// =================================================================

void WINAPI ClearListBox (HWND hWnd,
                          int  iDlgItem)
    {
    HWND hListBox = GetDlgItem (hWnd, iDlgItem);

    SendDlgItemMessage (hWnd,
                        iDlgItem,
                        LB_RESETCONTENT,
                        (WPARAM) 0,
                        (LPARAM) 0);

    if (hListBox != NULL) UpdateWindow (hListBox);
    return;
    }

// -----------------------------------------------------------------

DWORD WINAPI GetListBoxCount (HWND hWnd,
                              int  iDlgItem)
    {
    DWORD dCount;

    dCount = SendDlgItemMessage (hWnd,
                                 iDlgItem,
                                 LB_GETCOUNT,
                                 (WPARAM) 0,
                                 (LPARAM) 0);

    if (dCount == LB_ERR) dCount = 0;
    return dCount;
    }

// -----------------------------------------------------------------

DWORD WINAPI AddListBoxItem (HWND  hWnd,
                             int   iDlgItem,
                             PBYTE pbText,
                             DWORD dData)
    {
    DWORD dIndex;

    dIndex = SendDlgItemMessage (hWnd,
                                 iDlgItem,
                                 LB_ADDSTRING,
                                 (WPARAM) 0,
                                 (LPARAM) pbText);

    if ((dIndex == LB_ERR) || (dIndex == LB_ERRSPACE))
        {
        dIndex = MAXDWORD;
        }
    else
        {
        SendDlgItemMessage (hWnd,
                            iDlgItem,
                            LB_SETITEMDATA,
                            (WPARAM) dIndex,
                            (LPARAM) dData);
        }
    return dIndex;
    }

// -----------------------------------------------------------------

DWORD WINAPI SelectListBoxItem (HWND  hWnd,
                                int   iDlgItem,
                                DWORD dSelect)
    {
    HWND   hListBox = GetDlgItem (hWnd, iDlgItem);
    DWORD  dCount, dIndex;

    dCount = GetListBoxCount (hWnd, iDlgItem);

    dIndex = ((dSelect < dCount) || (dSelect == MAXDWORD)
              ? dSelect
              : dCount-1);

    SendDlgItemMessage (hWnd,
                        iDlgItem,
                        LB_SETCURSEL,
                        (WPARAM) dIndex,
                        (LPARAM) 0);

    if (hListBox != NULL)
        {
        PostMessage (hWnd,
                     WM_COMMAND,
                     (WPARAM) MAKELONG (iDlgItem, LBN_SELCHANGE),
                     (LPARAM) hListBox);
        }
    return dCount;
    }

// -----------------------------------------------------------------

void WINAPI DoubleClickListBox (HWND hWnd,
                                int  iDlgItem)
    {
    HWND hListBox = GetDlgItem (hWnd, iDlgItem);

    if (hListBox != NULL)
        {
        PostMessage (hWnd,
                     WM_COMMAND,
                     (WPARAM) MAKELONG (iDlgItem, LBN_DBLCLK),
                     (LPARAM) hListBox);
        }
    return;
    }

// -----------------------------------------------------------------

DWORD WINAPI FindListBoxText (HWND  hWnd,
                              int   iDlgItem,
                              PBYTE pbText)
    {
    DWORD dIndex;

    dIndex = SendDlgItemMessage (hWnd,
                                 iDlgItem,
                                 LB_FINDSTRINGEXACT,
                                 (WPARAM) -1,
                                 (LPARAM) pbText);

    if (dIndex == LB_ERR) dIndex = MAXDWORD;
    return dIndex;
    }

// -----------------------------------------------------------------

DWORD WINAPI GetListBoxText (HWND  hWnd,
                             int   iDlgItem,
                             PBYTE pbText,
                             DWORD dBytes)
    {
    PBYTE pbBuffer;
    DWORD dIndex, dSize;

    dSize = 0;

    if (dBytes)
        {
        pbText [0] = 0;

        dIndex = SendDlgItemMessage (hWnd,
                                     iDlgItem,
                                     LB_GETCURSEL,
                                     (WPARAM) 0,
                                     (LPARAM) 0);
        if (dIndex != LB_ERR)
            {
            dSize = SendDlgItemMessage (hWnd,
                                        iDlgItem,
                                        LB_GETTEXTLEN,
                                        (WPARAM) dIndex,
                                        (LPARAM) 0);

            if ((dSize != LB_ERR) || (dSize = 0))
                {
                pbBuffer = LocalAlloc (LMEM_FIXED, dSize+1);
                if (pbBuffer != NULL)
                    {
                    dSize = SendDlgItemMessage (hWnd,
                                                iDlgItem,
                                                LB_GETTEXT,
                                                (WPARAM) dIndex,
                                                (LPARAM) pbBuffer);

                    if ((dSize != LB_ERR) || (dSize = 0))
                        {
                        lstrcpyn (pbText, pbBuffer, dBytes);
                        }
                    LocalFree (pbBuffer);
                    }
                }
            }
        }
    return dSize;
    }

// -----------------------------------------------------------------

DWORD WINAPI GetListBoxData (HWND hWnd,
                             int  iDlgItem)
    {
    DWORD dIndex, dData;

    dIndex = SendDlgItemMessage (hWnd,
                                 iDlgItem,
                                 LB_GETCURSEL,
                                 (WPARAM) 0,
                                 (LPARAM) 0);
    if (dIndex != LB_ERR)
        {
        dData = SendDlgItemMessage (hWnd,
                                    iDlgItem,
                                    LB_GETITEMDATA,
                                    (WPARAM) dIndex,
                                    (LPARAM) 0);
        }
    else
        {
        dData = MAXDWORD;
        }
    return dData;
    }

// -----------------------------------------------------------------

HCURSOR WINAPI OpenListBox (HWND hWnd,
                            int  iDlgItem)
    {
    HCURSOR hCursor;

    hCursor = SetCursor (LoadCursor (NULL, IDC_WAIT));
    ClearListBox (hWnd, iDlgItem);

    SendDlgItemMessage (hWnd,
                        iDlgItem,
                        WM_SETREDRAW,
                        (WPARAM) FALSE,
                        (LPARAM) 0);
    return hCursor;
    }

// -----------------------------------------------------------------

DWORD WINAPI CloseListBox (HWND    hWnd,
                           int     iDlgItem,
                           DWORD   dSelect,
                           HCURSOR hCursor)
    {
    HWND  hListBox = GetDlgItem (hWnd, iDlgItem);
    DWORD dCount;

    SendDlgItemMessage (hWnd,
                        iDlgItem,
                        WM_SETREDRAW,
                        (WPARAM) TRUE,
                        (LPARAM) 0);

    dCount = SelectListBoxItem (hWnd, iDlgItem, dSelect);
    if (hListBox != NULL) UpdateWindow (hListBox);
    SetCursor (hCursor);
    return dCount;
    }

// -----------------------------------------------------------------

DWORD WINAPI EnumListBoxItems (HWND  hWnd,
                               int   iDlgItem,
                               PROC  Handler,
                               PVOID pData)
    {
    PBYTE pbBuffer;
    DWORD dCount, dOK, dIndex, dSize, dMaxSize;

    if (dCount = GetListBoxCount (hWnd, iDlgItem))
        {
        dMaxSize = 0;
        dOK      = 0;

        for (dIndex = 0; dIndex < dCount; dIndex++)
            {
            dSize = SendDlgItemMessage (hWnd,
                                        iDlgItem,
                                        LB_GETTEXTLEN,
                                        (WPARAM) dIndex,
                                        (LPARAM) 0);
            if (dSize != LB_ERR)
                {
                dMaxSize = max (dMaxSize, dSize);
                dOK++;
                }
            }
        if (dOK == dCount)
            {
            pbBuffer = LocalAlloc (LMEM_FIXED, ++dMaxSize);
            if (pbBuffer != NULL)
                {
                for (dIndex = 0; dIndex < dCount; dIndex++)
                    {
                    dSize = SendDlgItemMessage (hWnd,
                                                iDlgItem,
                                                LB_GETTEXT,
                                                (WPARAM) dIndex,
                                                (LPARAM) pbBuffer);
                    if (dSize != LB_ERR)
                        {
                        dOK = Handler (pbBuffer,
                                       dSize,
                                       dIndex,
                                       pData);

                        SendDlgItemMessage (hWnd,
                                            iDlgItem,
                                            LB_SETITEMDATA,
                                            (WPARAM) dIndex,
                                            (LPARAM) dOK);
                        }
                    }
                }
            }
        }
    return dCount;
    }

// =================================================================
// BROWSING
// =================================================================

DWORD WINAPI ListVolumes (HWND hWnd,
                          int  iCaption,
                          int  iListBox)
    {
    BYTE    abVolume [] = "?:";
    BYTE    abRoot   [] = "?:\\";
    HCURSOR hCursor;
    DWORD   dVolumes, dMask, dIndex, dCount;

    SetDlgItemText (hWnd, iCaption, abVolumes);

    hCursor  = OpenListBox (hWnd, iListBox);
    dMask    = 1;
    dVolumes = GetLogicalDrives ();

    for (abVolume [0] = 'A'; abVolume [0] <= 'Z'; abVolume [0]++)
        {
        if (dVolumes & dMask)
            {
            abRoot [0] = abVolume [0];

            AddListBoxItem (hWnd,
                            iListBox,
                            abVolume,
                            GetDriveType (abRoot));
            }
        dMask <<= 1;
        }
    abVolume [0] = GetDefaultVolume ();
    dIndex = FindListBoxText (hWnd, iListBox, abVolume);
    dCount = CloseListBox (hWnd, iListBox, dIndex, hCursor);

    printfDlgItem (hWnd,
                   iCaption,
                   "%lu &%s",
                   dCount,
                   (dCount == 1 ? abVolume : abVolumes));
    return dCount;
    }

// -----------------------------------------------------------------

DWORD WINAPI ListDirectories (HWND hWnd,
                              int  iCaption,
                              int  iListBox)
    {
    BYTE            abPath [MAX_PATH];
    HANDLE          hFindFile;
    WIN32_FIND_DATA FindData;
    HCURSOR         hCursor;
    DWORD           dCount;

    GetCurrentDirectory (MAX_PATH, abPath);
    SetDlgItemText (hWnd, IDT_PATH, abPath);
    SetDlgItemText (hWnd, iCaption, abDirectories);

    hCursor   = OpenListBox (hWnd, iListBox);
    hFindFile = FindFirstFile ("*", &FindData);

    while (hFindFile != INVALID_HANDLE_VALUE)
        {
        if (FindData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
            {
            if (lstrcmp (FindData.cFileName, "." ) &&
                lstrcmp (FindData.cFileName, ".."))
                {
                AddListBoxItem (hWnd,
                                iListBox,
                                FindData.cFileName,
                                MAXDWORD);
                }
            }
        if (!FindNextFile (hFindFile, &FindData))
            {
            FindClose (hFindFile);
            hFindFile = INVALID_HANDLE_VALUE;
            }
        }
    if (lstrcmp (abPath+1, ":\\"))
        {
        AddListBoxItem (hWnd, iListBox, "..", MAXDWORD);
        }
    dCount = CloseListBox (hWnd, iListBox, 0, hCursor);

    printfDlgItem (hWnd,
                   iCaption,
                   "%lu &%s",
                   dCount,
                   (dCount == 1 ? abDirectory : abDirectories));
    return dCount;
    }

// -----------------------------------------------------------------

DWORD WINAPI ListCabinets (HWND hWnd,
                           int  iCaption,
                           int  iListBox)
    {
    HANDLE          hFindFile;
    WIN32_FIND_DATA FindData;
    HCURSOR         hCursor;
    DWORD           dCount;

    SetDlgItemText (hWnd, iCaption, abCabinets);

    hCursor   = OpenListBox (hWnd, iListBox);
    hFindFile = FindFirstFile ("*", &FindData);

    while (hFindFile != INVALID_HANDLE_VALUE)
        {
        if (!(FindData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
            {
            if (TestCabinetHeader (FindData.cFileName))
                {
                AddListBoxItem (hWnd,
                                iListBox,
                                FindData.cFileName,
                                MAXDWORD);
                }
            }
        if (!FindNextFile (hFindFile, &FindData))
            {
            FindClose (hFindFile);
            hFindFile = INVALID_HANDLE_VALUE;
            }
        }
    dCount = CloseListBox (hWnd, iListBox, 0, hCursor);

    printfDlgItem (hWnd,
                   iCaption,
                   "%lu &%s",
                   dCount,
                   (dCount == 1 ? abCabinet : abCabinets));
    return dCount;
    }

// -----------------------------------------------------------------

DWORD WINAPI ListCabinetDir (PBYTE    pbCabinet,
                             DWORD    dBytes,
                             DWORD    dIndex,
                             PLISTCTL pListCtl)
    {
    HCABINET  hCabinet;
    CAB_ENTRY ce;
    BYTE      abFile [MAX_PATH];
    DWORD     dOffset = MAXDWORD;

    if ((hCabinet = OpenCabinet (pbCabinet)) != NULL)
        {
        while (ScanCabinet (hCabinet, &ce, abFile, MAX_PATH))
            {
            if (ce.iFolder != CAB_FILE_CONTINUED)
                {
                AddListBoxItem (pListCtl->hWnd,
                                pListCtl->iListBox,
                                abFile,
                                dIndex);
                }
            }
        dOffset = CloseCabinet (hCabinet);
        }
    return dOffset;
    }

// -----------------------------------------------------------------

DWORD WINAPI ListFiles (HWND hWnd,
                        int  iCaption,
                        int  iListBox)
    {
    LISTCTL ListCtl = {hWnd, iCaption, iListBox};
    HCURSOR hCursor;
    DWORD   dCount;

    SetDlgItemText (hWnd, iCaption, abFiles);

    hCursor = OpenListBox (hWnd, iListBox);

    dCount = EnumListBoxItems (hWnd,
                               IDL_CABINETS,
                               ListCabinetDir,
                               &ListCtl);

    dCount = CloseListBox (hWnd, iListBox, 0, hCursor);

    printfDlgItem (hWnd,
                   iCaption,
                   "%lu &%s",
                   dCount,
                   (dCount == 1 ? abFile : abFiles));
    return dCount;
    }

// =================================================================
// ABOUT DIALOG
// =================================================================

LRESULT CALLBACK AboutDlgInitDialog (HWND   hWnd,
                                     UINT   uMsg,
                                     WPARAM wParam,
                                     LPARAM lParam)
    {
    CenterWindow (hWnd);
    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK AboutDlgCtlColorEdit (HWND   hWnd,
                                       UINT   uMsg,
                                       WPARAM wParam,
                                       LPARAM lParam)
    {
    return (LRESULT) GetStockObject (LTGRAY_BRUSH);
    }

// -----------------------------------------------------------------

LRESULT CALLBACK AboutCmdEndDialog (HWND   hWnd,
                                    UINT   uMsg,
                                    WPARAM wParam,
                                    LPARAM lParam)
    {
    EndDialog (hWnd, (WORD) wParam);
    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

DISPATCH AboutCmd [] = {IDOK,           AboutCmdEndDialog,
                        IDCANCEL,       AboutCmdEndDialog,
                        DEFAULT_RESULT, FALSE};

LRESULT CALLBACK AboutDlgCommand (HWND   hWnd,
                                  UINT   uMsg,
                                  WPARAM wParam,
                                  LPARAM lParam)
    {
    return Dispatcher (AboutCmd, (WORD) wParam,
                       hWnd, uMsg, wParam, lParam);
    }

// -----------------------------------------------------------------

DISPATCH AboutDlg [] = {WM_INITDIALOG,   AboutDlgInitDialog,
                        WM_CTLCOLOREDIT, AboutDlgCtlColorEdit,
                        WM_COMMAND,      AboutDlgCommand,
                        DEFAULT_RESULT,  FALSE};

LRESULT CALLBACK AboutDlgProc (HWND   hWnd,
                               UINT   uMsg,
                               WPARAM wParam,
                               LPARAM lParam)
    {
    return Dispatcher (AboutDlg, uMsg,
                       hWnd, uMsg, wParam, lParam);
    }

// =================================================================
// INFO DIALOG
// =================================================================

LRESULT CALLBACK InfoDlgInitDialog (HWND   hWnd,
                                    UINT   uMsg,
                                    WPARAM wParam,
                                    LPARAM lParam)
    {
    BYTE     abCabinet [MAX_PATH];
    HCABINET hc;

    CenterWindow (hWnd);

    if (GetListBoxText ((HWND) lParam,
                        IDL_CABINETS,
                        abCabinet,
                        MAX_PATH))
        {
        printfDlgItem (hWnd, IDT_THIS, "%s", abCabinet);

        if ((hc = OpenCabinet (abCabinet)) != NULL)
            {
            printfDlgItem (hWnd,
                           IDT_PREVIOUS,
                           "%s%s%s%s",
                           hc->pbPrevCab,
                           (hc->pbPrevDisk [0] ? "  [" : abNull),
                           hc->pbPrevDisk,
                           (hc->pbPrevDisk [0] ?   "]" : abNull));

            printfDlgItem (hWnd,
                           IDT_NEXT,
                           "%s%s%s%s",
                           hc->pbNextCab,
                           (hc->pbNextDisk [0] ? "  [" : abNull),
                           hc->pbNextDisk,
                           (hc->pbNextDisk [0] ?   "]" : abNull));

            printfDlgItem (hWnd,
                           IDT_RESERVED,
                           "%lu",
                           hc->dReserve);

            printfDlgItem (hWnd,
                           IDT_SET_ID,
                           "%lu",
                           hc->dSetId);

            printfDlgItem (hWnd,
                           IDT_CAB_NUMBER,
                           "%lu",
                           hc->dCabNumber);

            printfDlgItem (hWnd,
                           IDT_NFOLDERS,
                           "%lu",
                           hc->dFolders);

            printfDlgItem (hWnd,
                           IDT_NFILES,
                           "%lu",
                           hc->dFiles);

            printfDlgItem (hWnd,
                           IDT_EXTRA_DATA,
                           "0x%08X",
                           hc->dExtraData);

            printfDlgItem (hWnd,
                           IDT_CAB_LIST,
                           "0x%08X",
                           hc->dCabList);

            printfDlgItem (hWnd,
                           IDT_FOLDER_LIST,
                           "0x%08X",
                           hc->dFolderList);

            printfDlgItem (hWnd,
                           IDT_FILE_LIST,
                           "0x%08X",
                           hc->dFileList);

            printfDlgItem (hWnd,
                           IDT_FILE_DATA,
                           "0x%08X",
                           hc->dFileData);

            printfDlgItem (hWnd,
                           IDT_CAB_SIZE,
                           "0x%08X",
                           hc->dCabSize);

            CloseCabinet (hc);
            }
        }
    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK InfoDlgCtlColorEdit (HWND   hWnd,
                                      UINT   uMsg,
                                      WPARAM wParam,
                                      LPARAM lParam)
    {
    return (LRESULT) GetStockObject (LTGRAY_BRUSH);
    }

// -----------------------------------------------------------------

LRESULT CALLBACK InfoCmdEndDialog (HWND   hWnd,
                                   UINT   uMsg,
                                   WPARAM wParam,
                                   LPARAM lParam)
    {
    EndDialog (hWnd, (WORD) wParam);
    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

DISPATCH InfoCmd [] = {IDOK,           InfoCmdEndDialog,
                       IDCANCEL,       InfoCmdEndDialog,
                       DEFAULT_RESULT, FALSE};

LRESULT CALLBACK InfoDlgCommand (HWND   hWnd,
                                 UINT   uMsg,
                                 WPARAM wParam,
                                 LPARAM lParam)
    {
    return Dispatcher (InfoCmd, (WORD) wParam,
                       hWnd, uMsg, wParam, lParam);
    }

// -----------------------------------------------------------------

DISPATCH InfoDlg [] = {WM_INITDIALOG,   InfoDlgInitDialog,
                       WM_CTLCOLOREDIT, InfoDlgCtlColorEdit,
                       WM_COMMAND,      InfoDlgCommand,
                       DEFAULT_RESULT,  FALSE};

LRESULT CALLBACK InfoDlgProc (HWND   hWnd,
                              UINT   uMsg,
                              WPARAM wParam,
                              LPARAM lParam)
    {
    return Dispatcher (InfoDlg, uMsg,
                       hWnd, uMsg, wParam, lParam);
    }

// =================================================================
// MAIN DIALOG
// =================================================================

LRESULT CALLBACK MainDlgInitDialog (HWND   hWnd,
                                    UINT   uMsg,
                                    WPARAM wParam,
                                    LPARAM lParam)
    {
    fMainDlgInit = TRUE;
    CenterWindow (hWnd);
    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainDlgPaint (HWND   hWnd,
                               UINT   uMsg,
                               WPARAM wParam,
                               LPARAM lParam)
    {
    LRESULT Result = DefWindowProc (hWnd, uMsg, wParam, lParam);

    if (fMainDlgInit)
        {
        fMainDlgInit = FALSE;
        UpdateWindow (hWnd);
        ListVolumes (hWnd, IDT_VOLUMES, IDL_VOLUMES);
        DoubleClickListBox (hWnd, IDL_VOLUMES);
        }
    return Result;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainDlgCtlColorEdit (HWND   hWnd,
                                      UINT   uMsg,
                                      WPARAM wParam,
                                      LPARAM lParam)
    {
    return (LRESULT) GetStockObject (LTGRAY_BRUSH);
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainDlgCtlColorListBox (HWND   hWnd,
                                         UINT   uMsg,
                                         WPARAM wParam,
                                         LPARAM lParam)
    {
    return (LRESULT) GetStockObject (WHITE_BRUSH);
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainDlgClose (HWND   hWnd,
                               UINT   uMsg,
                               WPARAM wParam,
                               LPARAM lParam)
    {
    HWND hButton = GetDlgItem (hWnd, IDB_EXIT);

    if (hButton != NULL)
        {
        PostMessage (hWnd,
                     WM_COMMAND,
                     (WPARAM) MAKELONG (IDB_EXIT, BN_CLICKED),
                     (LPARAM) hButton);
        }
    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainCmdVolumes (HWND   hWnd,
                                 UINT   uMsg,
                                 WPARAM wParam,
                                 LPARAM lParam)
    {
    DWORD dVolType;
    PBYTE pbVolType;
    BYTE  abVolume [] = "?:";

    if ((wParam >> 16) == LBN_SELCHANGE)
        {
        dVolType = GetListBoxData (hWnd, IDL_VOLUMES);
        switch (dVolType)
            {
            case DRIVE_REMOVABLE: pbVolType = "Removable"; break;
            case DRIVE_FIXED:     pbVolType = "Fixed";     break;
            case DRIVE_REMOTE:    pbVolType = "Remote";    break;
            case DRIVE_CDROM:     pbVolType = "CD-ROM";    break;
            case DRIVE_RAMDISK:   pbVolType = "RAM Disk";  break;
            default:              pbVolType = "Unknown";   break;
            }
        SetDlgItemText (hWnd, IDT_VOLTYPE, pbVolType);
        return (LRESULT) TRUE;
        }
    if ((wParam >> 16) == LBN_DBLCLK)
        {
        if (GetListBoxText (hWnd,
                            IDL_VOLUMES,
                            abVolume,
                            sizeof (abVolume)))
            {
            SetDefaultDirectory (abVolume);

            ListDirectories (hWnd,
                             IDT_DIRECTORIES,
                             IDL_DIRECTORIES);

            ListCabinets (hWnd,
                          IDT_CABINETS,
                          IDL_CABINETS);

            ListFiles (hWnd,
                       IDT_FILES,
                       IDL_FILES);
            }
        return (LRESULT) TRUE;
        }
    return (LRESULT) FALSE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainCmdDirectories (HWND   hWnd,
                                     UINT   uMsg,
                                     WPARAM wParam,
                                     LPARAM lParam)
    {
    BYTE abVolume [] = "?:";
    BYTE abDirectory [MAX_PATH];

    if ((wParam >> 16) == LBN_SELCHANGE)
        {
        return (LRESULT) TRUE;
        }
    if ((wParam >> 16) == LBN_DBLCLK)
        {
        if (GetListBoxText (hWnd,
                            IDL_VOLUMES,
                            abVolume,
                            sizeof (abVolume))
            &&
            GetListBoxText (hWnd,
                            IDL_DIRECTORIES,
                            abDirectory,
                            MAX_PATH))
            {
            SetDirectory (abVolume, abDirectory);

            ListDirectories (hWnd,
                             IDT_DIRECTORIES,
                             IDL_DIRECTORIES);

            ListCabinets (hWnd,
                          IDT_CABINETS,
                          IDL_CABINETS);

            ListFiles (hWnd,
                       IDT_FILES,
                       IDL_FILES);
            }
        return (LRESULT) TRUE;
        }
    return (LRESULT) FALSE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainCmdCabinets (HWND   hWnd,
                                  UINT   uMsg,
                                  WPARAM wParam,
                                  LPARAM lParam)
    {
    if ((wParam >> 16) == LBN_SELCHANGE)
        {
        return (LRESULT) TRUE;
        }
    if ((wParam >> 16) == LBN_DBLCLK)
        {
        DialogBoxParam (hInst,
                        INFO_DIALOG,
                        hWnd,
                        InfoDlgProc,
                        (LPARAM) hWnd);

        return (LRESULT) TRUE;
        }
    return (LRESULT) FALSE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainCmdFiles (HWND   hWnd,
                               UINT   uMsg,
                               WPARAM wParam,
                               LPARAM lParam)
    {
    CAB_ENTRY  ce;
    FILETIME   ft;
    SYSTEMTIME st;
    BYTE       abCabinet  [MAX_PATH];
    BYTE       abEntry    [MAX_PATH];
    BYTE       abCaption1 [MAX_MESSAGE];
    BYTE       abCaption2 [MAX_MESSAGE];
    DWORD      dOffset;

    if ((wParam >> 16) == LBN_SELCHANGE)
        {
        SelectListBoxItem (hWnd,
                           IDL_CABINETS,
                           GetListBoxData (hWnd, IDL_FILES));

        if (GetListBoxText (hWnd,
                            IDL_CABINETS,
                            abCabinet,
                            MAX_PATH)
            &&
            GetListBoxText (hWnd,
                            IDL_FILES,
                            abEntry,
                            MAX_PATH)
            &&
            ((dOffset =
              GetListBoxData (hWnd, IDL_CABINETS)) != MAXDWORD)
            &&
            FindCabinetEntry (abCabinet, abEntry, dOffset, &ce))
            {
            DosDateTimeToFileTime (ce.date, ce.time, &ft);
            FileTimeToSystemTime (&ft, &st);

            printfDlgItem (hWnd,
                           IDT_SIZE,
                           "%lu",
                           ce.cbFile);

            printfDlgItem (hWnd,
                           IDT_DATE,
                           "%02lu-%02lu-%04lu",
                           st.wMonth,
                           st.wDay,
                           st.wYear);

            printfDlgItem (hWnd,
                           IDT_TIME,
                           "%02lu:%02lu:%02lu",
                           st.wHour,
                           st.wMinute,
                           st.wSecond);

            printfDlgItem (hWnd,
                           IDT_ATTRIBUTES,
                           "%c %c %c %c %c %c",
                           (ce.attribs & CAB_ATTRIB_ARCHIVE
                            ? 'a'
                            : '-'),
                           (ce.attribs & CAB_ATTRIB_DIRECTORY
                            ? 'd'
                            : '-'),
                           (ce.attribs & CAB_ATTRIB_VOLUME
                            ? 'v'
                            : '-'),
                           (ce.attribs & CAB_ATTRIB_SYSTEM
                            ? 's'
                            : '-'),
                           (ce.attribs & CAB_ATTRIB_HIDDEN
                            ? 'h'
                            : '-'),
                           (ce.attribs & CAB_ATTRIB_READONLY
                            ? 'r'
                            : '-'));
            }
        else
            {
            ClearDlgItem (hWnd, IDT_SIZE);
            ClearDlgItem (hWnd, IDT_DATE);
            ClearDlgItem (hWnd, IDT_TIME);
            ClearDlgItem (hWnd, IDT_ATTRIBUTES);
            }
        return (LRESULT) TRUE;
        }
    if ((wParam >> 16) == LBN_DBLCLK)
        {
        SelectListBoxItem (hWnd,
                           IDL_CABINETS,
                           GetListBoxData (hWnd, IDL_FILES));

        if (GetListBoxText (hWnd,
                            IDL_CABINETS,
                            abCabinet,
                            MAX_PATH)
            &&
            GetListBoxText (hWnd,
                            IDL_FILES,
                            abEntry,
                            MAX_PATH))
            {
            GetWindowText (hWnd, abCaption1, MAX_MESSAGE);

            wsprintf (abCaption2,
                      "%s  -  %s \"%s\"",
                      abCaption1,
                      abExtracting,
                      abEntry);

            SetWindowText (hWnd, abCaption2);

            ExtractFileFromCabinet (hWnd,
                                    abCabinet,
                                    abEntry,
                                    GetFileName (abIniFile),
                                    abSectionDefaults,
                                    abKeyDestination);

            SetWindowText (hWnd, abCaption1);
            }
        return (LRESULT) TRUE;
        }
    return (LRESULT) FALSE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainCmdExtract (HWND   hWnd,
                                 UINT   uMsg,
                                 WPARAM wParam,
                                 LPARAM lParam)
    {
    DoubleClickListBox (hWnd, IDL_FILES);
    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainCmdAbout (HWND   hWnd,
                               UINT   uMsg,
                               WPARAM wParam,
                               LPARAM lParam)
    {
    DialogBox (hInst, ABOUT_DIALOG, hWnd, AboutDlgProc);
    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainCmdExit (HWND   hWnd,
                              UINT   uMsg,
                              WPARAM wParam,
                              LPARAM lParam)
    {
    EndDialog (hWnd, (WORD) wParam);
    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainCmdOk (HWND   hWnd,
                            UINT   uMsg,
                            WPARAM wParam,
                            LPARAM lParam)
    {
    HWND  hFocus;
    DWORD i;
    int   aiDlgItem [] = {IDL_VOLUMES,
                          IDL_DIRECTORIES,
                          IDL_CABINETS,
                          IDL_FILES,
                          0};

    hFocus = GetFocus ();
    if (hFocus != NULL)
        {
        i = 0;
        while (aiDlgItem [i++])
            {
            if (hFocus == GetDlgItem (hWnd, aiDlgItem [i-1]))
                {
                DoubleClickListBox (hWnd, aiDlgItem [i-1]);
                break;
                }
            }
        }
    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainCmdCancel (HWND   hWnd,
                                UINT   uMsg,
                                WPARAM wParam,
                                LPARAM lParam)
    {
    PostMessage (hWnd,
                 WM_NEXTDLGCTL,
                 (WPARAM) TRUE,
                 (LPARAM) FALSE);

    return (LRESULT) TRUE;
    }

// -----------------------------------------------------------------

DISPATCH MainCmd [] = {IDL_VOLUMES,     MainCmdVolumes,
                       IDL_DIRECTORIES, MainCmdDirectories,
                       IDL_CABINETS,    MainCmdCabinets,
                       IDL_FILES,       MainCmdFiles,
                       IDB_EXTRACT,     MainCmdExtract,
                       IDB_ABOUT,       MainCmdAbout,
                       IDB_EXIT,        MainCmdExit,
                       IDOK,            MainCmdOk,
                       IDCANCEL,        MainCmdCancel,
                       DEFAULT_RESULT,  FALSE};

LRESULT CALLBACK MainDlgCommand (HWND   hWnd,
                                 UINT   uMsg,
                                 WPARAM wParam,
                                 LPARAM lParam)
    {
    return Dispatcher (MainCmd, (WORD) wParam,
                       hWnd, uMsg, wParam, lParam);
    }

// -----------------------------------------------------------------

DISPATCH MainDlg [] = {WM_INITDIALOG,      MainDlgInitDialog,
                       WM_PAINT,           MainDlgPaint,
                       WM_CTLCOLOREDIT,    MainDlgCtlColorEdit,
                       WM_CTLCOLORLISTBOX, MainDlgCtlColorListBox,
                       WM_CLOSE,           MainDlgClose,
                       WM_COMMAND,         MainDlgCommand,
                       DEFAULT_RESULT,     FALSE};

LRESULT CALLBACK MainDlgProc (HWND   hWnd,
                              UINT   uMsg,
                              WPARAM wParam,
                              LPARAM lParam)
    {
    return Dispatcher (MainDlg, uMsg,
                       hWnd, uMsg, wParam, lParam);
    }

// =================================================================
// MAIN WINDOW
// =================================================================

LRESULT CALLBACK MainWndInit (HWND   hWnd,
                              UINT   uMsg,
                              WPARAM wParam,
                              LPARAM lParam)
    {
    DialogBox (hInst, MAIN_DIALOG, hWnd, MainDlgProc);
    PostMessage (hWnd, WM_CLOSE, (WPARAM) NULL, (LPARAM) NULL);
    return (LRESULT) NULL;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainWndQueryOpen (HWND   hWnd,
                                   UINT   uMsg,
                                   WPARAM wParam,
                                   LPARAM lParam)
    {
    DialogBox (hInst, ABOUT_DIALOG, hWnd, AboutDlgProc);
    return (LRESULT) FALSE;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK MainWndDestroy (HWND   hWnd,
                                 UINT   uMsg,
                                 WPARAM wParam,
                                 LPARAM lParam)
    {
    WNDCLASS WndClass;

    GetClassInfo (hInst, abMainClass, &WndClass);
    if (WndClass.hbrBackground != NULL)
        {
        DeleteObject (WndClass.hbrBackground);
        }
    PostQuitMessage (0);
    return (LRESULT) NULL;
    }

// -----------------------------------------------------------------

DISPATCH MainWnd [] = {WMU_INIT,        MainWndInit,
                       WM_QUERYOPEN,    MainWndQueryOpen,
                       WM_DESTROY,      MainWndDestroy,
                       DEFAULT_HANDLER, DefWindowProc};

LRESULT CALLBACK MainWndProc (HWND   hWnd,
                              UINT   uMsg,
                              WPARAM wParam,
                              LPARAM lParam)
    {
    return Dispatcher (MainWnd, uMsg,
                       hWnd, uMsg, wParam, lParam);
    }

// =================================================================
// MAIN PROGRAM
// =================================================================

HWND WINAPI CreateMainWindow (HINSTANCE hInstance,
                              PWNDCLASS pWndClass)
    {
    HWND hWndMain;

    pWndClass->style         = CS_HREDRAW | CS_VREDRAW;
    pWndClass->lpfnWndProc   = MainWndProc;
    pWndClass->cbClsExtra    = 0;
    pWndClass->cbWndExtra    = 0;
    pWndClass->hInstance     = hInstance;
    pWndClass->hIcon         = LoadIcon (hInstance, abMainIcon);
    pWndClass->hCursor       = LoadCursor (NULL, IDC_ARROW);
    pWndClass->lpszMenuName  = NULL;
    pWndClass->lpszClassName = abMainClass;
    pWndClass->hbrBackground = CreateSolidBrush
                                   (GetSysColor (COLOR_WINDOW));

    hWndMain = (RegisterClass (pWndClass)
                ? CreateWindow (abMainClass,
                                abMainCaption,
                                WS_OVERLAPPEDWINDOW,
                                CW_USEDEFAULT,
                                CW_USEDEFAULT,
                                CW_USEDEFAULT,
                                CW_USEDEFAULT,
                                NULL,
                                NULL,
                                hInstance,
                                NULL)
                : NULL);

    return hWndMain;
    }

// -----------------------------------------------------------------

BOOL WINAPI Initialize (HINSTANCE hInstance,
                        int       iCmdShow,
                        PWNDCLASS pWndClass)
    {
    OSVERSIONINFO ovi;
    BOOL          fOK = FALSE;

    hInst = hInstance;

    ovi.dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
    if (GetVersionEx (&ovi) &&
        (ovi.dwPlatformId == VER_PLATFORM_WIN32_NT) &&
        (ovi.dwMajorVersion < 4))
        {
        if (LoadSystemDll (&hCtl3d32, "CTL3D32.dll",
                           3,
                           &Ctl3dRegister,     "Ctl3dRegister",
                           &Ctl3dAutoSubclass, "Ctl3dAutoSubclass",
                           &Ctl3dUnregister,   "Ctl3dUnregister"))
            {
            Ctl3dRegister     (hInstance);
            Ctl3dAutoSubclass (hInstance);
            }
        else
            {
            UnloadSystemDll (&hCtl3d32);
            }
        }
    if (GetModuleFileName (hInstance, abExeFile, MAX_PATH) &&
        GetFile (abExeFile, ".ini", abIniFile, MAX_PATH))
        {
        hWndMain = CreateMainWindow (hInstance, pWndClass);
        if (hWndMain != NULL)
            {
            ShowWindow   (hWndMain, iCmdShow);
            UpdateWindow (hWndMain);
            fOK = TRUE;
            }
        }
    return fOK;
    }

// -----------------------------------------------------------------

BOOL WINAPI CleanUp (HINSTANCE hInstance)
    {
    BOOL fOK = TRUE;

    if (hCtl3d32 != NULL)
        {
        Ctl3dUnregister (hInstance);
        UnloadSystemDll (&hCtl3d32);
        }
    return fOK;
    }

// -----------------------------------------------------------------

int WINAPI WinMain (HINSTANCE hInstance,
                    HINSTANCE hPrevInstance,
                    PSTR      psCmdLine,
                    int       iShowCmd)
    {
    WNDCLASS WndClass;
    MSG      Msg;

    Msg.wParam = 0;

    if ((Initialize (hInstance, SW_SHOWMINIMIZED, &WndClass)) &&
        (PostMessage (hWndMain,
                      WMU_INIT,
                      (WPARAM) GetEnvironmentStrings(),
                      (LPARAM) psCmdLine)))
        {
        while (GetMessage (&Msg, NULL, 0, 0))
            {
            TranslateMessage (&Msg);
            DispatchMessage (&Msg);
            }
        }
    CleanUp (hInstance);
    return Msg.wParam;
    }

// =================================================================
// END OF PROGRAM
// =================================================================
