/* Lumberjack
 *
 * FX <fx@phenoelit.de>
 * Phenoelit (http://www.phenoelit.de)
 * (c) 1999, 2k
 *
 */
/* $Id: lc.c,v 2.7 1999/11/10 12:54:40 fx Exp fx $ */
#include "portable.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <math.h>
#include <sys/param.h>
#include <sys/times.h>

#include <ac/signal.h>
#include <ac/string.h>
#include <ac/unistd.h>

#include <lber.h>
#include <ldap.h>
#include <lutil.h>
#include <lutil_md5.h>
#include <lutil_sha1.h>

#include "ldapconfig.h"

// support for the guessed password list
#include "list.h"
#include "ldif_collect.h"

// my defs
#define PROGRAM_NAME "Lumberjack\n(c) 1999 by Phenoelit (http://www.phenoelit.de/)"
#define PROGRAM_VERSION "Version 0.2.7b"

// David E. Storey's def's
/* local macros */
#define CEILING(x)      ((double)x > (int)x ? (int)x + 1 : (int)x)
#define STRDUP(x)       (x ? strcpy(malloc(strlen(x) + 1), x) : NULL)

#define LDAP_PASSWD_ATTRIB "userPassword"

#define PASS_DEFAULT "userpassword:"
#define UID_DEFAULT "dn: uid="

#define HS_NONE  0
#define HS_PLAIN 1
#define HS_CONV  2

#define TIL_MAX 10
#define BUFFER_SIZE 16
#define FORCED_LENGTH 30

typedef enum
{
        HASHTYPE_NONE,
        HASHTYPE_CRYPT,
        HASHTYPE_MD5,
        HASHTYPE_SMD5,
        HASHTYPE_SHA1,
        HASHTYPE_SSHA1,
	HASHTYPE_NSMTA
}
HashTypes;

typedef struct salt_t
{
        unsigned char  *salt;
        unsigned int    len;
}
Salt;

typedef struct hash_t
{
        char           *name;
        unsigned int    namesz;
        char           *(*func) (const char *, Salt *);
        unsigned char   takes_salt;
        HashTypes       type;
        HashTypes       type_salted;
        unsigned int    default_salt_len;
}
Hash;

// Records for User/Hash infos
typedef struct t_recs {
        unsigned char  *name;
        unsigned char  *hash;
	HashTypes	type;
} RECS;

static int      verbose = 0;
static int      iverbose = 0;
static    int ldifuse=0;	// ldif file or hash string ?
static char *the_password;	// global var for the found password in ldif
static int force=0;		//force adding userless passwords
// the dict for brute force
static char dict[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!\"$%&/()=?`'\\}][{+*~#'-_.:,;<>|^ ";
static int dictsize=sizeof(dict);
static int marks=0;		// not really working

static char *buffer;		// the brute force dynamic buffer
static long int bf_counter;	// conter for brute force 

// I know - this is static and I should implement it dynamic ...
#define MAX_RECS 1048576
static RECS rec[MAX_RECS];
static int recs=0;
static int numeric=0;
static int num_depth=0;
static int numverbose=0;
static int prenum=1;

static double lt;
static int curdep;

// David E. Storey's functions (may be some are useless for my project
char *
pw_encode (unsigned char *passwd, Salt * salt, unsigned int len)
{
        int             salted = salt && salt->salt && salt->len;
        int             b64_len = 0;
        char           *base64digest = NULL;
        unsigned char  *npasswd = passwd;

        if (salted)
        {
                npasswd = (unsigned char *)malloc (len + salt->len);
                memcpy (npasswd, passwd, len);
                memcpy (&npasswd[len], salt->salt, salt->len);
                len += salt->len;
        }

        b64_len = CEILING (len / 3) * 4 + 1;
        base64digest = (char *)malloc (b64_len);
        if (lutil_b64_ntop (npasswd, len, base64digest, b64_len) < 0)
        {
                free (base64digest);
               base64digest = NULL;
        }

        if (salted)
                free (npasswd);

        return (base64digest);
}

void
make_salt (Salt * salt, unsigned int len)
{
        struct timeval  tv;

        if (!salt)
                return;

        /* seed random number generator */
        gettimeofday (&tv, NULL);
        srand (tv.tv_usec);

        salt->len = len;
        salt->salt = (unsigned char *)malloc (len);

        for (len = 0; len < salt->len; len++)
                salt->salt[len] = (tv.tv_usec ^ rand ()) & 0xff;
}

#ifdef SLAPD_CLEARTEXT
char *
hash_none (const char *pw_in, Salt * salt)
{
        return (STRDUP (pw_in));
}
#endif

#ifdef SLAPD_CRYPT
char *
hash_crypt (const char *pw_in, Salt * salt)
{
        const unsigned char crypt64[] =
                "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz1234567890./";
        char   *crypted_pw = NULL;
        Salt    lsalt;

	/* sanity check */
	if (!(isalnum(salt->salt[0]) || salt->salt[0] == '.' || salt->salt[0] == '/'))
		salt->salt[0] = crypt64[salt->salt[0] % (sizeof (crypt64) - 1)];
	if (!(isalnum(salt->salt[1]) || salt->salt[1] == '.' || salt->salt[1] == '/'))
		salt->salt[1] = crypt64[salt->salt[1] % (sizeof (crypt64) - 1)];

	crypted_pw = crypt (pw_in, (char *)salt->salt);
        
	return (STRDUP (&crypted_pw[2]));
}
#endif

char *
hash_md5 (const char *pw_in, Salt * salt)
{
        lutil_MD5_CTX   MD5context;
        unsigned char   MD5digest[16];

        lutil_MD5Init (&MD5context);
        lutil_MD5Update (&MD5context, pw_in, strlen(pw_in));
        if (salt && salt->salt && salt->len)
                lutil_MD5Update (&MD5context, salt->salt, salt->len);
        lutil_MD5Final (MD5digest, &MD5context);

        return (pw_encode (MD5digest, salt, sizeof (MD5digest)));
}

char *
hash_sha1 (const char *pw_in, Salt * salt)
{
        lutil_SHA1_CTX  SHA1context;
        unsigned char   SHA1digest[20];

        lutil_SHA1Init (&SHA1context);
        lutil_SHA1Update (&SHA1context, pw_in, strlen(pw_in));
        if (salt && salt->salt && salt->len)
                lutil_SHA1Update (&SHA1context, salt->salt, salt->len);
        lutil_SHA1Final (SHA1digest, &SHA1context);

        return (pw_encode (SHA1digest, salt, sizeof (SHA1digest)));
}

// NS-MTA-MD5 source from Antoon Bosselaers, ESAT-COSIC
static char * ns_mta_hextab = "0123456789abcdef";
void
ns_mta_hexify(char *buffer, char *str, int len)
{
  char *pch = str;
  char ch;
  int i;

  for(i = 0;i < len; i ++) {
    ch = pch[i];
    buffer[2*i] = ns_mta_hextab[(ch>>4)&15];
    buffer[2*i+1] = ns_mta_hextab[ch&15];
  }
  return;
}

char * ns_mta_hash_alg(char *buffer, char *salt, const char *passwd) {
  lutil_MD5_CTX context;
  char saltstr[2048];
  unsigned char digest[16];

  sprintf(saltstr,"%s%c%s%c%s",salt,89,passwd,247,salt);

  lutil_MD5Init(&context);
  lutil_MD5Update(&context,(unsigned char *)saltstr,strlen(saltstr));
  lutil_MD5Final(digest,&context);
  ns_mta_hexify(buffer,(char*)digest,16);
  buffer[32] = '\0';
  return(buffer);
}
// end of NS-MTA-MD5 source from Antoon Bosselaers

char *hash_ns_mta_md5 (const char *pw_in, Salt * salt) {
  char mta_salt[33];
  char buffer[65];
  char *ret1,*ret2;

  strcpy(mta_salt,salt->salt);
  mta_salt[32] = 0;
  ret1=ns_mta_hash_alg(buffer,mta_salt,pw_in);
  ret2=malloc(strlen(ret1)+1);
  strcpy(ret2,ret1);
  return (ret2);
}

static Hash hashes[] =
{
#ifdef SLAPD_CLEARTEXT
        {"none",  4, hash_none,  0, HASHTYPE_NONE,  HASHTYPE_NONE,  0},
#endif
#ifdef SLAPD_CRYPT
        {"crypt", 5, hash_crypt, 1, HASHTYPE_CRYPT, HASHTYPE_CRYPT, 2},
#endif
        {"md5",   3, hash_md5,   0, HASHTYPE_MD5,   HASHTYPE_SMD5,  0},
        {"smd5",  4, hash_md5,   1, HASHTYPE_SMD5,  HASHTYPE_SMD5,  4},
        {"sha",   3, hash_sha1,  0, HASHTYPE_SHA1,  HASHTYPE_SSHA1, 0},
        {"ssha",  4, hash_sha1,  1, HASHTYPE_SSHA1, HASHTYPE_SSHA1, 4},
        {"nsmtamd5",  8, hash_ns_mta_md5,  1, HASHTYPE_NSMTA, HASHTYPE_NSMTA, 31},
        {NULL,    0, NULL,       0, HASHTYPE_NONE,  HASHTYPE_NONE,  0}
};

// now to something completely different - my code

/* try(..)
 * tryes hashed password against wordlist
 * RETURNS: 
 * 	-1: on error
 * 	0: on no success
 * 	1: on found
 */
int try(char *try_hash1,int htype,FILE *wordlist) {

    char *testpass;
    char *hashed_pw=NULL;
    char *try_hash;
    Salt *st;

    rewind(wordlist);
    if ((testpass=malloc(256))==NULL) {
	fprintf(stderr,"try(): malloc failed\n");
	return (-1);
    }

    st=malloc(sizeof(Salt));
    if (htype==HASHTYPE_NSMTA) {
	try_hash=malloc(33);
	memset(try_hash,0,32);
	strncpy(try_hash,try_hash1,32);
	try_hash[32]=0;
	st->salt=malloc(33);
	memset(st->salt,0,33);
	strncpy(st->salt,&try_hash1[32],32);
	st->len=32;
    } else if (htype==HASHTYPE_CRYPT) {
	try_hash=malloc(strlen(try_hash1)-1);
	memset(try_hash,0,strlen(try_hash1)-1);
	st->salt=malloc(3);
	memset(st->salt,0,3);
	strncpy(st->salt,try_hash1,2);
	strncpy(try_hash,&try_hash1[2],strlen(try_hash1)-2);
	st->len=2;
    } else {
	try_hash=malloc(strlen(try_hash1)+1);
	memset(try_hash,0,strlen(try_hash1)+1);
	strcpy(try_hash,try_hash1);
	st->salt=NULL;
	st->len=0;
    }
    
    memset(testpass,0,256);
    while (fgets(testpass,255,wordlist)!=NULL) {
	testpass[strlen(testpass)-1]='\0';
	if (verbose) printf("Try: %s\n",testpass);

	hashed_pw = hashes[htype].func (testpass,st); 
	if (verbose) {
	    printf("HASH: {%s}%s\n",hashes[htype].name,hashed_pw);
	    printf("TARG: {%s}%s\n",hashes[htype].name,try_hash);
	}
	if (!strcmp(hashed_pw,try_hash)) {
	    if (ldifuse==0) { 
		printf("========================\nFOUND: %s\n=======================\n",testpass);
		free(hashed_pw);
		free(testpass);
		return 1;
	    } else {
		if ((the_password=malloc(strlen(testpass)+1))==NULL) {
		    fprintf(stderr,"PANIC: found password but malloc() failed\n");
		    fprintf(stderr,"password: %s\n",testpass);
		    return (-1);
		}
		memset(the_password,0,strlen(testpass)+1);
		strcpy(the_password,testpass);
	    }
	    free(hashed_pw);
	    free(testpass);
	    return 1;
	}
	free(hashed_pw);
    }
    if (marks) printf("Scan for %s failed\n",try_hash);
    free(st->salt);
    free(st);
    free(try_hash);

    return 0;
}

/* try_bf(..)
 * tryes hashed password against brute force generated string
 * RETURNS: 
 * 	-1: on error
 * 	0: on no success
 * 	1: on found
 */

int try_bf(char *try_hash1,int htype,int d) {

    char *hashed_pw=NULL;
    char *lb;
    char app[2];
    int i;
    char *savebuffer;
    int retval=0;
    clock_t t1,t2;

    char *try_hash;
    Salt *st;

    if (d==1) {
	st=malloc(sizeof(Salt));
	if (htype==HASHTYPE_NSMTA) {
	    try_hash=malloc(33);
	    memset(try_hash,0,32);
	    strncpy(try_hash,try_hash1,32);
	    try_hash[32]=0;
	    st->salt=malloc(33);
	    memset(st->salt,0,33);
	    strncpy(st->salt,&try_hash1[32],32);
	    st->len=32;
	} else if (htype==HASHTYPE_CRYPT) {
	    try_hash=malloc(strlen(try_hash1)-1);
	    memset(try_hash,0,strlen(try_hash1)-1);
	    st->salt=malloc(3);
	    memset(st->salt,0,3);
	    strncpy(st->salt,try_hash1,2);
	    strncpy(try_hash,&try_hash1[2],strlen(try_hash1)-2);
	    st->len=2;
	} else {
	    try_hash=malloc(strlen(try_hash1)+1);
	    memset(try_hash,0,strlen(try_hash1)+1);
	    strcpy(try_hash,try_hash1);
	    st->salt=NULL;
	    st->len=0;
	}

	for (i=0;i<dictsize-1;i++) {
	    memset(app,0,2);
	    app[0]=dict[i];
	    if ((lb=malloc(strlen(buffer)+2))==NULL) {
		fprintf(stderr,"try_bf(): malloc failed\n");
		return (-1);
	    }
	    memset(lb,0,strlen(buffer)+2);
	    strcpy(lb,buffer);
	    strcat(lb,app);
	    
	    if (verbose) printf("Try: %s\n",lb);
	    hashed_pw = hashes[htype].func (lb,st); 
	    if (verbose) {
		printf("HASH: {%s}%s\n",hashes[htype].name,hashed_pw);
		printf("TARG: {%s}%s\n",hashes[htype].name,try_hash);
	    }
	    if (marks==1) printf("%ld\t",bf_counter++);
	    if (!strcmp(hashed_pw,try_hash)) {
		printf("........................\nFOUND: %s\n........................\n",lb);
		free(hashed_pw);
		free(lb);
		free(st->salt);
		free(st);
		free(try_hash);
		return 1;
	    }
	}
	free(st->salt);
	free(st);
	free(try_hash);
    } else { // d!=1
	curdep++;
	if (d==2) t1=times(NULL);
	for (i=0;i<dictsize;i++) {
	    memset(app,0,2);
	    app[0]=dict[i];
	    if ((savebuffer=malloc(strlen(buffer)+1))==NULL) {
		fprintf(stderr,"try_bf(): malloc failed\n");
		return (-1);
	    }
	    memset(savebuffer,0,strlen(buffer)+1);
	    strcat(savebuffer,buffer);
	    strcat(buffer,app);
	    // recursive call
	    retval=try_bf(try_hash,htype,d-1);
	    if (retval<0) {
		free(savebuffer);
		return (-1);
	    }
	    if (retval>0) {
	        free(savebuffer);
		return (1);
	    }
	    memset(buffer,0,strlen(savebuffer)+1);
	    strcpy(buffer,savebuffer);
	    free(savebuffer);
	}
	if (d==2) t2=times(NULL);
	if (lt==0) lt=((double)t2-(double)t1);
	lt=((lt+((double)t2-(double)t1))/(double)2);
	if (d==3) {
	    printf("Estimated time: %8.2f sec ...\n",pow(lt,(curdep+d-3)));
	    printf("lt = %f\n",lt);
	}
	curdep--;
    }
    return 0;
}

/* try_bf_ldif(..)
 * tryes hashed password against brute force generated string
 * RETURNS: 
 * 	-1: on error
 * 	0: on no success
 * 	1: on found
 */

int try_bf_ldif(int d) {

    char *hashed_pw=NULL;
    char *lb;
    char app[2];
    int i,j;
    char *savebuffer;
    int retval=0;
    clock_t t1,t2;

    char *try_hash;
    Salt *st;

    if (d==1) {
	for (i=0;i<dictsize-1;i++) {

	    memset(app,0,2);
	    app[0]=dict[i];
	    if ((lb=malloc(strlen(buffer)+2))==NULL) {
		fprintf(stderr,"try_bf(): malloc failed\n");
		return (-1);
	    }
	    memset(lb,0,strlen(buffer)+2);
	    strcat(lb,buffer);
	    strcat(lb,app);
	    
	    for (j=0;j<recs;j++) {

		// we have it - why once more ?
		if (strstr(rec[j].name,"(cracked)")) continue;

		st=malloc(sizeof(Salt));
		if (rec[j].type==HASHTYPE_NSMTA) {
		    try_hash=malloc(33);
		    memset(try_hash,0,32);
		    strncpy(try_hash,rec[j].hash,32);
		    try_hash[32]=0;
		    st->salt=malloc(33);
		    memset(st->salt,0,33);
		    strncpy(st->salt,&rec[j].hash[32],32);
		    st->len=32;
		} else if (rec[j].type==HASHTYPE_CRYPT) {
		    try_hash=malloc(strlen(rec[j].hash)-1);
		    memset(try_hash,0,strlen(rec[j].hash)-1);
		    st->salt=malloc(3);
		    memset(st->salt,0,3);
		    strncpy(st->salt,rec[j].hash,2);
		    strncpy(try_hash,&rec[j].hash[2],strlen(rec[j].hash)-2);
		    st->len=2;
		} else {
		    try_hash=malloc(strlen(rec[j].hash)+1);
		    memset(try_hash,0,strlen(rec[j].hash)+1);
		    strcpy(try_hash,rec[j].hash);
		    st->salt=NULL;
		    st->len=0;
		}

		hashed_pw = hashes[rec[j].type].func (lb,st); 
		if (verbose) {
		    printf("%s\n",rec[j].name);
		    printf("Try: %s\n",lb);
		    printf("HASH: {%s}%s\n",hashes[rec[j].type].name,hashed_pw);
		    printf("TARG: {%s}%s\n",hashes[rec[j].type].name,rec[j].hash);
		}
		if (marks==1) printf("%ld\t",bf_counter++);
		if (!strcmp(hashed_pw,try_hash)) {
		    printf(">>>>>>>>>>>>>>>>>>>>>>>>\n");
		    printf("User: %s\n",rec[j].name);
		    printf("Pass: %s\n",lb);
		    printf(">>>>>>>>>>>>>>>>>>>>>>>>\n");
		    memset(rec[j].name,0,strlen(rec[j].name));
		    strcpy(rec[j].name,"(cracked)");
		    continue;
		}
		free(hashed_pw);
		free(st->salt);
		free(st);
		free(try_hash);
	    }
	    free(lb);
	}
    } else { // d!=1
	curdep++;
	if (d==2) t1=times(NULL);
	for (i=0;i<dictsize;i++) {
	    memset(app,0,2);
	    app[0]=dict[i];
	    if ((savebuffer=malloc(strlen(buffer)+1))==NULL) {
		fprintf(stderr,"try_bf_ldif(): malloc failed\n");
		return (-1);
	    }
	    memset(savebuffer,0,strlen(buffer)+1);
	    strcat(savebuffer,buffer);
	    strcat(buffer,app);
	    // printf("DEBUG: recursive call (Buffer=%s,d=%d)\n",buffer,d-1);
	    // recursive call
	    retval=try_bf_ldif(d-1);
	    if (retval<0) {
		free(savebuffer);
		return (-1);
	    }
	    if (retval>0) {
	        free(savebuffer);
		return (1);
	    }
	    memset(buffer,0,strlen(savebuffer)+1);
	    strcpy(buffer,savebuffer);
	    free(savebuffer);
	}
	if (d==2) t2=times(NULL);
	if (lt==0) lt=((double)t2-(double)t1);
	lt=((lt+((double)t2-(double)t1))/(double)2);
	if (d==3) {
	    printf("Estimated time: %8.2f sec ...\n",pow(lt,(curdep+d-3)));
	    printf("lt = %f\n",lt);
	}
	curdep--;
    }
    return 0;
}

/* try_guess(..)
 * tryes hashed password against guessed words
 * RETURNS: 
 * 	-1: on error
 * 	0: on no success
 * 	1: on found
 */
int try_guess(char *try_hash1,int htype) {

    char *hashed_pw=NULL;
    char *try_hash;
    Salt *st;

    st=malloc(sizeof(Salt));
    if (htype==HASHTYPE_NSMTA) {
	try_hash=malloc(33);
	memset(try_hash,0,32);
	strncpy(try_hash,try_hash1,32);
	try_hash[32]=0;
	st->salt=malloc(33);
	memset(st->salt,0,33);
	strncpy(st->salt,&try_hash1[32],32);
	st->len=32;
    } else if (htype==HASHTYPE_CRYPT) {
	try_hash=malloc(strlen(try_hash1)-1);
	memset(try_hash,0,strlen(try_hash1)-1);
	st->salt=malloc(3);
	memset(st->salt,0,3);
	strncpy(st->salt,try_hash1,2);
	strncpy(try_hash,&try_hash1[2],strlen(try_hash1)-2);
	st->len=2;
    } else {
	try_hash=malloc(strlen(try_hash1)+1);
	memset(try_hash,0,strlen(try_hash1)+1);
	strcpy(try_hash,try_hash1);
	st->salt=NULL;
	st->len=0;
    }

    list_rewind();
    while (list_next()==0) {
	if (verbose) printf("Try: %s\n",list_content());
	hashed_pw = hashes[htype].func ((char *)list_content(),st); 
	if (verbose) {
	    printf("HASH: {%s}%s\n",hashes[htype].name,hashed_pw);
	    printf("TARG: {%s}%s\n",hashes[htype].name,try_hash);
	}
	if (!strcmp(hashed_pw,try_hash)) {
	    if (ldifuse==0) { 
		printf("========================\nFOUND: %s\n=======================\n",list_content());
		free(hashed_pw);
		return 1;
	    } else {
		if ((the_password=malloc(strlen((char *)list_content())+1))==NULL) {
		    fprintf(stderr,"PANIC: found password but malloc() failed\n");
		    fprintf(stderr,"password: %s\n",list_content());
		    return (-1);
		}
		memset(the_password,0,strlen((char *)list_content())+1);
		strcpy(the_password,(char *)list_content());
	    }
	    free(hashed_pw);
	    return 1;
	}
	free(hashed_pw);
    }
    if (marks) printf("Scan for %s failed\n",try_hash);
    free(st->salt);
    free(st);
    free(try_hash);

    return 0;
}

/* try_num(..)
 * tryes hashed password against guesses with number extension
 * RETURNS: 
 * 	-1: on error
 * 	0: on no success
 * 	1: on found
 */

int try_num(char *try_hash1,int htype,int d) {

    char *hashed_pw=NULL;
    char *lb;
    char app[2];
    int i,j;
    char *savebuffer;
    int retval=0;

    // overrides the global dict for brute force
    static char dict[] = "0123456789";
    static int dictsize=sizeof(dict);

    char *try_hash;
    Salt *st;


    if (d==1) {
	list_rewind();

	// we do the salting here 'cause this will not truncate the hash later
	st=malloc(sizeof(Salt));
	if (htype==HASHTYPE_NSMTA) {
	    try_hash=malloc(33);
	    memset(try_hash,0,32);
	    strncpy(try_hash,try_hash1,32);
	    try_hash[32]=0;
	    st->salt=malloc(33);
	    memset(st->salt,0,33);
	    strncpy(st->salt,&try_hash1[32],32);
	    st->len=32;
	} else if (htype==HASHTYPE_CRYPT) {
	    try_hash=malloc(strlen(try_hash1)-1);
	    memset(try_hash,0,strlen(try_hash1)-1);
	    st->salt=malloc(3);
	    memset(st->salt,0,3);
	    strncpy(st->salt,try_hash1,2);
	    strncpy(try_hash,&try_hash1[2],strlen(try_hash1)-2);
	    st->len=2;
	} else {
	    try_hash=malloc(strlen(try_hash1)+1);
	    memset(try_hash,0,strlen(try_hash1)+1);
	    strcpy(try_hash,try_hash1);
	    st->salt=NULL;
	    st->len=0;
	}

	while (list_next()==0) {
	    for (i=0;i<dictsize-1;i++) {
		memset(app,0,2);
		app[0]=dict[i];
		if ((lb=malloc(strlen(buffer)+2+strlen((char *)list_content())))==NULL) {
		    fprintf(stderr,"try_num(): malloc failed\n");
		    return (-1);
		}
		memset(lb,0,strlen(buffer)+2+strlen((char *)list_content()));
		strcpy(lb,(char *)list_content());
		strcat(lb,buffer);
		strcat(lb,app);
		
		if (verbose||numverbose) printf("Try: %s\n",lb);
		hashed_pw = hashes[htype].func (lb,st); 
		if (verbose) {
		    printf("HASH: {%s}%s\n",hashes[htype].name,hashed_pw);
		    printf("TARG: {%s}%s\n",hashes[htype].name,try_hash);
		}
		if (marks==1) printf("%ld\t",bf_counter++);
		if (!strcmp(hashed_pw,try_hash)) {
		    if (ldifuse==0) { 
			printf("========================\nFOUND: %s\n=======================\n",lb);
			free(hashed_pw);
			free(lb);
			free(st->salt);
			free(st);
			free(try_hash);
			return 1;
		    } else {
			if ((the_password=malloc(strlen(lb)+1))==NULL) {
			    fprintf(stderr,"PANIC: found password but malloc() failed\n");
			    fprintf(stderr,"password: %s\n",lb);
			    return (-1);
			}
			memset(the_password,0,strlen(lb)+1);
			strcpy(the_password,lb);
		    }
		    free(hashed_pw);
		    free(lb);
		    free(st->salt);
		    free(st);
		    free(try_hash);
		    return 1;
		}
		free(hashed_pw);

		if (prenum) {
		    // and in the reverse (num_password)
		    memset(lb,0,strlen(buffer)+2+strlen((char *)list_content()));
		    strcpy(lb,buffer);
		    strcat(lb,app);
		    strcat(lb,(char *)list_content());
		    
		    if (verbose||numverbose) printf("Try: %s\n",lb);
		    hashed_pw = hashes[htype].func (lb,st); 
		    if (verbose) {
			printf("HASH: {%s}%s\n",hashes[htype].name,hashed_pw);
			printf("TARG: {%s}%s\n",hashes[htype].name,try_hash);
		    }
		    if (marks==1) printf("%ld\t",bf_counter++);
		    if (!strcmp(hashed_pw,try_hash)) {
			if (ldifuse==0) { 
			    printf("========================\nFOUND: %s\n=======================\n",lb);
			    free(hashed_pw);
			    free(lb);
			    free(st->salt);
			    free(st);
			    free(try_hash);
			    return 1;
			} else {
			    if ((the_password=malloc(strlen(lb)+1))==NULL) {
				fprintf(stderr,"PANIC: found password but malloc() failed\n");
				fprintf(stderr,"password: %s\n",lb);
				return (-1);
			    }
			    memset(the_password,0,strlen(lb)+1);
			    strcpy(the_password,lb);
			}
			free(hashed_pw);
			free(st->salt);
			free(st);
			free(try_hash);
			return (1);
		    } //if found
		    free(hashed_pw);

		    // and the ugly NNppppppppNNNN style
		    free(lb); //we have to reallocate 'cause it gets larger
		    if ((lb=malloc(strlen(buffer)+4+strlen((char *)list_content())))==NULL) {
			fprintf(stderr,"try_num(): malloc failed\n");
			return (-1);
		    }
		    for (j=0;j<99;j++) {
			memset(lb,0,strlen(buffer)+4+strlen((char *)list_content()));
			sprintf(lb,"%d",j);
			strcat(lb,(char *)list_content());
			strcat(lb,buffer);
			strcat(lb,app);
			
			if (verbose||numverbose) printf("Try: %s\n",lb);
			hashed_pw = hashes[htype].func (lb,st); 
			if (verbose) {
			    printf("HASH: {%s}%s\n",hashes[htype].name,hashed_pw);
			    printf("TARG: {%s}%s\n",hashes[htype].name,try_hash);
			}
			if (marks==1) printf("%ld\t",bf_counter++);
			if (!strcmp(hashed_pw,try_hash)) {
			    if (ldifuse==0) { 
				printf("========================\nFOUND: %s\n=======================\n",lb);
				free(hashed_pw);
				free(lb);
				free(st->salt);
				free(st);
				free(try_hash);
				return 1;
			    } else {
				if ((the_password=malloc(strlen(lb)+1))==NULL) {
				    fprintf(stderr,"PANIC: found password but malloc() failed\n");
				    fprintf(stderr,"password: %s\n",lb);
				    return (-1);
				}
				memset(the_password,0,strlen(lb)+1);
				strcpy(the_password,lb);
			    }
			    free(hashed_pw);
			    free(st->salt);
			    free(st);
			    free(try_hash);
			    return (1);
			} //if found
			free(hashed_pw);
		    } //for (the first 2 digits)
		} // if prenum
		free(lb);
	    } //for (the dict)
	} //while
	free(st->salt);
	free(st);
	free(try_hash);
    } else { // d!=1
	curdep++;
	for (i=0;i<dictsize;i++) {
	    memset(app,0,2);
	    app[0]=dict[i];
	    if ((savebuffer=malloc(strlen(buffer)+1))==NULL) {
		fprintf(stderr,"try_num(): malloc failed\n");
		return (-1);
	    }
	    memset(savebuffer,0,strlen(buffer)+1);
	    strcat(savebuffer,buffer);
	    strcat(buffer,app);
	    // recursive call
	    retval=try_num(try_hash1,htype,d-1);
	    if (retval<0) {
		free(savebuffer);
		return (-1);
	    }
	    if (retval>0) {
	        free(savebuffer);
		return (1);
	    }
	    memset(buffer,0,strlen(savebuffer)+1);
	    strcpy(buffer,savebuffer);
	    free(savebuffer);
	}
	curdep--;
    }
    return 0;
}

/* try_num_wl(..)
 * tryes hashed password against words in wordlist with number extension
 * RETURNS: 
 * 	-1: on error
 * 	0: on no success
 * 	1: on found
 */

int try_num_wl(char *try_hash1,int htype,int d,FILE *wordlist) {

    char *hashed_pw=NULL;
    char *lb;
    char app[2];
    int i,j;
    char *savebuffer;
    int retval=0;
    char *testpass;

    // overrides the global dict for brute force
    static char dict[] = "0123456789";
    static int dictsize=sizeof(dict);
    
    char *try_hash;
    Salt *st;


    if (d==1) {
	rewind(wordlist);

	// we do this only for the last level ...
	st=malloc(sizeof(Salt));
	if (htype==HASHTYPE_NSMTA) {
	    try_hash=malloc(33);
	    memset(try_hash,0,32);
	    strncpy(try_hash,try_hash1,32);
	    try_hash[32]=0;
	    st->salt=malloc(33);
	    memset(st->salt,0,33);
	    strncpy(st->salt,&try_hash1[32],32);
	    st->len=32;
	} else if (htype==HASHTYPE_CRYPT) {
	    try_hash=malloc(strlen(try_hash1)-1);
	    memset(try_hash,0,strlen(try_hash1)-1);
	    st->salt=malloc(3);
	    memset(st->salt,0,3);
	    strncpy(st->salt,try_hash1,2);
	    strncpy(try_hash,&try_hash1[2],strlen(try_hash1)-2);
	    st->len=2;
	} else {
	    try_hash=malloc(strlen(try_hash1)+1);
	    memset(try_hash,0,strlen(try_hash1)+1);
	    strcpy(try_hash,try_hash1);
	    st->salt=NULL;
	    st->len=0;
	}

	if ((testpass=malloc(265))==NULL) {
	    fprintf(stderr,"try_num_wl(): malloc() failed\n");
	    return (-1);
	}
	memset(testpass,0,256);
	
	while (fgets(testpass,255,wordlist)!=NULL) {
	    testpass[strlen(testpass)-1]='\0';
	    for (i=0;i<dictsize-1;i++) {
		memset(app,0,2);
		app[0]=dict[i];
		if ((lb=malloc(strlen(buffer)+2+strlen(testpass)))==NULL) {
		    fprintf(stderr,"try_num(): malloc failed\n");
		    return (-1);
		}
		memset(lb,0,strlen(buffer)+2+strlen(testpass));
		strcpy(lb,testpass);
		strcat(lb,buffer);
		strcat(lb,app);
		
		if (verbose||numverbose) printf("Try: %s\n",lb);
		hashed_pw = hashes[htype].func (lb,st); 
		if (verbose) {
		    printf("HASH: {%s}%s\n",hashes[htype].name,hashed_pw);
		    printf("TARG: {%s}%s\n",hashes[htype].name,try_hash);
		}
		if (marks==1) printf("%ld\t",bf_counter++);
		if (!strcmp(hashed_pw,try_hash)) {
		    if (ldifuse==0) { 
			printf("========================\nFOUND: %s\n=======================\n",lb);
			free(hashed_pw);
			free(lb);
			free(testpass);
			free(st->salt);
			free(st);
			free(try_hash);
			return 1;
		    } else {
			if ((the_password=malloc(strlen(lb)+1))==NULL) {
			    fprintf(stderr,"PANIC: found password but malloc() failed\n");
			    fprintf(stderr,"password: %s\n",lb);
			    return (-1);
			}
			memset(the_password,0,strlen(lb)+1);
			strcpy(the_password,lb);
		    }
		    free(hashed_pw);
		    free(lb);
		    free(testpass);
		    free(st->salt);
		    free(st);
		    free(try_hash);
		    return 1;
		}
		free(hashed_pw);

		if (prenum) {
		    // and in the reverse (num_password)
		    memset(lb,0,strlen(buffer)+2+strlen(testpass));
		    strcpy(lb,buffer);
		    strcat(lb,app);
		    strcat(lb,testpass);
		    
		    if (verbose||numverbose) printf("Try: %s\n",lb);
		    hashed_pw = hashes[htype].func (lb,st); 
		    if (verbose) {
			printf("HASH: {%s}%s\n",hashes[htype].name,hashed_pw);
			printf("TARG: {%s}%s\n",hashes[htype].name,try_hash);
		    }
		    if (marks==1) printf("%ld\t",bf_counter++);
		    if (!strcmp(hashed_pw,try_hash)) {
			if (ldifuse==0) { 
			    printf("========================\nFOUND: %s\n=======================\n",lb);
			    free(hashed_pw);
			    free(lb);
			    free(testpass);
			    free(st->salt);
			    free(st);
			    free(try_hash);
			    return 1;
			} else {
			    if ((the_password=malloc(strlen(lb)+1))==NULL) {
				fprintf(stderr,"PANIC: found password but malloc() failed\n");
				fprintf(stderr,"password: %s\n",lb);
				return (-1);
			    }
			    memset(the_password,0,strlen(lb)+1);
			    strcpy(the_password,lb);
			}
			free(hashed_pw);
			free(testpass);
			free(st->salt);
			free(st);
			free(try_hash);
			return (1);
		    } //if found
		    free(hashed_pw);

		    // and the ugly NNppppppppNNNN style
		    free(lb); //we have to reallocate 'cause it gets larger
		    if ((lb=malloc(strlen(buffer)+4+strlen(testpass)))==NULL) {
			fprintf(stderr,"try_num(): malloc failed\n");
			return (-1);
		    }
		    for (j=0;j<99;j++) {
			memset(lb,0,strlen(buffer)+4+strlen(testpass));
			sprintf(lb,"%d",j);
			strcat(lb,testpass);
			strcat(lb,buffer);
			strcat(lb,app);
			
			if (verbose||numverbose) printf("Try: %s\n",lb);
			hashed_pw = hashes[htype].func (lb,st); 
			if (verbose) {
			    printf("HASH: {%s}%s\n",hashes[htype].name,hashed_pw);
			    printf("TARG: {%s}%s\n",hashes[htype].name,try_hash);
			}
			if (marks==1) printf("%ld\t",bf_counter++);
			if (!strcmp(hashed_pw,try_hash)) {
			    if (ldifuse==0) { 
				printf("========================\nFOUND: %s\n=======================\n",lb);
				free(hashed_pw);
				free(lb);
				free(testpass);
				free(st->salt);
				free(st);
				free(try_hash);
				return 1;
			    } else {
				if ((the_password=malloc(strlen(lb)+1))==NULL) {
				    fprintf(stderr,"PANIC: found password but malloc() failed\n");
				    fprintf(stderr,"password: %s\n",lb);
				    return (-1);
				}
				memset(the_password,0,strlen(lb)+1);
				strcpy(the_password,lb);
			    } // if ldifuse
			    free(hashed_pw);
			    free(testpass);
			    free(st->salt);
			    free(st);
			    free(try_hash);
			    return (1);
			} //if found
			free(hashed_pw);
		    } //for (the first 2 digits)
		} // if prenum
		free(lb);
	    } //for (the dict)
	    memset(testpass,0,256);
	} //while
	free(testpass);
	free(st->salt);
	free(st);
	free(try_hash);
    } else { // d!=1
	curdep++;
	for (i=0;i<dictsize;i++) {
	    memset(app,0,2);
	    app[0]=dict[i];
	    if ((savebuffer=malloc(strlen(buffer)+1))==NULL) {
		fprintf(stderr,"try_num(): malloc failed\n");
		return (-1);
	    }
	    memset(savebuffer,0,strlen(buffer)+1);
	    strcat(savebuffer,buffer);
	    strcat(buffer,app);
	    // recursive call
	    retval=try_num_wl(try_hash1,htype,d-1,wordlist);
	    if (retval<0) {
		free(savebuffer);
		return (-1);
	    }
	    if (retval>0) {
	        free(savebuffer);
		return (1);
	    }
	    memset(buffer,0,strlen(savebuffer)+1);
	    strcpy(buffer,savebuffer);
	    free(savebuffer);
	}
	curdep--;
    }
    return 0;
}

void usage(char *n) {
    printf("\nusage: %s [options] -w wordlist.txt {-f ldiffile.ldif|[-c hash type] -p crypted_string}\n",n);
    printf("options:\n");
    printf(" P: \tprint guess word list and exit\n\n");
    printf(" F: \tforce adding of contextless passwords\n");
    printf(" C: \t(obsolete - does nothing)\n");
    printf(" n x: \tnumeric extensions for guessed words (max depth x)\n");
    printf(" s: \tdon't collect ldif guessing informations\n\
    \t(supply this option before -f !)\n");
    printf(" S: \tdon't do prefix number testing\n");
    printf(" b x: \tbrute force mode (start depth x)\n");
    printf(" B y: \tbrute force mode (stop depth x)\n");
    printf(" u: \tuid attribute in ldif file (default: '%s')\n",UID_DEFAULT);
    printf(" U: \tpassword attribute in ldif file (default: '%s')\n",PASS_DEFAULT);
    printf(" v: \tverbose\n");
    printf(" V: \tinformational verbose mode (just some infos)\n");
    printf(" M: \tMarks for completion of one brute force loop\n");
    printf("\nEXAMPLE:\n");
    printf("%s -w passwords.txt -f company.ldif\n",n);
    printf("%s -w passwords.txt -p EL3TaxUY1oPheRtO3TsrFMoZFqg= -c ssha\n",n);
    printf("%s -w passwords.txt -f company.ldif -b 5 -B 7 -M -F -u 'dn: cn='\n",n);
    exit (1);
}

#define STRINGSIZE 2048
#define BF_STOP 8

int main(int argc, char **argv) {

    int opt,j;
    char *target_hash=NULL,tstr[STRINGSIZE];
    FILE *wordlist=NULL;
    
    FILE *ldif=NULL;
    unsigned long int ldif_lineno=0;
    char *uid_attr=NULL, *pass_attr=NULL;
    int openuser=0;
    char *hashtypepos;
    
    FILE *crypts;
    char *vc;
    int nl;

    int printlist=0;

    // brute force
    int bf_start=0, bf_stop=0;
    int bfi;
    int retval;

    int hashtype=HASHTYPE_SHA1;
    int surpress_collect=0;
    
    printf("%s\n%s\n",PROGRAM_NAME,PROGRAM_VERSION);
    
    if (argc==1) usage(argv[0]);

    while ((opt=getopt(argc,argv,"sSPNn:MCVvFp:f:c:w:u:U:b:B:"))!=EOF) {
	switch (opt) {
	    case 'f': ldifuse=1;
		      if ((ldif=fopen(optarg,"rt"))==NULL) {
			  fprintf(stderr,"main: unable to open ldif file %s\n",optarg);
			  return (-1);
		      }
			if (verbose) printf("Create guessing list ...\n");
			list_verbose=1;
			list_create();
			if (!surpress_collect) ldif_collect(optarg);
		      break;
	    case 'p': if (ldifuse) {
			  fprintf(stderr,"ldif and hashtring are concurent!\n");
			  usage(argv[0]);
		      }
		      if ((target_hash=malloc(strlen(optarg)+1))==NULL) {
			  fprintf(stderr,"main: malloc() failed\n");
			  return (-1);
		      }
		      memset(target_hash,0,strlen(optarg)+1);
		      strcpy(target_hash,optarg);
		      break;
	    case 'P': printlist++;
		      break;
	    case 'V': iverbose++;
		      break;
	    case 'v': verbose++;
		      break;
	    case 'F': force++;
		      break;
	    case 'C': break;
	    case 's': surpress_collect++;
		      break;
	    case 'S': prenum=0;
		      break;
	    case 'N': numverbose++;
		      break;
	    case 'n': if ((num_depth=atoi(optarg))<=0) {
			  fprintf(stderr,"maximum numeric depth %s ?\n",optarg);
			  exit (-1);
		      }
		      numeric++;
		      break;
	    case 'M': marks++;
		      break;
	    case 'b': if ((bf_start=atoi(optarg))<=0) {
			  fprintf(stderr,"Depth 0 impossible\n");
			  exit (-1);
		      }
		      break;
	    case 'B': if ((bf_stop=atoi(optarg))<=0) {
			  fprintf(stderr,"Depth 0 impossible\n");
			  exit (-1);
		      }
		      break;
	    case 'w': if ((wordlist=fopen(optarg,"rt"))==NULL) {
			  fprintf(stderr,"main: unable to open %s\n",optarg);
			  return (-1);
		      }
		      break;
	    case 'c': for (j = 0; hashes[j].name; j++) {
			  if (!strncasecmp (optarg, hashes[j].name, hashes[j].namesz)) {
			      hashtype = hashes[j].type;
			      break;
			  }
		      }
		      if (!hashes[j].name) {
			  fprintf (stderr, "hash type: %s unknown\n", optarg);
                          usage (argv[0]);
		      }
		      break;
	    case 'u': if ((uid_attr=malloc(strlen(optarg)+1))==NULL) {
			  fprintf(stderr,"main: malloc() failed\n");
			  return (-1);
		      }
		      memset(uid_attr,0,strlen(optarg)+1);
		      strcpy(uid_attr,optarg);
		      break;
	    case 'U': if ((pass_attr=malloc(strlen(optarg)+1))==NULL) {
			  fprintf(stderr,"main: malloc() failed\n");
			  return (-1);
		      }
		      memset(pass_attr,0,strlen(optarg)+1);
		      strcpy(pass_attr,optarg);
		      break;
	    default:  usage(argv[0]);
	}
    }

    if (printlist) {
	list_display();
	list_destroy();
	exit (0);
    }

    // tests for non-optional stuff
    if (wordlist==NULL) usage(argv[0]);
    if ((ldifuse==0)&&(target_hash==NULL)) {
	fprintf(stderr,"What ? No hash and no file ?\n");
	exit (-1);
    }
    if ((ldifuse==1)&&(ldif==NULL)) {
	fprintf(stderr,"What ? Which file ?\n");
	exit (-1);
    }

    // make stdout line buffered for output in a file
#ifdef HAVE_SETLINEBUF
   setlinebuf(stdout);
#else
   setvbuf(stdout, NULL, _IOLBF, 0);
#endif


    if ((bf_stop==0)&&(bf_start!=0)) {
	printf("Setting brute force stop depth to %d\n",BF_STOP);
	bf_stop=BF_STOP;
    }

    if (!ldifuse) { 
	// password string 
	if (try(target_hash,hashtype,wordlist)==0) {
	    if (numeric) {
		if (iverbose) printf("Entering numeric wordlist mode ...\n");
		if ((buffer=malloc(BUFFER_SIZE))==NULL) {
		    fprintf(stderr,"main(): malloc failed\n");
		    exit (-1);
		}
		memset(buffer,0,BUFFER_SIZE);
		for (bfi=1;bfi<=num_depth;bfi++) {
		    if (iverbose) fprintf(stderr,"Wordlist numeric mode (depth %d)...\n",bfi);
		    if (try_num_wl(target_hash,hashtype,bfi,wordlist)!=0) 
			return (0);
		}
		free(buffer);
	    }

	    if (bf_start>0) {
		if (iverbose) printf("Entering brute force mode ...\n");
		if ((buffer=malloc(bf_stop+3))==NULL) {
		    fprintf(stderr,"main(): malloc failed\n");
		    exit (-1);
		}
		memset(buffer,0,bf_stop+3);
		bf_counter=0;
		for (bfi=bf_start;bfi<=bf_stop;bfi++) {
		    if (iverbose) fprintf(stderr,"Brute force mode (depth %d)\n",bfi);
		    curdep=0;
		    lt=0;
		    if (try_bf(target_hash,hashtype,bfi)!=0) return (0);
		}
		free(buffer);
	    }
	}
	return (0);
    } else {
	// ldif file 
	if (!uid_attr) {
	    if ((uid_attr=malloc(strlen(UID_DEFAULT)+1))==NULL) {
		  fprintf(stderr,"main: malloc() failed\n");
		  return (-1);
	    }
	    memset(uid_attr,0,strlen(UID_DEFAULT)+1);
	    strcpy(uid_attr,UID_DEFAULT);
	}
	if (!pass_attr) {
	    if ((pass_attr=malloc(strlen(PASS_DEFAULT)+1))==NULL) {
		  fprintf(stderr,"main: malloc() failed\n");
		  return (-1);
	    }
	    memset(pass_attr,0,strlen(PASS_DEFAULT)+1);
	    strcpy(pass_attr,PASS_DEFAULT);
	}
	if (verbose) {
	    printf("uid attribute name set to '%s'\n",uid_attr);
	    printf("password attribute name set to '%s'\n",pass_attr);
	}

	if (iverbose) printf("Collecting ldif user informations ...\n");
	fflush(stdout);
	rewind(ldif);
	memset(tstr,0,STRINGSIZE);
	openuser=0;
        while (fgets(tstr,STRINGSIZE-1,ldif)!=NULL) {
	    tstr[strlen(tstr)-1]='\0';
	    // if (strstr(tstr,uid_attr)) {
	    if ((tstr[0]==uid_attr[0])&&(strstr(tstr,uid_attr))) {
		// test for open user
		if (openuser==1) {
		    if (verbose) printf("%s is user without password - ignored\n",rec[recs-1].name);
		    if ((recs--)<0) {recs=0;}
		}

		openuser=1;
		rec[recs].name=malloc(strlen(tstr)+1);
		strcpy(rec[recs].name,tstr);
		if ((recs++)>MAX_RECS) {
		    fprintf(stderr,"maximum user information reached - ignoring everyting from now\n");
		    break;
		}
	    }
	    if (strstr(tstr,pass_attr)) {
		if ((openuser==0)&&(force==0)) {
		    printf("Password without context (line %ld) - ignored\n",ldif_lineno);
		} 
		if ((openuser==1)||(force>0)) {
		    if ((force>0)&&(openuser==0)) {
			rec[recs].name=malloc(FORCED_LENGTH+1);
			sprintf(rec[recs].name,"(forced #%ld)",ldif_lineno);
			if ((recs++)>MAX_RECS) {
			    fprintf(stderr,"maximum user information reached - ignoring everyting from now\n");
			    break;
			}
			printf("Password without context (line %ld) - adding forced\n",ldif_lineno);
		    }
		    openuser=0;
		    rec[recs-1].hash=malloc(strlen(tstr+1));
		    if ((hashtypepos=strstr(tstr,"{SHA}"))!=NULL) {
			rec[recs-1].type=HASHTYPE_SHA1;
			hashtypepos=hashtypepos+(5*sizeof(char)); 
			strcpy(rec[recs-1].hash,hashtypepos);
		        if (verbose) 
		 	    printf ("%s ({%s}%s) added\n",rec[recs-1].name,
				hashes[HASHTYPE_SHA1].name,rec[recs-1].hash);
		    } else if ((hashtypepos=strstr(tstr,"{SSHA}"))!=NULL) {
			rec[recs-1].type=HASHTYPE_SSHA1;
			hashtypepos=hashtypepos+(6*sizeof(char)); 
			strcpy(rec[recs-1].hash,hashtypepos);
		        if (verbose) {
		 	    printf ("%s ({%s}%s) added\n",rec[recs-1].name,
				    hashes[HASHTYPE_SSHA1].name,
				    rec[recs-1].hash);
			}
		    } else if ((hashtypepos=strstr(tstr,"{SMD5}"))!=NULL) {
			rec[recs-1].type=HASHTYPE_SMD5;
			hashtypepos=hashtypepos+(6*sizeof(char)); 
			strcpy(rec[recs-1].hash,hashtypepos);
		        if (verbose) {
		 	    printf ("%s ({%s}%s) added\n",rec[recs-1].name,
				    hashes[HASHTYPE_SMD5].name,
				    rec[recs-1].hash);
			}
		    } else if ((hashtypepos=strstr(tstr,"{MD5}"))!=NULL) {
			rec[recs-1].type=HASHTYPE_MD5;
			hashtypepos=hashtypepos+(5*sizeof(char)); 
			strcpy(rec[recs-1].hash,hashtypepos);
		        if (verbose) {
		 	    printf ("%s ({%s}%s) added\n",rec[recs-1].name,
				    hashes[HASHTYPE_MD5].name,
				    rec[recs-1].hash);
			}
		    } else if ((hashtypepos=strstr(tstr,"{NS-MTA-MD5}"))!=NULL) {
			rec[recs-1].type=HASHTYPE_NSMTA;
			hashtypepos=hashtypepos+(12*sizeof(char)); 
			strcpy(rec[recs-1].hash,hashtypepos);
		        if (verbose) {
		 	    printf ("%s ({%s}%s) added\n",rec[recs-1].name,
				    hashes[HASHTYPE_NSMTA].name,
				    rec[recs-1].hash);
			}
		    } else if (
			    ((hashtypepos=strstr(tstr,"{CRYPT}"))!=NULL)||
			    ((hashtypepos=strstr(tstr,"{crypt}"))!=NULL)) {
			rec[recs-1].type=HASHTYPE_CRYPT;
			hashtypepos=hashtypepos+(7*sizeof(char)); 
			strcpy(rec[recs-1].hash,hashtypepos);
		        if (verbose) {
		 	    printf ("%s ({%s}%s) added\n",rec[recs-1].name,
				    hashes[HASHTYPE_NSMTA].name,
				    rec[recs-1].hash);
			}
		    } else {
			fprintf(stderr,"Unknown hash type - ignored (%s)\n",tstr);
		        if ((recs--)<0) {recs=0;}
		    }
		}
	    }
	    ldif_lineno++;
	} // end of while for ldif
	fclose(ldif);
	printf("%d users with password found ...\n",recs);

	// test all users against all guessed words
	if (!surpress_collect) {
	    for (j=0;j<recs;j++) {
		if (verbose) {
		    printf("Try: %s ({%s}%s)\n",
			    rec[j].name,
			    hashes[rec[j].type].name,
			    rec[j].hash);
		}
		if (try_guess(rec[j].hash,rec[j].type)==1) {
		    printf(">>>>>>>>>>>>>>>>>>>>>>>>\n");
		    printf("User: %s\n",rec[j].name);
		    printf("Pass: %s\n",the_password);
		    printf(">>>>>>>>>>>>>>>>>>>>>>>>\n");
		    free(the_password);
		    memset(rec[j].name,0,strlen(rec[j].name));
		    strcpy(rec[j].name,"(cracked)");
		}
	    }
	}

	if (iverbose) printf("Entering wordlist mode ...\n");
	// test all users against all wordlist words
	for (j=0;j<recs;j++) {
	    if (!strcmp(rec[j].name,"(cracked)")) continue;
	    if (verbose) {
		printf("Try: %s ({%s}%s)\n",
			rec[j].name,
			hashes[rec[j].type].name,
			rec[j].hash);
	    }
	    if (try(rec[j].hash,rec[j].type,wordlist)==1) {
		printf(">>>>>>>>>>>>>>>>>>>>>>>>\n");
		printf("User: %s\n",rec[j].name);
		printf("Pass: %s\n",the_password);
		printf(">>>>>>>>>>>>>>>>>>>>>>>>\n");
		free(the_password);
		memset(rec[j].name,0,strlen(rec[j].name));
		strcpy(rec[j].name,"(cracked)");
	    }
	}

	// test all users against all guessed words with numeric extension
	if (numeric&&(!surpress_collect)) {
	    if (iverbose) printf("Entering numeric mode ...\n");
	    for (j=0;j<recs;j++) {
		if (!strcmp(rec[j].name,"(cracked)")) continue;
		if (verbose) {
		    printf("Try: %s ({%s}%s)\n",
			    rec[j].name,
			    hashes[rec[j].type].name,
			    rec[j].hash);
		}
		if ((buffer=malloc(TIL_MAX+3))==NULL) {
		    fprintf(stderr,"main(): malloc failed\n");
		    exit (-1);
		}
		memset(buffer,0,bf_stop+3);
		bf_counter=0;
		for (bfi=1;bfi<=num_depth;bfi++) {
		    if (iverbose) fprintf(stderr,"Numeric mode (depth %d) (%d/%d)\n",bfi,j+1,recs);
		    curdep=0;
		    if ((retval=try_num(rec[j].hash,rec[j].type,bfi))==1) {
			printf(">>>>>>>>>>>>>>>>>>>>>>>>\n");
			printf("User: %s\n",rec[j].name);
			printf("Pass: %s\n",the_password);
			printf(">>>>>>>>>>>>>>>>>>>>>>>>\n");
			free(the_password);
			memset(rec[j].name,0,strlen(rec[j].name));
			strcpy(rec[j].name,"(cracked)");
			break;
		    } else if (retval<0) {
			fprintf(stderr,"Exiting\n");
			exit (-1);
		    }
		} // for numeric
		free(buffer);
	    } //for all users
	}
	list_destroy();

	// test all users against all wordlist words with numeric extension
	if (numeric) {
	    if (iverbose) printf("Entering wordlist numeric mode ...\n");
	    for (j=0;j<recs;j++) {
		if (!strcmp(rec[j].name,"(cracked)")) continue;
		if (verbose) {
		    printf("Try: %s ({%s}%s)\n",
			    rec[j].name,
			    hashes[rec[j].type].name,
			    rec[j].hash);
		}
		if ((buffer=malloc(TIL_MAX+3))==NULL) {
		    fprintf(stderr,"main(): malloc failed\n");
		    exit (-1);
		}
		memset(buffer,0,bf_stop+3);
		bf_counter=0;
		for (bfi=1;bfi<=num_depth;bfi++) {
		    if (iverbose) fprintf(stderr,"Wordlist numeric mode (depth %d) (%d/%d)\n",bfi,j+1,recs);
		    curdep=0;
		    if (try_num_wl(rec[j].hash,rec[j].type,bfi,wordlist)==1) {
			printf(">>>>>>>>>>>>>>>>>>>>>>>>\n");
			printf("User: %s\n",rec[j].name);
			printf("Pass: %s\n",the_password);
			printf(">>>>>>>>>>>>>>>>>>>>>>>>\n");
			free(the_password);
			memset(rec[j].name,0,strlen(rec[j].name));
			strcpy(rec[j].name,"(cracked)");
			bfi=num_depth;
		    } else if (retval<0) {
			fprintf(stderr,"Exiting\n");
			exit (-1);
		    }
		} // for numeric
		free(buffer);
	    } //for all users
	} // if numeric
	
	// brute force mode 
	if (bf_start>0) {
	    if (iverbose) printf("Entering brute force mode ...\n");
	    if ((buffer=malloc(bf_stop+3))==NULL) {
		fprintf(stderr,"main(): malloc failed\n");
		exit (-1);
	    }
	    memset(buffer,0,bf_stop+3);
	    bf_counter=0;
	    for (bfi=bf_start;bfi<=bf_stop;bfi++) {
		fprintf(stderr,"Brute force mode (depth %d)\n",bfi);
		curdep=0;
		lt=0;
		if ((retval=try_bf_ldif(bfi))<0) break;
	    }
	}

	for (j=0;j<recs;j++) {
	    free(rec[j].name);
	    free(rec[j].hash);
	}

    } // end of ldif if

    return (0);
}
