Unit Btree;

{ btree1 -- a general unit for manipulating alphabetized strings
  on the heap }

{ Copyright 1989, by J. W. Rider  }

{ Modified for readability by Philip Howlett 1993 }

Interface

Type
  recptr = ^btrec;
  btrec = record
      upper,
      right,
      left      : recptr;
      counter   : integer;
      data      : string[15];
    end;

Procedure firstrec(var current : recptr);
Procedure lastrec(var current : recptr);
Procedure delrec(var root,current: recptr);
Function findrec(var current: recptr; s: string) : Boolean;
Procedure nextrec(var current: recptr);
Procedure prevrec(var current: recptr);
Function storec(var root: recptr; var s: string) : recptr;

Implementation

Const
  grain = 16 ; { heap granularity; usage here requires power of two }

{ Granularity of heap is set to 'grain' bytes, see Turbo Pascal
  Reference Guide, pg 199.

  SYSTEM.FREEMIN is also set to 16000 in initialization.  No
  investigation has been made as to whether or not these values
  are optimal.  Failure to set FREEMIN large enough will cause
  a run-time error for trees that are too large. }

Type
  lh = record
      upper,
      right,
      left        : recptr;
      counter     : Integer;
    end; { line header declaration }

{
Procedure getmem(var p : recptr; size : word);

Begin
  Getmem(p,(size+grain) and ($FFFF - grain + 1));
End;
}
{
Procedure freemem(var p : recptr; size : word);

Begin
  Freemem(p,(size+grain) and ($FFFF - grain +1));
End;
}

Procedure firstrec(var current : recptr);

{ returns the address of the first record in the btree branch
  from "current". if no btree has been entered, returns nil. }

Begin
  If current<>nil then
    While current^.left<>nil do
      current:=current^.left;
End;


Procedure lastrec(var current : recptr);

{ returns the address of the last record in the btree branch
  from "current". if no btree has been entered, returns nil. }

Begin
  If current<>nil then
    While current^.right<>nil do
      current:=current^.right;
End;


Procedure delrec(var root, current : recptr);

Var
  tr : recptr;

{ delete the current record }

Begin
  tr:=current;
  If tr<>nil then
    Begin

   { replace the current node with the right hand branches }

      If tr^.right<>nil then
        Begin
          current := tr^.right;
          firstrec(current);
          current^.left := tr^.left;
          tr^.right^.upper := tr^.upper;
          If tr^.left<>nil then
            tr^.left^.upper:=current;
          If tr^.upper<>nil then
            If tr^.upper^.left=tr then
              tr^.upper^.left := tr^.right
            Else
              tr^.upper^.right := tr^.right
          Else
            root:=tr^.right;
        End

   { replace the current node with the left hand branches }

      Else
        If tr^.left<>nil then
          Begin
            tr^.left^.upper  :=  tr^.upper;
            If tr^.upper<>nil then
              If tr^.upper^.right=tr then
                tr^.upper^.right := tr^.left
              Else
                tr^.upper^.left := tr^.left
            Else
              root := tr^.left;
          End

   { current node is a leaf }

        Else
          If tr^.upper<>nil then
            If tr^.upper^.right=tr then
              tr^.upper^.right := nil
            Else
              tr^.upper^.left := nil

   { current node is only node in tree }

          Else
            root:=nil;

   { always return the "parent" as current }

      current := tr^.upper;

   { free up the memory }

      freemem(tr,length(tr^.data)+1+sizeof(lh));
    End;
End;


Function findrec(var current : recptr; s : string) : boolean;

{ find the record that matches the string }

Begin
  findrec := false;
  While current<>nil do
    If current^.data=s then
      Begin
        findrec := true;
        exit;
      End
    Else
      If current^.data>s then
        If current^.left<>nil then
          current := current^.left
        Else
          exit
      Else
        If current^.right<>nil then
          current := current^.right
        Else
          exit;
End;


Procedure nextrec(var current : recptr);

{ makes current the record following current record }

Begin
  If current<>nil then
    If current^.right<>nil then
      Begin
        current := current^.right;
        While current^.left<>nil do
          current := current^.left;
      End
    Else
      Begin
        While (current^.upper<>nil) and (current^.upper^.right=current) do
          current := current^.upper;
        If current^.upper<>nil then
          current := current^.upper
        Else
          current := nil;
      End;
End;


Procedure prevrec(var current : recptr);

{ makes current the record preceding current record }

Begin
  If current<>nil then
    If current^.left<>nil then
      Begin
        current := current^.left;
        While current^.right<>nil do
          current := current^.right;
      End
    Else
      Begin
        While (current^.upper<>nil) and (current^.upper^.left=current) do
          current := current^.upper;
        If current^.upper<>nil then
          current := current^.upper
        Else current := nil;
      End;
End;


Function storec(var root : recptr; var s : string) : recptr;

{ stores a btree record }

Var
  current,
  newrec : recptr;

Function incrrec:boolean;

{ if record already exists, just increment its count '.c' }

Begin
  current := root;
  incrrec := true;
  If findrec(current,s) then
    inc(current^.counter)
  Else
    incrrec := false;
End;

Begin { procedure storec }

   { if the record already exists, just increment the count c }

  If not incrrec then

   { if there is enough room to store the record }

    If (maxavail>=(length(s)+1+grain+sizeof(lh))) then
      Begin

   { allocate room for the record }

        Getmem(newrec,length(s)+1+sizeof(lh));
        newrec^.counter := 1;
        newrec^.right := nil;
        newrec^.left := nil;
        storec := newrec;

   { store the record into the btree }

        newrec^.data := s;
        newrec^.upper := current;
        If current<>nil then
          If current^.data>s then
            current^.left := newrec
          Else
            current^.right := newrec
        Else
          root := newrec;
      End
    Else
      storec := nil
  Else
    storec := current;
End; { procedure storec }

End.
