// Copyright (C) 2000 Phil Howlett <pghowlett@ozemail.com.au>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.#

//
// database.cpp
//

#include <unistd.h>
#include <stdarg.h>
#include <time.h>
#include <stdio.h>

#include <mysql/mysql.h>

#include "nicole.h"
#include "database.h"
#include "dbresult.h"

/**
 * Constructs a MySQL database object. Also setups default
 * values for the object's variables
 */

Database::Database (  )
{
	connected = false;
	error = "No connection established";
	
	//set default user
	username = "root";
	password = "";
}

/**
 * Sets the username and password for the MySQL Database connection
 * \param *user the user connecting to the database
 * \param *pwd the password that this user will connect as.
 */
void Database::SetUser(const string *user, const string *pwd)
{
  username = *user;
  password = *pwd;
}

/**
 * Initialises the database connection (not in use yet)
 */
int Database::init (  )
{
	return DB_COMMAND_OK;
}

/**
 * Gets the error message from the mysql database
 *
 * \return the error message string
 */

string Database::error_message (  )
{
	if ( connected == false )
	{
		return error;
	}

	return mysql_error ( &mysql );

}

/**
 * Establish a connection to the database
 *
 * \param host The name of the host the database is located
 * \param db The name of the database we wish to connect to
 *
 * \retval DB_ERROR There was an error connecting to the requested database
 * \retval DB_CONNECTION_OK Connection successful.
 */

int Database::connect ( string host, string db )
{
	if ( mysql_connect ( &mysql, host.c_str (  ), DB_USER_NAME, DB_USER_PWD ) ==
			 NULL )
	{
		error = "Connect to database failed";
		return DB_ERROR;
	}

	if ( mysql_select_db ( &mysql, db.c_str (  ) ) )
	{
		mysql_close ( &mysql );
		error = "No such database";
		return DB_ERROR;
	}

	connected = true;
	return DB_CONNECTION_OK;
}

/**
 * Find the status of the current database
 *
 * \retval DB_CONNECTION_NONE No connection established
 * \retval DB_CONNECTION_OK Connection established
 */

int Database::status (  )
{
	if ( connected == false )
	{
		return DB_CONNECTION_NONE;
	}
	return DB_CONNECTION_OK;
}

/**
 * Disconnects from a MySQL database
 */

void Database::disconnect (  )
{
	if ( connected == false )
	{
		return;
	}
	mysql_close ( &mysql );
	connected = false;
}

/**
 * Resets the database connection (doesn't do anything at this stage)
 */

int Database::reset (  )
{
	return DB_COMMAND_OK;
}

/**
 * Executes a MySQL statement
 *
 * \param sqlFormat The SQL Command
 *
 * \return The DBResult Object
 */

DBResult *Database::exec ( char *sqlFormat, ... )
{
	va_list ap;
	char sqlCommand[5000];

	va_start ( ap, sqlFormat );
	vsnprintf ( sqlCommand, 5000, sqlFormat, ap );
	va_end ( ap );

	//cerr << "EXEC: " << sqlCommand << endl;

	return new DBResult ( &mysql, sqlCommand );
}

/**
 * Executes a MySQL statement
 *
 * \param res The DBResult object that will contain the query result
 * \param sqlFormat The SQL Command
 *
 * \return The DBResult Object
 */

void Database::exec ( DBResult * res, char *sqlFormat, ... )
{
	va_list ap;
	char sqlCommand[5000];

	va_start ( ap, sqlFormat );
	vsnprintf ( sqlCommand, 5000, sqlFormat, ap );
	va_end ( ap );

	//cerr << "EXEC: ->" << sqlCommand << "<-" << endl;

	res->init ( &mysql, sqlCommand );

}

/**
 * Destructs a MySQL database object
 */
Database::~Database (  )
{
	disconnect (  );							// Disconnect if connected to database
}
