// Copyright (C) 2000 Phil Howlett <pghowlett@ozemail.com.au>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.#

//
// parser.cpp
//

#include <string>
#include <iostream>
using namespace std;

#include "parser.h"
#include "dictionary.h"
#include "vocabulary.h"

// constructor

/**
 * Initialises the parser object, clearing member variables.
 */

Parser::Parser (  )
{
	inputline.erase ( inputline.begin (  ), inputline.end (  ) );
}

/**
 * resets the data values in the object
 */

void Parser::reset (  )
{
	inputline.erase ( inputline.begin (  ), inputline.end (  ) );
	cin.clear();
}

/**
 * this method accepts the raw input and outputs a formatted string, cleaning
 * it of unwanted characters
 * \return the new formatted string
 */

string Parser::clean_input ( string rawinput )
{
	int i;

	string newstring;

	int size = rawinput.size (  );

	
	//cout << "BEFORE: >" << rawinput << "<" << endl;
	
	// loop through the string

	for ( i = 0; i < size; i++ )
	{
		//cerr << "CHAR ->" << rawinput[i] << "<-" << endl;
		// accept alpha characters
		if ( ( isalpha ( rawinput[i] ) || ( rawinput[i] == '\'' ) ) )
		{
			newstring += toupper ( rawinput[i] );
		}
		else if ( rawinput[i] == ' ' )
		{
			// we only allow a space if the previous character isn't a
			// space and not the first char in our new string
			if ( ( newstring[newstring.size (  ) - 1] != ' ' ) &&
					 ( newstring.size (  ) != 0 ) )
			{
				newstring += rawinput[i];
			}
		}
		else if (rawinput[i] == '.')
		{
			newstring += " .";
		}
		else if ( (i == 0) && (rawinput[i] == '/') )
		{
			// allow the slash key because it is a command string
			newstring += rawinput[i];
		}
		else if ((i > 0) && (rawinput[i] == ','))
		{
		  // allow a comma as a seperate word
		  newstring += " ,";
		}
	}

	// strip off the trailing spaces
	while ( newstring[newstring.size (  ) - 1] == ' ' )
	{
		newstring[newstring.size (  ) - 1] = '\0';
	}

	//cout << "AFTER: >" << newstring << "<" << endl;
	
	return newstring;

}

/**
 * gets input from cin and places it into the rawinput string
 */

void Parser::input (  )
{
	string rawinput;
	getline ( cin, rawinput );
	
	// get the formated input
	inputline = clean_input ( rawinput );
	
}

/**
 * \return 1 if the input is empty
 * \return 0 if the input is not empty
 */

int Parser::empty (  )
{
	if ( inputline.size (  ) == 0 )
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

/**
 * \return the current value of the inputline
 */

string Parser::inputvalue (  )
{
	return inputline;
}

/**
 * \return the command parameter (the second word)
 */

string Parser::parameter (  )
{
	string param;
	
	int i;

	int foundspace = 0;
	
	for(i = 0; i< inputline.size(); i++)
	{
		if(foundspace)
		{
			param += inputline.at(i);
		}
		if(inputline.at(i) == ' ')
		{
			foundspace = 1;
		}
//		cout << "char >" << inputline.at(i) << "<" << endl;
	}

//	cout << "Parameter:>" << param << "<" << endl;
	return param;
}

/**
 * Works out if the currently input is a command.
 * \retval 1 Command was inputed
 * \retval 0 No command inputed.
 */

int Parser::is_command (  )
{
	if ( inputline.at ( 0 ) == '/' )	// we will use the slash '/' for now.
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

/**
 * Works out if the currently entered command is 'QUIT'
 * \retval 1 current command is 'QUIT'
 * \retval 0 current command is not 'QUIT'
 */

int Parser::cmd_quit (  )
{
	if ( inputline == "/QUIT" )
		return 1;
	else
		return 0;
}

/**
 * Works out if the currently entered command is 'HELP'
 * \retval 1 current command is 'HELP'
 * \retval 0 current command is not 'HELP'
 */

int Parser::cmd_help (  )
{
	if ( inputline == "/HELP" )
		return 1;
	else
		return 0;
}

/**
 * Works out if the currently entered command is 'STATUS'
 * \retval 1 current command is 'STATUS'
 * \retval 0 current command is not 'STATUS'
 */

int Parser::cmd_status (  )
{
	if ( inputline == "/STATUS" )
		return 1;
	else
		return 0;
}

/**
 * Works out if the currently entered command is 'INFO'
 * \retval 1 current command is 'INFO'
 * \retval 0 current command is not 'INFO'
 */

int Parser::cmd_info (  )
{
	if ( inputline.substr(0, 5) == "/INFO" )
		return 1;
	else
		return 0;
}

/**
 * Works out if the currently entered command is 'RESET'
 * \retval 1 current command is 'RESET'
 * \retval 0 current command is not 'RESET'
 */

int Parser::cmd_reset (  )
{
	if ( inputline == "/RESET" )
		return 1;
	else
		return 0;
}

/**
 * Works out if the currently entered command is 'ERASE ME'
 * \retval 1 current command is 'ERASE ME'
 * \retval 0 current command is not 'ERASE ME'
 */

int Parser::cmd_erase (  )
{
	if ( inputline == "/ERASE" )
	{
		cout << "If you REALLY want to erase NICOLE, then use /ERASE ME" << endl;
		return 0;
	}

	if ( inputline == "/ERASE ME" )
		return 1;
	else
		return 0;
}

/**
 * Display the help screen to stdout
 */

void Parser::show_help (  )
{
	cout << "Commands:" << endl
		<< "---------" << endl
		<< "/HELP     - Shows this help screen" << endl
		<< "/STATUS   - Display the current status of NICOLE" << endl
		<< "/INFO <w> - Show information about word <w>" << endl
		<< "/RESET    - Resets NICOLE's conversation path (warm boot)" << endl
		<< "/ERASE ME - Brainwashes NICOLE. **WARNING: Unrecoverable!**" << endl
		<< "/QUIT     - Quits NICOLE" << endl << endl;

}

/**
 * Display the status of NICOLE to stdout
 */

void Parser::show_status (  )
{
	Dictionary d;
	Vocabulary v;

	cout << "Generating status. Please wait..." << endl << endl;

	cout << "NICOLE Status:" << endl
		<< "--------------" << endl
		<< "Dictionary size: " << d.count (  ) << " words." << endl
		<< "Vocabulary size: " << v.count (  ) << " phrases." << endl << endl;

	cout << "done." << endl << endl;
}

/**
 * Display the info of a word to stdout
 */

void Parser::show_info( string s )
{
	Dictionary d;
	Vocabulary v;
	int i;
	
	if(s.size() != 0)
	{
		cout << endl;
		cout << "Info on word: \"" << s << "\"" << endl << endl;;
		
		cout << "Number of times used: " << d.freq(s) << endl;

// Comment'd out until I can look at this closer - Phil.
//
//		if (d.exists(s))
//		{
			// If the word doesn't exist, then there is no point in going further
			// is there?		
//			cout << "Matrix levels:" << endl;
//			for(i=1; i<=5; i++)
//			{
//				cout << "- Level " << i << ": " << v.freq(i, s) << endl;
//			}
//		}
		cout << endl;
	}
	else
	{
		cout << "Error: Missing parameter" << endl << endl;
		cout << "Usage:" << endl;
		cout << "      /INFO <word>   - where <word> is a word in the dictionary" << endl;
		cout << endl;
	}
}

/**
 * Erases the contents on NICOLE's brain. Not a very nice thing to do
 */

void Parser::erase_nicole (  )
{
	Dictionary d;
	Vocabulary v;

	cout << "Brainwashing NICOLE..." << endl;

	d.erase (  );

	cout << "AAAAAAAAAaaaaaaaaaaaaaaaaaaarrrrrrrrrrrgggggggggghhhhhhhhhh!!!!!"
		<< endl;

	v.erase (  );

	cout << "NICOLE is now brain dead." << endl;
}

/**
 * Destructs the parser object.
 * This doesn't do much as I don't dynamically create variables here.
 */

Parser::~Parser (  )
{
}
