/*  encrypt -- blowfish encrypt a file, include an SHA1 thumbprint
    Copyright (C) 2000 Synack Systems Corporation
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
 
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
 
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <time.h>
#include <netinet/in.h>
#include "common.h"
#include "sha1.h"
#include "blowfish.h"
#include "rijndael-api-fst.h"

char *path, *home;
u32 keylen;

int mangle_filenames = 1;
int wipe_plaintext = 0;
int unlink_plaintext = 1;
int be_verbose = 0;
int use_blowfish = 0;

int bfe(char *path, byte *key) {
    char *outpath=NULL, *asckey=NULL, b[3];
    byte *inbuf=NULL, *outbuf=NULL, *ptr, *s;
    int blocksz, i, status=0;
    FILE *in = NULL, *out = NULL;
    BLOWFISH_context bf;
    cipherInstance ci;
    keyInstance ki;
    size_t cnt;
    struct stat st;
    SHA1_CONTEXT sha1;
    u32 r;
    FileHeader fh;
    byte thumb[40];

    if(use_blowfish) {
        if(bf_setkey(&bf, key, keylen) != 0) {
            fprintf(stderr, "error setting blowfish key.\n");
            status=1; goto cleanup;
        }
        blocksz = 8;
        inbuf = malloc(blocksz);
        outbuf = malloc(blocksz);
        if(!(inbuf && outbuf)) {
            fprintf(stderr, "malloc didn't work.\n");
            status=1; goto cleanup;
        }
    } else {

        asckey = malloc(keylen*2 + 1);
        *asckey = '\0';
        for(i=0;i<keylen;i++) {
            sprintf(b, "%02X", key[i]);
            strcat(asckey, b);
        }

        switch(keylen) {
        case 24:
            i = 192; break;
        case 56:
            i = 256; break;
        default:
            i = 128;
        }

        if(makeKey(&ki, DIR_ENCRYPT, i, asckey) != TRUE) {
            fprintf(stderr, "error setting rijndael key.\n");
            status=1; goto cleanup;
        }
        if(cipherInit(&ci, MODE_ECB, NULL) != TRUE) {
            fprintf(stderr, "error initializing cipher.\n");
            status=1; goto cleanup;
        }
        blocksz = 16;
        inbuf=malloc(blocksz);
        outbuf=malloc(blocksz);
        if(!(inbuf && outbuf)) {
            fprintf(stderr, "malloc didn't work.\n");
            status=1; goto cleanup;
        }
    }

    outpath = malloc(strlen(path)+4);
    strcpy(outpath, path);
    strcat(outpath, ".r");

    if((in = fopen(path, "rw")) == NULL) {
        fprintf(stderr, "unable to open %s for reading\n", path);
        status=2; goto cleanup;
    }

    fstat(fileno(in), &st);
    if(!S_ISREG(st.st_mode)) {
        fprintf(stderr, "skipping %s; not a regular file\n", path);
        status=3; goto cleanup;
    }

    if(mangle_filenames) {
        out = frandcreate();
    } else {
        out = fcreate(outpath);
    }

    if(out == NULL) {
        fprintf(stderr, "unable to create a new file.\n");
        status=4; goto cleanup;
    }

    /* setup the file header */
    memset(&fh, 0, sizeof(fh));
    r = randu32();
    memcpy(fh.random, &r, 2);
    if(use_blowfish)
        fh.magic = BLOWFISH_MAGIC_NUM;
    else
        fh.magic = RIJNDAEL_MAGIC_NUM;
    fh.fsize = htonl(st.st_size);
    fh.fmode = htonl(st.st_mode);
    s = strrchr(path, '/');
    if(s) s++;
    else s = path;
    strncpy(fh.fname, s, 127);
    /* file header is setup */

    if(use_blowfish) cnt = BLOWFISH_HEADER_SZ;
    else cnt = RIJNDAEL_HEADER_SZ;
    ptr = (byte*) &fh;

    for(i=0;i<cnt;i+=blocksz) {
        if(use_blowfish) encrypt_block(&bf, outbuf, ptr+i);
        else blockEncrypt(&ci, &ki, ptr+i, blocksz*8, outbuf);
        fwrite(outbuf, 1, blocksz, out);

    }

    sha1_init(&sha1);

    do {
        memset(inbuf, 0, blocksz);
        cnt = fread(inbuf, 1, blocksz, in);
        sha1_write(&sha1, inbuf, cnt);
        if(cnt) {
            if(use_blowfish) encrypt_block(&bf, outbuf, inbuf);
            else blockEncrypt(&ci, &ki, inbuf, blocksz*8, outbuf);
            fwrite(outbuf, 1, blocksz, out);
        }
    } while(cnt);

    sha1_final(&sha1);
    ptr = sha1_read(&sha1);
    memset(thumb, 0, sizeof(thumb));
    for(i=0;i<5;i++)
        memcpy(thumb+(i*8), ptr+(i*4), 4);
    // memcpy(thumb, sha1_read(&sha1), sizeof(thumb));

    //if(be_verbose) {
    //    for(i=0;i<20;i++) fprintf(stderr, "%02X", thumb[i]);
    //    fprintf(stderr, "\n");
    //}
    for(i=0;i<40;i+=blocksz) {
        if(use_blowfish) encrypt_block(&bf, outbuf, thumb+i);
        else blockEncrypt(&ci, &ki, thumb+i, blocksz*8, outbuf);
        fwrite(outbuf, 1, blocksz, out);
    }

    /* if we have a blocksz of 16 then 4 bytes of the thumbprint will not
     * have been written by the above loop.. we write those now.. */
    if(blocksz == 16) {
        memset(inbuf, 0, blocksz);
        memcpy(inbuf, thumb+32, 8);
        blockEncrypt(&ci, &ki, inbuf, blocksz*8, outbuf);
        fwrite(outbuf, 1, blocksz, out);
    }

    if(wipe_plaintext) wipe(in, st.st_size);

cleanup:
    if(outpath) free(outpath);
    if(out) fclose(out);
    if(in) fclose(in);
    if(inbuf) {
        memset(inbuf, 0, blocksz);
        free(inbuf);
    }
    if(outbuf) {
        memset(outbuf, 0, blocksz);
        free(outbuf);
    }
    if(asckey) {
        memset(asckey, 0, keylen*2);
        memset(b, 0, 2);
    }
    memset(&ci, 0, sizeof(ci));
    memset(&ki, 0, sizeof(ki));
    memset(&bf, 0, sizeof(bf));
    memset(&sha1, 0, sizeof(sha1));
    memset(&fh, 0, sizeof(fh));
    memset(&thumb, 0, sizeof(thumb));

    return status;
}

void usage() {
    fprintf(stderr, "usage: encrypt [<options>] [file file file ...]\n");
    fprintf(stderr, "options:\n");
    fprintf(stderr, "    -b\t\tUse Blowfish to encrypt files\n");
    fprintf(stderr, "    -d\t\tDon't mangle file names\n");
    fprintf(stderr, "    -u\t\tDon't unlink plaintext files (-w overrides)\n");
    fprintf(stderr, "    -v\t\tEnable verbose output\n");
    fprintf(stderr, "    -w\t\tWipe away plain text after encrypting\n");
    exit(-1);
}

int main(int argc, char **argv) {
    int c;
    int i;
    byte * bfkey;

#ifndef DEBUGGING_NOW
    struct rlimit rlim;
    memset(&rlim, 0, sizeof(rlim));
    setrlimit(RLIMIT_CORE, &rlim);
#endif

    while(1) {
        c = getopt(argc,argv,"budvw");
        if(c == -1) break;
        switch(c) {
        case 'b':
            use_blowfish = 1;
            break;
        case 'd':
            mangle_filenames = 0;
            break;
        case 'u':
            unlink_plaintext = 0;
            break;
        case 'v':
            be_verbose = 1;
            break;
        case 'w':
            wipe_plaintext = 1;
            unlink_plaintext = 1;
            break;
        default :
            usage();
        }
    }

    if(optind >= argc) usage();

    bfkey = getbfkey(&keylen);
    if(!bfkey) {
        fprintf(stderr, "Couldn't get a key.  Run key.\n");
        exit(1);
    }

    for(i=optind;i<argc;i++) {
        if(!bfe(argv[i], bfkey)) {
            if(unlink_plaintext || wipe_plaintext) {
                unlink(argv[i]);
            }
        }
    }

    memset(bfkey, 0, keylen);


    return 0;
}
