/*  key -- accepts a password or consults other data sources and serves a
    variable size keys for Refugee programs

    Copyright (C) 2000 Synack Systems Corporation
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
 
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
 
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/stat.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <utime.h>
#include <signal.h>
#include <fcntl.h>
#include "common.h"
#include "sha1.h"
#include "tiger.h"
#include "md5.h"

byte * bfkey = NULL;
int bfsock = 0;
byte key_hashing = KEY_SHA1;
int go_daemon = 1;
int we_are_generating = 0;
int key_from_path = 0;
char *key_path = NULL;
int key_from_device = 0;
int be_verbose = 0;
char *key_device = NULL;
time_t key_lifespan = 60;
SHA1_CONTEXT sha1;
TIGER_CONTEXT tiger;
MD5_CONTEXT md5;

void quit(int sig) {
    if(bfkey) {
        if(be_verbose) fprintf(stderr, "\nDestroying key.\n");

        switch(key_hashing) {
        case KEY_SHA1:
            memset(bfkey, 0, 20);
            break;
        case KEY_TIGER:
            memset(bfkey, 0, 24);
            break;
        case KEY_LAME32:
            bfkey = NULL;
            break;
        case KEY_MD5:
            memset(bfkey, 0, 16);
            break;
        case KEY_RAND448:
            memset(bfkey, 0, sizeof(u32)*14);
            free(bfkey);
            break;
        }
    }

    if(key_path) {
        memset(key_path, 0, strlen(key_path));
        free(key_path);
    }

    if(key_device) {
        memset(key_device, 0, strlen(key_device));
        free(key_device);
    }

    if(bfsock) close(bfsock);

    exit(4);
}

void generate() {
    char path[1024];
    int fd;

    u32 *r;
    printf("I will now create a random key that you can use to encrypt\n");
    printf("and decrypt your files.  We recommend that you store this\n");
    printf("key on a removable device such as a floppy disk.\n\n");
prompter:
    printf("Where would you like your new key to be stored?\n");
    printf("path [/dev/fd0]: ");

    fgets(path, 1023, stdin);

    printf("\n");
    if(path[0] == '\n') strcpy(path, "/dev/fd0");

    fd = open(path, O_RDWR, 0600);
    if(fd < 0) {
        printf("\nCouldn't open %s for writing.\n", path);
        goto prompter;
    }


    r = random_key();

    /* identifying/placeholder data */
    write(fd, "REFUGEE", 8);
    write(fd ,r, sizeof(u32)*14);
    close(fd);

    memset(r, sizeof(u32)*14, 0);
    free(r);

    exit(0);
}

void usage() {
    fprintf(stderr, "usage: key [<options>]\n");
    fprintf(stderr, "options:\n");
    fprintf(stderr, "    -b\t\tDo not run in the background (daemonize)\n");
    fprintf(stderr, "    -d <dev>\tRead a key from specified device (eg. /dev/fd0)\n");
    fprintf(stderr, "    -f <path>\tMake a key from 'path' contents instead of a password\n");
    fprintf(stderr, "    -g\t\tGenerate a 448 bit blowfish key (implies -b)\n");
    fprintf(stderr, "    -l\t\tUse a lame hash to provide a 32 bit key\n");
    fprintf(stderr, "    -m\t\tUse MD5 to provide a 128 bit key\n");
    fprintf(stderr, "    -s\t\tUse SHA1 to provide a 160 bit key\n");
    fprintf(stderr, "    -t\t\tUse TIGER to provide a 192 bit key\n");
    fprintf(stderr, "    -v\t\tEnable verbose output\n");
    fprintf(stderr, "    -x <num>\tkey should expire (quit) in num minutes (default 60)\n");
    exit(-1);
}

void daemonize() {
    pid_t pid;
    if((pid=fork())) {
        printf("key is moving to the background.");
        printf(" (%d)\n", pid);
        fflush(stdout);

        if(bfkey) {
            switch(key_hashing) {
            case KEY_SHA1:
                memset(bfkey, 0, 20);
                break;
            case KEY_TIGER:
                memset(bfkey, 0, 24);
                break;
            case KEY_LAME32:
                bfkey = NULL;
                break;
            case KEY_MD5:
                memset(bfkey, 0, 16);
                break;
            case KEY_RAND448:
                memset(bfkey, 0, sizeof(u32)*14);
                free(bfkey);
                break;
            }
        }
        exit(0);
    }

    fclose(stdin);
    fclose(stdout);
    chdir("/");
}

void makekey_fromdevice() {
    int fd;
    char s[8];
    fd = open(key_device, O_RDONLY, 0);
    if(fd < 0) {
        fprintf(stderr, "Couldn't open %s for reading.\n", key_device);
        quit(23);
    }

    /* identifying/placeholder data */
    read(fd, s, 8);
    if(strcmp(s, "REFUGEE")) {
        fprintf(stderr, "%s doesn't is corrupt or doesn't contain a refugee key.\n", key_device);
        quit(24);
    }
    bfkey = malloc(sizeof(u32)*14);
    read(fd ,bfkey, sizeof(u32)*14);

    close(fd);
}

void makekey_fromfile() {
    int f;
    ssize_t cnt;
    byte buf[1024];
    struct stat st;
    struct utimbuf timbuf;

    printf("making a key from %s\n", key_path);

    stat(key_path, &st);

    switch(key_hashing) {
    case KEY_SHA1:
        sha1_init(&sha1);
        break;
    case KEY_TIGER:
        tiger_init(&tiger);
        break;
    case KEY_LAME32:
        fprintf(stderr,"You cannot use the -f and -l options together.\n");
        quit(100);
    case KEY_MD5:
        md5_init(&md5);
        break;
    }

    f = open(key_path, O_RDONLY, 0);
    while((cnt=read(f, buf, 1024))) {
        switch(key_hashing) {
        case KEY_SHA1:
            sha1_write(&sha1, buf, cnt);
            break;
        case KEY_TIGER:
            tiger_write(&tiger, buf, cnt);
            break;
        case KEY_MD5:
            md5_write(&md5, buf, cnt);
            break;
        }

    }

    switch(key_hashing) {
    case KEY_SHA1:
        sha1_final(&sha1);
        bfkey=sha1_read(&sha1);
        break;
    case KEY_TIGER:
        tiger_final(&tiger);
        bfkey=tiger_read(&tiger);
        break;
    case KEY_MD5:
        md5_final(&md5);
        break;
        bfkey=md5_read(&md5);
        break;
    }

    close(f);
    timbuf.actime = st.st_atime;
    timbuf.modtime = st.st_mtime;
    utime(key_path, &timbuf);

    memset(key_path, 0, strlen(key_path));
}

void makekey_frompassword() {
    char *pass, *repeat;
repeater:
    pass = strdup(getpass("Password: "));
    repeat = getpass("Repeat: ");
    if(strcmp(pass,repeat)) {
        fprintf(stderr, "Passwords didn't match.\n");
        goto repeater;
    }

    switch(key_hashing) {
    case KEY_SHA1:
        sha1_init(&sha1);
        sha1_write(&sha1, pass, strlen(pass));
        sha1_final(&sha1);
        bfkey = sha1_read(&sha1);
        break;
    case KEY_TIGER:
        tiger_init(&tiger);
        tiger_write(&tiger, pass, strlen(pass));
        tiger_final(&tiger);
        bfkey = tiger_read(&tiger);
        break;
    case KEY_LAME32:
        bfkey = (byte*)lamehash(pass);
        break;
    case KEY_MD5:
        md5_init(&md5);
        md5_write(&md5, pass, strlen(pass));
        md5_final(&md5);
        bfkey = md5_read(&md5);
        break;
    }

    memset(repeat, 0, strlen(repeat));
    memset(pass, 0, strlen(pass));
    free(pass);
}

int main(int argc, char **argv) {
    int i,c;

    /* for UNIX domain sockets */
    struct sockaddr_un saun, fsaun;
    int ns;
    size_t len;
    byte cmd;


#ifndef DEBUGGING_NOW
    struct rlimit rlim;
    memset(&rlim, 0, sizeof(rlim));
    setrlimit(RLIMIT_CORE, &rlim);
#endif

    for(i=0;i<_NSIG;i++)
        signal(i, quit);

    while(1) {
        c = getopt(argc,argv, "bd:f:glmstx:v");
        if(c == -1) break;
        switch(c) {
        case 'b':
            go_daemon = 0;
            break;
        case 'd':
            key_from_device = 1;
            key_device = malloc(strlen(optarg)+1);
            strcpy(key_device, optarg);
            memset(optarg, 0, strlen(optarg));
            key_hashing = KEY_RAND448;
            break;
        case 'f':
            key_from_path = 1;
            key_path = malloc(strlen(optarg)+1);
            strcpy(key_path, optarg);
            memset(optarg, 0, strlen(optarg));
            break;
        case 'g':
            we_are_generating = 1;
            break;
        case 'l':
            key_hashing = KEY_LAME32;
            break;
        case 'm':
            key_hashing = KEY_MD5;
            break;
        case 's':
            key_hashing = KEY_SHA1;
            break;
        case 't':
            key_hashing = KEY_TIGER;
            break;
        case 'v':
            be_verbose = 1;
            break;
        case 'x':
            key_lifespan = atoi(optarg);
            break;
        default :
            usage();
        }
    }

    if(we_are_generating) generate();
    else if(key_from_device) makekey_fromdevice();
    else if(key_from_path) makekey_fromfile();
    else makekey_frompassword();

    if((bfsock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
        perror("socket");
        quit(0);
    }

    saun.sun_family = AF_UNIX;
    strncpy(saun.sun_path, getenv("HOME"), sizeof(saun.sun_path)-1);
    strncat(saun.sun_path, "/.refugee_key", sizeof(saun.sun_path)-strlen(saun.sun_path)-1);
    unlink(saun.sun_path);

    len = sizeof(saun.sun_family) + strlen(saun.sun_path) + 1;
    if(bind(bfsock, (struct sockaddr *)&saun, len) < 0) {
        perror("bind");
        quit(0);
    }

    if(chmod(saun.sun_path, 0600)) {
        perror("fchmod");
        quit(0);
    }

    if(listen(bfsock,5) < 0) {
        perror("listen");
        quit(0);
    }

    if(go_daemon) daemonize();

    /* this program will terminate in 1 hour */
    alarm(key_lifespan*60);

    while(1) {
        if((ns = accept(bfsock, (struct sockaddr *)&fsaun, &len)) < 0) {
            perror("accept");
        } else {
            /* we are ignoring cmd for now.. it is simply here so
               bfkey may be able to do different things in 
               the future 
             */
            recv(ns, &cmd, 1, 0);
            send(ns, &key_hashing, 1, 0);
            switch(key_hashing) {
            case KEY_SHA1:
                send(ns, bfkey, 20, 0);
                break;
            case KEY_TIGER:
                send(ns, bfkey, 24, 0);
                break;
            case KEY_LAME32:
                send(ns, &bfkey, 4, 0);
                break;
            case KEY_MD5:
                send(ns, bfkey, 16, 0);
                break;
            case KEY_RAND448:
                send(ns, bfkey, 56, 0);
                break;
            }
            close(ns);
        }
    }


    return 0;
}
