/*  recat -- decrypt a Refugee encrypted file and dump it to stdout
    Copyright (C) 2000 Synack Systems Corporation
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
 
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
 
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <time.h>
#include <netinet/in.h>
#include "common.h"
#include "sha1.h"
#include "blowfish.h"
#include "rijndael-api-fst.h"

u32 keylen;
int use_blowfish = 0;

int bfd(char *path, byte *key) {
    byte *inbuf=NULL, *outbuf=NULL, *ptr, thumb[40], newthumb[40];
    FILE *in;
    size_t cnt;
    struct stat st;
    char *s, *asckey=NULL, b[3];
    int i, status=0, blocksz;
    SHA1_CONTEXT sha1;
    BLOWFISH_context bf;
    cipherInstance ci;
    keyInstance ki;
    u32 stsize;
    FileHeader fh;

    s = strstr(path, ".r");
    if((s == NULL) || (*(s+2) != '\0')) {
        /* for backwards compatibility  look for .bf */
        s = strstr(path, ".bf");
        if((s == NULL) || (*(s+3) != '\0')) {
            fprintf(stderr, "skipping %s because it doesn't end in .r\n", path);
            status=2; goto cleanup;
        }
    }

    if((in = fopen(path, "r")) == NULL) {
        fprintf(stderr, "unable to open %s for reading\n", path);
        status=3; goto cleanup;
    }

    fstat(fileno(in), &st);
    if(!S_ISREG(st.st_mode)) {
        fprintf(stderr, "skipping %s; not a regular file\n", path);
        status=4; goto cleanup;
    }


    /* we will try to blowfish decrypt this file first.. if that doesn't work
     * then we will try to rijndael decrypt it..
     *
     * as rijndael becomes more popular this will be reversed..
     */
    if(bf_setkey(&bf, key, keylen) != 0) {
        fprintf(stderr, "error setting blowfish key.\n");
        status=1; goto cleanup;
    }

    blocksz=8;
    inbuf = malloc(blocksz);
    outbuf = malloc(blocksz);
    if(!(inbuf && outbuf)) {
        fprintf(stderr, "malloc didn't work.\n");
        status=1; goto cleanup;
    }

    ptr = (byte*) &fh;
    for(i=0;i<BLOWFISH_HEADER_SZ;i+=blocksz) {
        fread(inbuf, 1, blocksz, in);
        decrypt_block(&bf, ptr+i, inbuf);
    }

    if(fh.magic == BLOWFISH_MAGIC_NUM) {
        use_blowfish = 1;
    } else {
        /* the file was couldn't be decrypted with blowfish so next we'll try
         * to decrypt it with rijndael
         */
        use_blowfish = 0;

        memset(inbuf, 0, blocksz);
        blocksz = 16;
        inbuf=malloc(blocksz);
        outbuf=malloc(blocksz);
        if(!(inbuf && outbuf)) {
            fprintf(stderr, "malloc didn't work.\n");
            status=1; goto cleanup;
        }

        asckey = malloc(keylen*2 + 1);
        if(!asckey) {
            fprintf(stderr, "malloc didn't work.\n");
            status=1; goto cleanup;
        }
        *asckey = '\0';
        for(i=0;i<keylen;i++) {
            sprintf(b, "%02X", key[i]);
            strcat(asckey, b);
        }

        switch(keylen) {
        case 24:
            i = 192; break;
        case 56:
            i = 256; break;
        default:
            i = 128;
        }           

        if(makeKey(&ki, DIR_DECRYPT, i, asckey) != TRUE) {
            fprintf(stderr, "error setting rijndael key.\n");
            status=1; goto cleanup;
        }
        if(cipherInit(&ci, MODE_ECB, NULL) != TRUE) {
            fprintf(stderr, "error initializing cipher.\n");
            status=1; goto cleanup;
        }

        /* okay.. all the rijndael stuff is setup.. let's try reading the
         * header again
         */
        fseek(in, 0, SEEK_SET);

        ptr = (byte*) &fh;
        for(i=0;i<RIJNDAEL_HEADER_SZ;i+=blocksz) {
            fread(inbuf, 1, blocksz, in);
            blockDecrypt(&ci, &ki, inbuf, blocksz*8, ptr+i);
        }

        if(fh.magic != RIJNDAEL_MAGIC_NUM) {
            fprintf(stderr, "bad magic number; is %s a refugee file?  right key?\n", path);
            status=5; goto cleanup;
        }
    }

    fh.fsize = ntohl(fh.fsize);
    fh.fmode = ntohl(fh.fmode);
    stsize = fh.fsize;

    sha1_init(&sha1);

    do {
        cnt = fread(inbuf, 1, blocksz, in);
        if(cnt == blocksz) {
            if(use_blowfish) decrypt_block(&bf, outbuf, inbuf);
            else blockDecrypt(&ci, &ki, inbuf, blocksz*8, outbuf);
            sha1_write(&sha1,outbuf,stsize < blocksz ? stsize : blocksz);
            fwrite(outbuf, 1, stsize > blocksz ? blocksz : stsize, stdout);
            stsize = cnt > stsize ? 0 : stsize-cnt;
        } else {
            fprintf(stderr, "unexpected end of file\n");
            status=5; goto cleanup;
        }
    } while(stsize);

    sha1_final(&sha1);
    ptr = sha1_read(&sha1);
    memset(newthumb, 0, sizeof(newthumb));
    for(i=0;i<5;i++)
        memcpy(newthumb+(i*8),ptr+(i*4), 4);

    for(i=0;i<40;i+=blocksz) {
        fread(inbuf, 1, blocksz, in);
        if(use_blowfish) decrypt_block(&bf, thumb+i, inbuf);
        else blockDecrypt(&ci, &ki, inbuf, blocksz*8, thumb+i);
    }

    /* WE DON'T NEED TO READ THE FINAL 8 bytes of the thumb right now
     * because it should be all zeros.
     *
     * if we have a blocksz of 16 then 4 bytes of the thumbprint will not
     * have been read by the above loop.. we read them now */
    /*
    if(blocksz == 16) {
        fread(inbuf, 1, blocksz, in);
        blockDecrypt(&ci, &ki, inbuf, blocksz*8, outbuf);
        memcpy(thumb+32, outbuf, 8);
        } */

    if(memcmp(thumb,newthumb, sizeof(thumb))) {
        fprintf(stderr, "SHA1 thumbprint doesn't match original.\n");
        fprintf(stderr, "Your data has most likely been corrupted/modified!\n");
        status=6; goto cleanup;
    }

cleanup:
    memset(&bf, 0, sizeof(bf));
    memset(&fh, 0, sizeof(fh));
    memset(&ci, 0, sizeof(ci));
    memset(&ki, 0, sizeof(ki));
    memset(&sha1, 0, sizeof(sha1));
    memset(newthumb, 0, sizeof(newthumb));
    memset(thumb, 0, sizeof(thumb));
    memset(b, 0, sizeof(b));

    if(inbuf) {
        memset(inbuf, 0, blocksz);
        free(inbuf);
    }
    if(outbuf) {
        memset(outbuf, 0, blocksz);
        free(outbuf);
    }

    if(asckey) {
        memset(asckey, 0, keylen*2);
        free(asckey);
    }

    blocksz=0;

    if(in) fclose(in);

    return status;
}

void usage() {
    fprintf(stderr, "usage: recat [file file file ...]\n");
    exit(-1);
}

int main(int argc, char **argv) {
    int i;
    byte *bfkey;


#ifndef DEBUGGING_NOW
    struct rlimit rlim;
    memset(&rlim, 0, sizeof(rlim));
    setrlimit(RLIMIT_CORE, &rlim);
#endif

    if(argc < 2) usage();

    if((bfkey = getbfkey(&keylen)) == NULL) {
        fprintf(stderr, "Couldn't get a key.  Run key.\n");
        exit(1);
    }

    for(i=1;i<argc;i++) {
        bfd(argv[i], bfkey);
    }

    memset(bfkey, 0, keylen);

    return 0;
}
