/*
**  COBOL-74 scanner 
**  Scanner para o Compilador Cobol
*/

%{
#include "htcoblib.h"
#include "htglobals.h"

// #define DEBUG_SCANNER 		1

#define MAX_COPY_DEPTH 		10
#define MAX_CPYNAME_LEN 	65
#define MAXLINEBUF         	256

void copy_switch( char *filename );
void scdebug( char *fmt, ... );
#define lexchar input

extern int lexgetc(void);

static struct {
	YY_BUFFER_STATE buffer;
	int line;
	struct copy_symbols *last_symbol;
} file_stack[10];
static int fsp=0;
int source_debug=0;

static struct copy_symbols {
	struct copy_symbols *next;
	char *value;
	char *replacement;
} *copy_symbols=NULL;
struct copy_symbols *append_copy_symbols=NULL;
	
char *include_filename;
char include_full_filename[256];
extern FILE *o_lst;
int lex_fgets( char *buf, int maxsize );

#define YY_INPUT(buf,result,max_size) \
	{ \
		result = lex_fgets( buf, max_size ); \
	}

%}

letters [A-Za-z]
alphanum [A-Za-z0-9]
exletter {letters}|-
digit [0-9]
pic_ch [9XVSBZAP.,-+*/$]
relop [<>=]
white [\r\t ]
lpar \(
rpar \)
sdelim \"
ldelim \'
any [!-~]
int {digit}({digit})*
decdelim [.,]
sign [+-]

%{
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "htcoboly.h"
#include "htcobol.h"	/* definitions from YACC */

char toktext[100];
char *tp;
unsigned int varnumber;
int reserved_token;
int lineno=0;			/* source line */
int previous_state = 0;

struct {
	char *name;
	int token;
	unsigned minor;
} reserved_symbols[] = {
	{"GLOBAL",GLOBAL,0},{"EXTERNAL",EXTERNAL,0},
	{"SEPARATE",SEPARATE,0},{"LEFT",LEFT,0},{"TRAILING",TRAILING,0},
	{"CHARACTER",CHARACTER,0},{"PIC",PIC,0},{"PICTURE",PIC,0},
	{"SCREEN",SCREEN,0},{"REVERSE-VIDEO",REVERSEVIDEO,0},
	{"BUMBERTOK",NUMBERTOK,0},{"PLUS",PLUS,0},{"MINUS",MINUS,0},
	{"FOREGROUND-COLOR",FOREGROUNDCOLOR,0},
	{"BACKGROUND-COLOR",BACKGROUNDCOLOR,0},{"COL",COLUMN,0},
	{"COLUMN",COLUMN,0},{"BELL",BELL,0},{"UNDERLINE",UNDERLINE,0},
	{"HIGHLIGHT",HIGHLIGHT,0},{"LOWLIGHT",LOWLIGHT,0},
	{"BLINK",BLINK,0},{"SECURE",SECURE,0},
	{"JUST",JUST,0},{"JUSTIFIED",JUST,0},{"RIGHT",RIGHT,0},{"AUTO",AUTO,0},
	{"REQURIED",REQUIRED,0},{"FULL",FULL,0},
	{"ASCENDING",DIRECTION,ASCENDING},{"DESCENDING",DIRECTION,DESCENDING},
	{"OR",OR,0},{"AND",AND,0},{"UPDATE",UPDATE,0},{"SORT",SORT,0},
	{"DATE",DATE_TIME,DATE},{"TIME",DATE_TIME,TIME},
	{"COMMAND-LINE",CMD_LINE,0},{"ENVIRONMENT-VARIABLES",ENVIRONMENT_VARIABLES,0},
	{"INKEY",DATE_TIME,INKEY},{"INSPECT",INSPECT,0},
	{"TALLYING",TALLYING,0},{"REPLACING",REPLACING,0},{"LEADING",LEADING,0},
	{"INITIAL",INITIALTOK,0},{"FIRST",FIRSTTOK,0},{"ALL",ALL,0},
	{"ESCAPE",ESCAPE,0},{"MOVE",MOVE,0},{"ACCEPT",ACCEPT,0},
	{"FD",FD,0},{"SD",SD,0},{"ECHO",ECHOT,0},{"REDEFINES",REDEFINES,0},
	{"FILLER",FILLER,0},{"OCCURS",OCCURS,0},{"TIMES",TIMES,0},
	{"GIVING",GIVING,0},{"ERASE",ERASE,0},{"COMP",USAGENUM,COMP},
	{"COMPUTATIONAL",USAGENUM,COMP},{"INDEX",USAGENUM,INDEX},
	{"DISPLAY",DISPLAY,0},{"UPON",UPON,0},{"CONSOLE",CONSOLE,0},
	{"STD-OUTPUT",STD_OUTPUT,0},{"STD-ERROR",STD_ERROR,0},
	{"TRACE",TRACE,0},{"READY",READY,0},{"RESET",RESET,0},
	{"COMP-3",USAGENUM,COMP3},{"COMPUTATIONAL-3",USAGENUM,COMP3},
	{"PROGRAM",PROGRAM,0},
	{"PROGRAM-ID",PROGRAM_ID,0},{"ENVIRONMENT",DIVISNUM,ENVIRONMENT},
	{"DATA",DIVISNUM,DATA},{"DIVISION",DIVISION,0},{"SECTION",SECTION,0},
	{"TO",CONNECTIVE,TOTOK},{"EXIT",EXIT,0},{"OF",OF,0},{"IN",IN,0},
	{"PROCEDURE",DIVISNUM,PROCEDURE},
	{"IDENTIFICATION",DIVISNUM,IDENTIFICATION},
	{"AUTHOR",COMMENTING,AUTHOR},{"CONFIGURATION",CONFIGURATION,0},
	{"SAME",SAME,0},{"FOR",CONNECTIVE,FOR},
	{"INPUT-OUTPUT",INPUT_OUTPUT,0},{"SPECIAL-NAMES",SPECIAL_NAMES,0},
	{"FILE-CONTROL",FILE_CONTROL,0},{"I-O-CONTROL",I_O_CONTROL,0},
	{"IS",CONNECTIVE,IS},{"ARE",CONNECTIVE,ARE},{"START",START,0},
	{"THROUGH",CONNECTIVE,THRU},{"THRU",CONNECTIVE,THRU},
	{"WORKING-STORAGE",WORKING_STORAGE,0},{"AREA",AREA,0},
	{"LINKAGE",LINKAGE,0},{"COMMA",COMMA,0},{"INVALID",INVALID,0},
	{"DECIMAL-POINT",DECIMAL_POINT,0},{"FILE",FILEN,0},{"BY",BY,0},
	{"GO",GO,0},{"USAGE",USAGE,0},{"BLANK",BLANK,0},{"SIGN",SIGN,0},
	{"VALUE",VALUE,0},{"OTHER",OTHER,0},{"CONTINUE",CONTINUE,0},
	{"THEN",THEN,0},{"EVALUATE",EVALUATE,0},{"ALSO",ALSO,0},
	{"END-ADD",END_ADD,0},{"END-CALL",END_CALL,0},
	{"END-COMPUTE",END_COMPUTE,0},{"END-DELETE",END_DELETE,0},
	{"END-DIVIDE",END_DIVIDE,0},{"END-EVALUATE",END_EVALUATE,0},
	{"END-IF",END_IF,0},{"END-MULTIPLY",END_MULTIPLY,0},{"END-PERFORM",END_PERFORM,0},
	{"END-READ",END_READ,0},{"END-REWRITE",END_REWRITE,0},{"END-SEARCH",END_SEARCH,0},
	{"END-START",END_START,0},{"END-STRINGCMD",END_STRINGCMD,0},
	{"END-SUBTRACT",END_SUBTRACT,0},{"END-UNSTRING",END_UNSTRING,0},
	{"END-WRITE",END_WRITE,0},{"TEST",TEST,0},
        {"SEARCH",SEARCH,0},{"WHEN",WHEN,0},
	{"LABEL",LABEL,0},{"RECORD",RECORD,0},{"OMITTED",OMITTED,0},
	{"STANDARD",STANDARD,0},{"RECORDS",RECORDS,0},{"BLOCK",BLOCK,0},
	{"CONTAINS",CONTAINS,0},{"CHARACTERS",CHARACTERS,0},
	{"ADD",ADD,0},{"SUBTRACT",SUBTRACT,0},{"DELETE",DELETE,0},
	{"MULTIPLY",MULTIPLY,0},{"DIVIDE",DIVIDE,0},
	{"COMPUTE",COMPUTE,0},{"STOP",STOP,0},{"RUN",RUN,0},
	{"PERFORM",PERFORM,0},{"VARYING",VARYING,0},{"FROM",FROM,0},
	{"UNTIL",UNTIL,0},{"IF",IF,0},
	{"NEXT",NEXT,1},{"PREV",NEXT,2},{"PREVIOUS",NEXT,2},
	{"ELSE",ELSE,0},{"LINE",LINE,0},{"LINES",LINE,0},{"PAGE",PAGETOK,0},
	{"SELECT",SELECT,0},{"ASSIGN",ASSIGN,0},{"SENTENCE",SENTENCE,0},
	{"EQUAL",CONDITIONAL,EQUAL},{"LESS",CONDITIONAL,LESS},
	{"THAN",CONNECTIVE,THAN},
	{"=",CONDITIONAL,EQUAL},{"<",CONDITIONAL,LESS},{">",CONDITIONAL,GREATER},
	{"<=",CONDITIONAL,LEQ},{">=",CONDITIONAL,GEQ},{"<>",CONDITIONAL,NEQ},
	{"GREATER",CONDITIONAL,GREATER},{"OPEN",OPEN,0},{"CLOSE",CLOSE,0},
	{"READ",READ,0},{"RETURN",READ,1},
	{"WRITE",WRITE,0},{"RELEASE",WRITE,1},{"REWRITE",REWRITE,0},
	{"AFTER",AFTER,0},{"BEFORE",BEFORE,0},
	{"ADVANCING",ADVANCING,0},
	{"INTO",INTO,0},{"AT",AT,0},{"END",END,0},{"NEGATIVE",NEGATIVE,0},
	{"ZERO",ZERONUM,ZERO},{"ZEROS",ZERONUM,ZEROS},
	{"ZEROES",ZERONUM,ZEROES},{"SPACE",SPACES,0},{"SPACES",SPACES,0},
	{"HIGH-VALUE",HIGHVALUES,0},{"HIGH-VALUES",HIGHVALUES,0},
	{"LOW-VALUE",LOWVALUES,0},{"LOW-VALUES",LOWVALUES,0},
	{"QUOTE",QUOTES,0},{"QUOTES",QUOTES,0},
	{"POSITIVE",POSITIVE,0},{"NOT",NOT,0},
	{"NO",CONNECTIVE,NO},
	{"INPUT",OPENMD,INPUT},{"I-O",OPENMD,I_O},
	{"OUTPUT",OPENMD,OUTPUT},{"EXTEND",OPENMD,EXTEND},
	{"CALL",CALL,0},{"USING",USING,0},{"WITH",WITH,0},
	{"DARK",DARK,0},
	{"SET",SET,0},{"UP",UP,0},{"DOWN",DOWN,0},
	{"SOURCE-COMPUTER",COMMENTING,SOURCE_COMPUTER},
	{"OBJECT-COMPUTER",COMMENTING,OBJECT_COMPUTER},
	{"DATE-WRITTEN",COMMENTING,DATE_WRITTEN},
	{"ORGANIZATION",ORGANIZATION,0},{"ACCESS",ACCESS,0},{"MODE",MODE,0},
	{"KEY",KEY,0},{"STATUS",STATUS,0},{"SEQUENTIAL",SEQUENTIAL,0},
	{"INDEXED",INDEXED,0},{"DYNAMIC",DYNAMIC,0},
	{"RANDOM",RANDOM,0},{"RELATIVE",RELATIVE,0},
	{"COM1",PORTNUM,8},{"COM2",PORTNUM,1},{"COM3",PORTNUM,2},
	{"COM4",PORTNUM,3},{"LPT1",PORTNUM,4},{"LPT2",PORTNUM,5},
	{"LPT3",PORTNUM,6},{"LPT4",PORTNUM,7},
	{"DISK",PORTNUM,0},{"PRINTER",PORTNUM,4},
	{"COPY",COPY,0},{"STRING",STRINGCMD,0},{"UNSTRING",UNSTRING,0},
	{"ON",ONTOK,0},{"POINTER",POINTER,0},{"DELIMITED",DELIMITED,0},
	{"OVERFLOW",OVERFLOWTK,0},{"DELIMITER",DELIMITER,0},
	{"COUNT",COUNT,0},{"DUPLICATES",DUPLICATES,0},
	{"ALTERNATE",ALTERNATE,0},{"CURRENCY",CURRENCY,0},
	{"REFERENCE",REFERENCE,0},{"INITIALIZE",INITIALIZE,0},
	{"",0,0}
};
struct reserved_sym {
	char *name;
	struct reserved_sym *next;
	int i; /* index on reserved symbol table */
};
#define HASHLEN 100
static struct reserved_sym *reserved_tab[ HASHLEN ]={NULL};

#define svtoken() strncpy(toktext,yytext,sizeof(toktext))  

%}

%x IDENT_ST ENVIR_ST DATA_ST COMMENT_ST FD_ST REDEF_ST
%x SUBSCRIPTS_ST COPYFILE_ST EXCEPTION_ST
%x PIC_ST PIC_ST2 COPY_ST

%%
%{
	extern int curr_division;
	struct copy_symbols *tmp;

	switch (curr_division) {
	case CDIV_IDENT:
		scdebug("-> IDENT_ST\n");
		BEGIN IDENT_ST;
		break;
	case CDIV_ENVIR:
		scdebug("-> ENVIR_ST\n");
		BEGIN ENVIR_ST;
		break;
	case CDIV_DATA:
		scdebug("-> DATA_ST\n");
		BEGIN DATA_ST;
		break;
	case CDIV_PROC:
		scdebug("-> INITIAL\n");
		BEGIN INITIAL;
		break;
	case CDIV_COMMENT:
		scdebug("-> COMMENT_ST\n");
		BEGIN COMMENT_ST;
		break;
	case CDIV_FD:
		scdebug("-> FD_ST\n");
		BEGIN FD_ST;
		break;
	case CDIV_REDEF:
		scdebug("-> REDEF_ST\n");
		BEGIN REDEF_ST;
		break;
	case CDIV_SUBSCRIPTS:
		scdebug("-> SUBSCRIPTS_ST\n");
		BEGIN SUBSCRIPTS_ST;
		break;
	case CDIV_COPYFILE:
		append_copy_symbols=NULL;
		scdebug("-> COPYFILE_ST\n");
		previous_state = YY_START;
		BEGIN COPYFILE_ST;
		break;
	case CDIV_EXCEPTION:
		scdebug("-> EXCEPTION_ST\n");
		BEGIN EXCEPTION_ST;
		break;
	case CDIV_PIC:
		scdebug("-> PIC_ST\n");
		BEGIN PIC_ST;
		break;
	}
	curr_division=0; /* avoid new state switch */
%}

<*>^[ \t]?"*".*$	{ 
	   /* allow floating comment line */  
		scdebug("<comment>: %s:\n", yytext);
	}

<*>^"D ".*	{ 
		if (!source_debug)
			scdebug("<comment>: %s:\n", yytext);
		else
			yyless(2);
	}

", " { return LISTSEP; }

{white}+ { }

<INITIAL,SUBSCRIPTS_ST>{letters}(({alphanum}|-)*{alphanum}+)? { {
		struct sym *symbol;
		svtoken();
		if ((reserved_token=reserved(toktext))!=0) 
		{
			scdebug(" <reserved>: %s\n", toktext);
			return reserved_token;
		}
		/* if symbol was defined at the data division */
		if (symbol=lookup(toktext,2)) {
			yylval.sval=symbol;
			scdebug("<VARIABLE>: %s\n", toktext);
			return(VARIABLE);
		} 
		/* otherwise, it should be a label */
		else {
			yylval.sval=install(toktext,SYTB_LAB,0); 
			scdebug("<LABELSTR>: %s\n", toktext);
			return(LABELSTR);
		}
	} }

<ENVIR_ST>{letters}(({alphanum}|-)*{alphanum}+)? {
		svtoken();
		if ((reserved_token=reserved(toktext))!=0) 
		{
			scdebug(" <reserved>: %s\n", toktext);
			return reserved_token;
		}
		scdebug("<STRING>: %s\n", toktext);
		yylval.sval=install(toktext,SYTB_VAR,0);
		return(STRING);
	}

<EXCEPTION_ST>{letters}(({alphanum}|-)*{alphanum}+)? { {
		struct sym *symbol;
		svtoken();
		if ((reserved_token=reserved(toktext))!=0) 
		{
			scdebug(" <reserved>: %s\n", toktext);
			if (reserved_token == NOT)
				reserved_token = NOTEXCEP;
			return reserved_token;
		}
		/* if symbol was defined at the data division */
		if (symbol=lookup(toktext,2)) {
			yylval.sval=symbol;
			scdebug("<variable>: %s\n", toktext);
			return(VARIABLE);
		} 
		/* otherwise, it should be a label */
		else {
			yylval.sval=install(toktext,SYTB_LAB,0);
			scdebug("<LABELSTR>: %s\n", toktext);
			return(LABELSTR);
		}
	} }

<FD_ST>{
{letters}(({alphanum}|-)*{alphanum}+)? {
		svtoken();
		if ((reserved_token=reserved(toktext))!=0) 
		{
			scdebug(" <reserved>: %s\n", toktext);
			return reserved_token;
		}
		scdebug("<STRING>: %s\n", toktext);
		/* install w/clone if neccessary */
		if ((yylval.sval=lookup(toktext,SYTB_VAR))==NULL)
			yyerror("undefined %s at FD/SD",toktext); 
		return(STRING);
	}
[ \t]* { }
}

<REDEF_ST>{letters}(({alphanum}|-)*{alphanum}+)? {
		svtoken();
		if ((reserved_token=reserved(toktext))!=0) 
		{
			scdebug(" <reserved>: %s\n", toktext);
			return reserved_token;
		}
		scdebug("<STRING>: %s\n", toktext);
		/* don't clone redefined var */
		yylval.sval=install(toktext,SYTB_VAR,0); 
		return(STRING);
	}

<DATA_ST>{letters}(({alphanum}|-)*{alphanum}+)? {
		svtoken();
		if ((reserved_token=reserved(toktext))!=0) 
		{
			scdebug(" <reserved>: %s\n", toktext);
			return reserved_token;
		}
		scdebug("<STRING>: %s\n", toktext);
		/* install w/clone if neccessary */
		yylval.sval=install(toktext,SYTB_VAR,1); 
		return(STRING);
	}

<SUBSCRIPTS_ST>{
{int}+	{
		svtoken();
		scdebug("<NLITERAL>: %s\n", toktext);
		yylval.lval = (struct lit *)install(toktext,SYTB_LIT,0);
		return(NLITERAL);
	}
{any}	{
		svtoken();
		scdebug("<0: any>: %s\n", toktext);
		return(toktext[0]); 
	}
{white}+ { }
}	

<IDENT_ST>{
[ \t]* { }
\./[ \t]*"\n" { 
		scdebug("<EOS>:\n");
		return(EOS); 
	}
{letters}(({alphanum}|-)*{alphanum}+)? {
		svtoken();
		if ((reserved_token=reserved(toktext))!=0) 
		{
			scdebug(" <reserved>: %s\n", toktext);
			return reserved_token;
		}
		scdebug(" <IDSTRING>: %s\n", toktext);
		yylval.str=toktext;
		return IDSTRING;
	}
{any}	{
		svtoken();
		scdebug("<0: any>: %s\n", toktext);
		return(toktext[0]); 
	}
"\n" { }
} /* end of <IDENT_ST> */

<COPY_ST>{
\./[ \t]*"\n" { 
		scdebug("<EOS>:\n");
		copy_switch(include_filename);
		scdebug("<-> previous_state>\n");
		BEGIN(previous_state);
		return(EOS); 
	}
[ \t]*","[ \t]* { scdebug("<COPY_ST: sep>: %s\n", toktext);  }
"==" {
		int eq_seen=0;
		tp=toktext;
		while (1) {
			while((*tp=lexchar())!='=') { tp++; eq_seen=1; }
			if ((*tp=lexchar())=='=' && eq_seen) {
				*tp=0;
				break;
			}
		}
		scdebug(" <IDSTRING>: %s\n", toktext);
		yylval.str = savename(toktext);
		return IDSTRING;
	}
{alphanum}(({alphanum}|-)*{alphanum}+)? {
		if ((reserved_token=reserved(yytext))!=0) 
		{
		   if (reserved_token==BY) {
				scdebug(" <reserved>: %s\n", yytext);
				return BY; 
		   }
		   if (reserved_token==REPLACING) {
				scdebug(" <reserved>: %s\n", yytext);
				return REPLACING; 
		   }
		}
		scdebug(" <IDSTRING>: %s\n", yytext);
		yylval.str = savename(yytext);
		return IDSTRING;
	}
{white}+ { }

} /* end of COPY_ST */

<COPYFILE_ST>{
{any}+{alphanum} {
		scdebug(" <IDSTRING>: %s\n", yytext);
		yylval.str = savename(yytext);
		scdebug("-> COPY_ST\n"); 
		BEGIN(COPY_ST);
		return IDSTRING;
  }
{white}+ { }
}

<INITIAL,ENVIR_ST,DATA_ST,REDEF_ST,EXCEPTION_ST>{
\./[ \t]*"\n" { 
		scdebug("<EOS>:\n");
		return(EOS); 
	}

{sdelim}	{
		tp=toktext;
		while((*tp=lexchar())!='"') tp++;
		*tp=0;
		scdebug("<CLITERAL>: %s\n", toktext);
		yylval.lval = (struct lit *)install(toktext,SYTB_LIT,0);
		return(CLITERAL);
	}

X{sdelim}	{ {
		unsigned char hnum[2];
		tp=toktext;
		hnum[0]=hnum[1]=0;
		while((*tp=toupper(lexchar()))!='"') {
			hnum[0] *= 0x10;
			hnum[0] += (*tp > '9') ?
				(*tp - 'A' + 0x0a) : (*tp - '0');
			tp++;
		}
		yylval.lval = (struct lit *)install(hnum,SYTB_LIT,0);
		return(CLITERAL);
	} }

{ldelim}	{
		tp=toktext;
		while((*tp=lexchar())!='\'') tp++;
		*tp=0;
		scdebug("<CLITERAL>: %s\n", toktext);
		yylval.lval = (struct lit *)install(toktext,SYTB_LIT,0);
		return(CLITERAL);
	}

{sign}?{digit}*{decdelim}?{int}+	{
		svtoken();
		scdebug("<NLITERAL>: %s\n", toktext);
		yylval.lval = (struct lit *)install(toktext,SYTB_LIT,0);
		return(NLITERAL);
	}

{digit}({exletter}|{digit})* { 
	svtoken();
	scdebug("<LABELSTR>: %s\n", toktext);
	yylval.sval=install(toktext,SYTB_LAB,0);
	return(LABELSTR); 
	}

{white}+ { }


{relop}{relop}? {
		svtoken();
		if ((reserved_token=reserved(toktext))!=0) {
			scdebug(" <reserved>: %s\n", toktext);
			return reserved_token;
		}
		yyerror("relational operator unknown");
	}

{any}	{
		svtoken();
		scdebug("<0: any>: %s\n", toktext);
		return(toktext[0]); 
	}

"\n"   { }

} /* end of <INITIAL,DATA_ST,ENVIR_ST> */


<COMMENT_ST>{
{letters}(({alphanum}|-)*{alphanum}+)? {
		svtoken();
		if ((reserved_token=reserved(toktext))!=0) 
		{
			scdebug(" <reserved>: %s\n", toktext);
			if (reserved_token == DIVISNUM) {
				BEGIN INITIAL;
				return reserved_token;
			}
		}
		scdebug(" <COMMENT_ST>: %s\n", toktext);
	}
{any}	{
	   svtoken();
	   scdebug(" <COMMENT_ST>: %s\n", toktext);
        }
[ \t]* { }
"\n"   { }
} /* end of COMMENT_ST */

<PIC_ST>{white}* { 
		BEGIN PIC_ST2;
		scdebug("<PICS -> PIC_ST2>:\n");
	}

<PIC_ST,PIC_ST2>{
{white}* { 
		BEGIN DATA_ST;
		scdebug("<PICS -> DATA_ST>:\n");
	}

\./[ \t]*"\n" { 
		BEGIN DATA_ST;
		scdebug("<PICS -> DATA_ST>: EOS\n");
		return EOS;
	}

{lpar}{digit}({digit})*{rpar} {	
		{ 
		char *s;
		yylval.ival = 0;
		s = yytext+1; /* bypass lpar */
		while (*s!=')')
			yylval.ival = 10 * yylval.ival + *s++ - '0';
		scdebug("<PICS: MULTIPLIER>: %d\n", yylval.ival);
		return(MULTIPLIER); 
		} 
	}

{any}	{ 
			scdebug("<PICS: any char>: %c\n", yytext[0]);
		yylval.ival=(int)yytext[0];
		return(CHAR); }
"\n" { }
} /* end of PICS */

<<EOF>> {
	int tmp;
	struct copy_symbols *cs,*cs1;
	if (--fsp<0) {
		//fprintf(stderr,"scanner receiving <<EOF>>\n");
		yyterminate();
	}
	else {
		if (HTG_verbose == TRUE ) {
		    printf("Lines included in file: %4d\n", lineno);
		}		
//		printf("Lines included in file: %4d\n", lineno);

		tmp = lineno;
		yy_delete_buffer( YY_CURRENT_BUFFER );
		yy_switch_to_buffer( file_stack[fsp].buffer );
		lineno = file_stack[ fsp  ].line;
		/* free all replacements in this dead context */
		cs1 = file_stack[ fsp ].last_symbol;
		cs = copy_symbols;
		if (cs == cs1)
			copy_symbols = NULL;
		while (cs) {
			if (cs->next == cs1)
				break;
			cs = cs->next;
		}
		if (cs)
			cs->next = NULL; 
		if (cs1) {
			while (cs1) {
				free( cs1->value );
				free( cs1->replacement );
				cs = cs1->next;
				free( cs1 );
				cs1 = cs;
			}
		}
		lineno = lineno + tmp;
	}
}

%%

int lex_fgets( char *buf, int maxsize ) {
	int result;
	char *s;
	s = fgets( buf,maxsize,yyin );
	result = (s==NULL) ? YY_NULL: strlen(s);
	if (result) { 
// 		fprintf(stderr,"lexbuf before: |%s|\n",buf); 
		copy_do_replacements( buf ); 
		result = strlen(buf);
// 		fprintf(stderr,"lexbuf after: |%s|\n",buf); 
		if (HTG_list_flag) 
			fprintf( o_lst,"%04d:%s",++lineno,buf ); 
		else 
			++lineno; 
	} 
	return result;
}

void dcs() {
fprintf(stderr,"===============\n");
if (fsp)
	fprintf(stderr,"last symbol: %s\n",
		file_stack[fsp-1].last_symbol ? 
		file_stack[fsp].last_symbol->value : "");
if (copy_symbols) {
	struct copy_symbols *ccs;
	ccs= copy_symbols;
	while (ccs) {
		fprintf(stderr,"copy symbolA: %s\n",ccs->value);
		ccs = ccs->next;
	}
}
}

/* despite the name, copy_strip_spaces eliminates ", " and "; "
	separators too, besides multiples spaces.
	This follows standard CD-1.2 pp.679, rule 7.b */
char *
copy_strip_spaces( char *s ) {
	char *s1,*new;
	s1 = new = malloc(strlen(s)+2);
	while (*s==' ' || *s=='\t') /* don't strip indenting */
		*s1++=*s++;
	while (*s) {
		if ((*s==',' || *s==';') && *(s+1)==' ') s++;
		if (*s==' ' || *s=='\t' || *s=='\n') {
			*s1++ = ' ';
			if (*s=='\n') break;
			while (*s && (*s==' ' || *s=='\t')) s++;
		}
		else
			*s1++ = *s++;
	}
	*s1=0;
	return new;
}

void add_copy_replacement( char *orig_value, char *replacement ) {
	struct copy_symbols *cs,*tmp;
	char *value;
//fprintf(stderr,"add_copy_replacement: \"%s\" BY \"%s\"\n",
//		orig_value,replacement);
	value=copy_strip_spaces(orig_value);
	free(orig_value);
	cs = malloc(sizeof(struct copy_symbols));
	cs->value = value;
	cs->replacement = replacement;
	cs->next = NULL;
	if (append_copy_symbols==NULL) 
		append_copy_symbols = cs;
	else {
		tmp=append_copy_symbols;
		while (tmp->next) 
			tmp=tmp->next;
		tmp->next = cs;
	}
}

void copy_do_replacements( char *buffer ) {
	struct copy_symbols *cs;
	char *stripped,*s,*tmp,*s1,c;
	
	// Replacement table empty
	if (copy_symbols==NULL) return;

	// skip comments
	if (*buffer == '*')  {
//	   fprintf(stderr,"compare ignore comments:%s:\n", buffer);
	   return;
	}

	s1 = tmp = malloc(MAXLINEBUF);
	s = stripped = copy_strip_spaces( buffer );
	while (*s && (*s == ' ' || *s == '\t')) /* bypass white */ 
		*s1++ = *s++; 
	while (*s) {
		for (cs=copy_symbols;cs;cs=cs->next) {
			if (!strncmp(s,cs->value,strlen(cs->value))) {
				c = *(s+strlen(cs->value));
				if (c!=' ' && c!='\0') continue;
				strcpy(s1,cs->replacement);
				s += strlen(cs->value);
				s1 += strlen(cs->replacement);
				*s1++ = ' ';
				break;
			}
		}
		if (cs==NULL) {
			while (*s && *s != ' ' && *s != '\t') 
				*s1++ = *s++;
			while (*s && (*s == ' ' || *s == '\t' || *s == '\n')) {
				if (*s == '\n') {
					*s1++ = *s++;
					break;
				}
				*s1++ = *s++;
			}
		}
	}
	*s1++='\n';
	*s1++=0;
	strcpy(buffer,tmp);
	free(stripped);
	free(tmp);
}

void copy_switch( char *filename ) {
	struct copy_symbols *tmp;
	if (fsp>=MAX_COPY_DEPTH) {
		/* yyerror("Impossible to redirect input!"); */
		fprintf(stderr, "Copy nested too deeply (max=%d)\n", MAX_COPY_DEPTH);
		exit(1);
	}
	file_stack[ fsp ].buffer = YY_CURRENT_BUFFER;
	file_stack[ fsp ].line = lineno;
	if ((tmp=copy_symbols)!=NULL) {
		while (tmp->next) tmp=tmp->next;
		tmp->next = append_copy_symbols;
	}
	else {
		copy_symbols = append_copy_symbols;
	}
	file_stack[ fsp++ ].last_symbol = append_copy_symbols;

        if (find_copybook_file(filename) != 0) {
	   if (HTG_verbose == TRUE ) {
	      printf("error: copy file '%s' not found\n", filename);
              printf("Include(copybooks) search path(s)=%s;\n", HTG_COPYDIR);
	   }
	   else {
	      fprintf(stderr, "error: copy file '%s' not found\n", filename);
              fprintf(stderr, "Include(copybooks) search path(s)=%s;\n", HTG_COPYDIR);
	   }
	   exit(1);
        }        

	if (HTG_verbose == TRUE ) {
	    printf("Including copybook: %s\n",include_full_filename);
//	    printf("Including copybook: %s\n",filename);
	}

//	printf("Including: %-20s\n",filename);
//	if ((yyin = fopen( filename,"r" ))==NULL) {
	if ((yyin = fopen( include_full_filename, "r" ))==NULL) {
//		fprintf(stderr, "Cannot open copy file %s\n", filename);
		fprintf(stderr, "Cannot open copy file %s\n", include_full_filename);
		exit(1);
	}

	lineno = 0;
	yy_switch_to_buffer( yy_create_buffer(yyin,YY_BUF_SIZE) );
}

char *savename( char *name );
int hash( char *name );

struct reserved_sym *lookup_reserved( char *s ) {
	struct reserved_sym *as;
	for ( as = reserved_tab[ hash(s) ] ; as != NULL ; as = as->next )
		if ( strcasecmp( s, as->name ) == 0 )
			return( as );
	return( NULL );
}

void install_reserved( void ) {
	struct reserved_sym *as;
	int val,i;
	for (i=0;strcmp(reserved_symbols[i].name,"");i++) {
		as = (struct reserved_sym *)
			malloc( sizeof(struct reserved_sym) );
		as->name = savename( reserved_symbols[i].name );
		val = hash( as->name );
		as->next = reserved_tab[ val ];
		reserved_tab[ val ] = as;
		as->i=i;
	}
}

int reserved( char *s ) {
	struct reserved_sym *r;
	if ((r=lookup_reserved(s))!=NULL) {
			yylval.ival=reserved_symbols[r->i].minor;
			return reserved_symbols[r->i].token;
	}
	return 0;
}

#if defined(SunOS)
va_list __builtin_va_alist;
#endif

void scdebug( char *fmt, ... ) {
	va_list args;
	va_start(args,fmt);
#ifdef DEBUG_SCANNER
	fprintf(stderr,"SCAN(%4d):",lineno);
	vfprintf(stderr,fmt,args);
#endif
	va_end(args);
}

int yywrap() { return 1; }

/* end of scan.l */
