// 
// 
//   COBOL Compiler Run Time Library -- Move Module
// 
// 

#include "htcoblib.h"

//#define DEBUG_RTS 0 

/*-----------------------------------------------------------------------*\
 |                                                                       |
 |       void cob_move( struct fld_desc *FieldDescSrc,                   |
 |                      char            *caSrcBuffer,                    |
 |                      struct fld_desc *FieldDescDest,                  |
 |                      char            *caDestBuffer)                   |
 |                                                                       |
 |   Copy the contents of the field described by FieldDescSrc (data is   |
 |   pointed to by caSrcBuffer) to the field described by FieldDescDest  |
 |   (data will be stored in memory pointed to by caDescBuffer).         |
 |                                                                       |
 |   This routine will appropriately convert data (or format it) as it   |
 |   potentially moves from one data type to another. Truncation and     |
 |   padding will also be done in this routine if field sizes differ.    |
 |   Data types that this routine will handle are:                       |
 |        Identifier      Source Type            Destination Type        |
 |            '9'         Numeric Display        Numeric Display         |
 |            'C'         Numeric Packed Decimal Numeric Packed Decimal  |
 |            'A'         Alpha                  Alpha                   |
 |            'X'         AlphaNumeric           AlphaNumeric            |
 |            'E'         AlphaNumeric           Numeric Edited          |
 |            'G'         AlphaNumeric           AlphaNumeric            |
 |            'B'         Numeric Bin Int        Numeric Bin Int         |
 |                                                                       |
\*-----------------------------------------------------------------------*/


void cob_move( pfldDesc1, caData1, pfldDesc2, caData2 )
struct fld_desc 	*pfldDesc1;
char 				*caData1;
struct fld_desc 	*pfldDesc2;
char 				*caData2;
{

	extern int		decimal_comma;
	int				i;
	int				iPadLength;
	unsigned char	iDestLength;
	unsigned char	iSrcLength;
	char			iSrcDecimals;
	char			iDestDecimals;
	char			*pSrcData;
	char			*pDstData;
	char			caWrkData[20];
	char			caPic[20];
	char			cWork;

	struct	fld_desc	*pSrcFld;
	struct	fld_desc	*pDstFld;
	struct	fld_desc	FldWrk;
	

	iSrcLength = pfldDesc1->len;
	iDestLength = pfldDesc2->len;
	iSrcDecimals = (char)pfldDesc1->decimals;
	iDestDecimals = (char)pfldDesc2->decimals;
	pSrcFld = pfldDesc1;			/* may be changed to point to work area */
	pDstFld = pfldDesc2;			/* may be changed to point to work area */
	pSrcData = caData1;				/* may be changed to point to work area */
	pDstData = caData2;				/* may be changed to point to work area */
	FldWrk.pic = caPic;

	switch(pSrcFld->type) {			/* source field type */
		case DTYPE_ALPHA:
		case DTYPE_ALPHANUMERIC:
		case DTYPE_EDITED:
		case DTYPE_GROUP:
			if((pSrcFld->all)		/* handle high/low values for */
			&& ((pDstFld->type == DTYPE_DISPLAY)
             || (pDstFld->type == DTYPE_PACKED))) {
				if((pSrcData[0] == (char)0)
				|| (pSrcData[0] == (char)0xFF)) {
					cWork = pSrcFld->type;
					pSrcFld->type = DTYPE_DISPLAY;
					cob_move(pSrcFld, pSrcData, pDstFld, pDstData);
					pSrcFld->type = cWork;
					return;
				}
			}
			switch(pDstFld->type) {	/* destination field type */
				case DTYPE_ALPHA:
				case DTYPE_ALPHANUMERIC:
				case DTYPE_GROUP:
					/*----------------------------------------------------*\
					 |                                                    |
                     |  Just move the source data to the destination area |
                     |  left to right, padding with spaces on the right   |
                     |  if the destination is larger than the source, or  |
                     |  truncating on the right if the destination is     |
                     |  smaller.  There is also some special logic that   |
                     |  checks to see if the 'all' attribute of the       |
                     |  source field is non zero. If it is, the source    |
                     |  contents are repeated as many times as necessary  |
                     |  to completely fill the destination area.          |
                     |                                                    |
                    \*----------------------------------------------------*/

					if(iSrcLength >= iDestLength) {
						memcpy(pDstData, pSrcData, iDestLength);
					}
					else {
						if(pSrcFld->all == 0) {
							memcpy(pDstData, pSrcData, iSrcLength);
							iPadLength = iDestLength - iSrcLength;
							for(i = 0; i < iPadLength; ++i) {
								pDstData[i + iSrcLength] = ' ';	/* pad rest */
							}
						}
						else {
							while(iDestLength) {
								if(iSrcLength <= iDestLength)
									i = iSrcLength;
								else
									i = iDestLength;
								memcpy(pDstData, pSrcData, i);
								pDstData += i;
								iDestLength -= i;
							}
						}
					}
					break;
				case DTYPE_DISPLAY: {
					/*----------------------------------------------------*\
					 |                                                    |
                     |  Edit and move alphanumeric contents into a        |
                     |  numeric display destination field.  If the        |
                     |  source field contains non numeric data, a runtime |
                     |  error is generated, and the destination field     |
                     |  is cleared to all zeroes.  The source field may   |
                     |  contain leading or trailing plus (+) or minus (-) |
                     |  signs, and if the receiving field is signed, its  |
                     |  sign will be set accordingly. If no sign is       |
                     |  present in the source data, it is assumed that    |
                     |  a positive number is represented.  Spaces and     |
                     |  tabs are ignored in the conversion.               |
                     |                                                    |
                    \*----------------------------------------------------*/

					char		cSign;
					char		cDecimalPoint;
					char		cChar;
					char		*pDecPortion;

					int			iIntCount;
					int			iDecCount;
					int			iWork;
					int			iSrcPtr;
					int			iWrkLength;
					int			bInDecPortion;
					int			bIsSigned;
					int			bLeadingWhite;
					int			bTrailingWhite;
		
					/*----------------------------------------------------*\
					 |                                                    |
                     |  If the source field is scaled (has 'P's in the    |
                     |  picture clause, move the source data to a like    |
                     |  sized non scaled numeric field, then call move    |
                     |  again using this non scaled field as the source   |
                     |  and the original destination field as the         |
                     |  destination.  This isn't the most efficient way   | 
                     |  to handle this type of move, but scaled fields    |
                     |  aren't heavily used.                              |
                     |                                                    |
                    \*----------------------------------------------------*/

					if(iDestDecimals < 0) { 		/* integer scaling */
						FldWrk.len = pDstFld->len;
						FldWrk.len += ((char)pDstFld->decimals * -1);
						FldWrk.type = DTYPE_DISPLAY;
						FldWrk.decimals = 0;
						FldWrk.all = 0;
						i = 0;
						if(pDstFld->pic[0] == 'S') {
							FldWrk.pic[0] = 'S';
							FldWrk.pic[1] = 1;
						 	i = 2;
						}
						FldWrk.pic[i++] = '9';
						FldWrk.pic[i++] = (char)FldWrk.len;
						FldWrk.pic[i] = (char)0;
						cob_move(pSrcFld, pSrcData, &FldWrk, caWrkData);
						cob_move(&FldWrk, caWrkData, pDstFld, pDstData);
						return;
					}
					if(iDestLength < iDestDecimals) {	/* fractional scaling */
						FldWrk.len = pDstFld->decimals;
						FldWrk.type = DTYPE_DISPLAY;
						FldWrk.decimals = pDstFld->decimals;
						FldWrk.all = 0;
						i = 0;
						if(pDstFld->pic[0] == 'S') {
							FldWrk.pic[0] = 'S';
							FldWrk.pic[1] = 1;
						 	i = 2;
						}
						FldWrk.pic[i++] = 'V';
						FldWrk.pic[i++] = (char)1;
						FldWrk.pic[i++] = '9';
						FldWrk.pic[i++] = (char)FldWrk.len;
						FldWrk.pic[i] = (char)0;
						cob_move(pSrcFld, pSrcData, &FldWrk, caWrkData);
						cob_move(&FldWrk, caWrkData, pDstFld, pDstData);
						return;
					}
					iDestDecimals = (int)pDstFld->decimals;
					iWrkLength = iSrcLength;
					iDecCount = 0;
					iIntCount = 0;
					iSrcPtr = 0;
					bInDecPortion = 0;
					bIsSigned = 0;
					bLeadingWhite = 1;
					bTrailingWhite = 0;
					pDecPortion = (char *)0;
					if(decimal_comma) {
						cDecimalPoint = ',';
					}
					else {
						cDecimalPoint = '.';
					}
					cSign = 0; 
					for(i = 0; i < iSrcLength; ++i) {
						cChar = pSrcData[i];
						switch(cChar) {
							case '+':
								cSign = 0;
								break;
							case '-':
								cSign = 1;
								break;
							case '.':
							case ',':
								if(cChar == cDecimalPoint) {
									if((!bInDecPortion)
									&& ((i + 1) < iSrcLength))
										pDecPortion = &caWrkData[iSrcPtr];
									bInDecPortion = 1;
								}
								break;
							case '0':
							case '1':
							case '2':
							case '3':
							case '4':
							case '5':
							case '6':
							case '7':
							case '8':
							case '9':
								if(!bInDecPortion) {
										iIntCount++;
								}
								else {
										iDecCount++;
								}
								caWrkData[iSrcPtr++] = cChar;
								break;
							case ' ':
							case '\t':
								break;
							default:
								runtime_error(RTERR_INVALID_DATA, pfldDesc1,
									(void *)pSrcData);
								memset(pDstData, '0', pDstFld->len);
								return;
						}
					}
					i = 0;
					iSrcLength = iSrcPtr;
					iSrcPtr = 0;
					if(iDestLength > iDestDecimals) {
                        iWork = iDestLength - iDestDecimals;
                        if(iIntCount < iWork) {     /* move pad first */
                            iPadLength = iWork - iIntCount;
                            while(i < iPadLength) {
                                caData2[i++] = '0';
                            }
                        }
                        if(iIntCount > iWork) {     /* truncate */
                            iSrcPtr = iIntCount - iWork;
                        }
                        while(iSrcPtr < iIntCount) {
                            pDstData[i++] = caWrkData[iSrcPtr++];
                        }
                    }
					iWork = 0;
					if(iDecCount > iDestDecimals) {
						iDecCount = iDestDecimals;
					}
					while(iWork < iDecCount) {
						pDstData[i++] = pDecPortion[iWork++];
					}
					while(iWork < iDestDecimals) {
						pDstData[i++] = '0';
						++iWork;
					}
					if(pDstFld->pic[0] == 'S')
						put_sign(pDstFld, pDstData, (int)cSign);
					break;
				}
				case DTYPE_PACKED: {
					/*----------------------------------------------------*\
					 |                                                    |
                     |  Moving alphanumeric data to a packed field is a   |
                     |  two step process, the first process moves it to   |
                     |  a work area defined like the destination field,   |
                     |  except that it will be display and contain no     |
                     |  scaling.  Then the contents of this work area     |
                     |  will then be moved to the original destination.   |
                     |  This is pretty inefficient, but, it will work     | 
                     |  for a first cut.                                  |
                     |                                                    |
                    \*----------------------------------------------------*/

					memcpy(&FldWrk, pSrcFld, sizeof(FldWrk));
					FldWrk.pic = caPic;
					FldWrk.type = DTYPE_DISPLAY;
					FldWrk.pic[0] = 'S';
					FldWrk.pic[1] = 1;
				 	i = 2;
					if(iSrcDecimals < 0) {
						FldWrk.pic[i++] = '9';
						FldWrk.pic[i++] = (char)((int)FldWrk.decimals * -1);
					}
					else {
						if(FldWrk.len > FldWrk.decimals) {
							FldWrk.pic[i++] = '9';
							FldWrk.pic[i++] = (char)(FldWrk.len - FldWrk.decimals);
						}
						if(FldWrk.decimals) {
							FldWrk.pic[i++] = 'V';
							FldWrk.pic[i++] = (char)1;
							FldWrk.pic[i++] = '9';
							FldWrk.pic[i++] = (char)FldWrk.decimals;
						}
					}
					FldWrk.pic[i] = (char)0;
					cob_move(pSrcFld, pSrcData, &FldWrk, caWrkData);
					cob_move(&FldWrk, caWrkData, pDstFld, pDstData);
					break;
				}
				case DTYPE_EDITED: 
					/*----------------------------------------------------*\
					 |                                                    |
                     |  We are handling edited moves in their own routine | 
                     |  since the logic involved is very complicated, and |
                     |  will be used in a similar fashion regardless of   |
                     |  source data type.                                 |
                    \*----------------------------------------------------*/

					move_edited(pSrcFld, pSrcData, pDstFld, pDstData);
					break;

				case DTYPE_BININT: {
					/*----------------------------------------------------*\
					 |                                                    |
                     |  Call cobmove to move the alphanumeric source into |
                     |  a work area of type display with an implied       |
                     |  picture of S9(10), then convert the value in the  |
                     |  work area to a 4 byte binary integer and store    |
                     |  in the receiving field.  The picture of the       |
                     |  receiving field is ignored.                       |
                     |                                                    |
                    \*----------------------------------------------------*/
					
					int					iWork;
					int					bIsNegative;
					char				caWork[10];

					FldWrk.len = 10;
					FldWrk.decimals = 0;
					FldWrk.type = DTYPE_DISPLAY;
					FldWrk.all = 0;
					FldWrk.pic[0] = 'S';
                   	cob_move(pSrcFld, pSrcData, &FldWrk, caWork);
					bIsNegative = extract_sign(&FldWrk, caWork);
					iWork = 0;
					for(i = 0; i < 10; ++i) {
						iWork *= 10;
						iWork += (int)(caWork[i] - '0');
					}
					if(bIsNegative)
						iWork *= -1;
					*(int *)pDstData = iWork;
					break;
				}
			}
			break;


/* Source type is Display */
		case DTYPE_DISPLAY:
			switch(pfldDesc2->type) {	
				case DTYPE_ALPHA:
				case DTYPE_GROUP:
				case DTYPE_ALPHANUMERIC: {
					int					j;

					/*----------------------------------------------------*\
					 |                                                    |
                     |  If the source field is scaled (has 'P's in the    |
                     |  picture clause, move the source data to a like    |
                     |  sized non scaled numeric field, then call move    |
                     |  again using this non scaled field as the source   |
                     |  and the original destination field as the         |
                     |  destination.  This isn't the most efficient way   | 
                     |  to handle this type of move, but scaled fields    |
                     |  aren't heavily used.                              |
                     |                                                    |
                    \*----------------------------------------------------*/

					if(iDestDecimals < 0) { 		/* integer scaling */
						FldWrk.len = pDstFld->len;
						FldWrk.len += ((char)pDstFld->decimals * -1);
						FldWrk.type = DTYPE_DISPLAY;
						FldWrk.decimals = 0;
						FldWrk.all = 0;
						i = 0;
						if(pDstFld->pic[0] == 'S') {
							FldWrk.pic[0] = 'S';
							FldWrk.pic[1] = 1;
						 	i = 2;
						}
						FldWrk.pic[i++] = '9';
						FldWrk.pic[i++] = (char)FldWrk.len;
						FldWrk.pic[i] = (char)0;
						cob_move(pSrcFld, pSrcData, &FldWrk, caWrkData);
						cob_move(&FldWrk, caWrkData, pDstFld, pDstData);
						return;
					}
					if(iDestLength < iDestDecimals) {	/* fractional scaling */
						FldWrk.len = pDstFld->decimals;
						FldWrk.type = DTYPE_DISPLAY;
						FldWrk.decimals = pDstFld->decimals;
						FldWrk.all = 0;
						i = 0;
						if(pDstFld->pic[0] == 'S') {
							FldWrk.pic[0] = 'S';
							FldWrk.pic[1] = 1;
						 	i = 2;
						}
						FldWrk.pic[i++] = 'V';
						FldWrk.pic[i++] = (char)1;
						FldWrk.pic[i++] = '9';
						FldWrk.pic[i++] = (char)FldWrk.decimals;
						FldWrk.pic[i] = (char)0;
						cob_move(pSrcFld, pSrcData, &FldWrk, caWrkData);
						cob_move(&FldWrk, caWrkData, pDstFld, pDstData);
						return;
					}

					/*----------------------------------------------------*\
					 |                                                    |
                     |  Else, just move the data left to right,           |
                     |  truncating or padding with spaces on the right    |
                     |  side of the destination field.                    |
                     |                                                    |
                    \*----------------------------------------------------*/

					if(iSrcLength >= iDestLength) {
						memcpy(pDstData, pSrcData, iDestLength);
					}
					else {
						if(pSrcFld->all == 0) {
							memcpy(pDstData, pSrcData, iSrcLength);
							iPadLength = iDestLength - iSrcLength;
							for(i = 0; i < iPadLength; ++i) {
								pDstData[i + iSrcLength] = ' ';	/* pad */
							}
						}
						else {
							j = 0;
							for(i = 0; i < iDestLength; ++i) {
								pDstData[i] = pSrcData[j++];
								if(j == iSrcLength)
									j = 0;
							}
						}
					}
					break;
				}

				case DTYPE_DISPLAY: {
					int			j;
					int			iSrcSign;
					int			iSrcIntDigits;
					int			iDestIntDigits;
					char		caWork[MAX_DIGITS * 2];

					/*--------------------------------------------*\
					 |                                            |
					 |   If source field is signed, extract the   |
                     |   value of the sign (zero is positive and  |
                     |   non zero is negative).  The extract      |
                     |   routine also 'unsigns' the overpuched    |
                     |   last digit of the field, so we will      |
                     |   need to re-sign the field later.         |
                     |   if the 'all' flag is set on, then set    | 
                     |   the dest type to alphanumeric, call      |
                     |   ourselves again, then change the type    |
                     |   back to numeric.  In the case of the     |
                     |   'all' flag, scaling and decimal points   |
                     |   are ignored. The 'all' flag is used      |
                     |   primarily for zero, high and low vales.  |
                     |                                            |
                    \*--------------------------------------------*/

					if(pSrcFld->pic[0] == 'S') {
						iSrcSign = extract_sign(pSrcFld, pSrcData);
					}
					else {
						iSrcSign = 0;
					}

					if(pSrcFld->all != 0) {
						pDstFld->type = DTYPE_ALPHANUMERIC;
						j = 0;
						if(pSrcData[0] == (char)0) {	/* low values */
							j = (int)pSrcData[0];
							j = j | 0x100;
							pSrcData[0] = '0';
						}
						if(pSrcData[0] == (char)0xFF) {	/* high values */
							j = (int)pSrcData[0];
							pSrcData[0] = '9';
						}
						cob_move(pSrcFld, pSrcData, pDstFld, pDstData);
						pDstFld->type = DTYPE_DISPLAY;
						if(j) 
							pSrcData[0] = (char)j;
						if(pSrcFld->pic[0] == 'S')
							put_sign(pSrcFld, pSrcData, iSrcSign);
						return;
					}

					/*--------------------------------------------*\
					 |                                            |
                     |  Move the contents of the source field to  |
                     |  our work area to create a normalized      | 
                     |  9(18)v9(18) view of the data.             |
                     |                                            |
                    \*--------------------------------------------*/

					for(i = 0; i < sizeof(caWork); ++i)
							caWork[i] = '0';

					if(iSrcDecimals < 0) {
						j = MAX_DIGITS + iSrcDecimals - 1;
						iSrcIntDigits = iSrcLength;
					}
					else {
						j = MAX_DIGITS - 1;
						iSrcIntDigits = iSrcLength - iSrcDecimals;
					}
					for(i = 0; i < iSrcIntDigits; ++i) 
						caWork[j--] = pSrcData[iSrcIntDigits - i - 1];
					
					if(iSrcDecimals > 0) {
						if(iSrcDecimals > iSrcLength) {
							j = iSrcDecimals - 1 + MAX_DIGITS;
							for(  ; i < iSrcLength; ++i)
								caWork[j++] = pSrcData[i];
						}
						else {
							j = MAX_DIGITS;
							for(i = 0; i < iSrcDecimals; ++i)
								caWork[j++] = pSrcData[i + iSrcIntDigits];
						}
					}
					if(pSrcFld->pic[0] == 'S') 
						put_sign(pSrcFld, pSrcData, iSrcSign);

					/*--------------------------------------------*\
					 |                                            |
                     |  Now move the contents of our normalized   |
                     |  data area to the receiving field.         |
                     |                                            |
                    \*--------------------------------------------*/

					if(iDestDecimals < 0) {
						j = MAX_DIGITS + iDestDecimals - 1;
						iDestIntDigits = iDestLength;
					}
					else {
						j = MAX_DIGITS - 1;
						iDestIntDigits = iDestLength - iDestDecimals;
					}
					for(i = 0; i < (iSrcIntDigits + iSrcDecimals); ++i) {
						if(i == pDstFld->len)
							break;
						pDstData[i] = '0';
					}
					for(i = 0; i < iDestIntDigits; ++i) 
						pDstData[iDestIntDigits - i - 1] = caWork[j--];
					
					if(iDestDecimals > 0) {
						if(iDestDecimals > iDestLength) {
							j = iDestDecimals - 1 + MAX_DIGITS;
							for(i = 0; i < iDestLength; ++i)
								pDstData[iDestLength - 1 - i] = caWork[j--];
						}
						else {
							j = MAX_DIGITS;
							for(i = 0; i < iDestDecimals; ++i)
								pDstData[i + iDestIntDigits] = caWork[j++];
						}
					}
					if(pDstFld->pic[0] == 'S')
						put_sign(pDstFld, pDstData, iSrcSign);
					break;
				}

				case DTYPE_PACKED: {
					int			j, k;
					int			iSrcSign;
					int			iSrcIntDigits;
					int			iDestIntDigits;
					char		caWork[MAX_DIGITS * 2];
					unsigned char	cPack;

					/*--------------------------------------------*\
					 |                                            |
					 |   If source field is signed, extract the   |
                     |   value of the sign (zero is positive and  |
                     |   non zero is negative).  The extract      |
                     |   routine also 'unsigns' the overpuched    |
                     |   last digit of the field, so we will      |
                     |   need to re-sign the field later.         |
                     |   If the source field has the 'all' flag   |
                     |   on, move to work area as type display,   |
                     |   then change work area type to alpha,     |
                     |   and call move again, change type back    |
                     |   to display, then call move using the     |
                     |   work area as source and the original     |
                     |   destination. This facilitates moving     |
                     |   of zero, high and low values.            |
                     |                                            |
                    \*--------------------------------------------*/

					if(pSrcFld->pic[0] == 'S') {
						iSrcSign = extract_sign(pSrcFld, pSrcData);
					}
					else {
						iSrcSign = 0;
					}

					if(pSrcFld->all != 0) {
						memcpy(&FldWrk, pDstFld, sizeof(FldWrk));
						FldWrk.type = DTYPE_ALPHANUMERIC;
						j = 0;
						if(pSrcData[0] == (char)0) {	/* low values */
							j = (int)pSrcData[0];
							j = j | 0x100;
							pSrcData[0] = '0';
						}
						if(pSrcData[0] == (char)0xFF) {	/* high values */
							j = (int)pSrcData[0];
							pSrcData[0] = '9';
						}
						cob_move(pSrcFld, pSrcData, &FldWrk, caWork);
						FldWrk.type = DTYPE_DISPLAY;
						cob_move(&FldWrk, caWork, pDstFld, pDstData);
						if(j) 
							pSrcData[0] = (char)j;
						return;
					}

					/*--------------------------------------------*\
					 |                                            |
                     |  Move the contents of the source field to  |
                     |  our work area to create a normalized      | 
                     |  9(18)v9(18) view of the data.             |
                     |                                            |
                    \*--------------------------------------------*/

					for(i = 0; i < sizeof(caWork); ++i)
							caWork[i] = '0';

					if(iSrcDecimals < 0) {
						j = MAX_DIGITS + iSrcDecimals - 1;
						iSrcIntDigits = iSrcLength;
					}
					else {
						j = MAX_DIGITS - 1;
						iSrcIntDigits = iSrcLength - iSrcDecimals;
					}
					for(i = 0; i < iSrcIntDigits; ++i) 
						caWork[j--] = pSrcData[iSrcIntDigits - i - 1];
					
					if(iSrcDecimals > 0) {
						if(iSrcDecimals > iSrcLength) {
							j = iSrcDecimals - 1 + MAX_DIGITS;
							for(  ; i < iSrcLength; ++i)
								caWork[j++] = pSrcData[i];
						}
						else {
							j = MAX_DIGITS;
							for(i = 0; i < iSrcDecimals; ++i)
								caWork[j++] = pSrcData[i + iSrcIntDigits];
						}
					}
					if(pSrcFld->pic[0] == 'S') 
						put_sign(pSrcFld, pSrcData, iSrcSign);


					/*--------------------------------------------*\
					 |                                            |
                     |  Now move the contents of our normalized   |
                     |  data area to the receiving field, packing |
                     |  the data as we go.  Packed fields are (at |
                     |  least at this time) always considered to  |
                     |  be signed, so we will also move a         |
                     |  trailing sign to the receiving area. A    |
                     |  value of 0xC represents a positive value  |
                     |  and 0xD represents a negative value.      |
                     |                                            |
                    \*--------------------------------------------*/

					k = 0;
					cPack = 0;
					if(iDestDecimals < 0) {
						j = MAX_DIGITS + iDestDecimals - 1;
						iDestIntDigits = iDestLength;
					}
					else {
						j = MAX_DIGITS;
						iDestIntDigits = iDestLength - iDestDecimals;
					}
					j -= iDestIntDigits;	/* j is pointer into work area */
											/* zero fill destination first */
					for(i = 0; i < (iSrcIntDigits + iSrcDecimals); ++i) {
						if(i == pDstFld->len)
							break;
						if(i & 1)			/* if lower (right) nibble */
							(unsigned char)pDstData[k++] = 0;
					}
					if(i & 1)
						(unsigned char)pDstData[k] = 0;

					k = 0;
					for(i = 0; i < iDestIntDigits; ++i) {
						if(k & 1) {			/* if lower (right) nibble */
							cPack = cPack | ((unsigned char)caWork[j++] & 0xF);
							(unsigned char)pDstData[k >> 1] = cPack;
							++k;
						}
						else {
							cPack = (unsigned char)caWork[j++] << 4;
							++k;
						}
					}
					
					if(iDestDecimals > 0) {
						if(iDestDecimals > iDestLength) {
							j = iDestDecimals - 1 + MAX_DIGITS - iDestLength;
							for(i = 0; i < iDestLength; ++i) {
								if(k & 1) {
									cPack |= ((unsigned char)caWork[j++] & 0xF);
									(unsigned char)pDstData[k >> 1] = cPack;
									++k;
								}
								else {
									cPack = (unsigned char)caWork[j++] << 4;
									++k;
								}
							}
						}
						else {
							j = MAX_DIGITS;
							for(i = 0; i < iDestDecimals; ++i) {
								if(k & 1) {
									cPack |= ((unsigned char)caWork[j++] & 0xF);
									(unsigned char)pDstData[k >> 1] = cPack;
									++k;
								}
								else {
									cPack = (unsigned char)caWork[j++] << 4;
									++k;
								}
							}
						}
					}
					if(k & 1) {
						cPack |= ((unsigned char)caWork[j++] & 0xF);
						(unsigned char)pDstData[k >> 1] = cPack;
					}
					put_sign(pDstFld, pDstData, iSrcSign);
					break;
				}

				case DTYPE_EDITED: 
					/*----------------------------------------------------*\
					 |                                                    |
                     |  We are handling edited moves in their own routine | 
                     |  since the logic involved is very complicated, and |
                     |  will be used in a similar fashion regardless of   |
                     |  source data type.                                 |
                    \*----------------------------------------------------*/

					move_edited(pSrcFld, pSrcData, pDstFld, pDstData);
					break;

				case DTYPE_BININT: {
					/*----------------------------------------------------*\
					 |                                                    |
                     |  Just scan through the integer portion of the      |
                     |  source field and build the resulting 4 byte       |
                     |  binary integer (which will be move to the         |
                     |  receiving field.  The picture of the receiving    |
                     |  field is ignored.                                 |
                     |                                                    |
                    \*----------------------------------------------------*/
		
					int					j;
					int					bIsNegative;
					int					iWork;
					
					*(int *)pDstData = 0;
					if(iSrcDecimals >= iSrcLength)
						break;			/* source has only fraction */

					if(pSrcFld->pic[0] == 'S')
						bIsNegative = extract_sign(pSrcFld, pSrcData);
					else
						bIsNegative = 0;
					if(iSrcDecimals < 0)
						j = iSrcLength;
					else
						j = iSrcLength - iSrcDecimals;

					iWork = 0;
					for(i = 0; i < j; ++i) {
						iWork *= 10;
						iWork += (int)(pSrcData[i] - '0');
					}
					if(iSrcDecimals < 0) {
						for(i = iSrcDecimals; i > 0; --i)
							iWork *= 10;
					}
					if(bIsNegative)
						iWork *= -1;
					*(int *)pDstData = iWork;
				
					if(pSrcFld->pic[0] == 'S')
						put_sign(pSrcFld, pSrcData, bIsNegative);
					break;
				}
			}
			break;

/* Source type is Packed */
		case DTYPE_PACKED:
			switch(pfldDesc2->type) {		/* destination field type */
				case DTYPE_ALPHA:
				case DTYPE_GROUP:
				case DTYPE_ALPHANUMERIC: {
					int				j;
					unsigned char	cWork;

					/*----------------------------------------------------*\
					 |                                                    |
                     |  If the source field is scaled (has 'P's in the    |
                     |  picture clause, move the source data to a like    |
                     |  sized non scaled numeric field, then call move    |
                     |  again using this non scaled field as the source   |
                     |  and the original destination field as the         |
                     |  destination.  This isn't the most efficient way   | 
                     |  to handle this type of move, but scaled fields    |
                     |  aren't heavily used.                              |
                     |                                                    |
                    \*----------------------------------------------------*/

					if(iDestDecimals < 0) { 		/* integer scaling */
						FldWrk.len = pDstFld->len;
						FldWrk.len += ((char)pDstFld->decimals * -1);
						FldWrk.type = DTYPE_DISPLAY;
						FldWrk.decimals = 0;
						FldWrk.all = 0;
						i = 0;
						if(pDstFld->pic[0] == 'S') {
							FldWrk.pic[0] = 'S';
							FldWrk.pic[1] = 1;
						 	i = 2;
						}
						FldWrk.pic[i++] = '9';
						FldWrk.pic[i++] = (char)FldWrk.len;
						FldWrk.pic[i] = (char)0;
						cob_move(pSrcFld, pSrcData, &FldWrk, caWrkData);
						cob_move(&FldWrk, caWrkData, pDstFld, pDstData);
						return;
					}
					if(iDestLength < iDestDecimals) {	/* fractional scaling */
						FldWrk.len = pDstFld->decimals;
						FldWrk.type = DTYPE_DISPLAY;
						FldWrk.decimals = pDstFld->decimals;
						FldWrk.all = 0;
						i = 0;
						if(pDstFld->pic[0] == 'S') {
							FldWrk.pic[0] = 'S';
							FldWrk.pic[1] = 1;
						 	i = 2;
						}
						FldWrk.pic[i++] = 'V';
						FldWrk.pic[i++] = (char)1;
						FldWrk.pic[i++] = '9';
						FldWrk.pic[i++] = (char)FldWrk.decimals;
						FldWrk.pic[i] = (char)0;
						cob_move(pSrcFld, pSrcData, &FldWrk, caWrkData);
						cob_move(&FldWrk, caWrkData, pDstFld, pDstData);
						return;
					}

					/*----------------------------------------------------*\
					 |                                                    |
                     |  Else, just move the data left to right,           |
                     |  truncating or padding with spaces on the right    |
                     |  side of the destination field.                    |
                     |                                                    |
                    \*----------------------------------------------------*/

					if(iSrcLength >= iDestLength) 
						j = iDestLength;
					else
						j = iSrcLength;

					for(i = 0; i < j; ++i) {
						if(i & 1) {
							cWork = (unsigned char)pSrcData[i >> 1];
							cWork = cWork & 0xF;
							pDstData[i] = cWork + '0';
						}
						else {
							cWork = (unsigned char)(pSrcData[i >> 1]);
							cWork = cWork >> 4;
							pDstData[i] = cWork + '0';
						}
					}
					while(i < iDestLength)
						pDstData[i++] = ' ';
					break;
				}

				case DTYPE_DISPLAY: {
					int			j;
					int			iSrcSign;
					int			iSrcIntDigits;
					int			iDestIntDigits;
					char		caWork[MAX_DIGITS * 2];
					unsigned char	cWork;

					/*--------------------------------------------------*\
					 |                                                  |
					 |   Currently, packed fields are assumed to be     |
                     |   signed, so extract the value of the sign from  |
                     |   the source field (zero represents positive,    | 
                     |   and non zero represents negative).             |
                     |                                                  |
                    \*--------------------------------------------------*/

					iSrcSign = extract_sign(pSrcFld, pSrcData);

					/*--------------------------------------------*\
					 |                                            |
                     |  Move the contents of the source field to  |
                     |  our work area to create a normalized      | 
                     |  9(18)v9(18) view of the data.             |
                     |                                            |
                    \*--------------------------------------------*/

					for(i = 0; i < sizeof(caWork); ++i)
							caWork[i] = '0';

					i = 0;
					if(iSrcDecimals < 0) {
						j = MAX_DIGITS + iSrcDecimals - 1;
						iSrcIntDigits = iSrcLength;
					}
					else {
						j = MAX_DIGITS - 1;
						iSrcIntDigits = iSrcLength - iSrcDecimals;
					}
					j = j - iSrcIntDigits + 1;
					for(i = 0; i < iSrcIntDigits; ++i) {
						if(i & 1) {
							cWork = (unsigned char)pSrcData[i >> 1];
							cWork = cWork & 0xF;
							caWork[j++] = cWork + '0';
						}
						else {
							cWork = (unsigned char)(pSrcData[i >> 1]);
							cWork = cWork >> 4;
							caWork[j++] = cWork + '0';
						}
					}
					if(iSrcDecimals > 0) {
						if(iSrcDecimals > iSrcLength) {
							j = iSrcDecimals - iSrcLength + MAX_DIGITS;
							for(  ; i < iSrcLength; ++i) {
								if(i & 1) {
									cWork = (unsigned char)pSrcData[i >> 1];
									cWork = cWork & 0xF;
									caWork[j++] = cWork + '0';
								}
								else {
									cWork = (unsigned char)(pSrcData[i >> 1]);
									cWork = cWork >> 4;
									caWork[j++] = cWork + '0';
								}
							}
						}
						else {
							j = MAX_DIGITS;
							for(  ; i < iSrcLength; ++i) {
								if(i & 1) {
									cWork = (unsigned char)pSrcData[i >> 1];
									cWork = cWork & 0xF;
									caWork[j++] = cWork + '0';
								}
								else {
									cWork = (unsigned char)(pSrcData[i >> 1]);
									cWork = cWork >> 4;
									caWork[j++] = cWork + '0';
								}
							}
						}
					}

					/*--------------------------------------------*\
					 |                                            |
                     |  Now move the contents of our normalized   |
                     |  data area to the receiving field.         |
                     |                                            |
                    \*--------------------------------------------*/

					if(iDestDecimals < 0) {
						j = MAX_DIGITS + iDestDecimals - 1;
						iDestIntDigits = iDestLength;
					}
					else {
						j = MAX_DIGITS - 1;
						iDestIntDigits = iDestLength - iDestDecimals;
					}
					for(i = 0; i < (iSrcLength); ++i) {
						if(i == pDstFld->len)
							break;
						pDstData[i] = '0';
					}
					for(i = 0; i < iDestIntDigits; ++i) 
						pDstData[iDestIntDigits - i - 1] = caWork[j--];
					
					if(iDestDecimals > 0) {
						if(iDestDecimals > iDestLength) {
							j = iDestDecimals - 1 + MAX_DIGITS;
							for(i = 0; i < iDestLength; ++i)
								pDstData[iDestLength - 1 - i] = caWork[j--];
						}
						else {
							j = MAX_DIGITS;
							for(i = 0; i < iDestDecimals; ++i)
								pDstData[i + iDestIntDigits] = caWork[j++];
						}
					}
					if(pDstFld->pic[0] == 'S')
						put_sign(pDstFld, pDstData, iSrcSign);
					break;
				}

				case DTYPE_PACKED: {
					int			j, k;
					int			iSrcSign;
					int			iSrcIntDigits;
					int			iDestIntDigits;
					char		caWork[MAX_DIGITS * 2];
					unsigned char	cPack;

					/*--------------------------------------------------*\
					 |                                                  |
					 |   Currently, packed fields are assumed to be     |
                     |   signed, so extract the value of the sign from  |
                     |   the source field (zero represents positive,    | 
                     |   and non zero represents negative).             |
                     |                                                  |
                    \*--------------------------------------------------*/

					iSrcSign = extract_sign(pSrcFld, pSrcData);

					/*--------------------------------------------*\
					 |                                            |
                     |  Move the contents of the source field to  |
                     |  our work area to create a normalized      | 
                     |  9(18)v9(18) view of the data.             |
                     |                                            |
                    \*--------------------------------------------*/

					for(i = 0; i < sizeof(caWork); ++i)
							caWork[i] = '0';

					i = 0;
					if(iSrcDecimals < 0) {
						j = MAX_DIGITS + iSrcDecimals - 1;
						iSrcIntDigits = iSrcLength;
					}
					else {
						j = MAX_DIGITS - 1;
						iSrcIntDigits = iSrcLength - iSrcDecimals;
					}
					j = j - iSrcIntDigits + 1;
					for(i = 0; i < iSrcIntDigits; ++i) {
						if(i & 1)  {
							cPack = (unsigned char)pSrcData[i>>1];
							cPack = (cPack & 0xF) + '0';
							caWork[j++] = cPack;
						}
						else {
							cPack = (unsigned char)pSrcData[i>>1];
							cPack = (cPack >> 4) + '0';
							caWork[j++] = cPack;
						}
					}
					if(iSrcDecimals > 0) {
						if(iSrcDecimals > iSrcLength) {
							j = iSrcDecimals - iSrcLength + MAX_DIGITS;
							for(  ; i < iSrcLength; ++i) {
								if(i & 1) {
									cPack = (unsigned char)pSrcData[i>>1];
									cPack = (cPack & 0xF) + '0';
									caWork[j++] = cPack;
								}
								else {
									cPack = (unsigned char)pSrcData[i>>1];
									cPack = (cPack >> 4) + '0';
									caWork[j++] = cPack;
								}
							}
						}
						else {
							j = MAX_DIGITS;
							for(  ; i < iSrcLength; ++i) {
								if(i & 1) {
									cPack = (unsigned char)pSrcData[i>>1];
									cPack = (cPack & 0xF) + '0';
									caWork[j++] = cPack;
								}
								else {
									cPack = (unsigned char)pSrcData[i>>1];
									cPack = (cPack >> 4) + '0';
									caWork[j++] = cPack;
								}
							}
						}
					}

					/*--------------------------------------------*\
					 |                                            |
                     |  Now move the contents of our normalized   |
                     |  data area to the receiving field, packing |
                     |  the data as we go.  Packed fields are (at |
                     |  least at this time) always considered to  |
                     |  be signed, so we will also move a         |
                     |  trailing sign to the receiving area. A    |
                     |  value of 0xC represents a positive value  |
                     |  and 0xD represents a negative value.      |
                     |                                            |
                    \*--------------------------------------------*/

					k = 0;
					cPack = 0;
					if(iDestDecimals < 0) {
						j = MAX_DIGITS + iDestDecimals - 1;
						iDestIntDigits = iDestLength;
					}
					else {
						j = MAX_DIGITS - 1;
						iDestIntDigits = iDestLength - iDestDecimals;
					}
											/* zero fill destination first */
					for(i = 0; i < (iSrcLength); ++i) {
						if(i == pDstFld->len)
							break;
						if(i & 1)			/* if lower (right) nibble */
							(unsigned char)pDstData[k++] = 0;
					}
					if(i & 1)
						(unsigned char)pDstData[k] = 0;

					k = 0;
					j = j - iDestIntDigits + 1;
					for(i = 0; i < iDestIntDigits; ++i) {
						if(k & 1) {			/* if lower (right) nibble */
							cPack = cPack | ((unsigned char)caWork[j++] & 0xF);
							(unsigned char)pDstData[k >> 1] = cPack;
							++k;
						}
						else {
							cPack = (unsigned char)caWork[j++] << 4;
							++k;
						}
					}
					
					if(iDestDecimals > 0) {
						if(iDestDecimals > iDestLength) {
							j = iDestDecimals + MAX_DIGITS - iDestLength;
							for(i = 0; i < iDestLength; ++i) {
								if(k & 1) {
									cPack |= ((unsigned char)caWork[j++] & 0xF);
									(unsigned char)pDstData[k >> 1] = cPack;
									++k;
								}
								else {
									cPack = (unsigned char)caWork[j++] << 4;
									++k;
								}
							}
						}
						else {
							j = MAX_DIGITS;
							for(i = 0; i < iDestDecimals; ++i) {
								if(k & 1) {
									cPack |= ((unsigned char)caWork[j++] & 0xF);
									(unsigned char)pDstData[k >> 1] = cPack;
									++k;
								}
								else {
									cPack = (unsigned char)caWork[j++] << 4;
									++k;
								}
							}
						}
					}
					if(k & 1) {
						(unsigned char)pDstData[k >> 1] = cPack;
					}
					put_sign(pDstFld, pDstData, iSrcSign);
					break;
				}

				case DTYPE_EDITED:
					/*----------------------------------------------------*\
					 |                                                    |
                     |  We are handling edited moves in their own routine | 
                     |  since the logic involved is very complicated, and |
                     |  will be used in a similar fashion regardless of   |
                     |  source data type.                                 |
                    \*----------------------------------------------------*/

					move_edited(pSrcFld, pSrcData, pDstFld, pDstData);
					break;

				case DTYPE_BININT: {
					/*----------------------------------------------------*\
					 |                                                    |
                     |  Call cobmove to move the packed  source into      |
                     |  a work area of type display with an implied       |
                     |  picture of S9(10), then convert the value in the  |
                     |  work area to a 4 byte binary integer and store    |
                     |  in the receiving field.  The picture of the       |
                     |  receiving field is ignored.                       |
                     |                                                    |
                    \*----------------------------------------------------*/
					
					int					iWork;
					int					bIsNegative;
					char				caWork[10];

					FldWrk.len = 10;
					FldWrk.decimals = 0;
					FldWrk.all = 0;
					FldWrk.pic[0] = 'S';
					FldWrk.pic[1] = (char)1;
					FldWrk.pic[2] = '9';
					FldWrk.pic[3] = (char)10;
					FldWrk.pic[4] = (char)0;
                   	cob_move(pSrcFld, pSrcData, &FldWrk, caWork);
					bIsNegative = extract_sign(&FldWrk, caWork);
					iWork = 0;
					for(i = 0; i < 10; ++i) {
						iWork *= 10;
						iWork += (int)(caWork[i] - '0');
					}
					if(bIsNegative)
						iWork *= -1;
					*(int *)pDstData = iWork;
					break;
				}
			}
			break;

/* source is binary integer */
		case DTYPE_BININT: {
			int					iWork;
			int					j;
			int					bIsNegative;
			extern int			_iIntValues_[10];
			char				caWork[11];
			
			iWork = *(int *)pSrcData;
			if(pDstFld->type != DTYPE_BININT) {
				if(iWork < 0) {
					bIsNegative = 1;
					iWork *= -1;
				}
				else
					bIsNegative = 0;
	
				for(i = 0; i < 10; ++i) {
					if(iWork >= _iIntValues_[i]) {
						j = iWork / _iIntValues_[i];
						caWork[i] = (char)(j + '0');
						iWork -= (_iIntValues_[i] * j);
					}
					else
						caWork[i] = '0';
				}
			}

			switch(pDstFld->type) {	/* destination field type */
				case DTYPE_ALPHA:
				case DTYPE_ALPHANUMERIC:
				case DTYPE_GROUP: {
					/*----------------------------------------------------*\
					 |                                                    |
					 |  Expand the source field into a display field of   |
                     |  type S9(10) SIGN IS TRAILING SEPARATE, then move  |
                     |  this work field to the receiving field,           |
                     |  truncating on the right if the receiving field is |
                     |  too small, or padding with spaces on the right if |
                     |  the receiving field is larger than 11 chars.      |
                     |                                                    |
                    \*----------------------------------------------------*/
					
					if(bIsNegative)
						caWork[10] = '-';
					else
						caWork[10] = '+';
		
					if(iDestLength <= 11)
						memcpy(pDstData, caWork, iDestLength);
					else {
						memcpy(pDstData, caWork, 11);
						for(i = 11; i < iDestLength; ++i)
							pDstData[i] = ' ';
					}
					break;
				}
			case DTYPE_DISPLAY: {
					/*----------------------------------------------------*\
					 |                                                    |
					 |  Expand the source field into a display field of   |
                     |  type S9(10) then move this work field to the      |
                     |  receiving field just like any other display type. |
                     |                                                    |
                    \*----------------------------------------------------*/

					int					k;
					
					memset(pDstData, '0', iDestLength);
					if(iDestDecimals >= iDestLength)
						break;			/* dest has only fraction */

					k = 9;
					if(iDestDecimals < 0) {
						j = iDestLength - 1;
						k = k - (iDestDecimals * -1);
					}
					else
						j = iDestLength - iDestDecimals - 1;
					if(j > k)
						iWork = k;
					else
						iWork = j;
					for(i = 0; i <= iWork; ++i)
						pDstData[j - i] = caWork[k - i];
					if(pDstFld->pic[0] == 'S')
						put_sign(pDstFld, pDstData, bIsNegative);
					break;
				}
				case DTYPE_PACKED: {
					/*----------------------------------------------------*\
					 |                                                    |
					 |  Call cobmove to move the pic S9(10) data to the   |
                     |  receiving field.                                  |
                     |                                                    |
                    \*----------------------------------------------------*/

					FldWrk.len = 10;
					FldWrk.decimals = 0;
					FldWrk.type = DTYPE_DISPLAY;
					FldWrk.all = 0;
					FldWrk.pic[0] = 'S';
					FldWrk.pic[1] = 1;

					put_sign(&FldWrk, caWork, bIsNegative);
					cob_move(&FldWrk, caWork, pDstFld, pDstData);
					break;
				}
				
				case DTYPE_EDITED: {
					/*----------------------------------------------------*\
					 |                                                    |
					 |  Call move_edited using the work area defined as   |
                     |  a PIC S9(10) source field.                        |
                     |                                                    |
                    \*----------------------------------------------------*/

					FldWrk.len = 10;
					FldWrk.decimals = 0;
					FldWrk.type = DTYPE_DISPLAY;
					FldWrk.all = 0;
					FldWrk.pic[0] = 'S';
					FldWrk.pic[1] = 1;

					put_sign(&FldWrk, caWork, bIsNegative);
					move_edited(&FldWrk, caWork, pDstFld, pDstData);
					break;
				}

				case DTYPE_BININT: {
					*(int *)pDstData = iWork;
					break;
				}
			}
		}
	}
#ifdef DEBUG_RTS
	fprintf(stdout, "Source field: ");
	fprintf(stdout, "len = %ld, type = %c, decimals = %d, all = %d\n",
		pfldDesc1->len, pfldDesc1->type, (char)pfldDesc1->decimals, pfldDesc1->all);
	for(i = 0; i < 20; ++i)
		fprintf(stdout, "%02X ", (unsigned char)caData1[i]);
	fprintf(stdout, "\n");
	for(i = 0; i < 20; ++i) {
		if((caData1[i] >= ' ') && (caData1[i] < 0x7f))
			fprintf(stdout, " %c ", caData1[i]);
		else
			fprintf(stdout, "   ");
	}
	fprintf(stdout, "\n");

	fprintf(stdout, "Destination field: ");
	fprintf(stdout, "len = %ld, type = %c, decimals = %d, all = %d\n",
		pfldDesc2->len, pfldDesc2->type, (char)pfldDesc2->decimals, pfldDesc2->all);
	for(i = 0; i < 20; ++i)
		fprintf(stdout, "%02X ", (unsigned char)caData2[i]);
	fprintf(stdout, "\n");
	for(i = 0; i < 20; ++i) {
		if((caData2[i] >= ' ') && (caData2[i] < 0x7f))
			fprintf(stdout, " %c ", caData2[i]);
		else
			fprintf(stdout, "   ");
	}
	fprintf(stdout, "\n\n");
	fflush(stdout);
#endif
	return;
}

/*-----------------------------------------------------------------------*\
 |                                                                       |
 |       char *pic_expand( struct fld_desc *FieldDesc)                   |
 |                                                                       |
 |  Return a string containing the picture clause for the field          |
 |  described in the field description structure pointed to by the       |
 |  argument FieldDesc.                                                  |
 |                                                                       |
 |  The string pointed to by the field 'pic' in the fld_desc structure   |
 |  has the following format:                                            |
 |         char           cPictChar;                                     |
 |         unsigned char  cLength;                                       |
 |            . . .                                                      |
 |            . . .                                                      |
 |         unsigned char(0);                                             |
 |  Where cPictChar is a character of the picture, for example 'X', '9', |
 |  'Z', etc. and cLength is the number of times that that character is  |
 |  repeated. These two fields repeat as many times as necessary to      |
 |  describe the entire picture clause.                                  |
 |                                                                       |
 |  Example:      PIC   Z(3),Z(2)9.99CR.                                 |
 |  Would be encoded as:                                                 |
 |         char('Z');                                                    |
 |         unsigned char(3);      the 'Z' occurrs 3 times                |
 |         char(',');                                                    |
 |         unsigned char(1);      comma occurs once                      |
 |         char('Z');                                                    |
 |         unsigned char(2);      second Z occurs 2 times                |
 |         char('9');                                                    |
 |         unsigned char(1);      followed by one 9                      |
 |         char('.');                                                    |
 |         unsigned char(1);      one decimal point                      |
 |         char('9');                                                    |
 |         unsigned char(2);      two digits following dec point         |
 |         char('C');                                                    |
 |         unsigned char(1);                                             |
 |         char('R');                                                    |
 |         unsigned char(1);                                             |
 |         unsigned char(0);      terminator for the string              |
 |                                                                       |
\*-----------------------------------------------------------------------*/

char *pic_expand( pfldDesc ) 
struct fld_desc *pfldDesc;
{
	char		*pic;
	int			i = 0;
	int			j;
	char		*result;

	pic = pfldDesc->pic;
	while (*pic) {
		pic++;
		i += *((unsigned char *)pic++);
	}
	result = malloc(i + 1);		/* i has length of encoded pic string */
	if(!result) {				/* if malloc failed, do something ? */
		runtime_error(RTERR_INVALID_PIC, pfldDesc, (void *)0);
		return((char *)0);
	}

	pic = pfldDesc->pic;
	j = 0;
	while (*pic) {				/* decode the picture from fld_struct */
		for (i = 0; i < *((unsigned char *)pic + 1); i++) {
			result[j++] = *pic;
		}
		pic += 2;
	}
	result[j] = '\0';				/* null terminator for decoded pic string */
	return(result);
}

void runtime_error(int iErrorNum, struct fld_desc *pField, void *pData)
{
	int				i, j;
	extern int		_iRtErrorNbr[];
	extern char		*_szRtErrorDesc[];

	for(i = 0; _iRtErrorNbr[i] != -1; ++i)
		if(iErrorNum == _iRtErrorNbr[i])
			break;

	fprintf(stderr, "\n\nRun Time Error - ");
	if(_iRtErrorNbr[i] != -1)
		fprintf(stderr, "%s\n", _szRtErrorDesc[i]);
	else
		fprintf(stderr, "Unknown Error\n");

	if(pField) {
		fprintf(stderr, "Field Description: ");
		fprintf(stderr, "len = %ld, type = %c, decimals = %d, all = %d\n",
			pField->len, pField->type, (char)pField->decimals, pField->all);
	}
	if(!pData) {
		fprintf(stderr, "No data pointer provided\n");
		return;
	}
	
	fprintf(stderr, "Data Dump, Address = %08lX", (unsigned long)pData);
	j = 0;
	while(j < pField->len) {
		fprintf(stderr, "\n%04X: ", (unsigned)j);
		for(i = 0; i < 24; ++i) {
			if((i + j) == pField->len)
				break;
			fprintf(stderr, "%02X ", ((unsigned char *)pData)[i]);
		}
		fprintf(stderr, "\n      ");
		for(i = 0; i < 24; ++i) {
			if((i + j) == pField->len)
				break;
			if((((unsigned char *)pData)[i] >= ' ') 
			&& (((unsigned char *)pData)[i] < 0x7f))
				fprintf(stderr, " %c ", ((unsigned char *)pData)[i]);
			else
				fprintf(stderr, "   ");
		}
		j += 24;
	}
	fprintf(stderr, "\n\n");
}

void move_edited(pSrcFld, pSrcData, pDstFld, pDstData)
struct fld_desc	*pSrcFld;
struct fld_desc	*pDstFld;
char			*pSrcData;
char			*pDstData;
{
	int					i, k;
	int					iSrcPtr;
	int					iDstPtr;
	int					iDecimalPos;
	int					bEndSuppression;
	int					bIsNegative;
	int					bInFraction;
	int					bIsAlphaEdited;
	int					bIsBlankWhenZero;
	extern int			decimal_comma;

	char				*caWorkData;
	char				*pPic;
	char				caPic[3];
	char				cChar;
	char				cFloatChar;
	char				cFillChar;
	char				cDecimalPoint;
	char				cComma;
	extern char			cCurrencySymbol;

	struct fld_desc		FldWrk;

	caPic[0] = 'S';
	caPic[1] = (char)1;
	caPic[2] = (char)0;
	FldWrk.pic = caPic;
	FldWrk.len = 0;
	FldWrk.decimals = 0;
	pPic = pDstFld->pic;
	bEndSuppression = 0;
	bIsNegative = 0;
	bInFraction = 0;
	bIsAlphaEdited = 0;
	bIsBlankWhenZero = 1;
	cFloatChar = (char)0;
	if(decimal_comma) {
		cComma = '.';
		cDecimalPoint = ',';
	}
	else {
		cComma = ',';
		cDecimalPoint = '.';
	}


	for(i = 0; pPic[i]; i += 2) {
		cChar = pPic[i];
		k = (int)pPic[i + 1];
		if((cChar == 'X')
		|| (cChar == '9')
		|| (cChar == 'A')) {
			if(cChar != '9')
				bIsAlphaEdited = 1;
			bIsBlankWhenZero = 0;
			FldWrk.len += (long)k;
			if(bInFraction)
				FldWrk.decimals += k;
			bEndSuppression = 1;
			continue;
		}
		if(cChar == 'Z') {
			FldWrk.len += (long)k;
			if(bInFraction)
				FldWrk.decimals += k;
			continue;
		}
		if(cChar == '*') {
			FldWrk.len += (long)k;
			if(bInFraction)
				FldWrk.decimals += k;
			bIsBlankWhenZero = 0;
			continue;
		}
		if(cChar == cCurrencySymbol) {
			if(cFloatChar) {
				if(cFloatChar == cCurrencySymbol) {
					FldWrk.len += (long)k;
					if(bInFraction)
						FldWrk.decimals += k;
				}
				continue;
			}
			if(k > 1) {
				cFloatChar = cChar;
				FldWrk.len += (k - 1);
				if(bInFraction)
					FldWrk.decimals += (k - 1);
				continue;
			}
			if(pPic[i + 2] == cComma) {
				cFloatChar = cChar;
				continue;
			}
			if(pPic[i + 2] == cDecimalPoint) {
				cFloatChar = cChar;
				continue;
			}
			bIsBlankWhenZero = 0;
			continue;
		}
		if(cChar == '+') {
			if(cFloatChar) {
				if(cFloatChar == '+') {
					FldWrk.len += (long)k;
					if(bInFraction)
						FldWrk.decimals += k;
				}
				continue;
			}
			if(k > 1) {
				cFloatChar = cChar;
				FldWrk.len += (k - 1);
				if(bInFraction)
					FldWrk.decimals += (k - 1);
				continue;
			}
			if(pPic[i + 2] == cComma) {
				cFloatChar = cChar;
				continue;
			}
			if(pPic[i + 2] == cDecimalPoint) {
				cFloatChar = cChar;
				continue;
			}
			bIsBlankWhenZero = 0;
			continue;
		}
		if(cChar == '-') {
			if(cFloatChar) {
				if(cFloatChar == '-') {
					FldWrk.len += (long)k;
					if(bInFraction)
						FldWrk.decimals += k;
				}
				continue;
			}
			if(k > 1) {
				cFloatChar = cChar;
				FldWrk.len += (k - 1);
				if(bInFraction)
					FldWrk.decimals += (k - 1);
				continue;
			}
			if(pPic[i + 2] == cComma) {
				cFloatChar = cChar;
				continue;
			}
			if(pPic[i + 2] == cDecimalPoint) {
				cFloatChar = cChar;
				continue;
			}
			bIsBlankWhenZero = 0;
			continue;
		}
		if(cChar == cComma) {
			continue;
		}
		if(cChar == cDecimalPoint) {
			bInFraction = 1;
			continue;
		}
		if(cChar == 'V') {
			bInFraction = 1;
			continue;
		}
		bIsBlankWhenZero = 0;
	}

	if(!FldWrk.len)
		return;

	caWorkData = (char *)malloc(FldWrk.len);
	if(!caWorkData)
		return;
	if(bIsAlphaEdited) 
		FldWrk.type = DTYPE_ALPHANUMERIC;
	else
		FldWrk.type = DTYPE_DISPLAY;
	cob_move(pSrcFld, pSrcData, &FldWrk, caWorkData);
	if(!bIsAlphaEdited)
		bIsNegative = extract_sign(&FldWrk, caWorkData);

	if(bIsBlankWhenZero) {
		for(i = 0; i < FldWrk.len; ++i)
			if(caWorkData[i] != '0')
				break;
		if(i == FldWrk.len) {		/* it's zero, blank it out */
			memset(pDstData, ' ', pDstFld->len);
			return;
		}
	}
				
	bEndSuppression = 0;
	bInFraction = 0;
	cFloatChar = (char)0;
	cFillChar = ' ';
	iSrcPtr = 0;
	iDstPtr = 0;
	iDecimalPos = 0;

	pPic = pic_expand(pDstFld);
	if(!pPic) {
		free(caWorkData);
		return;
	}
	for(i = 0; pPic[i]; ++i) {
		cChar = pPic[i];
		if((cChar == 'X')
		|| (cChar == '9')
		|| (cChar == 'A')) {
			if(!bEndSuppression) {		/* all suppressed so far */
				if(i != 0) { 			/* not first character */
					if(cFloatChar) {
						pDstData[iDstPtr - 1] = cFloatChar;
					}
					else {
						if(pPic[i - 1] == cComma)
							pDstData[iDstPtr - 2] = cFillChar;
						else
							pDstData[iDstPtr - 1] = cFillChar;
					}
				}
			}
			pDstData[iDstPtr++] = caWorkData[iSrcPtr++];
			bEndSuppression = 1;
			continue;
		}
		if((cChar == 'Z')
		|| (cChar == '*')) {
			if(bEndSuppression)
				pDstData[iDstPtr++] = caWorkData[iSrcPtr];
			else {
				if(caWorkData[iSrcPtr] != '0') {
					pDstData[iDstPtr++] = caWorkData[iSrcPtr];
					bEndSuppression = 1;
				}
				else {
					if(cChar == 'Z')
						pDstData[iDstPtr++] = ' ';
					else {
						cFillChar = '*';
						pDstData[iDstPtr++] = cChar;
					}
				}
			}
			++iSrcPtr;
			continue;
		}
		if(cChar == cCurrencySymbol) {
			if((cChar != cFloatChar)
			&& (i)) {
				if((pPic[i - 1] != cChar)
				&& (pPic[i + 1] != cChar)
				&& (pPic[i + 1] != cComma)) {	
					bEndSuppression = 1;
					pDstData[iDstPtr++] = cChar;
					continue;
				}
			}
			if((cChar != cFloatChar)
			&& (!i)) {
				if((pPic[i + 1] != cChar)
				&& (pPic[i + 1] != cComma)) {	
					bEndSuppression = 1;
					pDstData[iDstPtr++] = cChar;
					continue;
				}
			}
			if(bEndSuppression) {
				pDstData[iDstPtr++] = caWorkData[iSrcPtr++];
				continue;
			}
			if(cFloatChar) {
				if(cFloatChar == cCurrencySymbol) {
					if(caWorkData[iSrcPtr] != '0') {
						pDstData[iDstPtr - 1] = cChar;
						pDstData[iDstPtr++] = caWorkData[iSrcPtr++];
						bEndSuppression = 1;
					}
					else {
						if(pPic[i + 1] == cDecimalPoint) {
							pDstData[iDstPtr++] = cChar;
							bEndSuppression = 1;
							++iSrcPtr;
							continue;
						}
						pDstData[iDstPtr++] = ' ';
						++iSrcPtr;
					}
				}
				else
					pDstData[iDstPtr++] = cChar;
				continue;
			}
			if((pPic[i + 1] == cChar)
			|| (pPic[i + 1] == cComma)) {
				cFloatChar = cChar;
				pDstData[iDstPtr++] = ' ';
				continue;
			}
			pDstData[iDstPtr++] = cChar;
			continue;
		}
		if(cChar == '+') {
			if((cChar != cFloatChar)
			&& (i != 0)) {
				if((pPic[i - 1] != cChar)
				&& (pPic[i + 1] != cChar)
				&& (pPic[i + 1] != cComma)) {
					bEndSuppression = 1;
					if(bIsNegative)
						pDstData[iDstPtr++] = '-';
					else
						pDstData[iDstPtr++] = '+';
					continue;
				}
			}
			if((cChar != cFloatChar)
			&& (i == 0)) {
				if((pPic[i + 1] != cChar)
				&& (pPic[i + 1] != cComma)) {
					bEndSuppression = 1;
					if(bIsNegative)
						pDstData[iDstPtr++] = '-';
					else
						pDstData[iDstPtr++] = '+';
					continue;
				}
			}
			if(bEndSuppression) {
				pDstData[iDstPtr++] = caWorkData[iSrcPtr++];
				continue;
			}
			if(cFloatChar) {
				if(cFloatChar == '+') {
					if(caWorkData[iSrcPtr] != '0') {
						if(bIsNegative)
							pDstData[iDstPtr - 1] = '-';
						else
							pDstData[iDstPtr - 1] = '+';
						pDstData[iDstPtr++] = caWorkData[iSrcPtr++];
						bEndSuppression = 1;
					}
					else {
						if(pPic[i + 1] == cDecimalPoint) {
							if(bIsNegative)
								pDstData[iDstPtr++] = '-';
							else
								pDstData[iDstPtr++] = '+';
							++iSrcPtr;
							bEndSuppression = 1;
							continue;
						}
						pDstData[iDstPtr++] = ' ';
						++iSrcPtr;
					}
				}
				else
					pDstData[iDstPtr++] = cChar;
				continue;
			}
			if((pPic[i + 1] == cChar)
			|| (pPic[i + 1] == cComma)) {
				cFloatChar = cChar;
				pDstData[iDstPtr++] = ' ';
				continue;
			}
			pDstData[iDstPtr++] = cChar;
			continue;
		}
		if(cChar == '-') {
			if((cChar != cFloatChar)
			&& (i)) {
				if((pPic[i - 1] != cChar)
				&& (pPic[i + 1] != cChar)
				&& (pPic[i + 1] != cComma)) {		
					bEndSuppression = 1;
					if(bIsNegative)
						pDstData[iDstPtr++] = cChar;
					else
						pDstData[iDstPtr++] = ' ';
					continue;
				}
			}
			if((cChar != cFloatChar)
			&& (!i)) {
				if((pPic[i + 1] != cChar)
				&& (pPic[i + 1] != cComma)) {		
					bEndSuppression = 1;
					if(bIsNegative)
						pDstData[iDstPtr++] = cChar;
					else
						pDstData[iDstPtr++] = ' ';
					continue;
				}
			}
			if(bEndSuppression) {
				pDstData[iDstPtr++] = caWorkData[iSrcPtr++];
				continue;
			}
			if(cFloatChar) {
				if(cFloatChar == '-') {
					if(caWorkData[iSrcPtr] != '0') {
						if(bIsNegative)
							pDstData[iDstPtr - 1] = '-';
						else
							pDstData[iDstPtr - 1] = ' ';
						pDstData[iDstPtr++] = caWorkData[iSrcPtr++];
						bEndSuppression = 1;
					}
					else {
						if(pPic[i + 1] == cDecimalPoint) {
							if(bIsNegative)
								pDstData[iDstPtr++] = '-';
							else
								pDstData[iDstPtr++] = ' ';
							++iSrcPtr;
							bEndSuppression = 1;
							continue;
						}
						pDstData[iDstPtr++] = ' ';
						++iSrcPtr;
					}
				}
				else
					pDstData[iDstPtr++] = cChar;
				continue;
			}
			if((pPic[i + 1] == cChar)
			|| (pPic[i + 1] == cComma)) {
				cFloatChar = cChar;
				pDstData[iDstPtr++] = ' ';
				continue;
			}
			pDstData[iDstPtr++] = cChar;
			continue;
		}
		if(cChar == cComma) {
			if(bEndSuppression)
				pDstData[iDstPtr++] = cChar;
			else
				pDstData[iDstPtr++] = cFillChar;
			continue;
		}
		if(cChar == cDecimalPoint) {
			pDstData[iDstPtr++] = '.';
			iDecimalPos = iDstPtr;
			continue;
		}
		if((cChar == 'C')
		&& (pPic[i + 1] == 'R')) {
			if(bIsNegative) {
				pDstData[iDstPtr++] = 'C';
				pDstData[iDstPtr++] = 'R';
			}
			else {
				pDstData[iDstPtr++] = ' ';
				pDstData[iDstPtr++] = ' ';
			}
			++i;
			continue;
		}
		if((cChar == 'D')
		&& (pPic[i + 1] == 'B')) {
			if(bIsNegative) {
				pDstData[iDstPtr++] = 'D';
				pDstData[iDstPtr++] = 'B';
			}
			else {
				pDstData[iDstPtr++] = ' ';
				pDstData[iDstPtr++] = ' ';
			}
			++i;
			continue;
		}
		if(cChar == 'B') {
			pDstData[iDstPtr++] = ' ';
			continue;
		}
		if(cChar == 'V') {
			continue;
		}
		pDstData[iDstPtr++] = cChar;
	}
	if((iDecimalPos)
	&& (bEndSuppression)) {
		for(i = iDecimalPos; i < pDstFld->len; ++i) {
			if((pDstData[i] >= '0')
			&& (pDstData[i] <= '9'))
				break;
			pDstData[i] = '0';
		}
	}
	free(pPic);
	free(caWorkData);
	return;
}

		


int _iRtErrorNbr[] = 
{
	RTERR_INVALID_DATA,
	RTERR_INVALID_PIC,
	-1
};
char *_szRtErrorDesc[] =
{
	"Invalid Data Content",
	"Invalid Picture Structure",
	(char *)0
};

void _DUMP_(unsigned char *caData, char *szCount, char *caOut)
{
	int				i, k;
	unsigned char	c;

	k = 0;
	for(i = 0; i < 4; ++i) {
        if(szCount[i] == '\0')
			break;
		k *= 10;
		k += (szCount[i] - '0');
	}

	for(i = 0; i < k; ++i) {
		c = caData[i] & 0xf;
		if(c > 9)
			c = ('A' + (c - 10));
		else
			c += '0';
		caOut[(i * 2) + 1] = c;
		c = caData[i] >> 4;
		if(c > 9)
			c = ('A' + (c - 10));
		else
			c += '0';
		caOut[i * 2] = c;
	}
}




int decimal_comma = 0;
char cCurrencySymbol = '$';
int _iIntValues_[10] = {
	1000000000,
	100000000,
	10000000,
	1000000,
	100000,
	10000,
	1000,
	100,
	10,
	1
};
