// 
// 
//   Cobol Compiler Library -- File Handling Module
// 
// 

// Permissible statements -- table from CD-1.2 (cobol draft) pp. 450
//   ----------------------------------------------------------------- 
//   acc. mode | statement  | -------- open mode ------
//             |            | input output  i-o  extend
//   ----------+------------+--------------------------
//   sequential|  read      |   X            X
//             |  write     |          X           X
//             |  rewrite   |                X
//   ----------+------------+--------------------------
//   sequential|  delete    |                X
//   (relative |            |
//    & indexed|  start     |   X            X
//    files)   |            |
//   ----------+------------+--------------------------
//   random    |  read      |   X            X
//             |  write     |          X     X
//             |  rewrite   |                X
//             |  start     |
//             |  delete    |                X
//  ----------+------------+--------------------------
//   dynamic   |  read      |   X            X
//             |  write     |          X     X
//             |  rewrite   |                X
//             |  start     |   X            X
//             |  delete    |                X
//   ----------+------------+--------------------------


#include "htcoblib.h"

#if defined(SunOS)
va_list __builtin_va_alist;
#endif

#ifdef WANT_DYNAMIC_LIBS
DB *db_open_stub( const char *s, int i, int j, DBTYPE d, const void *p );
DB *(* db_open)( const char *, int, int, DBTYPE, const void * ) = 
	db_open_stub;

DB *
db_open_stub( const char *s, int i, int j, DBTYPE d, const void *p ) {
	char *libname = "libdb.so";
	void *handle = dlopen(libname,RTLD_LAZY);
	if (!handle) {
		fprintf(stderr,"*ERROR* loading %s: %s\n",libname,dlerror());
		return;
	}
	db_open = dlsym(handle,"dbopen");
	return (* db_open)( s,i,j,d,p );
}
#else
#define db_open dbopen
#endif

int cob_open( struct file_desc *f, char *record, char *fname, int mode )
{
	DBTYPE type=DB_BTREE;
	void *infop=NULL;
	int sflags=S_IRUSR | S_IWUSR; 
	int oflags;
	char *filename,*s;
	char alt_filename[128];
	int alt_key_no;
	int len;
	
	BTREEINFO alt_key;
        alt_key.flags = 0;
        alt_key.cachesize = 0;
        alt_key.maxkeypage = 0;
        alt_key.minkeypage = 0;
        alt_key.psize = 0;
        alt_key.compare = NULL;
        alt_key.prefix = NULL;
        alt_key.lorder = 0;

	/* beware: fname points to a field storage (non null-terminated) 
	   we must copy fname to a C string and terminate it with a \0
	   and also trim spaces at the end. */
	len = f->fname_desc->len;
	filename = malloc(len+1);
	memmove(filename,fname,len);
	s=filename+len;
	*s-- = 0;
	while (*s == ' ') *s-- = 0;
	
	/* Check to see if the file is already open. If so return
	   File Status 91 in according to the Ansi 74 Standard. */
	if (f->dbp != NULL)
		return 91;

	switch (mode) {
	case FMOD_INPUT: 
		oflags = O_RDONLY; break;
	case FMOD_IO: 
		oflags = O_RDWR; break;
	case FMOD_OUTPUT: 
		/* libdb doesn't support O_WRONLY */ 
		oflags = O_CREAT | O_TRUNC | O_RDWR; 
		break;
	case FMOD_EXTEND: 
		oflags = O_RDWR | O_APPEND; break;
	}
	if (f->organization == ORG_INDEXED) {
		type= DB_BTREE; 
	}
	else if (f->organization == ORG_RELATIVE) {
/*		type= DB_RECNO;  */
	}
	if (f->organization == ORG_INDEXED) {
		struct altkey_desc *akd;	
		akd = (struct altkey_desc *)( f+1 );
		alt_key_no=1;
		while (akd->offset != -1) {
//			printf("Alternate key: offset=%d, descriptor pic=0x%08x, dupl=%d\n",
//				akd->offset, (int)akd->descriptor->pic, akd->duplicates);
			if (akd->duplicates > 0)
			{
				alt_key.flags = R_DUP;
			}
			else
			{
				alt_key.flags = 0;
			}
			sprintf(alt_filename,"%s%d",filename,alt_key_no);
			akd->alt_dbp = db_open(alt_filename,oflags,sflags,type,&alt_key);					
			if (!akd->alt_dbp) {
				if (errno == EINVAL) return 37;
				return 91;
			}
			alt_key_no++;
			akd++;
		}
		f->dbp = db_open(filename,oflags,sflags,type,infop);
	}
	/* otherwise it is sequential or relative, save its file handle, converted */
	else
		f->dbp = (void *)open(filename, oflags, sflags);
	free(filename);
	if (!f->dbp) {
		if (errno == EINVAL) return 37;
		return 91;
	}
	/* save mode to check later (read,write,start,...) */
	f->open_mode = mode;
	return 0;
}

int cob_close( struct file_desc *f, char *record)
{
	/* Check to see if file is open. If not return File Status 91
	   In accordance with the Cobol 74 Standard. */

	if (f->dbp == NULL)
		return 91;

	if (f->organization == ORG_INDEXED)
		{
			struct altkey_desc *akd;	
			akd = (struct altkey_desc *)( f+1 );
			while (akd->offset != -1) {
				akd->alt_dbp->close(akd->alt_dbp);
				akd++;
			}
			f->dbp->close(f->dbp);
			f->dbp = NULL;
		}
	else	
		{
			close((int)f->dbp);
			f->dbp = NULL;
		}
	return 0;
}

int cob_read( struct file_desc *f, char *record, ... ) {
	int result;
	recno_t recno;
	struct fld_desc *fkey;
	char *keybuf;
	off_t	file_pos;
	DBT key, data;
	va_list args;
	char dummy[10];
	
	/* Check to see if file is open. If not return File Status 92
	   In accordance with the Cobol 74 Standard. */

	if (f->dbp == NULL)
		return 92;
	
	/* Check the mode the file is opened in to make sure that read
	   is Allowed */
	if (((f->open_mode != FMOD_INPUT) && (f->open_mode != FMOD_IO)))
		return 92;
	
	if (f->organization == ORG_RELATIVE)
	{
		va_start(args,record);
		recno = va_arg(args,recno_t);
		va_end(args);
		if (recno < 1)
			return 23;
		file_pos = lseek((int)f->dbp, ((recno)*((f->reclen))), SEEK_SET);
		result = read( (int)f->dbp, record, f->reclen );
		if (record[0]=='\0')
			return 23;
		if (result == f->reclen)
			return 0;
		if (result == 0)
			return 10;
		return 30;
	}
	else if (f->organization == ORG_INDEXED)
	{
		va_start(args,record);
		fkey = va_arg(args, struct fld_desc *);
		keybuf = va_arg(args, char *);
		va_end(args);
		key.data = record+f->rec_index;
		key.size = f->ixd_desc->len;
		if(fkey == NULL)
		{
			f->key_in_use=NULL;
		}
		else
		{
			struct altkey_desc *akd;	
			akd = (struct altkey_desc *)( f+1 );
			while (akd->offset != -1) {
				if (akd->descriptor->pic == fkey->pic)
				{
					f->key_in_use = akd;
					key.data = keybuf;
					key.size = akd->descriptor->len;
					result=akd->alt_dbp->seq(akd->alt_dbp,&key,&data,R_CURSOR);
					if (result)
						return 23;
					if (memcmp(key.data,keybuf,key.size)!=0)
						return 23;
					key.data = data.data;
					key.size = f->ixd_desc->len;
				}
				akd++;
			}
			
		}
		result = f->dbp->seq(f->dbp,&key,&data,R_CURSOR);
		if (result) 
			return 23; /* should have a better error info here */
		if (data.size < f->reclen)
			return 23;
		memmove( record, data.data, f->reclen );
		if (memcmp(key.data,record+f->rec_index,key.size)!=0)
			return 23;
		return 0;
	} else if (f->organization == ORG_LINESEQUENTIAL)
	/* If the file is LINE SEQUENTIAL we need to read in the <NL> as well */
	{
		result = read( (int)f->dbp, record, f->reclen );
		if (result == f->reclen)
			/* Throw the New Line Away */
			result = read( (int)f->dbp, dummy, 1 );
		if (result == 1)
			return 0;
		if (result == 0)
			return 10;
		return 30;
	} else { /* sequential files */
		result = read( (int)f->dbp, record, f->reclen );
		if (result == f->reclen)
			return 0;
		if (result == 0)
			return 10;
		return 30;
	}
}

int cob_read_next( struct file_desc *f, char *record) {
	int result;
	int flags=R_NEXT;
	DBT key,data;
	
	/* Check to see if file is open. If not return File Status 92
	   In accordance with the Cobol 74 Standard. */

	if (f->dbp == NULL)
		return 92;
	
	/* Check the mode the file is opened in to make sure that read
	   is Allowed */
	if (((f->open_mode != FMOD_INPUT) && (f->open_mode != FMOD_IO)))
		return 92;
	
	if (f->organization == ORG_SEQUENTIAL)
	{ 
		result = read( (int)f->dbp, record, f->reclen );
		if (result <= 0)
			return 10; /* what errors should I return? */
		return 0;
	}
	if (f->organization == ORG_RELATIVE)
	{ 	
		result=1;
		record[0]='\0';
		while (record[0]=='\0' && result >0)
		{
			result = read( (int)f->dbp, record, f->reclen );
		}
		if (result <= 0)
			return 10; /* what errors should I return? */
		return 0;
	}
        if (f->organization == ORG_INDEXED)
	{
        	if (f->key_in_use !=NULL)
		{
			struct altkey_desc *akd;	
			akd = (struct altkey_desc *)( f->key_in_use );
			result=akd->alt_dbp->seq(akd->alt_dbp,&key,&data,flags);
			if (result)
				return 10;
			key.data = data.data;
			key.size = f->ixd_desc->len;
			flags=0;
			result = f->dbp->get(f->dbp,&key,&data,flags);
        		if (result)
                		return 23; /* should have a better error info here */
        		if (data.size < f->reclen)
               			 return 23;
        		memmove( record, data.data, f->reclen );
        		return 0;
	   	}	
		result = f->dbp->seq(f->dbp,&key,&data,flags);
        	if (result)
                	return 23; /* should have a better error info here */
        	if (data.size < f->reclen)
               		 return 23;
        	memmove( record, data.data, f->reclen );
        	return 0;
	}
	return 99;
}

int cob_read_prev( struct file_desc *f, char *record) {
	int result;
        int flags=R_PREV;
	off_t file_pos;
        DBT key,data;
	
	/* Check to see if file is open. If not return File Status 92
	   In accordance with the Cobol 74 Standard. */

	if (f->dbp == NULL)
		return 92;
	
	/* Check the mode the file is opened in to make sure that read
	   is Allowed */
	if (((f->open_mode != FMOD_INPUT) && (f->open_mode != FMOD_IO)))
		return 92;

        if (f->organization == ORG_SEQUENTIAL)
        {
		/* Need some logic here to figure out if at beginning of file */
		/* As a result of the previous Read Previous */
		file_pos = lseek((int)f->dbp, (2*((f->reclen)* -1)), SEEK_CUR);
                result = read( (int)f->dbp, record, f->reclen );
                if (result <=0)
                        return 10;
                return 0;
        }
        if (f->organization == ORG_RELATIVE)
        {
		result=1;
		record[0]='\0';
		while (record[0]=='\0' && result >0)
		{
			file_pos = lseek((int)f->dbp, (2*((f->reclen)* -1)), SEEK_CUR);
                	if (file_pos == 0)
				return 10;
			result = read( (int)f->dbp, record, f->reclen );
		}
                if (result<=0)
                        return 10;
                return 0;
        }
        if (f->organization == ORG_INDEXED)
	{
        	if (f->key_in_use !=NULL)
		{
			struct altkey_desc *akd;	
			akd = (struct altkey_desc *)( f->key_in_use );
			result=akd->alt_dbp->seq(akd->alt_dbp,&key,&data,flags);
			if (result)
				return 10;
			key.data = data.data;
			key.size = f->ixd_desc->len;
			flags=0;
			result = f->dbp->get(f->dbp,&key,&data,flags);
        		if (result)
                		return 23; /* should have a better error info here */
        		if (data.size < f->reclen)
               			 return 23;
        		memmove( record, data.data, f->reclen );
        		return 0;
	   	}	
		result = f->dbp->seq(f->dbp,&key,&data,flags);
        	if (result)
                	return 23; /* should have a better error info here */
        	if (data.size < f->reclen)
               		 return 23;
        	memmove( record, data.data, f->reclen );
        	return 0;
	}
	return 99;
}

int cob_write( struct file_desc *f, char *record, ... )
{
	int result;
	recno_t recno;
	int flags=0;
	off_t file_pos;
	DBT key, data;
	va_list args;
	char *NL="\n";
	
	/* Check to see if file is open. If not return File Status 92
	   In accordance with the Cobol 74 Standard. */

	if (f->dbp == NULL)
		return 92;
	
	/* Check the mode the file is opened in to make sure that write
	   is Allowed */
	if (((f->open_mode != FMOD_OUTPUT) && (f->open_mode != FMOD_IO)))
		return 92;
	
	data.data = record;
	data.size = f->reclen; 
	if (f->organization == ORG_INDEXED) {
		key.data = record+f->rec_index;
		key.size = f->ixd_desc->len;
		result = f->dbp->put(f->dbp,&key,&data,flags);
		if (result) {
			return 99;
		}
		/* If the main write was successfull then we proceed to
		write out the alternate keys. Any Failure will mean that 
		we have to delete the writes we have just done.
		*/
		{
			struct altkey_desc *akd;	
			akd = (struct altkey_desc *)( f+1 );
			while (akd->offset != -1) {
				key.data = record+akd->offset;
				key.size = akd->descriptor->len;
				data.data = record+f->rec_index;
				data.size = f->ixd_desc->len;
				result=akd->alt_dbp->put(akd->alt_dbp,&key,&data,flags);
				/* If an error occurs we need code to back out.
				Will be done later.
				*/
				akd++;
			}
		}
		return 0;	
		
	}	
	else if (f->organization == ORG_RELATIVE) {
		va_start(args,record);
		recno = va_arg(args,recno_t);
		va_end(args);
		if (recno < 1)
			return 23;
		file_pos = lseek((int)f->dbp, ((recno)*((f->reclen))), SEEK_SET);
		result = write( (int)f->dbp, record, f->reclen );
		if (!result)
			return 99; /* what errors should I return? */
		return 0;
	}
	else if (f->organization == ORG_LINESEQUENTIAL) {
		result = write( (int)f->dbp, record, f->reclen );
		if (!result)
			return 99; /* what errors should I return? */
		result = write( (int)f->dbp, NL, 1 );
		if (!result)
			return 99; /* what errors should I return? */
		return 0;
	}
	else {
		result = write( (int)f->dbp, record, f->reclen );
		if (!result)
			return 99; /* what errors should I return? */
		return 0;
	}
}

int cob_delete( struct file_desc *f, char *record, ...) 
{
	int result;
	recno_t recno;
	int flags=0;
	off_t file_pos;
	va_list args;
	DBT key;
	
	/* Check to see if file is open. If not return File Status 92
	   In accordance with the Cobol 74 Standard. */

	if (f->dbp == NULL)
		return 92;
	
	/* Check the mode the file is opened in to make sure that delete
	   is Allowed */
	if (f->open_mode != FMOD_IO)
		return 92;
	

	switch (f->organization) {
	case ORG_INDEXED:
		{
			struct altkey_desc *akd;	
			akd = (struct altkey_desc *)( f+1 );
			while (akd->offset != -1) {
				key.data = record+akd->offset;
				key.size = akd->descriptor->len;
				result=akd->alt_dbp->del(akd->alt_dbp,&key,flags);
				akd++;
			}
		}
		key.data = record+f->rec_index;
		key.size = f->ixd_desc->len; 
		result = f->dbp->del(f->dbp,&key,flags);
		if (result != 0)
			return 23;
		return 0;
		break;	
	case ORG_RELATIVE: 
		va_start(args,record);
		recno = va_arg(args,recno_t);
		va_end(args);
		if (recno < 1)
			return 23;
/*		recno = recno - 1; */
		file_pos = lseek((int)f->dbp, ((recno)*((f->reclen))), SEEK_SET);
		memset(record,0,f->reclen);
		result = write( (int)f->dbp, record, f->reclen );
		if (!result)
			return 99; /* what errors should I return? */
		return 0;
	default: 
		return 37;
	}
}

//int cob_start( struct file_desc *f, char *record, ...)
int cob_start( struct file_desc *f, char *record, int cond, struct fld_desc *r, char *key_ptr, ...)
{
	int result;
	recno_t recno;
	int flags=R_CURSOR;
	off_t file_pos;
	DBT key, data;
	va_list args;
	int alternate=0;
	
	/* Check to see if file is open. If not return File Status 92
	   In accordance with the Cobol 74 Standard. */

	if (f->dbp == NULL)
		return 92;
	
	/* Check the mode the file is opened in to make sure that start
	   is Allowed */
	if (((f->open_mode != FMOD_INPUT) && (f->open_mode != FMOD_IO)))
		return 92;
	
	if (f->organization == ORG_RELATIVE)
	{
//		va_start(args,record);
		va_start(args,key_ptr);
		recno = va_arg(args,recno_t);
		va_end(args);
		file_pos = lseek((int)f->dbp, ((recno)*((f->reclen))), SEEK_SET);
		if ( file_pos > 0)
			return 0;
		else return 23;
	}
	else if (f->organization == ORG_INDEXED)
	{
		struct altkey_desc *akd;
		akd = (struct altkey_desc *)( f+1 );
		f->key_in_use = NULL;
		while (akd->offset != -1)
		{
			if (akd->descriptor == r)
			{
				f->key_in_use = akd;
				alternate = 1;
				break;
			}
			akd++;
		}
		switch(cond)
		{
		case 1:
			key.data = record+f->rec_index;
			key.size = f->ixd_desc->len;
			if (alternate ==  1)
			{
				key.data = key_ptr;
				key.size = akd->descriptor->len;
				result=akd->alt_dbp->seq(akd->alt_dbp,&key,&data,flags);
				if (result)
					return 23;
				if (memcmp(key.data,key_ptr,key.size)!=0)
					return 23;
				return 0;
			}
			else
			{
			result = f->dbp->seq(f->dbp,&key,&data,flags);
			if (result) 
				return 23; /* should have a better error info here */
			if (data.size < f->reclen)
				return 23;
			memmove( record, data.data, f->reclen );
			if (memcmp(key.data,record+f->rec_index,key.size)!=0)
				return 23; 
			return 0;
			}
			break;
		case 2:
		case 3:
		case 4:
		case 5:
		case 6:
		default:
			return 99;
		}
	} else { /* sequential files */
		return 0;
	}
}

int cob_save_status( char *status, int rt )
{
	*status++ = rt/10+'0';
	*status = rt%10+'0';
	return rt;
}

int cob_rewrite( struct file_desc *f, char *record, ... )
{
	int result;
	recno_t recno;
	int flags=0;
	off_t file_pos;
	DBT key, data;
	char newdata_data[65536];
	int newdata_size;
 	va_list args;
	char *NL="\n";
	
	/* Check to see if file is open. If not return File Status 92
	   In accordance with the Cobol 74 Standard. */

	if (f->dbp == NULL)
		return 92;
	
	/* Check the mode the file is opened in to make sure that rewrite
	   is Allowed */
	if (f->open_mode != FMOD_IO)
		return 92;

	switch (f->organization) {
	case ORG_INDEXED:
		break;
	case ORG_RELATIVE: 
		break;
	case ORG_SEQUENTIAL: 
		file_pos = lseek((int)f->dbp, ((f->reclen)* -1), SEEK_CUR);
		break;
	case ORG_LINESEQUENTIAL:
		file_pos = lseek((int)f->dbp, (((f->reclen+1))* -1), SEEK_CUR);
		break;
	default:
		return 30;
	}
	
	data.data = record;
	data.size = f->reclen; 
	/* Save the new record */
	memmove( newdata_data, record, f->reclen );
	newdata_size = f->reclen; 
	if (f->organization == ORG_INDEXED) {
		key.data = record+f->rec_index;
		key.size = f->ixd_desc->len;
		/* Get the origional Record so we can delete the
		   Alternate Keys is there is any change */
		result = f->dbp->get(f->dbp,&key,&data,flags);
		memmove( record, data.data, f->reclen );
		if (result)
			return 23;
		{
			struct altkey_desc *akd;	
			akd = (struct altkey_desc *)( f+1 );
			while (akd->offset != -1) {
				key.data = record+akd->offset;
				key.size = akd->descriptor->len;
				result=akd->alt_dbp->del(akd->alt_dbp,&key,flags);
				akd++;
			}
		}
		memmove( record, newdata_data, newdata_size );
		key.data = record+f->rec_index;
		key.size = f->ixd_desc->len;
		data.data = newdata_data;
		data.size = newdata_size; 
		/* Rewrite the Main Record */
		result = f->dbp->put(f->dbp,&key,&data,flags);
		if (!result) {
		{
			/* Rewrite the Alternate Keys */
			struct altkey_desc *akd;	
			akd = (struct altkey_desc *)( f+1 );
			while (akd->offset != -1) {
				key.data = record+akd->offset;
				key.size = akd->descriptor->len;
				data.data = record+f->rec_index;
				data.size = f->ixd_desc->len;
				result=akd->alt_dbp->put(akd->alt_dbp,&key,&data,flags);
				akd++;
			}
		}
			return 0;
		} else {
			return 99; /* ? error code to be determined */
		}
	}	
	else if (f->organization == ORG_RELATIVE) {
		va_start(args,record);
		recno = va_arg(args,recno_t);
		va_end(args);
		file_pos = lseek((int)f->dbp, ((recno)*((f->reclen))), SEEK_SET);
		result = write( (int)f->dbp, record, f->reclen );
		if (!result)
			return 99; /* what errors should I return? */
		return 0;
	}
	else if (f->organization == ORG_LINESEQUENTIAL) {
		result = write( (int)f->dbp, record, f->reclen );
		if (!result)
			return 99; /* what errors should I return? */
		result = write( (int)f->dbp, NL, 1 );
		if (!result)
			return 99; /* what errors should I return? */
		return 0;
	}
	else {
		result = write( (int)f->dbp, record, f->reclen );
		if (!result)
			return 99; /* what errors should I return? */
		return 0;
	}
}

int cob_write_adv( struct file_desc *f, char *record, int opt, ... )
{
	int result;
	va_list args;
	struct fld_desc *cnt_desc;
	char *cnt_buf;
        int lines=0;
	
	/* Check to see if file is open. If not return File Status 92
	   In accordance with the Cobol 74 Standard. */

	if (f->dbp == NULL)
		return 92;
	
	/* Check the mode the file is opened in to make sure that write
	   is Allowed */
	if (((f->open_mode != FMOD_OUTPUT) && (f->open_mode != FMOD_IO) && (f->open_mode != FMOD_EXTEND)))
		return 92;
	
	va_start( args, opt );
        if ( opt>0 ) {
                cnt_desc = va_arg( args, struct fld_desc * );
                cnt_buf = va_arg( args,char * );
                lines=get_index( cnt_desc,cnt_buf );
                if (opt==1) {
			result = write( (int)f->dbp, record, f->reclen );
                        for (;lines;lines--)
				write( (int)f->dbp, "\x0a", 1 );
                }
                else { /* opt==2 */
                        for (;lines;lines--)
				write( (int)f->dbp, "\x0a", 1 );
			result = write( (int)f->dbp, record, f->reclen );
                }
        }
        else if ( opt==-1 ) { /* before advancing page */
		result = write( (int)f->dbp, record, f->reclen );
		write( (int)f->dbp, "\x0a\x0c", 2 );
        }
        else if ( opt==-2 ) { /* after advancing page */
		write( (int)f->dbp, "\x0a\x0c", 2 );
		result = write( (int)f->dbp, record, f->reclen );
        }
	else /* only normal write (error?) */
		result = write( (int)f->dbp, record, f->reclen );
        return result;
}

int sort_open (struct file_desc *f, char *record, char *fname)  
{
	DBTYPE type=DB_BTREE;
	int sflags=S_IRUSR | S_IWUSR;
 	int oflags=O_CREAT | O_RDWR;
	char *filename,*s;
	int len;
	BTREEINFO b;

        b.flags = R_DUP;
        b.cachesize = 0;
        b.maxkeypage = 0;
        b.minkeypage = 0;
        b.psize = 0;
        b.compare = NULL;
        b.prefix = NULL;
        b.lorder = 0;


	/* beware: fname points to a field storage (non null-terminated) 
	   we must copy fname to a C string and terminate it with a \0
	   and also trim spaces at the end. */
	len = f->fname_desc->len;
	filename = malloc(len+1);
	memmove(filename,fname,len);
	s=filename+len;
	*s-- = 0;
	while (*s == ' ') *s-- = 0;

	type= DB_BTREE;
	f->dbp = db_open(NULL,oflags,sflags,type,&b);
	if (!f->dbp) {
		if (errno == EINVAL) return 37;
		return 30;
	}
	return 0;	
}

int sort_release  (struct file_desc *f, char *record, char *sd, ...) {
	int result;
	int flags=0;
	int key_size=0;
	int counter;
	int sort_direction;
	int key_ptr=0;
	DBT key, data;
	va_list	args;
	char *fld;
 	char key1[65536];
	char fld_new[65535];	
	
	data.data = record;
	data.size = f->reclen;
	
	va_start(args, *sd);
	sort_direction = *sd;
	while(*sd++) {
		fld = va_arg (args, char * );
		key_size = key_size + *sd;
		move_bytes (&fld_new,fld,*sd);
		if (sort_direction == 2)
		{
		/* If the key is descending then for each character we
		   subtract its value from 255 giving the final value.
		   This way we can combine the Ascending and Descending 
		   Keys into one long key. The result is the final key and
		   saves us from having to run sort. 
		*/
			for (counter=0; counter <=(*sd); counter++)
			{
				fld_new[counter] = 255 - fld_new[counter];
			}	
		}
		move_bytes (&key1[key_ptr],fld_new,*sd);
		sort_direction = sd[1];
		key_ptr += *sd++;
	}
	key.data = &key1;
	key.size = key_size;
	result = f->dbp->put(f->dbp, &key,&data,flags);
	if (!result) {
		return 0;
	} else {
		return 99;
	}
}
int sort_return (struct file_desc *f, char *record) {
	int result;
	DBT key,data;
	int flags=R_NEXT;
	
	result = f->dbp->seq(f->dbp,&key,&data,flags);
	if (result)
		return 10;
	if (data.size < f->reclen)
		return 10;
	move_bytes(record, data.data, f->reclen);
	return 0;

}
/* EOF fileio.c */

