/******************************** -*- C -*- ****************************
 *
 *	sbrk-like behavior for separate mmap'ed regions
 *
 *	$Revision: 1.8.3$
 *	$Date: 2000/09/05 16:16:17$
 *	$Author: pb$
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
 * Written by Paolo Bonzini (redisorganization of GNU mmalloc).
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 ***********************************************************************/


#include "gst.h"
#include "heap.h"
#include "memzero.h"

#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>       /* dpx/2 needs this after sys/types.h */
#include <string.h>

#ifdef _WIN32
#include <windows.h>
#else
#include <sys/mman.h>
#endif

#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif

#ifdef HAVE_STDDEF_H
#  include <stddef.h>
#endif

#ifdef HAVE_LIMITS_H
#  include <limits.h>
#else
#  ifndef CHAR_BIT
#    define CHAR_BIT 8
#  endif
#endif

#ifndef MIN
#  define MIN(A, B) ((A) < (B) ? (A) : (B))
#endif

/* Internal structure that defines the format of the heap descriptor.
 * This gets written to the base address of the region that we are
 * managing. */

struct heap
  {
    size_t areasize;

    /* The base address of the memory region for this malloc heap.  This
     * is the location where the bookkeeping data for mmap and for malloc
     * begins. */

    char *base;

    /* The current location in the memory region for this malloc heap which
     * represents the end of memory in use. */

    char *breakval;

    /* The end of the current memory region for this malloc heap.  This is
     * the first location past the end of mapped memory. */

    char *top;
  };

/* Prototypes for local functions */

static voidPtr heap_sbrk_internal ();

/* Cache pagesize-1 for the current host machine.  Note that if the host
 * does not readily provide a getpagesize() function, we need to emulate it
 * elsewhere, not clutter up this file with lots of kluges to try to figure
 * it out.
 */

static size_t pageround, pagesize;
#ifndef HAVE_GETPAGESIZE
extern int getpagesize ();
#endif

#define PAGE_ALIGN(addr) ((caddr_t) (((long)(addr) + pageround) & ~pageround))

/* We allocate extra pages for the heap descriptor and answer an address
 * that is HEAP_DELTA bytes past the actual beginning of the allocation.
 */
#define HEAP_DELTA	 ((long) PAGE_ALIGN(sizeof (struct heap)))

/* Wrappers for Windows and POSIX */
#ifdef _WIN32

#define _heap_reserve(hdp, size) \
  ((hdp) -> base = VirtualAlloc(NULL, (size), MEM_RESERVE, PAGE_NOACCESS))

#define _heap_release(hdp) \
  VirtualFree((hdp) -> base, (hdp) -> areasize, MEM_RELEASE)

#define _heap_map_in(hdp, size) \
  VirtualAlloc ((hdp) -> top, (size), MEM_COMMIT, PAGE_EXECUTE_READWRITE)

#define _heap_map_out(hdp, size) \
  VirtualFree(PAGE_ALIGN ((hdp) -> breakval), (size), MEM_DECOMMIT)

#else

static char *baseaddr = MMAP_BASE;

#define _heap_reserve(hdp, size) \
  ( ((hdp) -> base = baseaddr, baseaddr += (size), 1) )

#define _heap_release(hdp) \
  ( ((baseaddr == (hdp) -> base + (hdp) -> areasize) \
      ? baseaddr = (hdp) -> base : 0))

#if defined(MAP_ANONYMOUS) && !defined(MAP_ANON)
#define MAP_ANON MAP_ANONYMOUS
#endif

#ifdef MAP_ANON
#define anon_mmap(where, size, flags)				\
  mmap ((where), (size), PROT_READ | PROT_WRITE | PROT_EXEC,	\
	(flags) | MAP_PRIVATE | MAP_ANON, -1, 0)

#else /* !MAP_ANON */

/* Open file descriptor for the file to which malloc heaps is mapped
 * (/dev/zero). */
static int fd;

#define anon_mmap(where, size, flags)				\
  mmap ((where), (size), PROT_READ | PROT_WRITE | PROT_EXEC,	\
	(flags) | MAP_PRIVATE, fd, 0)

#endif /* !MAP_ANON */

/* Map in enough more memory to satisfy the request. */
#define _heap_map_in(hdp, size) \
  anon_mmap ((hdp) -> top, (size), MAP_FIXED)

#define _heap_map_out(hdp, size) \
  munmap (PAGE_ALIGN ((hdp) -> breakval), (size))

#endif


/* Initialize access to a heap managed region.

   The heap managed region is mapped to "/dev/zero", and the data
   will not exist in any filesystem object.

   On success, returns a "heap descriptor" which is used in subsequent
   calls to other heap package functions.  It is explicitly "void *"
   ("char *" for systems that don't fully support void) so that users
   of the package don't have to worry about the actual implementation
   details.

   On failure returns NULL. */

heap
heap_create (size)
     int size;
{
  struct heap mtemp;
  struct heap *hdp;
  char *mbase;

  if (!pageround) {
    pagesize = getpagesize ();
    pageround = pagesize - 1;

#if !defined(_WIN32) && !defined(MAP_ANON)
    /* Open the file descriptor we need for anonymous mmaps */
    fd = open ("/dev/zero", O_RDWR);
#endif
  }

  /* We start off with the heap descriptor allocated on the stack, until
   * we build it up enough to call heap_sbrk_internal() to allocate the
   * first page of the region and copy it there.  Ensure that it is zero'd and
   * then initialize the fields that we know values for. */

  hdp = &mtemp;
  memzero ((char *) hdp, sizeof (mtemp));
  hdp -> areasize = size;

  if (!_heap_reserve(hdp, size))
    {
      errno = ENOMEM;
      return (NULL);
    }

  hdp -> breakval = hdp -> top = hdp -> base;

  /* Now try to map in the first page, copy the heap descriptor structure
   * there, and arrange to return a pointer to this new copy.  If the mapping
   * fails, then close the file descriptor if it was opened by us, and arrange
   * to return a NULL. */

  if ((mbase = heap_sbrk_internal (hdp, sizeof (mtemp))) != NULL)
    {
      memcpy (mbase, hdp, sizeof (mtemp));
      hdp = (struct heap *) (mbase + HEAP_DELTA);
    }
  else
    {
      _heap_release (hdp);
      hdp = NULL;
    }

  return ((heap) hdp);
}

/* Terminate access to a heap managed region by unmapping all memory pages
 * associated with the region, and closing the file descriptor if it is one
 * that we opened.
 *
 * Returns NULL on success.
 *
 * Returns the heap descriptor on failure, which can subsequently be used
 * for further action, such as obtaining more information about the nature of
 * the failure by examining the preserved errno value.
 *
 * Note that the heap descriptor that we are using is currently located in
 * region we are about to unmap, so we first make a local copy of it on the
 * stack and use the copy. */

heap
heap_destroy (hd)
     heap hd;
{
  struct heap mtemp;

  if (hd != NULL)
    {

      mtemp = *(struct heap *) (hd - HEAP_DELTA);
      
      /* Now unmap all the pages associated with this region by asking for a
       * negative increment equal to the current size of the region. */

      if ((heap_sbrk_internal (&mtemp, mtemp.base - mtemp.top)) == NULL)
	{
	  /* Update the original heap descriptor with any changes */
	  *(struct heap *) (hd - HEAP_DELTA) = mtemp;
	}
      else
	{
	  _heap_release (&mtemp);
	  hd = NULL;
	}
    }

  return (hd);
}

/* Get core for the memory region specified by MDP, using SIZE as the
 * amount to either add to or subtract from the existing region.  Works
 * like sbrk(), but using mmap(). */

voidPtr
heap_sbrk (hd, size)
     heap hd;
     size_t size;
{
  struct heap *hdp;

  hdp = (struct heap *) (hd - HEAP_DELTA);
  return heap_sbrk_internal (hdp, size);
}

voidPtr
heap_sbrk_internal (hdp, size)
     struct heap *hdp;
     int size;
{
  char *result = NULL;
  size_t mapbytes;	/* Number of bytes to map */
  caddr_t moveto;	/* Address where we wish to move "break value" to */
  caddr_t mapto;	/* Address we actually mapped to */

  if (size == 0)
    /* Just return the current "break" value. */
    result = hdp -> breakval;

  else if (size < 0)
    {
      /* We are deallocating memory.  If the amount requested would cause
       * us to try to deallocate back past the base of the mmap'd region
       * then do nothing, and return NULL.  Otherwise, deallocate the
       * memory and return the old break value. */
      if (hdp -> breakval + size >= hdp -> base)
	{
	  result = (voidPtr) hdp -> breakval;
	  hdp -> breakval += size;
	  moveto = PAGE_ALIGN (hdp -> breakval);
	  _heap_map_out(hdp, (size_t) (hdp -> top - moveto));
	  hdp -> top = moveto;
	}
    }
  else if (hdp -> breakval + size > hdp -> top)
    {
      moveto = PAGE_ALIGN (hdp -> breakval + size);
      mapbytes = moveto - hdp -> top;
      mapto = _heap_map_in(hdp, mapbytes);
      if (mapto == hdp -> top)
	{
	  hdp -> top = moveto;
	  result = (voidPtr) hdp -> breakval;
	  hdp -> breakval += size;
	}
      else
	errno = ENOMEM;
    }
  else
    {
      result = (voidPtr) hdp -> breakval;
      hdp -> breakval += size;
    }

  return (result);
}
