/*
 * Copyright (c) 1992 by Sun Microsystems, Inc.
 */

#include <sys/param.h>
#include <sys/errno.h>
#include <sys/proc.h>
#include <sys/vfs.h>
#include <sys/vnode.h>
#include <sys/uio.h>
#include <sys/kmem.h>
#include <sys/cred.h>
#include <sys/statvfs.h>
#include <sys/mount.h>
#include <sys/tiuser.h>
#include <sys/cmn_err.h>
#include <sys/debug.h>
#include <sys/mkdev.h>
#include <sys/systm.h>
#include <sys/sysmacros.h>
#include <sys/pathname.h>
#include <sys/fs_subr.h>

#include <fist.h>
#include <rot1fs.h>

int fist_rot1_init(vfssw_t *, int);

/*
 * This is the loadable module fist_rot1per.
 */
#include <sys/modctl.h>

extern vfsops_t fist_rot1_vfsops;
int fist_rot1fs_major;
int fist_rot1fs_minor;
kmutex_t fist_rot1fs_minor_lock;

static vfssw_t vfw =
{
  "fist_rot1fs",
  fist_rot1_init,
  &fist_rot1_vfsops,
  0
};

/*
 * Module linkage information for the kernel.
 */
extern struct mod_ops mod_fsops;

static struct modlfs modlfs =
{
  &mod_fsops,
  "filesystem for fist_rot1fs",
  &vfw
};

static struct modlinkage modlinkage =
{
  MODREV_1,
  {
    (void *) &modlfs,
    NULL,
  }
};

/*
 * This is the module initialization routine.
 */

/*
 * A counter for how many filesystems are mounted using this counter.
 * If greater than 0, it means the module is unloadable.
 */
static int module_keepcnt = 0;

/*
 * There are not enough stubs for rpcmod so we must force load it
 */
char _depends_on[] = "strmod/rpcmod";

int
_init(void)
{
  print_location();
  return (mod_install(&modlinkage));
}

int
_fini(void)
{
  if (module_keepcnt != 0) {
    fist_dprint(4, "fist_rot1_vfsops: _fini(), module_keepcnt=%d.\n",
		module_keepcnt);
    return (EBUSY);
  }
  fist_dprint(4, "fist_rot1_vfsops: _fini(): removing module.\n");

  return (mod_remove(&modlinkage));
}

int
_info(struct modinfo *modinfop)
{
  int error;

  fist_dprint(4, "fist_rot1_vfsops: _info(): returning info on module.\n");
  error = mod_info(&modlinkage, modinfop);
  fist_dprint(4, "fist_rot1_vfsops: module rot1fs successfully loaded at address 0x%x, size 0x%x\n",
	      modinfop->mi_base, modinfop->mi_size);
  return error;
}

static kmutex_t fist_rot1node_count_lock;
extern struct vnodeops fist_rot1_vnodeops;
static int fist_rot1fs_fstype;

/*
 * fist_rot1fs vfs operations.
 */
static int fist_rot1_mount(vfs_t *, vnode_t *, struct mounta *, cred_t *);
static int fist_rot1_unmount(vfs_t *, cred_t *);
static int fist_rot1_root(vfs_t *, vnode_t **);
static int fist_rot1_statvfs(vfs_t *, struct statvfs64 *);
static int fist_rot1_sync(vfs_t *, short, cred_t *);
static int fist_rot1_vget(vfs_t *, vnode_t **, fid_t *);
static int fist_rot1_mountroot(vfs_t *, enum whymountroot);
static int fist_rot1_swapvp(vfs_t *, vnode_t **, char *);

struct vfsops fist_rot1_vfsops =
{
  fist_rot1_mount,		/* mount */
  fist_rot1_unmount,		/* unmount */
  fist_rot1_root,		/* root */
  fist_rot1_statvfs,		/* statvfs */
  fist_rot1_sync,		/* sync */
  fist_rot1_vget,		/* vget */
  fist_rot1_mountroot,		/* mountroot */
  fist_rot1_swapvp		/* swapvp */
};

#define	MAJOR_MIN 128
#define	vfsNVFS   (&vfssw[nfstype])

int
fist_rot1_init(
		 vfssw_t * vswp,
		 int fstype
)
{

  fist_dprint(5, "fist_rot1_init: vspw %x, fstype %d\n",
	      (int) vswp, fstype);
  fist_set_debug_value(4);

  if (vswp) {
    if (vswp->vsw_name)
      fist_dprint(5, "fist_rot1_init: vswp->vsw_name \"%s\"\n",
		  vswp->vsw_name);
    if (vswp->vsw_init)
      fist_dprint(5, "fist_rot1_init: vswp->vsw_init %x\n",
		  (int) vswp->vsw_init);
    fist_dprint(5, "fist_rot1_init: fist_rot1_init %x\n",
		(int) fist_rot1_init);
    if (vswp->vsw_vfsops) {
      fist_dprint(5, "fist_rot1_init: vswp->vsw_vfsops %x\n",
		  (int) vswp->vsw_vfsops);
      fist_dprint(5, "fist_rot1_init: fist_rot1_vfsops %x\n",
		  (int) &fist_rot1_vfsops);
      if (vswp->vsw_vfsops->vfs_mount)
	fist_dprint(5, "fist_rot1_init: vswp->vsw_vfsops->vfs_mount %x\n",
		    (int) vswp->vsw_vfsops->vfs_mount);
      fist_dprint(5, "fist_rot1_init: fist_rot1_mount %x\n",
		  (int) fist_rot1_mount);
    }
  }
  fist_rot1fs_fstype = fstype;
  ASSERT(fist_rot1fs_fstype != 0);
  /*
   * Associate VFS ops vector with this fstype
   */
  vswp->vsw_vfsops = &fist_rot1_vfsops;

  mutex_init(&fist_rot1fs_minor_lock, "fist_rot1fs minor lock",
	     MUTEX_DEFAULT, NULL);
  mutex_init(&fist_rot1node_count_lock, "fist_rot1node count lock",
	     MUTEX_DEFAULT, NULL);

  /*
   * Assign unique major number for all fist_rot1fs mounts
   */
  if ((fist_rot1fs_major = getudev()) == -1) {
    cmn_err(CE_WARN,
	    "fist_rot1fs: fist_rot1_init: can't get unique device number");
    fist_rot1fs_major = 0;
  }
  fist_rot1fs_minor = 0;

  print_location();
  return (0);
}

static int
fist_rot1_mount(
		  vfs_t * vfsp,	/* pre-made vfs structure to mount */
		  vnode_t * vp,	/* existing vnode to mount on */
		  struct mounta *uap,	/* user-area mount(2) arguments */
		  cred_t * cr	/* user credentials */
)
{
  int error = 0;
  char namebuf[TYPICALMAXPATHLEN + 4];	/* +4 because of bug 1170077 */
  int len = 0;
  /* struct fist_rot1_args args; */
  /* char datalen = uap->datalen; */
  struct fist_rot1info *fwip;
  /* fist_rot1node_t *fwnp; */
  dev_t fist_rot1fs_dev;
  struct vnode *rootvp;
  vnode_t *interposed_vp;	/* interposed vnode */

  if (vfsp)
    fist_print_vfs("fist_rot1_mount", vfsp);
  if (vp) {
    fist_dprint(4,
		"%s: fist_rot1_vnodeops %x\n",
		"fist_rot1_mount",
		(int) &fist_rot1_vnodeops);
    fist_print_vnode("fist_rot1_mount", vp);
  }
  if (uap)
    fist_print_uap("fist_rot1_mount", uap);

  /*
   * Now we can increment the count of module instances.
   * meaning that from now, the mounting cannot fail.
   */
  ++module_keepcnt;

  /* the if 0 is here by sauce because I like working as sauce  *rather than
   * rsauce */
#if 0
  /*
   * Make sure we're root
   */
  if (!suser(cr)) {
    error = EPERM;
    goto out;
  }
  /* Make sure we mount on a directory */
  if (vp->v_type != VDIR) {
    error = ENOTDIR;
    goto out;
  }
#endif

  /*
   * check if vnode is already a root of a filesystem (i.e., there
   * is already a mount on this vnode).
   */
  mutex_enter(&vp->v_lock);
  if ((uap->flags & MS_REMOUNT) == 0 &&
      (uap->flags & MS_OVERLAY) == 0 &&
      (vp->v_count != 1 || (vp->v_flag & VROOT))) {
    mutex_exit(&vp->v_lock);
    error = EBUSY;
    goto out;
  }
  mutex_exit(&vp->v_lock);

  /*
   * Get arguments: XXX: NOT NEEDED YET.
   */

  /*
   * Get vnode for interposed directory.
   */

  /* make sure special dir is a valid absolute pathname string */
  if (!uap || !uap->spec) {
    error = EINVAL;
    goto out;
  }
  /* check if special dir starts with a '/' */
  error = copyinstr(uap->spec, namebuf, TYPICALMAXPATHLEN, &len);
  if (error)
    goto out;
  if (namebuf[0] != '/') {
    error = EINVAL;
    goto out;
  }
  error = lookupname(uap->spec, UIO_USERSPACE, FOLLOW,
		     NULLVPP, &interposed_vp);
  if (error)
    goto out;
  /* Make sure the thing we just looked up is a directory */
  if (interposed_vp->v_type != VDIR) {
    VN_RELE(interposed_vp);
    error = ENOTDIR;
    goto out;
  }
/**************************************************************************
 * FIST_ROT1INFO:
 * The private information stored by the vfs for fist_rot1fs.
 */
  /* this implicitly allocates one vnode to be used for root vnode */
  /* XXX: enter this vnode in dnlc? */
  fwip = (struct fist_rot1info *)
    kmem_alloc(sizeof(struct fist_rot1info), KM_SLEEP);
  /* store the vfs of the stacked filesystem (pushed onto "stack") */
  fwip->fwi_mountvfs = vp->v_vfsp;
  /* initialize number of interposed vnodes */
  fwip->fwi_num_vnodes = 0;
  bzero((caddr_t) fwip->fwi_buckets, sizeof(fist_bucket_t *) * FIST_HT_SIZE);
  mutex_init(&fwip->fwi_ht_lock, "rot1fs HT lock", MUTEX_DEFAULT, NULL);
  /* fwip->fwi_rootvnode: is setup in the "root vnode" section below */

/**************************************************************************
 * VFS FOR THE FIST_ROT1 FILESYSTEM:
 */
  vfsp->vfs_bsize = vp->v_vfsp->vfs_bsize;	/* was 1024, now inherited */
  vfsp->vfs_fstype = fist_rot1fs_fstype;
  /* Assign a unique device id to the mount */
  mutex_enter(&fist_rot1fs_minor_lock);
  do {
    fist_rot1fs_minor = (fist_rot1fs_minor + 1) & MAXMIN;
    fist_rot1fs_dev = makedevice(fist_rot1fs_major, fist_rot1fs_minor);
  } while (vfs_devsearch(fist_rot1fs_dev));
  mutex_exit(&fist_rot1fs_minor_lock);
  /* set the rest of the fields */
  vfsp->vfs_dev = fist_rot1fs_dev;
  vfsp->vfs_fsid.val[0] = fist_rot1fs_dev;
  vfsp->vfs_fsid.val[1] = fist_rot1fs_fstype;
  vfsp->vfs_bcount = 0;
  /* store private fist_rot1 info in the pre-made vfs */
  vfsp->vfs_data = (caddr_t) fwip;
  /* fill in the vnode we are mounted on, in the vfs */
  vfsp->vfs_vnodecovered = vp;

/**************************************************************************
 * FIST_ROT1NODE:
 * The private information stored by interposing vnodes.
 * The interposing vnode here is the new root vnode of fist_rot1fs.  It
 * interposes upon the uap->spec vnode we are mounting on (the directory,
 * or partition interposed upon).
 */

/**************************************************************************
 * ROOT VNODE OF FIST_ROT1FS:
 */
  rootvp = fist_rot1_interpose(interposed_vp, vfsp);
  /* this is a root vnode of this filesystem */
  rootvp->v_flag |= VROOT;
  fwip->fwi_rootvnodep = rootvp;

/**************************************************************************
 * VNODE MOUNTED UPON:
 */
  /* this vnode to mount on is a mount point for fist_rot1 */
  vp->v_vfsmountedhere = vfsp;

  /* print values after we change them */
  if (vfsp)
    fist_print_vfs("fist_rot1_mount2", vfsp);
  if (vp)
    fist_print_vnode("fist_rot1_mount2", vp);
  if (rootvp)
    fist_print_vnode("fist_rot1_mount2rvp", rootvp);
  fist_print_vnode("fist_rot1_mount2rvn", fwip->fwi_rootvnodep);

  print_location();
  return (error);

out:
  /*
   * Cleanup our mess
   */
  module_keepcnt--;
  print_location();
  return (error);
}

/*
 * Undo fist_rot1fs mount
 */
static int
fist_rot1_unmount(
		    vfs_t * vfsp,
		    cred_t * cr
)
{
  struct fist_rot1info *fwip;
  vnode_t *vnp;

  fist_dprint(4, "fist_rot1_unmount vfsp %x\n", vfsp);

#if 0
  /* check if permitted */
  if (!suser(cr)) {
    return (EPERM);
  }
#endif

  /* free memory used by private data of vfs */
  if (!vfsp->vfs_data) {
    goto out;
  }
  fwip = vfstofwi(vfsp);
  if (fwip->fwi_num_vnodes > 1) {	/* always at least 1 b/c of rootvp */
    return (EBUSY);
  }
  fist_print_vnode("fist_rot1_unmount rvp", fwip->fwi_rootvnodep);
  if (fwip->fwi_rootvnodep->v_count != 1) {
    fist_dprint(4,
		"fist_rot1_unmount root vnode is busy %d\n",
		fwip->fwi_rootvnodep->v_count);
    return (EBUSY);
  }

  vnp = vfsp->vfs_vnodecovered;
  fist_dprint(4, "ROT1_UMOUNT: vnodecovered 0x%x has count %d\n", vnp, vnp->v_count);
  /* XXX: need to VN_RELE(vnp); */
  vnp->v_vfsmountedhere = NULL;

  VN_RELE(fwip->fwi_rootvnodep);

  /* check if HT is really empty */
  ASSERT(fist_ht_empty_assert(vfsp) == 1);
  mutex_destroy(&fwip->fwi_ht_lock);

  kmem_free(fwip, sizeof(struct fist_rot1info));

out:
  module_keepcnt--;
  return (0);
}

/*
 * find root of fist_rot1fs
 */
static int
fist_rot1_root(
		 vfs_t * vfsp,
		 vnode_t ** vpp
)
{
  struct fist_rot1info *fwip = NULL;

  fist_dprint(4, "fist_rot1_root vfsp %x vnode %x\n",
	      vfsp, *vpp);

  fwip = vfstofwi(vfsp);
  ASSERT(fwip != NULL);
  *vpp = fwip->fwi_rootvnodep;
  fist_dprint(4, "fist_rot1_root rootvnode %x\n", *vpp);

  /*
   * Should I apply the same operation to the hidden filesystem?
   * probably not, since this root vnode can lead me to the root vnode it
   * was interposed upon in vnodep->v_data->...
   * ANSWER: no!
   */
  /* error = VFS_ROOT(vfsp, vpp); */
  fist_dprint(4, "ROT1_ROOT1 root count %d, hidden count %d\n",
	      (*vpp)->v_count, vntofwn(*vpp)->fwn_vnodep->v_count);
#if 1
  /* XXX: is this right?  Should I HOLD a vnode everytime root is called? */
  /* probably ok, since we are holding the root vnode for fist_rot1... */
  VN_HOLD(*vpp);
#else
  if (*vpp) {
    fist_print_vnode("fist_rot1_root", *vpp);
  }
  /* mutex_enter(&(*vpp)->v_lock); */
  /* (*vpp)->v_count++; */
  /* mutex_exit(&(*vpp)->v_lock); */
#endif

  fist_dprint(4, "ROT1_ROOT2 root count %d, hidden count %d\n",
	      (*vpp)->v_count, vntofwn(*vpp)->fwn_vnodep->v_count);

  print_location();
  return (0);
}

/*
 * Get file system statistics.
 */
static int
fist_rot1_statvfs(
		    register vfs_t * vfsp,
		    struct statvfs64 * sbp
)
{
  int error = EPERM;
  struct fist_rot1info *fwip = NULL;
  vfs_t *hidden_vfsp;

  fist_dprint(4, "fist_rot1_statvfs %x\n", vfsp);

  if (vfsp) {
    /* find hidden vfs */
    fwip = vfstofwi(vfsp);
    ASSERT(fwip != NULL);
    hidden_vfsp = fwip->fwi_mountvfs;

    /* pass operation to hidden filesystem, and return status */
    error = VFS_STATVFS(hidden_vfsp, sbp);
  } else {			/* error: vfsp is invalid */
    error = EINVAL;
  }

  print_location();
  return (error);
}

/*
 * Flush any pending I/O to file system vfsp.
 * The fist_rot1_update() routine will only flush *all* fist_rot1 files.
 * If vfsp is non-NULL, only sync this fist_rot1 (in preparation
 * for a umount).
 */
static int
fist_rot1_sync(struct vfs *vfsp, short flag, struct cred *cr)
{
  /* commented the print of the sync since i got annoyed a  message when we
   * only return 0 */
#if 0

  fist_dprint(4, "fist_rot1_sync %x\n", vfsp);
#endif
  /* do not sync() lower layer f/s.  It'll be sync'ed on its own */
  return 0;

#if 0
  /* if vfsp is NULL, then this is a generic "sync()", treat it as ok */
  if (vfsp) {
    /* find hidden vfs */
    fwip = vfstofwi(vfsp);
    ASSERT(fwip != NULL);
    hidden_vfsp = fwip->fwi_mountvfs;

    /* pass operation to hidden filesystem, and return status */
    error = VFS_SYNC(hidden_vfsp, flag, cr);
  }
  print_location();
  return (error);
#endif
}

static int
fist_rot1_vget(
		 struct vfs *vfsp,
		 struct vnode **vpp,
		 struct fid *fidp
)
{
  int error = EPERM;
  struct fist_rot1info *fwip = NULL;
  vfs_t *hidden_vfsp;

  fist_dprint(4, "fist_rot1_vget %x\n", vfsp);

  if (vfsp) {
    /* find hidden vfs */
    fwip = vfstofwi(vfsp);
    ASSERT(fwip != NULL);
    hidden_vfsp = fwip->fwi_mountvfs;

    /* pass operation to hidden filesystem, and return status */
    error = VFS_VGET(hidden_vfsp, vpp, fidp);
  } else {			/* error: vfsp is invalid */
    error = EINVAL;
  }

  print_location();
  return (error);
}

/*
 * Mount root file system.
 * "why" is ROOT_INIT on initial call, ROOT_REMOUNT if called to
 * remount the root file system, and ROOT_UNMOUNT if called to
 * unmount the root (e.g., as part of a system shutdown).
 *
 * XXX - this may be partially machine-dependent; it, along with the
 * VFS_SWAPVP operation, goes along with auto-configuration.  A mechanism
 * should be provided by which machine-INdependent code in the kernel can
 * say "get me the right root file system" and "get me the right initial
 * swap area", and have that done in what may well be a machine-dependent
 * fashion.  Unfortunately, it is also file-system-type dependent (NFS gets
 * it via bootparams calls, FIST_ROT1 gets it from various and sundry
 * machine-dependent mechanisms, as SPECFS does for swap).
 */
static int
fist_rot1_mountroot(
		      struct vfs *vfsp,
		      enum whymountroot why
)
{
  /*
   * XXX: stackable filesystems cannot be used to as root filesystems, for
   * now.
   * -Erez Zadok <ezk@cs.columbia.edu>
   */
  fist_dprint(4, "fist_rot1_mountroot %x\n", vfsp);

  print_location();
  return (EINVAL);
}

static int
fist_rot1_swapvp(
		   struct vfs *vfsp,
		   struct vnode **vpp,
		   char *nm
)
{
  int error = EPERM;
  struct fist_rot1info *fwip = NULL;
  vfs_t *hidden_vfsp;

  fist_dprint(4, "fist_rot1_swapvp %x\n", vfsp);

  if (vfsp) {
    /* find hidden vfs */
    fwip = vfstofwi(vfsp);
    ASSERT(fwip != NULL);
    hidden_vfsp = fwip->fwi_mountvfs;

    /* pass operation to hidden filesystem, and return status */
    error = VFS_SWAPVP(hidden_vfsp, vpp, nm);
  } else {			/* error: vfsp is invalid */
    error = EINVAL;
  }

  print_location();
  return (error);
}
